/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>

/* These random numbers are NOT the same as the ones used in the
 * MacDonald libxdelta implementation. We cannot reuse those due to
 * GPL restrictions. Thank heavens those bits are not externally
 * exposed by the XDelta algorithm!
 *
 * The reason to use random numbers is that it flattens the entropy of
 * the input bytes across a larger numeric space, and therby reduced
 * collisions. Without this, the adler32 of  "bcx" and "adx" are the
 * same.
 */

static uint16_t magic[] = {
  0x1be1U, 0x85e3U, 0x93e3U, 0x77eaU, 0x8effU, 0x9af0U, 0x25faU, 0xc9faU,
  0xd6faU, 0xb14fU, 0x7340U, 0x3c47U, 0x0f48U, 0x1d48U, 0x0254U, 0x2a55U, 
  0x6355U, 0xee5cU, 0x7167U, 0xf57dU, 0x687eU, 0x927cU, 0xdb8fU, 0x148eU, 
  0xe58eU, 0x8f88U, 0xf488U, 0xfe8aU, 0x9f8bU, 0x1b93U, 0x2796U, 0xfa97U, 
  0x4aaeU, 0xeaa3U, 0x8ea7U, 0x64beU, 0xa1beU, 0x2ab1U, 0x65b3U, 0x4eb7U, 
  0x14b8U, 0xd9bbU, 0x8ccfU, 0x98cdU, 0xb9c3U, 0x9ccbU, 0xdcccU, 0x5dddU, 
  0x33d3U, 0xf7d4U, 0xf10dU, 0xb000U, 0x4b02U, 0x8505U, 0x5606U, 0x9e0aU, 
  0x971fU, 0xdc1eU, 0x5b15U, 0x9019U, 0x932fU, 0xbb23U, 0xa724U, 0xfb2bU, 
  0x012cU, 0x963eU, 0x2031U, 0x6d36U, 0x5c4dU, 0xb24eU, 0xf142U, 0x0c49U, 
  0x7e50U, 0xa850U, 0xcc50U, 0x1e51U, 0xe156U, 0xbb57U, 0xaf59U, 0xe55aU, 
  0x366dU, 0x176eU, 0x6c67U, 0x0b6bU, 0xd471U, 0x6b76U, 0x6b77U, 0x0f7bU, 
  0x4f7bU, 0xde7bU, 0xc981U, 0x8d83U, 0xe086U, 0x9787U, 0x5c88U, 0x9f8aU, 
  0x9e9eU, 0xf490U, 0x1092U, 0xbc96U, 0x78aeU, 0xf2b3U, 0xe4b4U, 0x5fbbU, 
  0x58cdU, 0x8ecdU, 0xf8cdU, 0xd3c6U, 0xc9ccU, 0x1cd1U, 0xaddaU, 0x63eeU, 
  0xe9e3U, 0xf9e6U, 0xb3ebU, 0x4effU, 0xa0f0U, 0xa5f6U, 0x77f8U, 0x46f8U, 
  0xc6fcU, 0xd50fU, 0x8a04U, 0x2509U, 0x9309U, 0xea09U, 0x0f0bU, 0x9a1eU, 
  0xe711U, 0x9718U, 0xec19U, 0x0e1bU, 0xa81cU, 0xe82fU, 0x372fU, 0x5b28U, 
  0xc32cU, 0x8c3dU, 0x2a31U, 0x1532U, 0xcf38U, 0xfc3cU, 0x124dU, 0x2d45U, 
  0x3547U, 0x6448U, 0xca4cU, 0x6450U, 0xc253U, 0x3d55U, 0x6255U, 0x1d5bU, 
  0x3a5cU, 0x5867U, 0xd667U, 0x4278U, 0xb079U, 0x6888U, 0xad8bU, 0xe19dU, 
  0x4190U, 0x3594U, 0xec96U, 0xfeaeU, 0x23a1U, 0xfba1U, 0x6da2U, 0x48beU, 
  0xd0beU, 0x44b3U, 0x1eb5U, 0x8fbbU, 0xe0bbU, 0xefceU, 0xe1c2U, 0xd8c9U, 
  0x37caU, 0x2dd6U, 0x89e6U, 0x58fdU, 0xebf2U, 0xc1f5U, 0x84f6U, 0x5cfbU, 
  0xfafcU, 0x960fU, 0x1e0dU, 0xcf0eU, 0x0103U, 0x1205U, 0x411aU, 0x961bU, 
  0xe61bU, 0x0b2dU, 0x3221U, 0x0522U, 0xc222U, 0x7b23U, 0xee28U, 0x072aU, 
  0xa432U, 0xc13bU, 0x023cU, 0x5a4eU, 0x8646U, 0x1c5fU, 0x1d53U, 0x3f56U, 
  0x6256U, 0x2664U, 0xed65U, 0x9767U, 0xa96aU, 0x3770U, 0x8f71U, 0xa673U, 
  0xf783U, 0xb885U, 0x5f9eU, 0xef9eU, 0x179fU, 0xd690U, 0x1091U, 0xc4adU, 
  0x80aeU, 0x72a3U, 0x3ea4U, 0x8ea6U, 0xe3a9U, 0x55b2U, 0xeab7U, 0x99b9U, 
  0x29c2U, 0xe8c6U, 0xa3cbU, 0x32d5U, 0xe7d8U, 0x3cedU, 0x50edU, 0xf5e2U, 
  0xb1e5U, 0xbaebU, 0x2cf3U, 0xb203U, 0x0604U, 0x6508U, 0x9408U, 0x830bU, 
  0x850cU, 0x391dU, 0x8510U, 0x792dU, 0xd424U, 0x3e29U, 0x2931U, 0xbc32U,
};

#define MAGIC(x) magic[x]

/* This is a sequential checksum based loosely on adler32. 
 *
 * It is NOT the same as the one used in Josh MacDonald's XDFS,
 * because it does not incorporate the same randomly generated 16 bit
 * numbers (because of GPL copyright restrictions). Thank heavens
 * those bits are not externally exposed by the XDelta family of
 * programs!
 *
 * The reason to use random numbers is that it flattens the entropy of
 * the input bytes across a larger numeric space, and therby reduced
 * collisions. Without this, the adler32 of  "bcx" and "adx" are the
 * same.
 */
phash_t
psum_incremental(phash_t sum, const unsigned char *buf, size_t len)
{
  uint32_t low = sum & 0xffffu;
  uint32_t hi = sum >> 16;

  while (len --) {
    low += MAGIC(*buf++);
    low &= 0xffffu;
    hi += low;
  }

  return (hi << 16) | low;
}

phash_t
psum_progressive(phash_t sum, unsigned lg2chunkSz,
		 unsigned char cold, unsigned char cnew)
{
  uint32_t low = sum & 0xffffu;
  uint32_t hi = sum >> 16;

  low -= MAGIC(cold);
  low += MAGIC(cnew);
  low &= 0xffffu;

  hi -= ( ((uint32_t) MAGIC(cold)) << lg2chunkSz);
  hi += low;

  return (hi << 16) | low;
}

