/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>

void
null_serialize(SDR_stream *strm, const void *ob)
{
  /* Nothing to do */
}

void
null_mark(Repository *r, const void *container, const void *ob, rbtree *memObs)
{
  assert(container == ob);
  /* nothing to do */
}

void *
null_deserialize(const DeserializeInfo *di, SDR_stream *strm)
{
  /* Nothing to do */
  return 0;
}

void
null_show(const void *ob)
{
  report(0, "<NULL>\n");
}

OC_bool
null_check(const void *ob)
{
  return TRUE;
}

#define OBTYPE(code,schema,nm,prefix,er) extern SerialType nm##_SerType;
#include "OBTYPES.def"

SerialType *serTypes[TY_ntypes] = {
#define OBTYPE(code,schema,nm,prefix,ver) &nm##_SerType,
#include "OBTYPES.def"
} ;

#define OBTYPE(code,schema,nm,prefix,ver) \
  struct SerialType nm##_SerType = \
     { TY_##nm, ver, #nm, \
       prefix##_deserialize, \
       prefix##_serialize, \
       prefix##_show, \
       prefix##_mark, \
       prefix##_check };
#include "OBTYPES.def"

SerialType *
ser_find_type(uint32_t t)
{
  uint32_t u;

  for (u = 0; u < TY_ntypes; u++) {
    if (serTypes[u] && (serTypes[u]->typ) == t)
      return serTypes[u];
  }

  return 0;
}

/* #define CROSS_CHECK */
const char *
ser_getTrueName(const void *s)
{
  /* We must always recompute this, because allegedly frozen objects
     are modified in-place on the client and then re-presented at the
     repository interface for upload. */

  Serializable *ser = (Serializable *)s;
  const char *tn = ser->trueName;
#ifdef CROSS_CHECK
  const char *tnchk;
#endif

#ifdef CROSS_CHECK
  /* The definitive computation of a truename is to write the object
     to a raw stream and then compute an SHA1 hash on the resulting
     data.  This creates a duplicated buffer, and is therefore memory
     intensive. We have therefore implemented a special stream that
     doesn't actually store a duplicate buffer, but computes the SHA1
     "on the fly". This check is here to allow us to verify that the
     computed result is identical to the definitive one. */
  {
    SDR_stream *strm = stream_createBuffer(SDR_BINARY);
    char *buf;
    unsigned len;

    sdr_write("ENTITY", strm, s);
    stream_close(strm);
    stream_reread(strm);

    len = stream_length(strm);
    buf = buffer_asString(stream_asBuffer(strm));

    tnchk = 
      truename_FromBytes(((Serializable *)s)->ser_type->prefix, len, buf);
  }
#endif

  if (ser->canFreeze == FALSE || tn == 0) {
    SDR_stream *shastrm = stream_create_sha1();
    char buf[30];
    unsigned len;
    sdr_write("ENTITY", shastrm, s);
    stream_reread(shastrm);

    len = stream_length(shastrm);

    assert(len < 29);

    stream_read(shastrm, buf, len);
    buf[len] = 0;

    tn = truename_FromHash(buf);

    stream_close(shastrm);
  }

#ifdef CROSS_CHECK
  if (!nmequal(tn, tnchk)) {
    log_trace(TRC_BUG, "Truename mismatch: %s, %s\n", tn, tnchk);
    assert(FALSE);
  }
#endif

  if (ser->canFreeze)
    ser->trueName = tn;

  return tn;
}

/* Returns an in-memory shallow copy of an existing serializable
   object */
void *
ser_dup(const void *ob)
{
  void *newObject;
  const char *wrap;
  Buffer *b;
  SDR_stream *strm = stream_createBuffer(SDR_BINARY);

  if (GETTYPE(ob) == TY_Mutable)
    wrap = "MUTABLE";
  else if (GETTYPE(ob) == TY_Revision)
    wrap = "REVISION";
  else
    wrap = "ENTITY";

  sdr_write(wrap, strm, ob);

  stream_reread(strm);
  b = stream_asBuffer(strm);
  stream_close(strm);

  strm = stream_fromBuffer(b, SDR_AUTO);
  newObject = sdr_read(wrap, strm);
  stream_close(strm);

  return newObject;
}

void 
ser_init(void *s, SerialType *ty, uint32_t v)
{
  Serializable *ser = s;
  ser->ser_type = ty;
  ser->ser_ver = v;
  ser->canFreeze = FALSE;
  ser->trueName = 0;
}
