/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>

#define NHASH 256

static unsigned 
obhash(ObDictEntry *ode)
{
  return ode->tnHash % NHASH;
}


struct ObDict {
  ObDictEntry *bucket[NHASH];
} ;

static void
obdict_check(ObDict *dict)
{
  int i;
  for (i = 0; i < NHASH; i ++) {
    ObDictEntry *entry;
    for(entry = dict->bucket[i]; entry; entry = entry->next )
      assert(entry->trueName);
  }
}

ObDictEntry *
ode_create(const char *tn, void *vp, unsigned w)
{
  ObDictEntry *ode = GC_MALLOC(sizeof(ObDictEntry));
  ode->trueName = tn;

  ode->tnHash = truename_hash(tn);
  ode->w = w;

  ode->next = 0;
  ode->value = vp;

  return ode;
}

ObDict *
obdict_create(void)
{
  ObDict *dict = GC_MALLOC(sizeof(ObDict));
  return dict;
}

void
obdict_insert(ObDict *dict, ObDictEntry *ode)
{
  int ndx = obhash(ode);
  assert(ode->trueName);
  assert(ode->next == 0);

  ode->next = dict->bucket[ndx];
  dict->bucket[ndx] = ode;

  obdict_check(dict);
}

ObDictEntry *
obdict_lookup(ObDict *dict, ObDictEntry *ode)
{
  int ndx = obhash(ode);

  ObDictEntry *entry;

  for(entry = dict->bucket[ndx]; entry; entry = entry->next) {
    assert(entry->trueName);
    if ((entry->tnHash == ode->tnHash) &&
	nmequal(entry->trueName, ode->trueName))
      return entry;
  }

  return 0;
}
