/*
 * Copyright (c) 2002, The EROS Group, LLC and Johns Hopkins
 * University. All rights reserved.
 * 
 * This software was developed to support the EROS secure operating
 * system project (http://www.eros-os.org). The latest version of
 * the OpenCM software can be found at http://www.opencm.org.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials
 *    provided with the distribution.
 * 
 * 3. Neither the name of the The EROS Group, LLC nor the name of
 *    Johns Hopkins University, nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <opencm.h>

/* The CommitInfo object does NOT name the branch/version into which
 * the commit occurs. This is because doing so would interfere with
 * recovery from partial uploads.
 *
 * It *cannot* include the name of the Change object because
 * cryptographically hashed names and circular reference really don't
 * mix. It could harmlessly name the containing branch, but I'm trying
 * hard to keep all of the pointers going one way (which is probably a
 * mistake for purposes of recoverability, but for now I want to keep
 * things simple). The CommitInfo object *should not* contain the
 * version number of this commit. Here's why:
 *
 * Suppose a commit fails because the branch top has moved. This can
 * occur between the time that the upload starts and the upload
 * completes. Then we would need to re-perform the commit against a
 * different base version. But if we do this:
 *
 *   1. We would then need to change the version number in the
 *      CommitInfo record.
 *   2. This would change it's true name.
 *   3. This would in turn change the content of every Entity and
 *      Change object.
 *   4. Which would in turn change *their* true names.
 *
 * all of which would ultimately force us to perform the whole damned
 * upload again from scratch.
 *
 * Thus: no version number here.
 */

OC_bool
commitinfo_check(const void *v)
{
  return TRUE;
}

void
commitinfo_show(const void *vp)
{
  const CommitInfo *ci = vp;
  
  report(0, "Date:          %s\n", CMGET(ci,time));
  report(0, "AuthorURI:     %s\n", CMGET(ci,authorURI));
  report(0, "BranchURI:     %s\n", CMGET(ci,branchURI));
  report(0, "Branch Version: %s\n", 
	  xunsigned64_str(CMGET(ci,branchVersion)));
  report(0, "Descrip:\n%s\n", buffer_asString(CMGET(ci,descrip)));
}

void
commitinfo_serialize(SDR_stream *strm, const void *vp)
{
  const CommitInfo *ci = vp;

  sdr_w_string("time", strm, CMGET(ci,time));
  sdr_w_obname("authorURI", strm, CMGET(ci,authorURI));
  sdr_write("descrip", strm, CMGET(ci,descrip));
  sdr_w_obname("branchURI", strm, CMGET(ci,branchURI));
  sdr_w_u64("rev", strm, CMGET(ci,branchVersion));
}

void *
commitinfo_deserialize(const DeserializeInfo *di, SDR_stream *strm)
{
  CommitInfo *ci = (CommitInfo *) GC_MALLOC(sizeof(CommitInfo));

  ser_init(ci, &CommitInfo_SerType, di->ver);
  SER_MODIFIED(ci);
  
  CMSET(ci,time,          sdr_r_string("time", strm));
  CMSET(ci,authorURI,     sdr_r_obname("authorURI", strm));
  CMSET(ci,descrip,       sdr_read("descrip", strm));
  CMSET(ci,branchURI,     sdr_r_obname("branchURI", strm));
  CMSET(ci,branchVersion, sdr_r_u64("rev", strm));

  CM_CANFREEZE(ci);

  return ci;
}

void
commitinfo_mark(Repository *r, const void *container,
		const void *ob, rbtree *memObs)
{
  const CommitInfo *ci = ob;

  assert(container == ob);

  mark_addmark(container, memObs, CMGET(ci,authorURI));
  mark_addmark(container, memObs, CMGET(ci,branchURI));
}

CommitInfo *
commitinfo_create(const Buffer *descrip, const char *authorURI,
		  const char *branchURI, uint64_t rev)
{
  CommitInfo *ci = (CommitInfo *) GC_MALLOC(sizeof(CommitInfo));

  assert(authorURI);

  ser_init(ci, &CommitInfo_SerType, CommitInfo_SerType.ver);
  SER_MODIFIED(ci);

  CMSET(ci, time, os_GetISOTime());
  CMSET(ci, descrip, descrip);
  CMSET(ci, authorURI, authorURI);
  CMSET(ci, branchURI, branchURI);
  CMSET(ci, branchVersion, rev);
  
  CM_CANFREEZE(ci);

  return ci;
}

