/*

Copyright (C) 2007 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include <nd.h>
#include <nd_gui.h>

#include <nd_pcap.h>
#include <nd_pcap_callbacks.h>

static ND_ProtoField pcap_fields[] = {
  { ND_VAL_FIELD, N_("Timstamp secs\n(%s)"),  N_("Seconds part of packet timestamp"),      32,  nd_pcap_tv_sec_cb  },
  { ND_VAL_FIELD, N_("Timstamp usecs\n(%s)"), N_("Microseconds part of packet timestamp"), 32,  nd_pcap_tv_usec_cb },
  { ND_VAL_FIELD, N_("Capture length\n(%s)"), N_("Size of packet part that was captured"), 32,  nd_pcap_caplen_cb },
  { ND_VAL_FIELD, N_("Packet length\n(%s)"), N_("Size of packet on the wire"),             32,  nd_pcap_len_cb },
  { 0, NULL, NULL, 0, NULL }
};

static LND_Protocol *pcap;
static ND_Protocol  *pcap_gui;


/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return ("Pcap Plugin");
}


const char *
description(void)
{
  return ("A plugin providing support for editing pcap packet header fields.\n");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return VERSION_MAJOR;
}


LND_Protocol *
init(void)
{
  /* A pseudo-protocol to fit into the real-proto environment */
  pcap = libnd_proto_new("pcap", LND_PROTO_LAYER_LINK, 0xa1b2c3d4);
  libnd_proto_registry_register(pcap);

  pcap_gui = nd_proto_new(pcap);
  pcap_gui->create_gui   = nd_pcap_create_gui;
  pcap_gui->set_gui      = nd_pcap_set_gui;
  pcap_gui->fields       = pcap_fields;
  pcap_gui->header_width = 128;
  
  return pcap;
}


/* Protocol method implementations: ------------------------------------ */

GtkWidget *
nd_pcap_create_gui(LND_Trace *trace, LND_ProtoInfo *pinf)
{
  GtkWidget *table;

  table = nd_gui_proto_table_create(trace, pinf);
  return table;
}


void       
nd_pcap_set_gui(const LND_Packet *packet, LND_ProtoInfo *pinf)
{
  nd_pcap_set_tv_sec(pinf, &packet->ph);
  nd_pcap_set_tv_usec(pinf, &packet->ph);
  nd_pcap_set_caplen(pinf, &packet->ph);
  nd_pcap_set_len(pinf, &packet->ph);

  return;
  TOUCH(pinf);
}


/* Misc helper stuff below --------------------------------------------- */

void
nd_pcap_set_tv_sec(LND_ProtoInfo *pinf, const struct pcap_pkthdr *hdr)
{
  char s[MAXPATHLEN];  
  g_snprintf(s, MAXPATHLEN, "%lu", (unsigned long) hdr->ts.tv_sec);
  nd_proto_field_set(pinf, &pcap_fields[0], s);
}

void
nd_pcap_set_tv_usec(LND_ProtoInfo *pinf, const struct pcap_pkthdr *hdr)
{
  char s[MAXPATHLEN];  
  g_snprintf(s, MAXPATHLEN, "%lu", (unsigned long) hdr->ts.tv_usec);
  nd_proto_field_set(pinf, &pcap_fields[1], s);
}

void
nd_pcap_set_caplen(LND_ProtoInfo *pinf, const struct pcap_pkthdr *hdr)
{
  char s[MAXPATHLEN];  
  g_snprintf(s, MAXPATHLEN, "%lu", (unsigned long) hdr->caplen);
  nd_proto_field_set(pinf, &pcap_fields[2], s);
}

void
nd_pcap_set_len(LND_ProtoInfo *pinf, const struct pcap_pkthdr *hdr)
{
  char s[MAXPATHLEN];  
  g_snprintf(s, MAXPATHLEN, "%lu", (unsigned long) hdr->len);
  nd_proto_field_set(pinf, &pcap_fields[3], s);
}


ND_Protocol *
nd_pcap_get_gui(void)
{
  return pcap_gui;
}


LND_Protocol *
nd_pcap_get(void)
{
  return pcap;
}

