<?php
/**
 * @package Kolab_Storage
 *
 * $Horde: framework/Kolab_Storage/lib/Horde/Kolab/Storage/Session.php,v 1.3.2.2 2008/09/22 16:21:23 wrobel Exp $
 */

/** We need the Auth library */
require_once 'Horde/Auth.php';

/**
 * The Kolab_Session class allows the Kolab classes to use a central
 * IMAP connection per session.
 *
 * $Horde: framework/Kolab_Storage/lib/Horde/Kolab/Storage/Session.php,v 1.3.2.2 2008/09/22 16:21:23 wrobel Exp $
 *
 * Copyright 2008 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * @author  Gunnar Wrobel <wrobel@pardus.de>
 * @package Kolab_Storage
 */
class Kolab_Session {

    /**
     * The name of the IMAP server.
     *
     * @var string
     */
    var $_imap_server;

    /**
     * The IMAP port of the server.
     *
     * @var int
     */
    var $_imap_port;

    /**
     * Primary user mail address.
     *
     * @var string
     */
    var $user_mail;

    /**
     * Our Kolab_IMAP object, used to communicate with the IMAP server.
     *
     * @var Kolab_IMAP
     */
    var $_imap;

    /**
     * Constructor.
     */
    function Kolab_Session()
    {
        global $conf;

        $user = $this->_fetchUser();
        if (!is_a($user, 'PEAR_Error')) {
            $result = $user->get(KOLAB_ATTR_MAIL);
            if (!empty($result) && !is_a($result, 'PEAR_Error')) {
                $this->user_mail = $result;
            }

            if (!empty($conf['kolab']['misc']['multidomain'])) {
                $result = $user->getServer('imap');
                if (!empty($result) && !is_a($result, 'PEAR_Error')) {
                    $server = explode(':', $result, 2);
                    if (!empty($server[0])) {
                        $this->_imap_server = $server[0];
                    }
                    if (!empty($server[1])) {
                        $this->_imap_port = $server[1];
                    }
                }
            }
        }

        if (!isset($this->_imap_server)
            && isset($conf['kolab']['imap']['server'])) {
            $this->_imap_server = $conf['kolab']['imap']['server'];
        }

        if (!isset($this->_imap_port)
            && isset($conf['kolab']['imap']['port'])) {
            $this->_imap_port = $conf['kolab']['imap']['port'];
        } else {
            $this->_imap_port = 143;
        }

        if (!isset($this->user_mail)) {
            $auth = Auth::getAuth();
            if (empty($auth)) {
                $auth = 'anonymous';
            }
            if (strpos($auth, '@')) {
                $this->user_mail = $auth;
            } else {
                $this->user_mail = $auth . '@' . (!empty($_SERVER['SERVER_NAME']) ? $_SERVER['SERVER_NAME'] : 'localhost');
            }
        }
    }

    /**
     * Returns the properties that need to be serialized.
     *
     * @return array  List of serializable properties.
     */
    function __sleep()
    {
        $properties = get_object_vars($this);
        unset($properties['_imap']);
        $properties = array_keys($properties);
        return $properties;
    }

    /**
     * Fetch the Kolab_Object representing the current user.
     *
     * @return Kolab_Object_user|PEAR_Error The object representing
     *                                      the current user.
     */
    function _fetchUser()
    {
        $user = Auth::getAuth();
        if (empty($user)) {
            return PEAR::raiseError(_('Anonymous user.'));
        }

        /** We need the Kolab Server access. */
        require_once 'Horde/Kolab/Server.php';
        $server = Horde_Kolab_Server::singleton();
        if (is_a($server, 'PEAR_Error')) {
            return $server;
        }

        $dn = $server->dnForUidOrMail($user);
        if (empty($dn)) {
            return PEAR::raiseError(_('No such user.'));
        }
        if (is_a($dn, 'PEAR_Error')) {
            return $dn;
        }

        $user = $server->fetch($dn);
        if (is_a($user, 'PEAR_Error')) {
            return $user;
        }

        global $conf;

        if (empty($conf['kolab']['misc']['allow_special'])
            && !is_a($user, 'Horde_Kolab_Server_Object_user')) {
            return PEAR::raiseError(_('Current user is not a standard Kolab user.'));
        }
        return $user;
    }

    /**
     * Returns the current IMAP connection.
     *
     * @return Kolab_IMAP|PEAR_Error An open IMAP connection.
     */
    function &getImap()
    {
        if (isset($this->_imap)) {
            return $this->_imap;
        }

        if (!isset($this->_imap_server)) {
            $this->_imap = PEAR::raiseError(_("The URL for the Kolab IMAP Server is not available!"));
            return $this->_imap;
        }

        /** We need the Kolab IMAP library now. */
        require_once 'Horde/Kolab/Storage/IMAP.php';

        $this->_imap = &Kolab_IMAP::singleton($this->_imap_server, $this->_imap_port,
                                              true, false);
        if (is_a($this->_imap, 'PEAR_Error')) {
            return $this->_imap;
        }
        $result = $this->_imap->connect(Auth::getAuth(),
                                        Auth::getCredential('password'));
        if (is_a($result, 'PEAR_Error')) {
            $this->_imap = &$result;
            return $result;
        }

        return $this->_imap;
    }

    /**
     * Attempts to return a reference to a concrete Kolab_Session instance.
     *
     * It will only create a new instance if no Kolab_Session instance
     * currently exists.
     *
     * This method must be invoked as:
     *   <code>$var = &Kolab_Session::singleton();</code>
     *
     * @static
     *
     * @return Kolab_Session  The concrete Session reference.
     */
    function &singleton()
    {
        static $session;

        if (!isset($session)) {
            /**
             * Horde_Kolab_Server currently has no caching so we mainly
             * cache some user information here as reading this data
             * may be expensive when running in a multi-host
             * environment.
             */
            require_once 'Horde/SessionObjects.php';
            $hs = &Horde_SessionObjects::singleton();
            $session = $hs->query('kolab_session');
        }

        if (empty($session)) {
            $session = new Kolab_Session();
        }

        register_shutdown_function(array(&$session, 'shutdown'));

        return $session;
    }

    /**
     * Stores the object in the session cache.
     */
    function shutdown()
    {
        require_once 'Horde/SessionObjects.php';
        $session = &Horde_SessionObjects::singleton();
        $session->overwrite('kolab_session', $this, false);
    }
}
