<?php
/**
 * A driver for simulating a Kolab user database stored in LDAP.
 *
 * $Horde: framework/Kolab_Server/lib/Horde/Kolab/Server/test.php,v 1.2.2.2 2008/08/01 07:56:19 wrobel Exp $
 *
 * @package Kolab_Server
 */

/** Require the LDAP based class as our base class */
require_once 'Horde/Kolab/Server/ldap.php';

/**
 * This class provides a class for testing the Kolab Server DB.
 *
 * $Horde: framework/Kolab_Server/lib/Horde/Kolab/Server/test.php,v 1.2.2.2 2008/08/01 07:56:19 wrobel Exp $
 *
 * Copyright 2008 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * @author  Gunnar Wrobel  <wrobel@pardus.de>
 * @package Kolab_Server
 */
class Horde_Kolab_Server_test extends Horde_Kolab_Server_ldap {

    /**
     * Array holding the current result set.
     *
     * @var array
     */
    var $_result;

    /**
     * Buffer for error numbers.
     *
     * @var int
     */
    var $_errno = 0;

    /**
     * Buffer for error descriptions.
     *
     * @var int
     */
    var $_error = '';

    /**
     * Parse LDAP filter.
     * Partially derived from Net_LDAP_Filter.
     *
     * @param string $filter The filter string.
     *
     * @return array|PEAR_Error An array of the parsed filter.
     */
    function _parse($filter)
    {
        $result = array();
        if (preg_match('/^\((.+?)\)$/', $filter, $matches)) {
            if (in_array(substr($matches[1], 0, 1), array('!', '|', '&'))) {
                $result['op']  = substr($matches[1], 0, 1);
                $result['sub'] = $this->_parseSub(substr($matches[1], 1));
                return $result;
            } else {
                if (stristr($matches[1], ')(')) {
                    return PEAR::raiseError("Filter parsing error: invalid filter syntax - multiple leaf components detected!");
                } else {
                    $filter_parts = preg_split('/(?<!\\\\)(=|=~|>|<|>=|<=)/', $matches[1], 2, PREG_SPLIT_DELIM_CAPTURE);
                    if (count($filter_parts) != 3) {
                        return PEAR::raiseError("Filter parsing error: invalid filter syntax - unknown matching rule used");
                    } else {
                        $result['att'] = $filter_parts[0];
                        $result['log'] = $filter_parts[1];
                        $result['val'] = $filter_parts[2];
                        return $result;
                    }
                }
            }
        } else {
            return PEAR::raiseError(sprintf("Filter parsing error: %s - filter components must be enclosed in round brackets",
                                            $filter));
        }
    }

    /**
     * Parse a LDAP subfilter.
     *
     * @param string $filter The subfilter string.
     *
     * @return array|PEAR_Error An array of the parsed subfilter.
     */
    function _parseSub($filter)
    {
        $result = array();
        $level = 0;
        $collect = '';
        while (preg_match('/^(\(.+?\))(.*)/', $filter, $matches)) {
            if (in_array(substr($matches[1], 0, 2), array('(!', '(|', '(&'))) {
                $level++;
                $collect .= $matches[1];
            } else if ($level) {
                $collect .= $matches[1];
                if (substr($matches[2], 0, 1) == ')') {
                    $collect .= ')';
                    $matches[2] = substr($matches[2], 1);
                    $level--;
                    if (!$level) {
                        $result[] = $this->_parse($collect);
                    }
                }
            } else {
                $result[] = $this->_parse($matches[1]);
            }
            $filter = $matches[2];
        }
        return $result;
    }

    /**
     * Search for an object.
     *
     * @param string  $filter     Filter criteria.
     * @param array   $attributes Restrict the search result to
     *                            these attributes.
     *
     * @return array|PEAR_Error A LDAP serach result.
     */
    function _search($filter, $attributes = null) {
        $filter = $this->_parse($filter);
        if (is_a($filter, 'PEAR_Error')) {
            return $filter;
        }
        $result = $this->_doSearch($filter, $attributes);
        if (empty($result)) {
            return null;
        }
        return $result;
    }

    /**
     * Perform the search.
     *
     * @param array   $filter     Filter criteria-
     * @param array   $attributes Restrict the search result to
     *                            these attributes.
     *
     * @return array|PEAR_Error A LDAP serach result.
     */
    function _doSearch($filter, $attributes = null)
    {
        if (isset($filter['log'])) {
            $result = array();
            foreach ($this->_params as $element) {
                if (isset($element['data'][$filter['att']])) {
                    switch ($filter['log']) {
                    case '=':
                        $value = $element['data'][$filter['att']];
                        if ($value == $filter['val']
                            || (is_array($value)
                                && in_array($filter['val'], $value))) {
                            if (empty($attributes)) {
                                $result[] = $element;
                            } else {
                                $selection = $element;
                                foreach ($element['data'] as $attr => $value) {
                                    if (!in_array($attr, $attributes)) {
                                        unset($selection['data'][$attr]);
                                    }
                                }
                                $result[] = $selection;
                            }
                        }
                        break;
                    default:
                        return PEAR::raiseError(_("Not implemented!"));
                    }
                }
            }
            return $result;
        } else {
            $subresult = array();
            $filtercount = count($filter['sub']);
            foreach ($filter['sub'] as $subfilter) {
                $subresult = array_merge($subresult, $this->_doSearch($subfilter, $attributes));
            }
            $result = array();
            $dns = array();
            foreach ($subresult as $element) {
                $dns[] = $element['dn'];
                $result[$element['dn']] = $element;
            }
            switch ($filter['op']) {
            case '&':
                $count = array_count_values($dns);
                $selection = array();
                foreach ($count as $dn => $value) {
                    if ($value == $filtercount) {
                        $selection[] = $result[$dn];
                    }
                }
                return $selection;
            case '|':
                return array_values($result);
            default:
                return PEAR::raiseError(_("Not implemented!"));
            }
        }
    }

    /**
     * Read object data.
     *
     * @param string $dn    The object to retrieve.
     * @param string $attrs Restrict to these attributes
     *
     * @return array|PEAR_Error An array of attributes.
     */
    function _read($dn, $attrs = null)
    {
        if (!isset($this->_params[$dn])) {
            return PEAR::raiseError(sprintf("LDAP Error: No such object: %s: No such object",
                                            $dn));
        }
        if (empty($attrs)) {
            return $this->_params[$dn]['data'];
        } else {
            $result = array();
            $data = $this->_params[$dn]['data'];

            foreach ($attrs as $attr) {
                if (isset($data[$attr])) {
                    $result[$attr] = $data[$attr];
                    array_push($result, $attr);
                }
            }
            $result['count'] = 1;
            return $result;
        }
    }

    /**
     * Count the number of results.
     *
     * @param array $result   The LDAP search result.
     *
     * @return int The number of records found.
     */
    function _count($result)
    {
        if (is_array($result)) {
            return count($result);
        } else {
            return false;
        }
    }

    /**
     * Return the dn of an entry.
     *
     * @param array $entry  The LDAP entry.
     *
     * @return string  The DN of the entry.
     */
    function _getDn($entry)
    {
        if (is_array($entry) && isset($entry['dn'])) {
            if (isset($entry['count'])) {
                return $entry['dn'][0];
            } else {
                return $entry['dn'];
            }
        }
        return false;
    }

    /**
     * Return the attributes of an entry.
     *
     * @param array $entry   The LDAP entry.
     *
     * @return array  The attributes of the entry.
     */
    function _getAttributes($entry)
    {
        if (is_array($entry)) {
            return $entry;
        }
        return false;
    }

    /**
     * Return the first entry of a result.
     *
     * @param array $result   The LDAP search result.
     *
     * @return mixe  The first entry of the result or false.
     */
    function _firstEntry($result)
    {
        if (is_array($result)) {
            $data = array_keys($result[0]['data']);
            $data['count'] = 1;
            $data['dn'] = array($result[0]['dn']);
            $data['dn']['count'] = 1;
            foreach ($result[0]['data'] as $attr => $value) {
                if (!is_array($value)) {
                    $value = array($value);
                }
                $data[$attr] = $value;
                $data[$attr]['count'] = count($value);
            }
            return $data;
        }
        return false;
    }

    /**
     * Return the entries of a result.
     *
     * @param array $result   The LDAP search result.
     *
     * @return mixed The entries of the result or false.
     */
    function _getEntries($result)
    {
        if (is_array($result)) {
            $data = array();
            $data['count'] = count($result);
            foreach ($result as $entry) {
                $t = $entry['data'];
                $t['dn'] = $entry['dn'];
                $data[] = $t;
            }
            return $data;
        }
        return false;
    }

    /**
     * Return the current LDAP error number.
     *
     * @return int  The current LDAP error number.
     */
    function _errno()
    {
        return $this->_errno;
    }

    /**
     * Return the current LDAP error description.
     *
     * @return string  The current LDAP error description.
     */
    function _error()
    {
        return $this->_error;
    }
}
