<?php
/**
 * A library for accessing the Kolab user database.
 *
 * $Horde: framework/Kolab_Server/lib/Horde/Kolab/Server.php,v 1.2.2.3 2008/09/22 12:22:33 wrobel Exp $
 *
 * @package Kolab_Server
 */

/** We need PEAR */
require_once 'PEAR.php';

/** Provide access to the Kolab specific objects. */
require_once 'Horde/Kolab/Server/Object.php';

/** Define types of return values. */
define('KOLAB_SERVER_RESULT_SINGLE', 1);
define('KOLAB_SERVER_RESULT_STRICT', 2);
define('KOLAB_SERVER_RESULT_MANY',   3);

/**
 * This class provides methods to deal with Kolab objects stored in
 * the Kolab object db.
 *
 * $Horde: framework/Kolab_Server/lib/Horde/Kolab/Server.php,v 1.2.2.3 2008/09/22 12:22:33 wrobel Exp $
 *
 * Copyright 2008 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * @author  Gunnar Wrobel  <wrobel@pardus.de>
 * @package Kolab_Server
 */
class Horde_Kolab_Server {

    /**
     * Server parameters.
     *
     * @var array
     */
    var $_params = array();

    /**
     * Valid Kolab object types
     *
     * @var array
     */
    var $valid_types = array(
        KOLAB_OBJECT_ADDRESS,
        KOLAB_OBJECT_ADMINISTRATOR,
        KOLAB_OBJECT_DOMAINMAINTAINER,
        KOLAB_OBJECT_GROUP,
        KOLAB_OBJECT_MAINTAINER,
        KOLAB_OBJECT_SERVER,
        KOLAB_OBJECT_SHAREDFOLDER,
        KOLAB_OBJECT_USER,
    );

    /**
     * Construct a new Horde_Kolab_Server object.
     *
     * @param array $params  Parameter array.
     */
    function Horde_Kolab_Server($params = array())
    {
        $this->_params = $params;
    }

    /**
     * Attempts to return a concrete Horde_Kolab_Server instance based
     * on $driver.
     *
     * @param mixed $driver  The type of concrete Horde_Kolab_Server subclass to
     *                       return.
     * @param array $params  A hash containing any additional
     *                       configuration or connection parameters a subclass
     *                       might need.
     *
     * @return Horde_Kolab_Server|PEAR_Error The newly created concrete
     *                                       Horde_Kolab_Server instance.
     */
    function &factory($driver, $params = array())
    {
        $driver = basename($driver);
        if (empty($driver) || $driver == 'none') {
            $db = new Horde_Kolab_Server($params);
            return $db;
        }

        if (file_exists(dirname(__FILE__) . '/Server/' . $driver . '.php')) {
            include_once dirname(__FILE__) . '/Server/' . $driver . '.php';
        }

        $class = 'Horde_Kolab_Server_' . $driver;
        if (class_exists($class)) {
            $db = new $class($params);
        } else {
            $db = PEAR::raiseError('Class definition of ' . $class . ' not found.');
        }

        return $db;
    }

    /**
     * Attempts to return a reference to a concrete Horde_Kolab_Server
     * instance based on $driver. It will only create a new instance
     * if no Horde_Kolab_Server instance with the same parameters currently
     * exists.
     *
     * This method must be invoked as:
     * $var = &Horde_Kolab_Server::singleton()
     *
     * @return Horde_Kolab_Server|PEAR_Error The concrete Horde_Kolab_Server
     *                                       reference.
     */
    function &singleton()
    {
        global $conf;

        static $instances = array();

        if (isset($conf['kolab']['server'])
            && isset($conf['kolab']['server']['driver'])) {
            $driver = $conf['kolab']['server']['driver'];
            if (isset($conf['kolab']['server']['params'])) {
                $params = $conf['kolab']['server']['params'];
            } else {
                $params = array();
            }
        } else if (isset($conf['kolab']['ldap']['server'])
                   && isset($conf['kolab']['ldap']['basedn'])
                   && isset($conf['kolab']['ldap']['phpdn'])
                   && isset($conf['kolab']['ldap']['phppw'])) {
            $driver = 'ldap';
            $params = array('server' => $conf['kolab']['ldap']['server'],
                            'base_dn' => $conf['kolab']['ldap']['basedn'],
                            'bind_dn' => $conf['kolab']['ldap']['phpdn'],
                            'bind_pw' => $conf['kolab']['ldap']['phppw']);
        } else {
            $driver = null;
            $params = array();
        }

        $signature = serialize(array($driver, $params));
        if (empty($instances[$signature])) {
            $instances[$signature] = &Horde_Kolab_Server::factory($driver,
                                                                  $params);
        }

        return $instances[$signature];
    }

    /**
     * Fetch a Kolab object.
     *
     * @param string $dn   The DN of the object to fetch.
     * @param string $type The type of the object to fetch.
     *
     * @return Kolab_Object|PEAR_Error The corresponding Kolab object.
     */
    function &fetch($dn, $type = null)
    {
        if (empty($type)) {
            $type = $this->_determineType($dn);
            if (is_a($type, 'PEAR_Error')) {
                return $type;
            }
        } else {
            if (!in_array($type, $this->valid_types)) {
                return PEAR::raiseError(sprintf(_("Invalid Kolab object type \"%s\"."),
                                                $type));
            }
        }

        $object = &Horde_Kolab_Server_Object::factory($type, $dn, $this);
        return $object;
    }

    /**
     * Get the groups for this object
     *
     * @param string $dn   The DN of the object to fetch.
     *
     * @return array|PEAR_Error An array of group ids.
     */
    function getGroups($dn)
    {
        return array();
    }

    /**
     * Read object data.
     *
     * @param string $dn    The object to retrieve.
     * @param string $attrs Restrict to these attributes.
     *
     * @return array|PEAR_Error An array of attributes.
     */
    function read($dn, $attrs = null) {
        return $this->_read($dn, $attrs);
    }

    /**
     * Stub for reading object data.
     *
     * @param string $dn    The object to retrieve.
     * @param string $attrs Restrict to these attributes.
     *
     * @return array|PEAR_Error An array of attributes.
     */
    function _read($dn, $attrs = null) {
        return PEAR::raiseError(_("Not implemented!"));
    }

    /**
     * Determine the type of a Kolab object.
     *
     * @param string $dn The DN of the object to examine.
     *
     * @return string The corresponding Kolab object type.
     */
    function _determineType($dn)
    {
        return KOLAB_OBJECT_USER;
    }

    /**
     * Identify the primary mail attribute for the first object found
     * with the given uid or mail.
     *
     * @param string $id  Search for objects with this uid/mail.
     *
     * @return mixed|PEAR_Error The mail address or false if there was
     *                          no result.
     */
    function mailForUidOrMail($id)
    {
        /* In the default class we just return the id */
        return $id;
    }

    /**
     * Returns a list of allowed email addresses for the given user.
     *
     * @param string $sasluser The user name.
     *
     * @return array|PEAR_Error An array of allowed mail addresses.
     */
    function addrsForUid($sasluser)
    {
        /* In the default class we just return the mail address */
        return $sasluser;
    }

    /**
     * Return the DN for a given primary mail, uid, or alias.
     *
     * @param string $mail  A valid mail address for the user.
     *
     * @return mixed|PEAR_Error The DN or false if there was no result.
     */
    function dnForMailAddress($mail)
    {
        /* In the default class we just return the mail address */
        return $mail;
    }

    /**
     * Identify the DN for the first object found using a specified
     * attribute value.
     *
     * @param string $attr     The name of the attribute used for searching.
     * @param string $value    The desired value of the attribute.
     * @param int    $restrict A KOLAB_SERVER_RESULT_* result restriction.
     *
     * @return mixed|PEAR_Error The DN or false if there was no result.
     */
    function dnForAttr($attr, $value,
                       $restrict = KOLAB_SERVER_RESULT_SINGLE)
    {
        /* In the default class we just return false */
        return false;
    }

    /**
     * Is the given DN member of the group with the given mail address?
     *
     * @param string $dn       DN of the user.
     * @param string $mail     Search the group with this mail address.
     *
     * @return boolean|PEAR_Error True in case the user is in the
     *                            group, false otherwise.
     */
    function memberOfGroupAddress($dn, $mail)
    {
        /* No groups in the default class */
        return false;
    }

    /**
     * Identify the DN for the first object found with the given uid.
     *
     * @param string $uid      Search for objects with this uid.
     * @param int    $restrict A KOLAB_SERVER_RESULT_* result restriction.
     *
     * @return mixed|PEAR_Error The DN or false if there was no result.
     */
    function dnForUid($uid,
                      $restrict = KOLAB_SERVER_RESULT_SINGLE)
    {
        return $this->dnForAttr('uid', $uid);
    }

    /**
     * Identify the DN for the first object found with the given mail.
     *
     * @param string $mail     Search for objects with this mail address.
     * @param int    $restrict A KOLAB_SERVER_RESULT_* result restriction.
     *
     * @return mixed|PEAR_Error The DN or false if there was no result.
     */
    function dnForMail($mail,
                      $restrict = KOLAB_SERVER_RESULT_SINGLE)
    {
        return $this->dnForAttr('mail', $mail);
    }

    /**
     * Identify the DN for the first object found with the given uid or mail.
     *
     * @param string $id       Search for objects with this uid/mail.
     * @param int    $restrict A KOLAB_SERVER_RESULT_* result restriction.
     *
     * @return mixed|PEAR_Error The DN or false if there was no result.
     */
    function dnForUidOrMail($id)
    {
        $dn = $this->dnForAttr('uid', $id);
        if (!$dn) {
            $dn = $this->dnForAttr('mail', $id);
        }
        return $dn;
    }

    /**
     * Identify the DN for the first object found with the given alias.
     *
     * @param string $mail     Search for objects with this mail alias.
     * @param int    $restrict A KOLAB_SERVER_RESULT_* result restriction.
     *
     * @return mixed|PEAR_Error The DN or false if there was no result.
     */
    function dnForAlias($mail,
                      $restrict = KOLAB_SERVER_RESULT_SINGLE)
    {
        return $this->dnForAttr('alias', $mail);
    }

    /**
     * Identify the DN for the first object found with the given mail
     * address or alias.
     *
     * @param string $mail Search for objects with this mail address
     * or alias.
     *
     * @return mixed|PEAR_Error The DN or false if there was no result.
     */
    function dnForMailOrAlias($mail) {
        $dn = $this->dnForAttr('alias', $mail);
        if (!$dn) {
            $dn = $this->dnForAttr('mail', $mail);
        }
        return $dn;
    }

    /**
     * Identify the DN for the first object found with the given uid,
     * mail or alias.
     *
     * @param string $id       Search for objects with this uid/mail/alias.
     * @param int    $restrict A KOLAB_SERVER_RESULT_* result restriction.
     *
     * @return mixed|PEAR_Error The DN or false if there was no result.
     */
    function dnForMailOrUidOrAlias($id)
    {
        $dn = $this->dnForAttr('uid', $id);
        if (!$dn) {
            $dn = $this->dnForAttr('mail', $id);
            if (!$dn) {
                $dn = $this->dnForAttr('alias', $id);
            }
        }
        return $dn;
    }
};
