/* $Id: main.c 2114 2006-10-22 14:44:42Z nick $
 *
 * Copyright (c) 2006 Nick Schermer <nick@xfce.org>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <gtk/gtk.h>
#include <libxfcegui4/libxfcegui4.h>
#include <libxfce4util/libxfce4util.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>

#ifdef HAVE_LIBNOTIFY
#include <libnotify/notify.h>
#endif

#define BORDER                     6
#define DEFAULT_POSITION           "buttom_right"
#define NOTIFY_RC_FILE             "xfce4/notication-daemon-xfce/settingsrc"
#define NOTIFY_DBUS_NAME           "org.freedesktop.Notifications"
#define NOTIFY_DBUS_CORE_INTERFACE "org.freedesktop.Notifications"
#define NOTIFY_DBUS_CORE_OBJECT    "/org/freedesktop/Notifications"

typedef struct
{
  const gchar *name;
  const gchar *identifier;
}
PopupLocation;



static const PopupLocation popup_locations[] =
{
  { N_("Top Left"),     "top_left"     },
  { N_("Top Right"),    "top_right"    },
  { N_("Bottom Left"),  "bottom_left"  },
  { N_("Bottom Right"), "bottom_right" },
  { NULL,               NULL           }
};



static void             notification_settings_init_daemon    (void);
static void             notification_settings_reload_daemon  (void);
#ifdef HAVE_LIBNOTIFY
static void             notification_settings_preview_closed (GObject *object,
                                                              gpointer user_data);
static gboolean         notification_settings_preview        (GtkWidget   *button);
#endif
static void             notification_settings_save_position  (GtkComboBox *combobox);
static void             notification_settings_write_setting  (const gchar *key,
                                                              const gchar *value);
static gchar           *notification_settings_read_setting   (const gchar *key,
                                                              const gchar *fallback);
static GtkWidget       *notifications_settings_dialog_new    (void);



/* globals */
static DBusGProxy *proxy = NULL;
#ifdef HAVE_LIBNOTIFY
static NotifyNotification *preview = NULL;
static gboolean show_preview_again = FALSE;
#endif


static void
notification_settings_init_daemon (void)
{
  GError *error = NULL;
  DBusGConnection *bus = NULL;

  g_type_init ();

  bus = dbus_g_bus_get (DBUS_BUS_SESSION, &error);

  if (G_UNLIKELY (error != NULL))
    {
       g_message ("Unable to get session bus: %s", error->message);
       g_error_free (error);
       return;
    }

  proxy = dbus_g_proxy_new_for_name (bus,
                                     NOTIFY_DBUS_NAME,
                                     NOTIFY_DBUS_CORE_OBJECT,
                                     NOTIFY_DBUS_CORE_INTERFACE);
  dbus_g_connection_unref (bus);
}



static void
notification_settings_reload_daemon (void)
{
  GError *error = NULL;

  if (G_UNLIKELY (proxy == NULL))
    return;

  if (!dbus_g_proxy_call(proxy, "ReloadSettings", &error,
                         G_TYPE_INVALID, G_TYPE_INVALID))
  {
     g_message("Unable to reload the daemon: %s", error->message);
     g_error_free(error);
  }
}


#ifdef HAVE_LIBNOTIFY
static void
notification_settings_preview_closed (GObject *object,
                                      gpointer data)
{
  GtkWidget *button;
  g_return_if_fail (object == G_OBJECT (preview));

  button = (GtkWidget *)data;

  /* destroy the notification */
  g_object_unref (object);
  preview = NULL;

  /* show preview again, or active the button again */
  if (show_preview_again)
    {
       /* reset bool and show again */
       show_preview_again = FALSE;
       g_idle_add ((GSourceFunc) notification_settings_preview,
                   button);
    }
  else
    {
       gtk_widget_set_sensitive (button, TRUE);
    }
}



static gboolean
notification_settings_preview (GtkWidget *button)
{
  /* return if the notification is already visible */
  if (preview != NULL)
    return FALSE;

  /* init libnotify */
  if (G_UNLIKELY (!notify_init("notification-settings")))
    {
       g_warning ("Unable to init the notification library");
       return FALSE;
    }

  /* disable preview button */
  gtk_widget_set_sensitive (button, FALSE);

  /* create new preview notification */
  preview = notify_notification_new (_("Preview notification"),
    _("Lorem Ipsum is simply dummy text of the printing and typesetting industry."),
      "notification-settings",
      NULL);

  notify_notification_set_urgency (preview, NOTIFY_URGENCY_NORMAL);
  notify_notification_set_category(preview, "preview");
  notify_notification_set_timeout (preview, 5000);

  /* attach close signal */
  g_signal_connect(G_OBJECT(preview), "closed",
    G_CALLBACK(notification_settings_preview_closed), button);

  /* show the notification */
  notify_notification_show (preview, NULL);

  /* uninit libnotify */
  notify_uninit ();

  return FALSE;
}
#endif



static void
notification_settings_save_position (GtkComboBox *combobox)
{
  const PopupLocation *l;
  gint                 i = 0, selected;
  gchar               *identifier;

  /* get active selected item */
  selected = gtk_combo_box_get_active (combobox);

  /* lookup the identifier name */
  for (l = popup_locations; l->name != NULL; l++)
    {
       if (i == selected)
         {
            identifier = g_strdup (l->identifier);
            break;
         }

       i++;
    }

  /* security */
  if (G_UNLIKELY (identifier == NULL))
    identifier = g_strdup (DEFAULT_POSITION);

  /* save identifier */
  notification_settings_write_setting ("popup_location", identifier);

  /* cleanup */
  g_free (identifier);

  /* reload daemon */
  notification_settings_reload_daemon ();


#ifdef HAVE_LIBNOTIFY
  /* display the preview again on new position */
  if (preview)
    {
       show_preview_again = TRUE;
       notify_notification_close (preview, NULL);
    }
#endif
}



static void
notification_settings_write_setting (const gchar *key,
                                     const gchar *value)
{
  gchar  *file;
  XfceRc *rc;

  file = xfce_resource_save_location (XFCE_RESOURCE_CONFIG, NOTIFY_RC_FILE, TRUE);
  if (G_LIKELY (file))
    {
       /* open rc file, writable */
       rc = xfce_rc_simple_open (file, FALSE);
       g_free (file);

       if (G_LIKELY (rc))
         {
            xfce_rc_write_entry (rc, key, value);
            xfce_rc_close (rc);
         }
    }
}



static gchar *
notification_settings_read_setting (const gchar *key,
                                    const gchar *fallback)
{
  gchar  *file;
  XfceRc *rc;
  gchar  *value = NULL;

  file = xfce_resource_save_location (XFCE_RESOURCE_CONFIG, NOTIFY_RC_FILE, TRUE);
  if (G_LIKELY (file))
    {
       /* open rc file, read-only */
       rc = xfce_rc_simple_open (file, TRUE);
       g_free (file);

       if (G_LIKELY (rc))
         {
            value = g_strdup (xfce_rc_read_entry (rc, key, fallback));

            xfce_rc_close (rc);
         }
    }

  if (G_UNLIKELY (value == NULL))
    value = g_strdup (fallback);

  return value;
}



static GtkWidget *
notifications_settings_dialog_new (void)
{
  GtkWidget           *dialog, *dialog_vbox, *hbox, *label, *combobox, *button, *image;
  const PopupLocation *l;
  gint                 i = 0;
  gchar               *active;

  dialog = xfce_titled_dialog_new_with_buttons (_("Notification Settings"),
                                                NULL,
                                                GTK_DIALOG_NO_SEPARATOR,
                                                NULL);
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
  xfce_titled_dialog_set_subtitle (XFCE_TITLED_DIALOG (dialog),
    _("Configure the Xfce notification daemon"));
  gtk_window_set_icon_name (GTK_WINDOW (dialog), "notification-settings");

  dialog_vbox = GTK_DIALOG (dialog)->vbox;

  hbox = gtk_hbox_new (FALSE, BORDER);
  gtk_widget_show (hbox);
  gtk_box_pack_start (GTK_BOX (dialog_vbox), hbox, FALSE, FALSE, 0);;
  gtk_container_set_border_width (GTK_CONTAINER (hbox), BORDER);

  label = gtk_label_new_with_mnemonic (_("Notification _position:"));
  gtk_widget_show (label);
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

  combobox = gtk_combo_box_new_text ();
  gtk_widget_show (combobox);
  gtk_box_pack_start (GTK_BOX (hbox), combobox, TRUE, TRUE, 0);
  gtk_label_set_mnemonic_widget (GTK_LABEL (label), combobox);

  /* connect combobox change signal */
  g_signal_connect(G_OBJECT(combobox), "changed",
    G_CALLBACK(notification_settings_save_position), NULL);

  /* append all position values */
  active = notification_settings_read_setting ("popup_location", DEFAULT_POSITION);
  for (l = popup_locations; l->name != NULL; l++)
    {
       gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), l->name);

       if (!strcmp(active, l->identifier))
         gtk_combo_box_set_active (GTK_COMBO_BOX (combobox), i);

       i++;
    }
  g_free (active);

#ifdef HAVE_LIBNOTIFY
  /* preview button */
  hbox = GTK_DIALOG (dialog)->action_area;

  button = gtk_button_new_with_mnemonic (_("Pre_view"));
  gtk_widget_show (button);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);

  image = gtk_image_new_from_icon_name ("notification-settings", GTK_ICON_SIZE_BUTTON);
  gtk_widget_show (image);
  gtk_button_set_image (GTK_BUTTON (button), image);

  g_signal_connect(G_OBJECT(button), "clicked",
    G_CALLBACK(notification_settings_preview), NULL);

  /* button box layout */
  gtk_button_box_set_layout (GTK_BUTTON_BOX (GTK_DIALOG (dialog)->action_area),
                             GTK_BUTTONBOX_EDGE);
#endif

  /* close button */
  gtk_dialog_add_button (GTK_DIALOG (dialog), GTK_STOCK_CLOSE, GTK_RESPONSE_OK);

  return dialog;
}

int
main (int argc, char **argv)
{
  GtkWidget *dialog;

#ifdef GETTEXT_PACKAGE
  xfce_textdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");
#endif

  gtk_init (&argc, &argv);

  /* init the connection with notification-daemon-xfce */
  notification_settings_init_daemon ();

  /* default icon name */
  gtk_window_set_default_icon_name ("notification-settings");

  /* show settings dialog */
  dialog = notifications_settings_dialog_new ();
  gtk_dialog_run (GTK_DIALOG (dialog));

  /* close dialog */
  gtk_widget_destroy (dialog);

#ifdef HAVE_LIBNOTIFY
  /* close the preview notification */
  if (preview)
    {
       /* the close signal is not triggered, so also unref */
       notify_notification_close (preview, NULL);
       g_object_unref (G_OBJECT (preview));
    }
#endif

  /* unref proxy */
  g_object_unref (G_OBJECT (proxy));

  return 0;
}
