// Implementation of a path linker that allows to directly go to any parent directory
// Proposed and coded by Julian Mitchell <jupeos@gmail.com>, many thanks to him!


#include <sstream>

#include <fox-1.6/fx.h>

#include "xfedefs.h"
#include "PathLinker.h"
#include "XFileExplorer.h"

extern FXMainWindow* mainWindow;

FXDEFMAP(PathLinker) PathLinkerMap[] = {
                                           FXMAPFUNC(SEL_FOCUSIN, PathLinker::ID_FOCUS_BUTTON, PathLinker::onCmdFocusButton),
                                           FXMAPFUNCS(SEL_ENTER, PathLinker::ID_START_LINK, PathLinker::ID_END_LINK, PathLinker::pathButtonEnter),
                                           FXMAPFUNCS(SEL_LEAVE, PathLinker::ID_START_LINK, PathLinker::ID_END_LINK, PathLinker::pathButtonEnter),
                                           FXMAPFUNCS(SEL_LEFTBUTTONPRESS, PathLinker::ID_START_LINK, PathLinker::ID_END_LINK, PathLinker::pathButtonPressed),
                                           FXMAPFUNCS(SEL_LEFTBUTTONRELEASE, PathLinker::ID_START_LINK, PathLinker::ID_END_LINK, PathLinker::pathButtonReleased),
                                       };

FXIMPLEMENT(PathLinker, FXHorizontalFrame, PathLinkerMap, ARRAYNUMBER(PathLinkerMap))

PathLinker::PathLinker(FXComposite* a): FXHorizontalFrame(a, LAYOUT_FILL_X, 0, 0, 0, 0, 0, 0, 2, 2, 0, 2)
{
	// Add some path links
    int varId = ID_START_LINK;
    for(int i = 0; i < MAX_LINKS; i++)
    {
        std::stringstream ss;
        ss << i;
        mPath.push_back(new FXButton(this, (ss.str() + PATHSEPSTRING).c_str(), NULL, this, (ButtonIds)varId, BUTTON_TOOLBAR, 0, 0, 0, 0, 0, 0, 0, 0));
        varId++;
        mPath[i]->hide();
        mPath[i]->setDefaultCursor(getApp()->getDefaultCursor(DEF_HAND_CURSOR));
	}
    mFullPath = "";
    activeButtons = 0;
	
	// Right most button is used for focus
	focusButton = new FXButton(this, " ", NULL, this, ID_FOCUS_BUTTON, BUTTON_TOOLBAR|LAYOUT_FILL_X, 0, 0, 0, 0, 0, 0, 0, 0);	
}


PathLinker::~PathLinker()
{
}


void PathLinker::setText(FXString text)
{
    int nextPos = 0;
    int previousPos = 0;
    int frameWidth = getParent()->getParent()->getWidth();


    // Hide all of the link buttons
    for(int i = 0; i < MAX_LINKS; i++)
    {
        mPath[i]->hide();
    }

    mFullPath = text.text();
    FXFont* normalFont = getApp()->getNormalFont();
    int textWidth = normalFont->getTextWidth(text.text(), text.length());

    FXString displayText = mFullPath;

    if(textWidth > frameWidth)
    {
        // The magic '4' is to compenstate for the '/...' that is introduced at the front
        // of the the path. This calculation doesn't seem to work quite right as it
        // shifts the splitter's grab bar a small amount, for the left panel.
        displayText = displayText.right(frameWidth / (textWidth / text.length()) - 4);
        displayText = "..." + displayText;
        displayText = PATHSEPSTRING + displayText;
    }

    unsigned int index = 0;
    nextPos = displayText.find(PATHSEPSTRING, 0);
    while(nextPos >= 0)
    {
        setText(index, displayText.mid(previousPos, nextPos - previousPos + 1));
        index++;
        previousPos = nextPos + 1;
        nextPos = displayText.find(PATHSEPSTRING, previousPos);
    }
    activeButtons = index;
	
    setText(index, displayText.mid(previousPos, displayText.length()));
}


void PathLinker::setText(unsigned int index, FXString displayText)
{	
    if(index < MAX_LINKS)
    {
        mPath[index]->setText(displayText);
		mPath[index]->show();
    }
}

long PathLinker::pathButtonEnter(FXObject * obj ,FXSelector sel ,void* ptr)
{
    switch(FXSELTYPE(sel))
    {
		case SEL_ENTER:
		{
			SetPathHighlight(FXSELID(sel));
			break;
		}
		case SEL_LEAVE:
		{
			RevertPathHighlight(FXSELID(sel));
			break;
		}
		default:
			break;
    }
    return 0;
}


void PathLinker::SetPathHighlight(int endId)
{
    for(int i=0; i <= (endId - ID_START_LINK); i++)
    {
        mPath[i]->setTextColor(FXRGB(0,0,255));
    }
}


void PathLinker::RevertPathHighlight(int endId)
{
	FilePanel* panel=(FilePanel*)getParent()->getParent();
	FilePanel* current=((XFileExplorer*) mainWindow)->lpanel->current;
 
	for(int i=0; i <= (endId - ID_START_LINK); i++)
    {
        if (panel==current)
			mPath[i]->setTextColor(getApp()->getForeColor());
		else
			mPath[i]->setTextColor(FXRGB(255,255,255));
    }
}


long PathLinker::pathButtonPressed(FXObject * obj ,FXSelector sel ,void* ptr)
{
    // Set the correct parent focus
    long retVal = 0;

	// Get parent panel and set the focus on it
	FilePanel* panel=(FilePanel*)getParent()->getParent();
	panel->focus();
	
	FXString filePath("");
    int endId = FXSELID(sel);

    // No change in the directory if this is the last button being pressed.
    if(endId - ID_START_LINK != activeButtons)
    {
        if(endId == ID_START_LINK)
        {
            // Selecting root dir
            filePath = PATHSEPSTRING;
        }
        else
        {
            int rpos = 0;
            char PathSepChar[] = PATHSEPSTRING;
            rpos = mFullPath.rfind(PathSepChar[0], 0x7FFFFFFF, activeButtons+1 - (endId - ID_START_LINK + 1));
            filePath = mFullPath.left(rpos);
        }

        // Update all windows concerned with this change in directory
        setText(filePath);
        ((XFileExplorer*) mainWindow)->dirpanel->list->setDirectory(filePath,TRUE);
        ((XFileExplorer*) mainWindow)->lpanel->current->list->setDirectory(filePath);

		retVal = 1;
    }
    return retVal;
}


long PathLinker::pathButtonReleased(FXObject * obj ,FXSelector sel ,void* ptr)
{
    FXString filepath("");
    int endId = FXSELID(sel);
    mPath[endId - ID_START_LINK]->killFocus();
    return 1;
}


// Gives the focus to the parent panel when clicking on the focus button
long PathLinker::onCmdFocusButton(FXObject * obj ,FXSelector sel ,void* ptr)
{	
	// Get parent panel and set the focus on it
	FilePanel* panel=(FilePanel*)getParent()->getParent();
	panel->focus();
    return 1;
}


void PathLinker::unfocus(void)
{
	this->setBackColor(FXRGB(128,128,128));
    for(int i=0; i <MAX_LINKS; i++)
    {
        mPath[i]->setBackColor(FXRGB(128,128,128));
		mPath[i]->setTextColor(FXRGB(255,255,255));
	}
	focusButton->setBackColor(FXRGB(128,128,128));
}


void PathLinker::focus(void)
{
	this->setBackColor(getApp()->getBaseColor());
    for(int i=0; i <MAX_LINKS; i++)
    {
        mPath[i]->setBackColor(getApp()->getBaseColor());
		mPath[i]->setTextColor(getApp()->getForeColor());
    }
	focusButton->setBackColor(getApp()->getBaseColor());
}
