/*
 * Copyright (C), 2000-2003 by the monit project group.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <config.h>

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif 

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif 

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#include "event.h"
#include "alert.h"
#include "monitor.h"
#include "engine.h"


/**
 *  Function for spawning of a process. This function fork's twice to
 *  avoid creating any zombie processes. Inspired by code from
 *  W. Richard Stevens book, APUE.
 *
 *  @author Jan-Henrik Haukeland, <hauk@tildeslash.com>
 *
 *  @version \$Id: spawn.c,v 1.36 2003/10/25 19:17:11 hauk Exp $
 *  @file
 */


/* -------------------------------------------------------------- Prototypes */


static void exec_alert_mail(Service_T s, Command_T c);
static void set_monit_environment(Service_T s, const char *event);


/* ------------------------------------------------------------------ Public */


/**
 * Execute the given command. If the execution fails, an alert message
 * is sent to the email addresses found in the service object that has
 * registred interest for restart alerts.
 * @param P A Service object
 * @param C A Command object
 * @param E An optional event string, specifying why this function was
 * called. May be NULL.
 */
void  spawn(Service_T S, Command_T C, const char *E) {
  
  pid_t pid;
  sigset_t mask, save;

  ASSERT(S);
  ASSERT(C);

  /*
   * Block SIGCHLD
   */
  sigemptyset(&mask);
  sigaddset(&mask, SIGCHLD);
  pthread_sigmask(SIG_BLOCK, &mask, &save);

  if( (pid= fork()) < 0 ) {

    log("Cannot fork of a new process\n");  
    exit(1);
    
  }  else if(pid == 0) {

    if((pid= fork()) < 0) {
      log("Cannot fork of a new process\n");  
      exit(1);
    }
    
    else if(pid > 0) {
      
      _exit(0);
      
    } else  {

      /*
       * Reset all signals, so the child process is
       * *not* created with any inherited SIG_BLOCK
       */
      sigemptyset(&mask);
      pthread_sigmask(SIG_SETMASK, &mask, NULL);
      
      signal(SIGINT, SIG_DFL);
      signal(SIGHUP, SIG_DFL);
      signal(SIGTERM, SIG_DFL);
      signal(SIGUSR1, SIG_DFL);
      signal(SIGPIPE, SIG_DFL);

      /*
       * Reset to the original umask so programs will inherit the
       * same file creation mask monit was started with
       */
      umask(Run.umask);

      /*
       * Switch uid/gid if requested
       */
      if(C->has_gid) {
	if(0 != setgid(C->gid)) {
	  log("Failed to change gid to '%d' for '%s'\n",
	      C->gid, C->arg[0]);
	}
      }
      if(C->has_uid) {
	if(0 != setuid(C->uid)) {
	  log("Failed to change uid to '%d' for '%s'\n",
	      C->uid, C->arg[0]);
	}
      }

      set_monit_environment(S, E);
      
      if(! Run.isdaemon) {
	redirect_stdfd();
      }
      
      log_close();

      fd_close();

      (void) execv(C->arg[0], C->arg);

      exec_alert_mail(S, C);
      if(log_init()) {
        log("Error: Could not execute %s\n", C->arg[0]);
        log_close();
      }
      _exit(1);
      
    }
  }

  /* Wait for first child - aka second parent, to exit */
  if(waitpid(pid, NULL, 0) != pid) {
    
      log("Waitpid error\n");

  }

  /*
   * Restore the signal mask
   */
  pthread_sigmask(SIG_SETMASK, &save, NULL);
  
  /*
   * We do not need to wait for the second child since we forked twice,
   * the init system-process will wait for it. So we just return
   */
  return ;
  
} 


/* ----------------------------------------------------------------- Private */


/*
 * Send an exec failed alert mail to those mail objects that has
 * registred interest for start/restart/stop notification.
 */
static void exec_alert_mail(Service_T s, Command_T c) {

  Mail_T a= s->maillist;

  for(; a; a= a->next)
      if(IS_EVENT_SET(a->events, EVENT_RESTART) ||
	 IS_EVENT_SET(a->events, EVENT_START)   ||
	 IS_EVENT_SET(a->events, EVENT_STOP)) {
	struct mymail m;
	char message[STRLEN];
	
	m.to= a->to;
	m.from= "monit@localhost";
	m.subject= "monit alert -- exec error";
	snprintf(message, STRLEN,
		 "Could not execute program '%s' for service %s\n\n"
		 "Your faithful employee,\nmonit",
		 c->arg[0]?c->arg[0]:"null", s->name);
	m.message= message;
	sendmail(&m);
      }
  
}


/*
 * Setup the environment with special MONIT_xxx variables. The program
 * executed may use such variable for various purposes.
 */
static void set_monit_environment(Service_T s, const char *event) {
  
  int i, j;
  char buf[10][STRLEN];
  char *date= get_RFC822date(NULL);
  
  i= 0;
  
  snprintf(buf[i++], STRLEN, "MONIT_DATE=%s", date);
  snprintf(buf[i++], STRLEN, "MONIT_SERVICE=%s", s->name);
  snprintf(buf[i++], STRLEN, "MONIT_HOST=%s", Run.localhostname);
  snprintf(buf[i++], STRLEN, "MONIT_EVENT=%s", event?event:"No Event");
  
  if(s->type == TYPE_PROCESS) {
    snprintf(buf[i++], STRLEN, "MONIT_PROCESS_PID=%d", is_process_running(s));
    snprintf(buf[i++], STRLEN, "MONIT_PROCESS_MEMORY=%ld",
	     s->procinfo->mem_kbyte);
    snprintf(buf[i++], STRLEN, "MONIT_PROCESS_CHILDREN=%d",
	     s->procinfo->children);
    snprintf(buf[i++], STRLEN, "MONIT_PROCESS_CPU_PERCENT=%d",
	     s->procinfo->cpu_percent);
  }
  
  buf[i][0]= 0;
  
  for(j= 0; *buf[j]; j++)
      putenv(buf[j]);

  FREE(date);
  
}
