/*
 * SiSCtrl - SiS Display Control Panel
 * for the SiS X.org/XFree86 driver
 *
 * MAIN part
 *
 * (C) 2003-2005 Thomas Winischhofer <thomas@winischhofer.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "sisctrl.h"

/* Forward declararions */
static void setxvval(Atom atom, int val);
static Bool sendsdcommand(unsigned int cmd);

static Bool getmfbdetails(int index);
static int calcrefresh(XF86VidModeModeInfo *modeinfo);


#ifdef USE_DEPRECATED_XV /*-------------------------------------------- */
/* libc etc wrappers */
static int
get_pid(void)
{
#ifdef HAVE_GETPID
   return(getpid());
#else
   return 0;
#endif
}

static void
setseed(void)
{
#if defined(HAVE_GETTIMEOFDAY) || defined(X_GETTIMEOFDAY)
   struct timeval mytimeval;
#if defined(HAVE_GETTIMEOFDAY)
   struct timezone mytimezone;
   gettimeofday(&mytimeval, &mytimezone);
#else
   X_GETTIMEOFDAY(&mytimeval);
#endif
#if defined(HAVE_SRAND)
   srand(mytimeval.tv_usec);
#elif defined(HAVE_SRANDOM)
   srandom(mytimeval.tv_usec);
#else
#warning no srandom() or srand()!
#endif
#else
#if defined(HAVE_SRAND)
   srand(time(0) + get_pid());
#elif defined(HAVE_SRANDOM)
   srandom(mytimeval.tv_usec);
#else
#warning no srandom() or srand()!
#endif
#endif
}

static int
getrandom(void)
{
#if defined(HAVE_RAND)
   return(rand());
#elif defined(HAVE_RANDOM)
   return(random());
#else
   return(7653176 + get_pid());
#endif
}

static void
waitrandom(void)
{
   if(getrandom() & 1) sleep(1);
}
#endif	/* ----------------------------------------------------- */

int
multby1000(double in)
{
   int result;
   Bool isneg = FALSE;

   if(in < 0.0) { isneg = TRUE; in = -in; }

   result = (int)in * 1000;
   in -= (double)((int)in);
   in *= 10.0;
   result += (int)in * 100;
   in -= (double)((int)in);
   in *= 10.0;
   result += (int)in * 10;
   in -= (double)((int)in);
   in *= 10.0;
   result += (int)in;

   if((result - ((result / 10) * 10)) >= 5)
      result = (result / 10) * 10 + 10;
   else
      result = (result / 10) * 10;

   if(isneg) result = -result;



   return result;
}

/**************************************************
 *               Clean up routines                *
 **************************************************/

static void
freemodestrings(void)
{
    int i;

    for(i = 0; i < SISCTRL_MAXMODES; i++) {
       if(gl.modetextptr[i]) {
          free(gl.modetextptr[i]);
          gl.modetextptr[i] = NULL;
       }
#ifdef HAVERANDR
       if(gl.sizetextptr[i]) {
          free(gl.sizetextptr[i]);
	  gl.sizetextptr[i] = NULL;
       }
#endif
    }
}

static void
freemodelines(void)
{
    int i;

    if(sc.vmmodecount) {
       for(i = 0; i < sc.vmmodecount; i++) {
          if(sc.vmmodelines[i]->privsize > 0)
             XFree(sc.vmmodelines[i]->private);
       }
       XFree(sc.vmmodelines);
       sc.vmmodecount = 0;
    }
}

/* Our exit function */
static void
cleanupandexit(int errcode)
{
    int i;

    /* Get rid of out xv image */
#ifdef USEXV
    gui_destroy_xv_image();
#endif

    /* GUI clean-ups */
    gui_clean_ups();

    /* Reset LOG verbosity level */
    if(sc.dpy && sc.sdinterface && sc.xlogquiet) {
       sc.sdstruct.sdc_parm[0] = 0;
       sendsdcommand(SDC_CMD_LOGQUIET);
    }

    /* Restore video-only Xv attributes */
    if(sc.dpy) {
       sisctrl_restore_disable_gfx();
    }

    /* Free mode name strings */
    freemodestrings();

    /* Free modeinfo array returned from GetAllModeLines */
    freemodelines();

    /* Free RandR related stuff */
#ifdef HAVE_RANDR
    if(sc.sconf) {
       XRRFreeScreenConfigInfo(sc.sconf);
       sc.sconf = NULL;
    }
#endif

#ifdef HAVE_XINERAMA
    if(sc.xineinfo) {
       XFree(sc.xineinfo);
       sc.xineinfo = NULL;
    }
#endif

    /* Close display */
    if(sc.dpy) {
       XCloseDisplay(sc.dpy);
       sc.dpy = NULL;
    }

    /* Close and remove our lockfile */
    if(sc.lockfile != -1) {
       close(sc.lockfile);
       remove(sc.filenamebuffer);
       sc.lockfile = -1;
    }

    if(errcode) exit(errcode);
}

/**************************************************
 *        OS signal and error handlers            *
 **************************************************/

RETSIGTYPE
signal_handler(int sig)
{
   switch(sig) {
   case SIGHUP:
   case SIGTERM:
   case SIGINT:
      cleanupandexit(0);
      gui_main_quit();
      exit(0);
   }
}

#ifdef HAVE_RANDR
static int
myerrorhandler(Display *dpy, XErrorEvent *error)
{
   sc.xerror = TRUE;
   return 0;
}
#endif

/**************************************************
 *          SiSCtrl interface routines            *
 **************************************************/

static int
mysderrorhandler(Display *dpy, XErrorEvent *error)
{
   sc.sderror = 1;
   return 0;
}

static Bool
sendsdcommand(unsigned int cmd)
{
   XErrorHandler oldErrHandler;
   int i;

   if(!sc.sdinterface) return FALSE;

   sc.sdstruct.sdc_command = cmd;
   sc.sdstruct.sdc_id = SDC_ID;
   sc.sdstruct.sdc_chksum = sc.sdstruct.sdc_command;
   for(i = 0; i < SDC_NUM_PARM_RESULT; i++) {
      sc.sdstruct.sdc_chksum += sc.sdstruct.sdc_parm[i];
   }

   oldErrHandler = XSetErrorHandler(mysderrorhandler);
   sc.sderror = 0;

   if(!SiSCtrlCommand(sc.dpy, sc.myscreen, &sc.sdstruct)) {
      XSetErrorHandler(oldErrHandler);
      return FALSE;
   }

   XSetErrorHandler(oldErrHandler);

   if(sc.sderror) return FALSE;

   XFlush(sc.dpy);

   if(sc.sdstruct.sdc_result_header != SDC_RESULT_OK) return FALSE;
   return TRUE;
}

static Bool
UnlockSiSDirect(int unlock)
{
   Bool ret = TRUE;
   int key = unlock ? SIS_DIRECTKEY : 0;
   int result;

   if(sc.sdinterface) {
      sc.sdstruct.sdc_parm[0] = key;
      if(!sendsdcommand(SDC_CMD_SETLOCKSTATUS)) {
         fprintf(stderr, "Failed to unlock/lock SiS direct interface (0x%x)\n", sc.sdstruct.sdc_result[0]);
         ret = FALSE;
      }
#ifdef USE_DEPRECATED_XV
   } else {
      if(XvSetPortAttribute(sc.dpy, sc.xv_port_nr, sc.atom_usd, key) != Success) {
         fprintf(stderr, "Failed to unlock/lock SiS direct interface\n");
         ret = FALSE;
      }
      XFlush(sc.dpy);
#endif
   }

   return ret;
}

/* Get/set value using xv. Here we do proper unlock/locking */

int
getxvval(Atom atom)
{
   int val = 0;

   if(UnlockSiSDirect(1)) {
      if(XvGetPortAttribute(sc.dpy, sc.xv_port_nr, atom, &val) != Success) {
         fprintf(stderr, xverrorstring);
      }
      XFlush(sc.dpy);
      UnlockSiSDirect(0);
   }
   return val;
}

static void
setxvval(Atom atom, int val)
{
   if(UnlockSiSDirect(1)) {
      if(XvSetPortAttribute(sc.dpy, sc.xv_port_nr, atom, val) != Success) {
         fprintf(stderr, xverrorstring);
      }
      XFlush(sc.dpy);
      UnlockSiSDirect(0);
   }
}

/* Get/set value using either sd or xv. Here we do proper unlock/locking */

int
newgetxvval(unsigned int cmd, Atom atom)
{
   int val = 0;

   if(!sc.sdinterface) return getxvval(atom);

   if(UnlockSiSDirect(1)) {
      if(!sendsdcommand(cmd)) {
         fprintf(stderr, xverrorstring);
      } else {
         val = sc.sdstruct.sdc_result[0];
      }
      UnlockSiSDirect(0);
   }
   return val;
}

static void
newsetxvval(unsigned int cmd, Atom atom, int val)
{
   if(!sc.sdinterface || cmd == 0) {
      setxvval(atom, val);
   } else {
      if(UnlockSiSDirect(1)) {
         sc.sdstruct.sdc_parm[0] = (unsigned int)val;
         if(!sendsdcommand(cmd)) {
	    fprintf(stderr, xverrorstring);
         }
         UnlockSiSDirect(0);
      }
   }
}

int
newgetrealxvval(unsigned int cmd, Atom atom)
{
   int val = 0;

   if(!sc.sdinterface || cmd == 0) return getxvval(atom);

   if(UnlockSiSDirect(1)) {
      if(!sendsdcommand(cmd)) {
         fprintf(stderr, xverrorstring);
      } else {
         val = sc.sdstruct.sdc_result[0] - 32768;
      }
      UnlockSiSDirect(0);
   }
   return val;
}

static void
newsetrealxvval(unsigned int cmd, Atom atom, int val)
{
   if(!sc.sdinterface || cmd == 0) {
      setxvval(atom, val);
   } else {
      if(UnlockSiSDirect(1)) {
         sc.sdstruct.sdc_parm[0] = (unsigned int)(val + 32768);
         if(!sendsdcommand(cmd)) {
	    fprintf(stderr, xverrorstring);
         }
         UnlockSiSDirect(0);
      }
   }
}

/* Get/Set gamma brightness. Does proper unlocking/locking */

static Bool
sd_writegammabr(unsigned int br, unsigned int bg, unsigned int bb)
{
    if(UnlockSiSDirect(1)) {
       sc.sdstruct.sdc_parm[0] = br;
       sc.sdstruct.sdc_parm[1] = bg;
       sc.sdstruct.sdc_parm[2] = bb;
       if(!sendsdcommand(sc.cmd_set_br)) {
          fprintf(stderr, xverrorstring);
       }
       UnlockSiSDirect(0);
    } else fprintf(stderr, xverrorstring);
}

static Bool
sd_writenewgammabr(unsigned int br, unsigned int bg, unsigned int bb,
		   unsigned int cr, unsigned int cg, unsigned int cb)
{
    if(UnlockSiSDirect(1)) {
       sc.sdstruct.sdc_parm[0] = br;
       sc.sdstruct.sdc_parm[1] = bg;
       sc.sdstruct.sdc_parm[2] = bb;
       sc.sdstruct.sdc_parm[3] = cr;
       sc.sdstruct.sdc_parm[4] = cg;
       sc.sdstruct.sdc_parm[5] = cb;
       if(!sendsdcommand(sc.cmd_set_newbr)) {
          fprintf(stderr, xverrorstring);
       }
       UnlockSiSDirect(0);
    } else fprintf(stderr, xverrorstring);
}

static void
sd_writegamma2(void)
{
    if(UnlockSiSDirect(1)) {
       sc.sdstruct.sdc_parm[0] = multby1000(sc.newgamma2r);
       sc.sdstruct.sdc_parm[1] = multby1000(sc.newgamma2g);
       sc.sdstruct.sdc_parm[2] = multby1000(sc.newgamma2b);
       if(sc.usenewgamma) {
          sc.sdstruct.sdc_parm[3] = multby1000(sc.newgamma_bri2[0]) + 1000;
          sc.sdstruct.sdc_parm[4] = multby1000(sc.newgamma_bri2[1]) + 1000;
          sc.sdstruct.sdc_parm[5] = multby1000(sc.newgamma_bri2[2]) + 1000;
          sc.sdstruct.sdc_parm[6] = multby1000(sc.newgamma_con2[0]) + 1000;
          sc.sdstruct.sdc_parm[7] = multby1000(sc.newgamma_con2[1]) + 1000;
          sc.sdstruct.sdc_parm[8] = multby1000(sc.newgamma_con2[2]) + 1000;
          if(!sendsdcommand(SDC_CMD_SETGETNEWGAMMACRT2)) {
             fprintf(stderr, xverrorstring);
          }
       } else {
          sc.sdstruct.sdc_parm[3] = multby1000(sc.gamma_bri2[0]);
          sc.sdstruct.sdc_parm[4] = multby1000(sc.gamma_bri2[1]);
          sc.sdstruct.sdc_parm[5] = multby1000(sc.gamma_bri2[2]);
          if(!sendsdcommand(SDC_CMD_SETGETGAMMACRT2)) {
             fprintf(stderr, xverrorstring);
          }
       }
       UnlockSiSDirect(0);
    } else fprintf(stderr, xverrorstring);
}

int
sd_readcolorcalib(Atom myatom, Bool IsCoarse)
{
    int val = 0;

    if(sc.sdinterface) {
       if(sendsdcommand(SDC_CMD_GETTVCOLORCALIB)) {
          if(IsCoarse) val = sc.sdstruct.sdc_result[0];
          else         val = sc.sdstruct.sdc_result[1];
          val -= 32768;
       }
#ifdef USE_DEPRECATED_XV
    } else {
       val = getxvval(myatom);
#endif
    }
    return val;
}

static void
sd_writecolorcalib(unsigned int coarse, unsigned int fine)
{
    if(UnlockSiSDirect(1)) {
       sc.sdstruct.sdc_parm[0] = coarse;
       sc.sdstruct.sdc_parm[1] = fine;
       if(!sendsdcommand(SDC_CMD_SETTVCOLORCALIB)) {
          fprintf(stderr, xverrorstring);
       }
       UnlockSiSDirect(0);
    } else fprintf(stderr, xverrorstring);
}

/* Read/write a ulong. These do NOT unlock/lock! */

static Bool
sd_readulong(unsigned int cmd, Atom xv_atom, unsigned int *result)
{
    if(sc.sdinterface) {
       if(!sendsdcommand(cmd)) {
          return FALSE;
       }
       *result = sc.sdstruct.sdc_result[0];
       return TRUE;
#ifdef USE_DEPRECATED_XV
    } else {
       int xv_val;
       if(XvGetPortAttribute(sc.dpy, sc.xv_port_nr, xv_atom, &xv_val) != Success) {
          XFlush(sc.dpy);
          return FALSE;
       }
       XFlush(sc.dpy);
       *result = xv_val;
       return TRUE;
#endif
    }
    return FALSE;
}

static Bool
sd_writeulong(unsigned int cmd, Atom xv_atom, unsigned int parm)
{
    if(sc.sdinterface) {
       sc.sdstruct.sdc_parm[0] = parm;
       sc.sdstruct.sdc_parm[1] = 0;
       if(!sendsdcommand(cmd)) {
          return FALSE;
       }
       return TRUE;
#ifdef USE_DEPRECATED_XV
    } else {
       int xv_val = parm;
       if(XvSetPortAttribute(sc.dpy, sc.xv_port_nr, xv_atom, xv_val) != Success) {
          XFlush(sc.dpy);
          return FALSE;
       }
       XFlush(sc.dpy);
       return TRUE;
#endif
    }
    return FALSE;
}

/* Get/set the TV position or scaling. These do NOT unlock/lock! */

int
sd_gettvposscale(unsigned int cmd, Atom atom)
{
   if(sc.sdinterface) {

      if(!sendsdcommand(cmd)) return 0;
      return (sc.sdstruct.sdc_result[0] - 32768);

#ifdef USE_DEPRECATED_XV
   } else {

      int value;
      if(XvGetPortAttribute(sc.dpy, sc.xv_port_nr, atom, &value) != Success) {
         XFlush(sc.dpy);
         return 0;
      }
      XFlush(sc.dpy);
      return value;
#endif
   }

   return 0;
}

static Bool
sd_settvposscale(unsigned int cmd, Atom atom, int value)
{
   if(sc.sdinterface) {

      sc.sdstruct.sdc_parm[0] = value + 32768;
      if(!sendsdcommand(cmd)) return FALSE;
      return TRUE;

#ifdef USE_DEPRECATED_XV
   } else {

      if(XvSetPortAttribute(sc.dpy, sc.xv_port_nr, atom, value) != Success) {
         XFlush(sc.dpy);
         return FALSE;
      }
      XFlush(sc.dpy);
      return TRUE;
#endif
   }
   return FALSE;
}

/* Search for atom name and make Atom for it */

Atom
searchandmakeatom(char *name, int num_attrs, XvAttribute *attributes, Bool printwarning)
{
    int i;
    Bool found = FALSE;

    for(i = 0; i < num_attrs; i++) {
       if(strcmp(attributes[i].name, name) == 0) {
          found = TRUE;
	  break;
       }
    }

    if(found) return(XInternAtom(sc.dpy, name, False));
    else {
       fprintf(stderr, "Could not find %s attribute, update your X driver\n", name);
       if(printwarning) {
          gui_showerror_pregui("Required Xv attribute not found, update your X driver\n", FALSE, FALSE);
       }
       return 0;
    }
}

/* Set/restore "disablegfx" and "disablegfxlr" xv attributes */

void
sisctrl_set_disable_gfx(int s1, int s2)
{
   if(sc.havedglgfx) {
      newsetxvval(SDC_CMD_SETXVDISABLEGFXLR, sc.atom_dgl, 0);
      newsetxvval(SDC_CMD_SETXVDISABLEGFX, sc.atom_gfx, 0);
   }
}

void
sisctrl_restore_disable_gfx_lr(void)
{
   if(!gui_vi_get_xv_shown()) {
      if(sc.havedglgfx) {
         newsetxvval(SDC_CMD_SETXVDISABLEGFXLR, sc.atom_dgl, sc.backup_dgl);
      }
   }
}

void
sisctrl_restore_disable_gfx(void)
{
   if(sc.havedglgfx) {
      newsetxvval(SDC_CMD_SETXVDISABLEGFXLR, sc.atom_dgl, sc.backup_dgl);
      newsetxvval(SDC_CMD_SETXVDISABLEGFX, sc.atom_gfx, sc.backup_gfx);
   }
}

/**************************************************
 *     Convert chiptype/vb type to string         *
 **************************************************/

static char *
getchiptype(Bool *IsSiS)
{
   char *chip;
   unsigned short chiptype, chipflags;

   if(sc.havenewhwinfo) {
      chiptype  = sc.chipType;
      chipflags = sc.chipFlags;
#ifdef USE_DEPRECATED_XV
   } else {
      chiptype  = (sc.hwinfo >> 16) & 0xff;
      chipflags = sc.hwinfo & 0xffff;
#endif
   }

   *IsSiS = TRUE;

   switch(chiptype) {
   case SIS_300:    chip = "300";    break;
   case SIS_540:    chip = "540";    break;
   case SIS_630:    chip = "630";    break;
   case SIS_730:    chip = "730";    break;
   case SIS_315H:   chip = "315H";   break;
   case SIS_315:    chip = "315";    break;
   case SIS_315PRO: if(!(chipflags & SiSCF_Is315USB)) {
		       if(!(chipflags & SiSCF_Is315E))	chip = "315PRO";
		       else				chip = "315E";
		    } else {
		       if(!(chipflags & SiSCF_Is315E))	chip = "315PRO (USB)";
		       else				chip = "315E (USB)";
		    }
		    break;
   case SIS_550:    chip = "550";    break;
   case SIS_650:    if(chipflags & SiSCF_IsM650)	chip = "M650";
	            else if(chipflags & SiSCF_Is651)	chip = "651";
		    else if(chipflags & SiSCF_Is652)	chip = "652";
		    else if(chipflags & SiSCF_IsM652)	chip = "M652";
		    else if(chipflags & SiSCF_IsM653)	chip = "M653";
		    else				chip = "650";
		    break;
   case SIS_740:    chip = "740";    break;
   case SIS_330:    chip = "Xabre";  break;
   case SIS_661:    if(chipflags & SiSCF_IsM661)	chip = "M661FX";
		    else if(chipflags & SiSCF_IsM661M)	chip = "M661MX";
		    else				chip = "661FX";
		    break;
   case SIS_670:    chip = "670";  break;
   case SIS_741:    if(chipflags & SiSCF_IsM741)	chip = "M741";
		    else 				chip = "741/741GX";
		    break;
   case SIS_660:    chip = "660";  break;
   case SIS_760:    if(chipflags & SiSCF_IsM760)	chip = "M760";
		    else				chip = "760/760GX";
		    break;
   case SIS_761:    chip = "761/761GX"; break;
   case SIS_770:    chip = "770";    break;
   case SIS_340:    chip = "340";    break;
   case XGI_20:	    chip = "Z7";
		    *IsSiS = FALSE;
		    break;
   case XGI_40:	    if(chipflags & SiSCF_IsXGIV3)	chip = "V3XT";
		    else if(chipflags & SiSCF_IsXGIV5)	chip = "V5";
		    else if(chipflags & SiSCF_IsXGIDUO)	chip = "Duo";
		    else				chip = "V8";
		    *IsSiS = FALSE;
		    break;
   default:         chip = "UNKNOWN";
   }
   return chip;
}

static const char *
getvbtype(void)
{
   static const char *vb_301     = "; SiS 301 video bridge";
   static const char *vb_301bdh  = "; SiS 301B-DH video bridge";
   static const char *vb_301b    = "; SiS 301B video bridge";
   static const char *vb_301c    = "; SiS 301C video bridge";
   static const char *vb_302b    = "; SiS 302B video bridge";
   static const char *vb_301lv   = "; SiS 301LV video bridge";
   static const char *vb_302lv   = "; SiS 302LV video bridge";
   static const char *vb_302elv  = "; SiS 302ELV video bridge";
   static const char *vb_307t    = "; SiS 307 video bridge";
   static const char *vb_307lv   = "; SiS 307LV video bridge";
   static const char *vb_lvdsch  = "; LVDS transmitter; Chrontel TV encoder";
   static const char *vb_lvds    = "; LVDS transmitter";
   static const char *vb_ch      = "; Chrontel TV encoder";
   static const char *vb_conn    = "; Conexant ???";
   static const char *vb_unknown = "";


   if(sc.sd2flags & SiS_SD2_USEVBFLAGS2) {
      switch(sc.vbflags2 & (VB2_VIDEOBRIDGE | VB2_30xBDH)) {
         case VB2_301:    		return vb_301;
         case (VB2_301B | VB2_30xBDH):	return vb_301bdh;
         case VB2_301B:   		return vb_301b;
         case VB2_301C:   		return vb_301c;
	 case VB2_307T:   		return vb_307t;
         case VB2_302B:    		return vb_302b;
         case VB2_301LV:  		return vb_301lv;
         case VB2_302LV:  		return vb_302lv;
         case VB2_302ELV:  		return vb_302elv;
	 case VB2_307LV:  		return vb_307lv;
         case (VB2_LVDS | VB2_CHRONTEL):return vb_lvdsch;
         case VB2_LVDS:			return vb_lvds;
         case VB2_CHRONTEL:		return vb_ch;
         case VB2_CONEXANT:		return vb_conn;
         default: 			return vb_unknown;
      }
   } else {
      switch(sc.currentvbflags & (VB_VIDEOBRIDGE | VB_30xBDH)) {
         case VB_301:    		return vb_301;
         case (VB_301B | VB_30xBDH):	return vb_301bdh;
         case VB_301B:   		return vb_301b;
         case VB_301C:   		return vb_301c;
         case VB_302B:    		return vb_302b;
         case VB_301LV:  		return vb_301lv;
         case VB_302LV:  		return vb_302lv;
         case VB_302ELV:  		return vb_302elv;
         case (VB_LVDS | VB_CHRONTEL):	return vb_lvdsch;
         case VB_LVDS:			return vb_lvds;
         case VB_CHRONTEL:		return vb_ch;
         case VB_CONEXANT:		return vb_conn;
         default: 			return vb_unknown;
      }
   }
}

/**************************************************
 *          Query hardware capabilities           *
 **************************************************/

Bool
isusb(void)
{
   unsigned short chiptype, chipflags;

   if(sc.havenewhwinfo) {
      chiptype  = sc.chipType;
      chipflags = sc.chipFlags;
#ifdef USE_DEPRECATED_XV
   } else {
      chiptype  = (sc.hwinfo >> 16) & 0xff;
      chipflags = sc.hwinfo & 0xffff;
#endif
   }

   if(chiptype != SIS_315PRO) return FALSE;
   if(!(chipflags & SiSCF_Is315USB)) return FALSE;
   return TRUE;
}

Bool
supportvihuesat(void)
{
   if(sc.sdinterface) {
      if(sc.sd2flags & SiS_SD2_SUPPORTXVHUESAT) return TRUE;
    } else {
      if(sc.sdflags & (SiS_SD_IS315SERIES |
		       SiS_SD_IS330SERIES |
		       SiS_SD_IS340SERIES)) return TRUE;
    }
    return FALSE;
}

Bool
vbhavevb(void)
{
   if(sc.havesdflags2) {
      if(sc.sd2flags & SiS_SD2_VIDEOBRIDGE) return TRUE;
   } else {
      if(sc.currentvbflags & VB_VIDEOBRIDGE) return TRUE;
   }
   return FALSE;
}

Bool
vbissisbridge(void)
{
   if(sc.havesdflags2) {
      if(sc.sd2flags & SiS_SD2_SISBRIDGE) return TRUE;
   } else {
      if(sc.currentvbflags & VB_SISBRIDGE) return TRUE;
   }
   return FALSE;
}

Bool
vbischrontel(void)
{
   if(sc.havesdflags2) {
      if(sc.sd2flags & SiS_SD2_CHRONTEL) return TRUE;
   } else {
      if(sc.currentvbflags & VB_CHRONTEL) return TRUE;
   }
   return FALSE;
}

Bool
vbsupportsvga2(void)
{
   if(sc.sdflags & SiS_SD_ADDLSUPFLAG) {
      if(sc.sdflags & SiS_SD_SUPPORTVGA2) return TRUE;
   } else {
      if(sc.currentvbflags & (VB_301|VB_301B|VB_301C|VB_302B)) return TRUE;
   }
   return FALSE;
}

Bool
vbsupportsscart(void)
{
   if(sc.sdflags & SiS_SD_ADDLSUPFLAG) {
      if(sc.sdflags & SiS_SD_SUPPORTSCART) return TRUE;
   } else {
      if(sc.currentvbflags & (VB_301|VB_301B|VB_301C|VB_302B)) return TRUE;
   }
   return FALSE;
}

Bool
vbsupportstv(void)
{
   if(sc.sdflags & SiS_SD_ADDLSUPFLAG) {
      if(sc.sdflags & SiS_SD_SUPPORTTV) return TRUE;
   } else {
      if(sc.currentvbflags & (VB_SISTVBRIDGE | VB_CHRONTEL)) return TRUE;
   }
   return FALSE;
}

Bool
vbsupportsoverscan(void)
{
   if(sc.sdflags & SiS_SD_ADDLSUPFLAG) {
      if(sc.sdflags & SiS_SD_SUPPORTOVERSCAN) return TRUE;
   } else {
      if(sc.currentvbflags & VB_CHRONTEL) return TRUE;
   }
   return FALSE;
}

Bool
vbsupportstmds(void)
{
   if(sc.havesdflags2) {
      if(sc.sd2flags & SiS_SD2_LCDTMDS) return TRUE;
   } else {
      if(sc.currentvbflags & (VB_301|VB_301C|VB_302B)) return TRUE;
      if((sc.currentvbflags & VB_301B) && (!(sc.currentvbflags & VB_30xBDH))) return TRUE;
   }
   return FALSE;
}

Bool
vbsupportsettvtype(void)
{
   if(sc.havesdflags2) {
      if(sc.sd2flags & SiS_SD2_SUPPORTTVTYPE) return TRUE;
   } else {
      if(sc.currentvbflags & VB_SISBRIDGE) return TRUE;
   }
   return FALSE;
}

Bool
vbsupportgamma2(void)
{
   if(sc.havesdflags2) {
      if(sc.sd2flags & SiS_SD2_SUPPORTGAMMA2) return TRUE;
   } else {
      if(sc.currentvbflags & VB_SISBRIDGE) return TRUE;
   }
   return FALSE;
}

Bool
vbsupporttvedgeenhance(void)
{
   if(sc.havesdflags2) {
      if(sc.sd2flags & SiS_SD2_SUPPTVEDGE) return TRUE;
   } else {
      if(sc.currentvbflags & VB_301) return TRUE;
   }
   return FALSE;
}

Bool
vbsupporttvsaturation(void)
{
   if(sc.havesdflags2) {
      if(sc.sd2flags & SiS_SD2_SUPPTVSAT) return TRUE;
   } else {
      if(sc.currentvbflags & VB_SISBRIDGE) {
         if(!(sc.currentvbflags & VB_301)) return TRUE;
      }
   }
   return FALSE;
}

Bool
vbsupporttvsize(void)
{
   if(sc.havesdflags2) {
      if(sc.sd2flags & SiS_SD2_SUPPORTTVSIZE) return TRUE;
   } else {
      if(sc.currentvbflags & VB_SISBRIDGE) return TRUE;
   }
   return FALSE;
}

/* Warning of new version */

static Bool
sisctrl_warn_version(void)
{
   return (gui_showerror_pregui("SiSCtrl found some flags set which it doesn't understand. "
	                        "This means that the X driver is newer than this version of "
	                        "SiSCtrl and the two might be incompatible.\n\n"
				"Start sisctrl anyway?", TRUE, TRUE));
}

/**************************************************
 *          Displaymode related helpers           *
 **************************************************/

int finddmhistory(void)
{
    int index = 0, i, vmdotclock;
    XF86VidModeModeLine vmmodeline;

    if(XF86VidModeGetModeLine(sc.dpy, sc.myscreen, &vmdotclock, &vmmodeline)) {
       for(i = 0; (i < sc.vmmodecount) && (i < SISCTRL_MAXMODES); i++) {
          if((vmdotclock == sc.vmmodelines[i]->dotclock)          &&
	     (vmmodeline.hdisplay == sc.vmmodelines[i]->hdisplay) &&
	     (vmmodeline.hsyncstart == sc.vmmodelines[i]->hsyncstart) &&
	     (vmmodeline.hsyncend == sc.vmmodelines[i]->hsyncend) &&
	     (vmmodeline.htotal == sc.vmmodelines[i]->htotal) &&
	     (vmmodeline.vdisplay == sc.vmmodelines[i]->vdisplay) &&
	     (vmmodeline.vsyncstart == sc.vmmodelines[i]->vsyncstart) &&
	     (vmmodeline.vsyncend == sc.vmmodelines[i]->vsyncend) &&
	     (vmmodeline.vtotal == sc.vmmodelines[i]->vtotal) &&
	     (vmmodeline.flags == sc.vmmodelines[i]->flags)) {
	     index = i;
	     break;
	  }
       }
    } else {
       fprintf(stderr, "Failed to query current modeline\n");
    }
    return index;
}

static Bool
checkmodeforscreen(int dm_index, Bool havemfbdetails, int c1x, int c1y, int c2x, int c2y)
{
   int sizeX, sizeY;
   unsigned int val;

   if(sc.HaveGss) {
      val = newgetxvval(SDC_CMD_GETSCREENSIZE, sc.atom_gss);
      sizeX = val >> 16;
      sizeY = val & 0xffff;

#if 0 /* Does not work. VidMode ext fails if modeline's dim < virtual */
      if(sc.sdflags & SiS_SD_ISMERGEDFB) {
	 if(!havemfbdetails) {
	    if(getmfbdetails(dm_index)) {
	       c1x = sc.sdstruct.sdc_result[1];
	       c1y = sc.sdstruct.sdc_result[2];
	       c2x = sc.sdstruct.sdc_result[4];
	       c2y = sc.sdstruct.sdc_result[5];
	       havemfbdetails = TRUE;
	    }
	 }
	 if(havemfbdetails) {
	    if((c1x <= sizeX) && (c1y <= sizeY) && (c2x <= sizeX) && (c2y <= sizeY)) {
	       return TRUE;
	    }
	 }
      }
#endif

      if((sc.vmmodelines[dm_index]->hdisplay > sizeX) ||
         (sc.vmmodelines[dm_index]->vdisplay > sizeY)) {
	 return FALSE;
      }
   }
   return TRUE;
}

#ifdef HAVE_RANDR
static short
searchrrrate(int desiredrate)
{
   int i, j;

   for(j = 0; j < 2; j++) {
      for(i = 0; i < sc.rrnrate; i++) {
         if((desiredrate == sc.rrrates[i] + j) ||
	    (desiredrate == sc.rrrates[i] - j)) return sc.rrrates[i];
      }
   }
   return 0;
}

static void
updaterandrconfig(void)
{
   if(sc.sconf) XRRFreeScreenConfigInfo(sc.sconf);
   sc.sconf = XRRGetScreenInfo(sc.dpy, sc.root);
}

static void
updaterandrsizes(void)
{
   updaterandrconfig();
   sc.rrsizes = XRRConfigSizes(sc.sconf, &sc.rrnsize);
}
#endif

/* The old (Xv) API for mode checking was not entirely bullet-proof.
 * If more than one program is querying the server, this might
 * lead to incorrect results. I tried to circumvent this by
 * using a (hopefully) unique ID and checking this ID with
 * the result, but congestions might happen anyway.
 * Since we now use the SiSCtrl extension, this problem is void.
 */

static int
checkmodeforgivencrt2(int index, unsigned short cond)
{
   unsigned int myid, i = 25;
   int myidr, myerror;
   Bool warned = FALSE, doneit = FALSE;
   char *retstr = NULL;
   char msgval[256];

   if(index > 255) {
      fprintf(stderr, "Cannot check mode, index too high (%d)\n", index);
      return 0x80;
   }

   if(sc.sdinterface) {
      sc.sdstruct.sdc_parm[0] = cond & 0xff;
      sc.sdstruct.sdc_parm[1] = sc.vmmodelines[index]->dotclock;
      sc.sdstruct.sdc_parm[2] = sc.vmmodelines[index]->hdisplay;
      sc.sdstruct.sdc_parm[3] = sc.vmmodelines[index]->hsyncstart;
      sc.sdstruct.sdc_parm[4] = sc.vmmodelines[index]->hsyncend;
      sc.sdstruct.sdc_parm[5] = sc.vmmodelines[index]->htotal;
      sc.sdstruct.sdc_parm[6] = sc.vmmodelines[index]->vdisplay;
      sc.sdstruct.sdc_parm[7] = sc.vmmodelines[index]->vsyncstart;
      sc.sdstruct.sdc_parm[8] = sc.vmmodelines[index]->vsyncend;
      sc.sdstruct.sdc_parm[9] = sc.vmmodelines[index]->vtotal;
      /* Do not need to unlock */
      if(sendsdcommand(SDC_CMD_CHECKMODEFORCRT2)) {
         return(sc.sdstruct.sdc_result[0] & 0xff);
      }
   }

#ifdef USE_DEPRECATED_XV
   if(!doneit) {

      do {

         if(!UnlockSiSDirect(1)) return 0x80;

         setseed();
         myid = getrandom();
         myid = ((((myid & 0xffff0000) >> 16) ^ (myid & 0xffff)) + get_pid()) & 0xffff;

         myid = (myid << 16) | ((index & 0xff) << 8) | (cond & 0xff);

         if(XvSetPortAttribute(sc.dpy, sc.xv_port_nr, sc.atom_cmd, myid) != Success) {
            fprintf(stderr, xverrorstring);
         }
         XFlush(sc.dpy);
         if(XvGetPortAttribute(sc.dpy, sc.xv_port_nr, sc.atom_cmdr, &myidr) != Success) {
            fprintf(stderr, xverrorstring);
         }
         UnlockSiSDirect(0);

         if((myid & 0xffffff00) == (myidr & 0xffffff00)) {
            return(myidr & 0xff);
         } else {
            if(!warned) fprintf(stderr, "SiS SD API congestion - is there another session running?\n");
	    warned = TRUE;
            waitrandom();
         }

      } while(i--);

      fprintf(stderr, "25 attempts to check mode failed, giving up\n");
      return 0x80;
   }
#else
   return 0x80;
#endif
}

static int
checkmodeforcrt2(int index)
{
   return(checkmodeforgivencrt2(index, 0));
}

static void
getmfbleftrightdetails(int *lefth, int *leftv, int *righth, int *rightv)
{
   int *crt1h, *crt1v, *crt2h, *crt2v, test;

   if(sc.sdstruct.sdc_result[0] == SDC_MMODE_POS_ERROR) {
      *lefth = *leftv = *righth = *rightv = 0;
      return;
   }

   test = sc.sdstruct.sdc_result[0];
   if(test == SDC_MMODE_POS_CLONE) {
      test = sc.sdstruct.sdc_result[7];
   }

   switch(test) {
   case SDC_MMODE_POS_LEFTOF:
   case SDC_MMODE_POS_ABOVE:
	crt1h = righth;
	crt1v = rightv;
	crt2h = lefth;
	crt2v = leftv;
	break;
   default:
	crt1h = lefth;
	crt1v = leftv;
	crt2h = righth;
	crt2v = rightv;
   }

   *crt1h = sc.sdstruct.sdc_result[1];
   *crt1v = sc.sdstruct.sdc_result[2];

   *crt2h = sc.sdstruct.sdc_result[4];
   *crt2v = sc.sdstruct.sdc_result[5];
}

static void
rebuildmodestrings(void)
{
    int i;
    char buffer[64];

    for(i = 0; (i < sc.vmmodecount) && (i < SISCTRL_MAXMODES); i++) {
       if(sc.sdflags & SiS_SD_ISMERGEDFB) {
          if((sc.sdinterface) &&
	     getmfbdetails(i) &&
	     (sc.sdstruct.sdc_result[0] != SDC_MMODE_POS_ERROR) &&
	     (sc.sdstruct.sdc_result[7] != SDC_MMODE_POS_CLONE)) {
             int lefth, leftv, righth, rightv;
             getmfbleftrightdetails(&lefth, &leftv, &righth, &rightv);
             if((sc.sdstruct.sdc_result[0] == SDC_MMODE_POS_CLONE) &&
                (lefth == righth) && (leftv == rightv)) {
                sprintf(buffer, "%dx%d (Cloned)",
			sc.vmmodelines[i]->hdisplay,
			sc.vmmodelines[i]->vdisplay);
             } else {
                sprintf(buffer, "%dx%d (%dx%d%c%dx%d)",
			sc.vmmodelines[i]->hdisplay,
			sc.vmmodelines[i]->vdisplay,
			lefth, leftv,
			(sc.sdstruct.sdc_result[0] == SDC_MMODE_POS_CLONE) ? '+' :
			   ((sc.sdstruct.sdc_result[0] != sc.sdstruct.sdc_result[7]) ? '!' : '-'),
			righth, rightv);
	     }
          } else {
             sprintf(buffer, "%dx%d",
		sc.vmmodelines[i]->hdisplay,
		sc.vmmodelines[i]->vdisplay);
	  }
	  sc.moderate[i] = 0;
	  if(sc.sd2flags & SiS_SD2_MERGEDUCLOCK) {
	     sc.moderate[i] = sc.vmmodelines[i]->dotclock / 1000;
	  }
       } else {
          int refresh = calcrefresh(sc.vmmodelines[i]);
	  sc.moderate[i] = refresh;
	  if(sc.sdflags & SiS_SD_ISDUALHEAD) {
	     if(!sc.sdinterface && (sc.sdflags & SiS_SD_ISDHXINERAMA)) {
		sprintf(buffer, "Screen %d: %dx%d (%d Hz%s%s)",
		   sc.myscreen,
		   sc.vmmodelines[i]->hdisplay,
		   sc.vmmodelines[i]->vdisplay,
		   refresh,
		   (sc.vmmodelines[i]->flags & V_INTERLACE) ? " I" : "",
		   (sc.vmmodelines[i]->flags & V_DBLSCAN) ? " D" : "");
	     } else {
	        sprintf(buffer, "CRT%d: %dx%d (%d Hz%s%s)",
		   (sc.sdflags & SiS_SD_ISDHSECONDHEAD) ? 1 : 2,
		   sc.vmmodelines[i]->hdisplay,
		   sc.vmmodelines[i]->vdisplay,
		   refresh,
		   (sc.vmmodelines[i]->flags & V_INTERLACE) ? " I" : "",
		   (sc.vmmodelines[i]->flags & V_DBLSCAN) ? " D" : "");
	     }
	  } else {
             sprintf(buffer, "%dx%d (%d Hz%s%s)",
		sc.vmmodelines[i]->hdisplay,
		sc.vmmodelines[i]->vdisplay,
		refresh,
		(sc.vmmodelines[i]->flags & V_INTERLACE) ? " I" : "",
		(sc.vmmodelines[i]->flags & V_DBLSCAN) ? " D" : "");
          }
       }
       if(!(gl.modetextptr[i] = malloc(strlen(buffer)+1)))
          cleanupandexit(3);

       strcpy(gl.modetextptr[i], buffer);
    }
}

static unsigned short
rebuildmodefield(int i)
{
   unsigned short field = 0;

   if(!(checkmodeforgivencrt2(i, SiS_CF2_LCD) & 0x01)) {
      field |= SiS_CRT2_LCD;
   }
   if(sc.sdflags & SiS_SD_SUPPORTLCDA) {
      if(!(checkmodeforgivencrt2(i, SiS_CF2_CRT1LCDA) & 0x01)) {
         field |= SiS_CRT2_CRT1LCDA;
      }
   }
   if(!(checkmodeforgivencrt2(i, (SiS_CF2_TV | SiS_CF2_TVPAL)) & 0x01)) {
      field |= (SiS_CRT2_TV | SiS_CRT2_TVPAL);
   }
   if(!(checkmodeforgivencrt2(i, (SiS_CF2_TV | SiS_CF2_TVNTSC)) & 0x01)) {
      field |= (SiS_CRT2_TV | SiS_CRT2_TVNTSC);
   }
   if(sc.sdflags & SiS_SD_SUPPORTPALMN) {
      if(!(checkmodeforgivencrt2(i, (SiS_CF2_TV | SiS_CF2_TVPALM)) & 0x01)) {
         field |= (SiS_CRT2_TV | SiS_CRT2_TVPALM);
      }
      if(!(checkmodeforgivencrt2(i, (SiS_CF2_TV | SiS_CF2_TVPALN)) & 0x01)) {
         field |= (SiS_CRT2_TV | SiS_CRT2_TVPALN);
      }
   }
   if(sc.sdflags & SiS_SD_SUPPORTHIVISION) {
      if(!(checkmodeforgivencrt2(i, (SiS_CF2_TVSPECIAL | SiS_CF2_TVHIVISION)) & 0x01)) {
         field |= SiS_CRT2_HIVISION;
      }
   }
   if(sc.sdflags & SiS_SD_SUPPORTYPBPR) {
      if(!(checkmodeforgivencrt2(i, (SiS_CF2_TVSPECIAL | SiS_CF2_TVYPBPR525I)) & 0x01)) {
         field |= SiS_CRT2_YPBPR525I;
      }
      if(!(checkmodeforgivencrt2(i, (SiS_CF2_TVSPECIAL | SiS_CF2_TVYPBPR525P)) & 0x01)) {
         field |= SiS_CRT2_YPBPR525P;
      }
      if(sc.sd2flags & SiS_SD2_SUPPORT625I) {
         if(!(checkmodeforgivencrt2(i, (SiS_CF2_TVSPECIAL | SiS_CF2_TVYPBPR625I)) & 0x01)) {
            field |= SiS_CRT2_YPBPR625I;
         }
      }
      if(sc.sd2flags & SiS_SD2_SUPPORT625P) {
         if(!(checkmodeforgivencrt2(i, (SiS_CF2_TVSPECIAL | SiS_CF2_TVYPBPR625P)) & 0x01)) {
            field |= SiS_CRT2_YPBPR625P;
         }
      }
      if(!(checkmodeforgivencrt2(i, (SiS_CF2_TVSPECIAL | SiS_CF2_TVYPBPR750P)) & 0x01)) {
         field |= SiS_CRT2_YPBPR750P;
      }
      if(!(checkmodeforgivencrt2(i, (SiS_CF2_TVSPECIAL | SiS_CF2_TVYPBPR1080I)) & 0x01)) {
         field |= SiS_CRT2_YPBPR1080I;
      }
   }
   if(vbsupportsvga2()) {
      if(!(checkmodeforgivencrt2(i, SiS_CF2_VGA2) & 0x01)) {
         field |= SiS_CRT2_VGA2;
      }
   }

   return field;
}

static void
rebuildmodefieldlist(void)
{
   int i;

   for(i = 0; (i < sc.vmmodecount) && (i < SISCTRL_MAXMODES); i++) {
      sc.modecheckfield[i] = 0;
      if((vbhavevb()) && (!(sc.sdflags & SiS_SD_ISDUALHEAD))) {
         sc.modecheckfield[i] = rebuildmodefield(i);
      }
   }
}

Bool
sisctrl_get_mfb_devices(int index, int rotindex,
		int *leftdev, int *rightdev, int *topdev, int *bottomdev, int *rot, Bool *cloned)
{
   int test, c1idx, c2idx;
   const int temparray[] = { 1, 2, 3, 0, 4 };

   *leftdev = *rightdev = *topdev = *bottomdev = -1;
   *rot = 0;
   *cloned = FALSE;

   if((sc.sdinterface) && getmfbdetails(index)) {
      test = sc.sdstruct.sdc_result[0];
      if(test == SDC_MMODE_POS_CLONE) {
         *cloned = TRUE;
         test = sc.sdstruct.sdc_result[7];
      }
      if(test == SDC_MMODE_POS_CLONE) {
         return FALSE;
      }

      if(sc.CRT1isoff) c1idx = -1;
      else if(sc.currentvbflags & CRT1_LCDA) c1idx = DM_GI_LCD;
      else c1idx = DM_GI_CRT;

      c2idx = -1;
      if(sc.currentvbflags & CRT2_LCD) c2idx = DM_GI_LCD;
      else if(sc.currentvbflags & CRT2_TV) c2idx = DM_GI_TV;
      else if(sc.currentvbflags & CRT2_VGA) c2idx = DM_GI_CRT;

#ifdef HAVE_RANDR
      if(gl.supportedrots & 0x0e) {
         if(rotindex == 0) {
            rotindex = temparray[(sc.rrcurrent_rotation & 0x0f) >> 1];
         }
      } else
#endif
         rotindex = 1;

      switch(rotindex) {
         case 2: *rot = 1; break;
         case 3: *rot = 2; break;
         case 4: *rot = 3; break;
      }

      switch(test) {
      case SDC_MMODE_POS_LEFTOF:
         switch(rotindex) {
#ifdef HAVE_RANDR
         case 2:	/* 90 */
            *topdev = c1idx;
            *bottomdev = c2idx;
            break;
         case 3: 	/* 180 */
            *leftdev = c1idx;
            *rightdev = c2idx;
            break;
         case 4:	/* 270 */
            *topdev = c2idx;
            *bottomdev = c1idx;
            break;
#endif
         default:
            *leftdev = c2idx;
            *rightdev = c1idx;
         }
         break;
      case SDC_MMODE_POS_RIGHTOF:
         switch(rotindex) {
#ifdef HAVE_RANDR
         case 2:	/* 90 */
            *topdev = c2idx;
            *bottomdev = c1idx;
            break;
         case 3: 	/* 180 */
            *leftdev = c2idx;
            *rightdev = c1idx;
            break;
         case 4:	/* 270 */
            *topdev = c1idx;
            *bottomdev = c2idx;
            break;
#endif
         default:
            *leftdev = c1idx;
            *rightdev = c2idx;
         }
         break;
      case SDC_MMODE_POS_ABOVE:
         switch(rotindex) {
#ifdef HAVE_RANDR
         case 2:	/* 90 */
            *leftdev = c2idx;
            *rightdev = c1idx;
            break;
         case 3: 	/* 180 */
            *topdev = c1idx;
            *bottomdev = c2idx;
            break;
         case 4:	/* 270 */
            *leftdev = c1idx;
            *rightdev = c2idx;
            break;
#endif
         default:
            *topdev = c2idx;
            *bottomdev = c1idx;
         }
         break;
      case SDC_MMODE_POS_BELOW:
         switch(rotindex) {
#ifdef HAVE_RANDR
         case 2:	/* 90 */
            *leftdev = c1idx;
            *rightdev = c2idx;
            break;
         case 3: 	/* 180 */
            *topdev = c2idx;
            *bottomdev = c1idx;
            break;
         case 4:	/* 270 */
            *leftdev = c2idx;
            *rightdev = c1idx;
            break;
#endif
         default:
            *topdev = c1idx;
            *bottomdev = c2idx;
         }
         break;
      default:
         return FALSE;
      }

      return TRUE;

   } else {
      return FALSE;
   }
}

#ifdef HAVE_RANDR
static void
build_randr_sizes(void)
{
    int i;
    char buffer[80];

    /* This is re-entrant. */

    updaterandrsizes();

    if(sc.rrnsize > SISCTRL_MAXMODES)
       sc.rrnsize = SISCTRL_MAXMODES;

    for(i=0; i < sc.rrnsize; i++) {
        sprintf(buffer, "%dx%d",
		sc.rrsizes[i].width,sc.rrsizes[i].height/*, i*/);
	if(gl.sizetextptr[i]) free(gl.sizetextptr[i]);
	gl.sizetextptr[i] = malloc(strlen(buffer)+1);
	if(gl.sizetextptr[i]) {
           strcpy(gl.sizetextptr[i], buffer);
	}
    }
    strcpy(sc.leaveunchangedtext, "Leave unchanged");
}
#endif

/**************************************************
 *               Read driver flags                *
 **************************************************/

static Bool
updatevbandcrt1(void)
{
    unsigned int val;

    if(!sd_readulong(SDC_CMD_GETVBFLAGS, sc.atom_qvf, &val)) {
       fprintf(stderr, "Couldn't read vbflags\n");
       return FALSE;
    }
    sc.currentvbflags = (unsigned int)val;

    if(!sd_readulong(SDC_CMD_GETCRT1STATUS, sc.atom_ct1, &val)) {
       fprintf(stderr, "Couldn't read CRT1 status\n");
       return FALSE;
    }
    sc.CRT1isoff = val ? 0 : 1;

    return TRUE;
}

/**************************************************
 *       Gamma ramp handling (VidModExtension)    *
 **************************************************/

static void
getgamma(void)
{
   if(!XF86VidModeGetGamma(sc.dpy, sc.myscreen, &sc.currentgamma)) {
      fprintf(stderr, "Unable to get gamma correction\n");
   }
}

static Bool
allocgramps(void)
{
   int i;

   for(i=0; i<3; i++) {
      gl.ramp[i] = (unsigned short *)malloc(gl.nramp * sizeof(unsigned short));
      if(!gl.ramp[i]) {
         if(gl.ramp[0]) { free(gl.ramp[0]); gl.ramp[0] = NULL; }
	 if(gl.ramp[1]) { free(gl.ramp[1]); gl.ramp[1] = NULL; }
	 fprintf(stderr, "Memory allocation error\n");
         return FALSE;
      }
   }
   return TRUE;
}

static void
freegramps(void)
{
   int i;

   for(i = 0; i < 3; i++) {
       if(gl.ramp[i]) free(gl.ramp[i]);
       gl.ramp[i] = NULL;
   }
}

static unsigned short
calcgammaval(int j, int nramp, float invgamma, float bri, float c)
{
    float k = (float)j;
    float nrm1 = (float)(nramp - 1);
    float con = c * nrm1 / 3.0;
    float l, v;

    if(con != 0.0) {
       l = nrm1 / 2.0;
       if(con <= 0.0) {
          k -= l;
          k *= (l + con) / l;
       } else {
          l -= 1.0;
          k -= l;
          k *= l / (l - con);
       }
       k += l;
       if(k < 0.0) k = 0.0;
    }

    if(invgamma == 1.0) {
       v = k / nrm1 * 65535.0;
    } else {
       v = pow(k / nrm1, invgamma) * 65535.0 + 0.5;
    }

    v += (bri * (65535.0 / 3.0)) ;

    if(v < 0.0) v = 0.0;
    else if(v > 65535.0) v = 65535.0;

    return (unsigned short)v;
}

static void
setgamma(void)
{
   int i, j, k;

   if(!XF86VidModeSetGamma(sc.dpy, sc.myscreen, &sc.newgamma)) {
      fprintf(stderr, "Unable to set gamma correction\n");
      return;
   }
   XFlush(sc.dpy);

   if(!XF86VidModeGetGammaRampSize(sc.dpy, sc.myscreen, &gl.nramp)) {
      fprintf(stderr, "Unable to get gamma ramp size\n");
      return;
   }

   if(!(allocgramps())) return;

   if(sc.usenewgamma) {

      for(i = 0; i < 3; i++) {

         float invgamma = 0.0;

         switch(i) {
         case 0: invgamma = 1. / sc.newgamma.red;
		 break;
         case 1: invgamma = 1. / sc.newgamma.green;
		 break;
         case 2: invgamma = 1. / sc.newgamma.blue;
		 break;
         }

         for(j = 0; j < gl.nramp; j++) {
	    gl.ramp[i][j] = calcgammaval(j, gl.nramp, invgamma, sc.newgamma_bri[i], sc.newgamma_con[i]);
	 }

      }

   } else {

      for(i = 0; i < 3; i++) {
         float dramp = 1. / (gl.nramp - 1);
         int fullscale = 65535 * sc.gamma_max[i];
         float invgamma, v;

         switch(i) {
         case 0: invgamma = 1. / sc.newgamma.red; break;
         case 1: invgamma = 1. / sc.newgamma.green; break;
         case 2: invgamma = 1. / sc.newgamma.blue; break;
         }

         for(j = 0; j < gl.nramp; j++) {
            sc.framp = pow(j * dramp, invgamma);

            v = (fullscale < 0) ? (65535 + fullscale * sc.framp) :
	 		       fullscale * sc.framp;
	    if(v < 0) v = 0;
	    else if(v > 65535) v = 65535;
	    gl.ramp[i][j] = (unsigned short)v;
         }
      }
   }

   gui_gamma_drawggfx();

   gui_gamma_update_gamma_gfx();

   if(!XF86VidModeSetGammaRamp(sc.dpy, sc.myscreen, gl.nramp, gl.ramp[0], gl.ramp[1], gl.ramp[2])) {
      fprintf(stderr, "Unable to set gamma correction\n");
   }
   XFlush(sc.dpy);

   freegramps();

   memcpy(&sc.currentgamma, &sc.newgamma, sizeof(XF86VidModeGamma));

   if(sc.HaveBri) {
      if(sc.sdinterface) {
         if(sc.usenewgamma) {
            sd_writenewgammabr((unsigned int)multby1000(sc.newgamma_bri[0]) + 1000,
			       (unsigned int)multby1000(sc.newgamma_bri[1]) + 1000,
			       (unsigned int)multby1000(sc.newgamma_bri[2]) + 1000,
			       (unsigned int)multby1000(sc.newgamma_con[0]) + 1000,
			       (unsigned int)multby1000(sc.newgamma_con[1]) + 1000,
			       (unsigned int)multby1000(sc.newgamma_con[2]) + 1000);
         } else {
            sd_writegammabr((unsigned int)multby1000(sc.gamma_max[0]),
			    (unsigned int)multby1000(sc.gamma_max[1]),
			    (unsigned int)multby1000(sc.gamma_max[2]));
         }
#ifdef USE_DEPRECATED_XV
      } else {
         setxvval(sc.atom_brr, multby1000(sc.gamma_max[0]));
         setxvval(sc.atom_brg, multby1000(sc.gamma_max[1]));
         setxvval(sc.atom_brb, multby1000(sc.gamma_max[2]));
#endif
      }
   }
}

static void
setgamma2(void)
{
   int i, j;

   gl.nramp = 64;

   if(!(allocgramps())) return;

   if(sc.usenewgamma) {

      for(i = 0; i < 3; i++) {

         float invgamma = 0.0;

         switch(i) {
         case 0: invgamma = 1. / sc.newgamma2r; break;
         case 1: invgamma = 1. / sc.newgamma2g; break;
         case 2: invgamma = 1. / sc.newgamma2b; break;
         }

         for(j = 0; j < gl.nramp; j++) {
	    gl.ramp[i][j] = calcgammaval(j, gl.nramp, invgamma, sc.newgamma_bri2[i], sc.newgamma_con2[i]);
	 }

      }

   } else {

      for(i = 0; i < 3; i++) {
         float dramp = 1. / (gl.nramp - 1);
         int fullscale = 65535 * sc.gamma_bri2[i];
         float invgamma, v;

         switch(i) {
         case 0: invgamma = 1. / sc.newgamma2r; break;
         case 1: invgamma = 1. / sc.newgamma2g; break;
         case 2: invgamma = 1. / sc.newgamma2b; break;
         }

         for(j = 0; j < gl.nramp; j++) {
            sc.framp = pow(j * dramp, invgamma);

            v = (fullscale < 0) ? (65535 + fullscale * sc.framp) :
	 		       fullscale * sc.framp;
	    if(v < 0) v = 0;
	    else if(v > 65535) v = 65535;
	    gl.ramp[i][j] = (unsigned short)v;
         }
      }
   }

   gui_gamma_drawggfx();

   gui_gamma_update_gamma_gfx();

   freegramps();

   sc.gamma2r = sc.newgamma2r;
   sc.gamma2g = sc.newgamma2g;
   sc.gamma2b = sc.newgamma2b;
}

static void
resetgamma(void)
{
   if(!XF86VidModeGetGamma(sc.dpy, sc.myscreen, &sc.currentgamma)) {
      fprintf(stderr, "Unable to get gamma correction\n");
      return;
   }
   XFlush(sc.dpy);


   setgamma();
   if(sc.HaveSepGC2) {
      /* Redraw for CRT2 */
      if(sc.GammaFor2) setgamma2();
   }
}

static void
endisablegamma1(Bool enable)
{
   sc.gammaflags = newgetxvval(SDC_CMD_GETGAMMASTATUS, sc.atom_sga) & ~0x01;
   if(enable) sc.gammaflags |= 0x01;
   else       sc.gammaflags &= ~0x08;
   if(sc.sdflags & SiS_SD_ISDUALHEAD) {
      sc.backupgammaflags &= 0x09;
      sc.backupgammaflags |= sc.gammaflags & ~0x09;
   }
   newsetxvval(SDC_CMD_SETGAMMASTATUS, sc.atom_sga, sc.gammaflags);
   resetgamma();
   sisctrl_vi_set_xv_gamma_status();
}

static void
endisablegamma2(Bool enable)
{
   sc.gammaflags = newgetxvval(SDC_CMD_GETGAMMASTATUS, sc.atom_sga) & ~0x02;
   if(enable) sc.gammaflags |= 0x02;
   else       sc.gammaflags &= ~0x08;
   if(sc.sdflags & SiS_SD_ISDUALHEAD) {
      sc.backupgammaflags &= 0x0a;
      sc.backupgammaflags |= sc.gammaflags & ~0x0a;
   }
   newsetxvval(SDC_CMD_SETGAMMASTATUS, sc.atom_sga, sc.gammaflags);
   resetgamma();
}

static void
endisablesepgamma2(Bool enable)
{
   sc.gammaflags = newgetxvval(SDC_CMD_GETGAMMASTATUS, sc.atom_sga) & ~0x08;
   if(enable) sc.gammaflags |= 0x08;
   if(sc.sdflags & SiS_SD_ISDUALHEAD) {
      sc.backupgammaflags &= 0x08;
      sc.backupgammaflags |= sc.gammaflags & ~0x08;
   }
   newsetxvval(SDC_CMD_SETGAMMASTATUS, sc.atom_sga, sc.gammaflags);
   if((!enable) && (!(sc.sdflags & SiS_SD_CANSETGAMMA))) {
      resetgamma();
   }
}

static void
endisablexvgamma(Bool enable)
{
   sc.gammaflags = newgetxvval(SDC_CMD_GETGAMMASTATUS, sc.atom_sga) & ~0x04;
   if(enable) sc.gammaflags |= 0x04;
   if(sc.sdflags & SiS_SD_ISDUALHEAD) {
      sc.backupgammaflags &= 0x04;
      sc.backupgammaflags |= sc.gammaflags & ~0x04;
   }
   newsetxvval(SDC_CMD_SETGAMMASTATUS, sc.atom_sga, sc.gammaflags);
   setxvgammasensitive();
   if(!(sc.gammaflags & 0x04)) resetgamma();
}

static void
setxvgamma(Atom myatom, int value)
{
   if(!(sc.sdflags & SiS_SD_SUPPORTXVGAMMA1)) return;

   if(sc.sdinterface) {
      sc.sdstruct.sdc_parm[0] = (unsigned int)multby1000(sc.xvgammared);
      sc.sdstruct.sdc_parm[1] = (unsigned int)multby1000(sc.xvgammagreen);
      sc.sdstruct.sdc_parm[2] = (unsigned int)multby1000(sc.xvgammablue);
      if(!sendsdcommand(SDC_CMD_SETXVGAMMA)) {
         fprintf(stderr, xverrorstring);
      }
#ifdef USE_DEPRECATED_XV
   } else {
      if(value < 100) value = 100;
      if(value > 10000) value = 10000;
      setxvval(myatom, value);
#endif
   }
}

static void
setxvgammared(void)
{
   setxvgamma(sc.atom_vgr, multby1000(sc.xvgammared));
}

static void
setxvgammagreen(void)
{
   setxvgamma(sc.atom_vgg, multby1000(sc.xvgammagreen));
}

static void
setxvgammablue(void)
{
   setxvgamma(sc.atom_vgb, multby1000(sc.xvgammablue));
}

/**************************************************
 *         Read/write TV position and size        *
 **************************************************/

static void
dotvposscale(unsigned int getcmd, unsigned int setcmd, Atom myatom, Bool Up, int limit)
{
   int val;

   if(sc.sdinterface && getcmd) {
      if(!sendsdcommand(getcmd)) {
         fprintf(stderr, xverrorstring);
         return;
      }
      val = sc.sdstruct.sdc_result[0] - 32768;
   } else {
      if(XvGetPortAttribute(sc.dpy, sc.xv_port_nr, myatom, &val) != Success) {
         fprintf(stderr, xverrorstring);
         return;
      }
   }
   if(Up) {
      if(val >= limit) return;
      val++;
   } else {
      if(val <= limit) return;
      val--;
   }
   if(UnlockSiSDirect(1)) {
      if(sc.sdinterface && setcmd) {
         sc.sdstruct.sdc_parm[0] = val + 32768;
         if(!sendsdcommand(setcmd)) {
            fprintf(stderr, xverrorstring);
            return;
         }
      } else {
         if(XvSetPortAttribute(sc.dpy, sc.xv_port_nr, myatom, val) != Success) {
            fprintf(stderr, xverrorstring);
         }
         XFlush(sc.dpy);
      }
      UnlockSiSDirect(0);
   }
}

static void
tvdefaultposscale(void)
{
   if(UnlockSiSDirect(1)) {

      if(!sd_settvposscale(SDC_CMD_SETTVXPOS, sc.atom_tvx, 0)) {
         fprintf(stderr, xverrorstring);
      }
      if(!sd_settvposscale(SDC_CMD_SETTVYPOS, sc.atom_tvy, 0)) {
         fprintf(stderr, xverrorstring);
      }

      if(vbsupporttvsize()) {
         if(!sd_settvposscale(SDC_CMD_SETTVXSCALE, sc.atom_txs, 0)) {
            fprintf(stderr, xverrorstring);
	 }
	 if(!sd_settvposscale(SDC_CMD_SETTVYSCALE, sc.atom_tys, 0)) {
            fprintf(stderr, xverrorstring);
	 }
      }
      UnlockSiSDirect(0);
   }
}

/**************************************************
 *   Get menu indices for CRT2 type and TV std    *
 **************************************************/

/* Get CRT2 type menu index from vbflags */
static int
getcrt2typeindex(unsigned int vbflags)
{
  int index;

  if(vbflags & CRT2_LCD)             index = C2_MI_TV_LCD;
  else if(vbflags & CRT2_VGA)        index = C2_MI_TV_VGA;
  else if(vbflags & CRT2_TV) {
     if(vbsupportsettvtype()) {
        if((vbflags & (TV_SVIDEO | TV_AVIDEO)) == (TV_SVIDEO | TV_AVIDEO)) index = C2_MI_TV_CVSV;
	else if(vbflags & TV_SVIDEO)   index = C2_MI_TV_SVHS;
        else if(vbflags & TV_AVIDEO)   index = C2_MI_TV_CVBS;
        else if(vbflags & TV_SCART)    index = C2_MI_TV_SCART;
	else if(vbflags & TV_HIVISION) index = C2_MI_TV_HIVI;
	else if(vbflags & TV_YPBPR)    index = C2_MI_TV_YPBPR;
	else		               index = C2_MI_TV_SVHS;
     } else                            index = C2_MI_TV_TV;
  } else                               index = C2_MI_TV_OFF;

  return index;
}

/* Get TV standard menu index from vbflags */
static int
gettvtypeindex(unsigned int vbflags)
{
    int index = C2_MI_TV_PAL;

    if(!(vbflags & TV_YPBPR)) {
       if(vbflags & TV_NTSC) {
          if(vbflags & TV_NTSCJ) index = C2_MI_TV_NTSCJ;
          else index = C2_MI_TV_NTSC;
       } else {
          if(vbflags & TV_PALM) index = C2_MI_TV_PALM;
          else if(vbflags & TV_PALN) index = C2_MI_TV_PALN;
          else index = C2_MI_TV_PAL;
       }
    }
    return index;
}

/* Get TV YPbPr aspect menu index from vbflags */
static int
gettvasindex(unsigned int vbflags)
{
    int index = C2_MI_AS_43;

    if(vbflags & TV_YPBPR) {
       if((vbflags & TV_YPBPRAR) == TV_YPBPR43)        index = C2_MI_AS_43;
       else if((vbflags & TV_YPBPRAR) == TV_YPBPR43LB) index = C2_MI_AS_43LB;
       else if((vbflags & TV_YPBPRAR) == TV_YPBPR169)  index = C2_MI_AS_169;
    }

    return index;
}

/* Get TV YPbPr signal type menu index from vbflags */
static int
gettvstindex(unsigned int vbflags)
{
    int index = C2_MI_ST_525I;

    if(vbflags & TV_YPBPR) {
       if(vbflags & TV_YPBPR525I)       index = C2_MI_ST_525I;
       else if(vbflags & TV_YPBPR525P)  index = C2_MI_ST_525P;
       else if((vbflags & TV_YPBPR625I) && (sc.sd2flags & SiS_SD2_SUPPORT625I)) index = C2_MI_ST_625I;
       else if((vbflags & TV_YPBPR625P) && (sc.sd2flags & SiS_SD2_SUPPORT625P)) index = C2_MI_ST_625P;
       else if(vbflags & TV_YPBPR750P)  index = C2_MI_ST_750P;
       else if(vbflags & TV_YPBPR1080I) index = C2_MI_ST_1080I;
    }

    return index;
}

/**************************************************
 *             GUI abstraction layer              *
 **************************************************/

/**************** Displaymode page ***************/

/* Set Displaymode history (currently selected item) */
static int
setdmhistory(void)
{
    int index = finddmhistory();

    gui_dm_set_current_entry(index);

    return index;
}

int
sisctrl_dm_get_current_mode_index(void)
{
    return finddmhistory();
}

#ifdef HAVE_RANDR
static void
updateleaveunchanged(void)
{
   static const char *myrotations[] = {
   	"0",
   	"90",
   	"180",
   	"270"
   };
   static const int temparray[] = {
   	0, 1, 2, 0, 3
   };

   updaterandrconfig();
   updaterandrsizes();
   sc.rrcurrent_size = XRRConfigCurrentConfiguration(sc.sconf, &sc.rrcurrent_rotation);
   sc.rrcurrent_rate = XRRConfigCurrentRate(sc.sconf);
   sprintf(sc.leaveunchangedtext,
       (sc.supprotations & 0x0e) ? "Keep %dx%d" : "Leave unchanged at %dx%d",
       sc.rrsizes[sc.rrcurrent_size].width,
       sc.rrsizes[sc.rrcurrent_size].height/*,
       sc.rrcurrent_size*/);

   if(sc.supprotations & 0x0e) {
      sprintf(sc.currentrottext, "Keep %s\302\260",
          myrotations[temparray[(sc.rrcurrent_rotation & 0x0f) >> 1]]);
   }

   gui_dm_update_leaveunchanged(sc.leaveunchangedtext, sc.currentrottext);
}

void
sisctrl_dm_update_leaveunchanged(void)
{
   updateleaveunchanged();
}
#endif

unsigned short
sisctrl_dm_get_modefield_filtered(int index)
{
    unsigned short field = sc.modecheckfield[index];

    if(!(sc.sdflags & SiS_SD_SUPPORTLCDA))
       field &= ~SiS_CRT2_CRT1LCDA;

    if(!(field & SiS_CRT2_TV))
       field &= ~(SiS_CRT2_TVPAL  |
		  SiS_CRT2_TVNTSC |
		  SiS_CRT2_TVPALM |
		  SiS_CRT2_TVPALN);
    else if(!(sc.sdflags & SiS_SD_SUPPORTPALMN))
       field &= ~(SiS_CRT2_TVPALM | SiS_CRT2_TVPALN);

    if(!vbsupportsvga2())
       field &= ~SiS_CRT2_VGA2;

    return field;
}

/* Add/update crt2 type icons to display mode field list */
void
sisctrl_set_crt2_types_for_modelist(void)
{
    int i, j;
    unsigned short field;

    gui_dm_create_gfx();

    for(i = 0, j = 0; i < SISCTRL_MAXMODES; i++) {

       if(gl.modetextptr[i]) {

          field = sisctrl_dm_get_modefield_filtered(i);

	  gui_dm_set_mode_list_attributes(j, field);

	  j++;

       }
    }
}

unsigned short
sisctrl_dm_get_modefield(int index)
{
    return sc.modecheckfield[index];
}

static void
sisctrl_rebuild_mode_and_size_list(void)
{
    if(sc.guimode) {
       gui_dm_remove_mode_list_and_randr_sizes();
    }

    freemodestrings();

    freemodelines();

    if(!XF86VidModeGetAllModeLines(sc.dpy, sc.myscreen, &sc.vmmodecount, &sc.vmmodelines)) {
       fprintf(stderr, "Failed to read modelines. Panic.\n");
       /* What now */
    }

    rebuildmodestrings();

    rebuildmodefieldlist();

#ifdef HAVE_RANDR
    if(sc.haverandr) {
       if(sc.guimode) {
          build_randr_sizes();
          gl.numrandrsizes = sc.rrnsize;
          gl.supportedrots = sc.supprotations;
          updateleaveunchanged();
       } else {
          updaterandrsizes();
       }
    }
#endif

   sc.backupmodeindex = sc.currentmodeindex = gl.startmodeindex = finddmhistory();

   if(sc.guimode) {
      gui_dm_rebuild_mode_list_and_randr_sizes();

      setdmhistory();
   }
}

/******************* CRT1 page ******************/

/* Select & show crt1 tab variant (with or without LCDA) */
static void
showcrt1tab(void)
{
   int variant = 0;

   if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!(vbhavevb()))) return;

   if(sc.sdflags & SiS_SD_SUPPORTLCDA) variant = 1;

   gui_crt1_show_variant(variant);
}

/* Set CRT1 on/off/lcd history (currently selected item) */
int
sisctrl_get_current_crt1type_index(void)
{
   int index = sc.CRT1isoff ? C1_MI_OFF : C1_MI_VGA;

   if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!(vbhavevb()))) return index;

   if(sc.sdflags & SiS_SD_SUPPORTLCDA) {
      if(sc.currentvbflags & CRT1_LCDA) index = C1_MI_LCDA;
   }
   return index;
}

static void
setcrt1history()
{
    int index;

    if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!(vbhavevb()))) return;

    index = sisctrl_get_current_crt1type_index();

    gui_crt1_set_type_menu_index(index);

    gui_crt1_update_gfx();
}


/******************* CRT2 page ******************/

/* Hide all CRT2 types in menu */
static void
hidecrt2typemenu(void)
{
    if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!(vbhavevb()))) return;

    gui_crt2_hide_crt2_menu_items();
}

/* Show CRT2 type items in menu (depending on hw and detection) */
static void
showcrt2typemenu(unsigned int vbflags, Bool detectedonly)
{
    Bool showsometv = FALSE, showsvid = FALSE, showcvbs = FALSE;

    if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!(vbhavevb()))) return;

    if((vbflags & CRT2_LCD) || (sc.currentvbflags & CRT2_LCD)) {
       gui_crt2_show_crt2_menu_item(C2_MI_TV_LCD);
    }
    if(vbsupportstv()) {
       if((vbflags & CRT2_TV) || (sc.currentvbflags & CRT2_TV) || (!detectedonly)) {
          if(vbsupportsettvtype()) {
	     if((vbflags & TV_SVIDEO) || (sc.currentvbflags & TV_SVIDEO) || (!detectedonly)) {
	        gui_crt2_show_crt2_menu_item(C2_MI_TV_SVHS);
	        showsometv = TRUE;
	        showsvid = TRUE;
	     }
	     if((vbflags & TV_AVIDEO) || (sc.currentvbflags & TV_AVIDEO) || (!detectedonly)) {
                gui_crt2_show_crt2_menu_item(C2_MI_TV_CVBS);
	        showsometv = TRUE;
	        showcvbs = TRUE;
	     }
	     if(showsvid && showcvbs) {
	        gui_crt2_show_crt2_menu_item(C2_MI_TV_CVSV);
	     }
	     if(sc.sdflags & SiS_SD_SUPPORTHIVISION) {
	        if((vbflags & TV_HIVISION) || (sc.currentvbflags & TV_HIVISION) || (!detectedonly)) {
		   gui_crt2_show_crt2_menu_item(C2_MI_TV_HIVI);
		   showsometv = TRUE;
		}
	     }
	     if(sc.sdflags & SiS_SD_SUPPORTYPBPR) {
	        if((vbflags & TV_YPBPR) || (sc.currentvbflags & TV_YPBPR) || (!detectedonly)) {
		   gui_crt2_show_crt2_menu_item(C2_MI_TV_YPBPR);
		   showsometv = TRUE;
		}
	     }
	     if((vbsupportsscart()) && (sc.sdflags & SiS_SD_VBHASSCART)) {
                if((vbflags & TV_SCART) || (sc.currentvbflags & TV_SCART) || (!detectedonly)) {
	           gui_crt2_show_crt2_menu_item(C2_MI_TV_SCART);
		   showsometv = TRUE;
	        }
             }
	     if(detectedonly && (!showsometv)) {  /* TV set, but neither SVIDEO, CVBS or SCART -> show all */
	        gui_crt2_show_crt2_menu_item(C2_MI_TV_SVHS);
	        gui_crt2_show_crt2_menu_item(C2_MI_TV_CVBS);
	        gui_crt2_show_crt2_menu_item(C2_MI_TV_CVSV);
	        if((vbsupportsscart()) &&
	           (sc.sdflags & SiS_SD_VBHASSCART)) {
	           gui_crt2_show_crt2_menu_item(C2_MI_TV_SCART);
	        }
		if(sc.sdflags & SiS_SD_SUPPORTHIVISION) {
		   gui_crt2_show_crt2_menu_item(C2_MI_TV_HIVI);
		}
		if(sc.sdflags & SiS_SD_SUPPORTYPBPR) {
		   gui_crt2_show_crt2_menu_item(C2_MI_TV_YPBPR);
		}
	     }
          } else {
	     gui_crt2_show_crt2_menu_item(C2_MI_TV_TV);
          }
       }
    }
    if((vbsupportsvga2()) && (!(sc.sdflags & SiS_SD_VBHASSCART))) {
       if((vbflags & CRT2_VGA) || (sc.currentvbflags & CRT2_VGA) || (!detectedonly)) {
	  gui_crt2_show_crt2_menu_item(C2_MI_TV_VGA);
       }
    }

    if(!sc.CRT1isoff) {
       gui_crt2_show_crt2_menu_item(C2_MI_TV_OFF);
    }

    gui_crt2_finished_show_crt2_menu_items();
}

/* Set CRT2 type history (currently selected item) */
static void
setcrt2typehistory(unsigned int vbflags)
{
    int index = 0;

    if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!(vbhavevb()))) return;

    index = getcrt2typeindex(vbflags);

    gui_crt2_set_type_menu_index(index);

    gui_crt2_update_gfx();
}

/* Callbacks from GUI part */


/* Set LCD menu history (currently selected item) */
int
sisctrl_get_lcd_scaling_history(void)
{
   int index = CO_MI_SC_ON;

   if(sc.sdflags & SiS_SD_ISDUALHEAD) return index;
   if(!(sc.sdflags & SiS_SD_SUPPORTSCALE)) return index;

   if(sc.lcdscaler & 0x01)      index = CO_MI_SC_DEFAULT;
   else if(sc.lcdscaler & 0x02) index = CO_MI_SC_OFF;

   return index;
}

int
sisctrl_get_lcd_center_history(void)
{
   int index = CO_MI_CE_DEFAULT;

   if(sc.sdflags & SiS_SD_ISDUALHEAD) return index;
   if(!(sc.sdflags & SiS_SD_SUPPORTSCALE)) return index;

   if(sc.sdflags & SiS_SD_SUPPORTCENTER) {
      if(sc.lcdscaler & 0x04)      index = CO_MI_CE_DEFAULT;
      else if(sc.lcdscaler & 0x08) index = CO_MI_CE_CENTER;
      else                         index = CO_MI_CE_CENTEROFF;
   }
   return index;
}

int
sisctrl_get_overscan_history(void)
{
   int index = 0;

   if(!(sc.sdflags & SiS_SD_SUPPORTSOVER)) {
      if(sc.choverscanstatus == 3) sc.choverscanstatus = 2;
   }

   switch(sc.choverscanstatus) {
      case 0: index = C2_MI_OV_BIOS;      break;
      case 1: index = C2_MI_OV_UNDERSCAN; break;
      case 2: index = C2_MI_OV_OVERSCAN;  break;
      case 3: index = C2_MI_OV_SUPEROVERSCAN;
   }
   return index;
}

/* End of callbacks */

static void
setlcdhistory(void)
{
    if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!(vbhavevb()))) return;
    if(!(sc.sdflags & SiS_SD_SUPPORTSCALE)) return;

    gui_crt2_set_scaling_menu_index(sisctrl_get_lcd_scaling_history());

    if(sc.sdflags & SiS_SD_SUPPORTLCDA) {
       gui_crt1_set_scaling_menu_index(sisctrl_get_lcd_scaling_history());
    }

    if(sc.sdflags & SiS_SD_SUPPORTCENTER) {
       gui_crt2_set_centering_menu_index(sisctrl_get_lcd_center_history());
       if(sc.sdflags & SiS_SD_SUPPORTLCDA) {
          gui_crt1_set_centering_menu_index(sisctrl_get_lcd_center_history());
       }
    }
}

/* Set Signal Type menu history (currently selected item) */
static void
settvsthistory(unsigned int vbflags)
{
    int index = C2_MI_ST_525I;

    if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!(vbhavevb()))) return;
    if(!(sc.sdflags & SiS_SD_SUPPORTYPBPR)) return;

    index = gettvstindex(vbflags);

    gui_crt2_set_signal_menu_index(index);
}

/* Set Aspect Ratio menu history (currently selected item) */
static void
settvashistory(unsigned int vbflags)
{
    int index = 0;

    if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!(vbhavevb()))) return;
    if(!(sc.sdflags & SiS_SD_SUPPORTYPBPRAR)) return;

    index = gettvasindex(vbflags);

    gui_crt2_set_aspect_menu_index(index);
}

/* Set Chrontel-Overscan menu history (currently selected item) */
static void
settvoverhistory(void)
{
    if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!(vbhavevb()))) return;
    if(!vbsupportsoverscan()) return;

    gui_crt2_set_overscan_menu_index(sisctrl_get_overscan_history());
}

static void
settvtypehistory(unsigned int vbflags)
{
    int index = 0;

    if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!(vbhavevb()))) return;

    index = gettvtypeindex(vbflags);

    gui_crt2_set_tvstd_menu_index(index);

    if(vbsupportsoverscan()) {
       if(gui_crt2_get_overscan_menu_index() == C2_MI_OV_SUPEROVERSCAN) {
          if((index != C2_MI_TV_PAL) || (!(sc.sdflags & SiS_SD_SUPPORTSOVER))) {
	     gui_crt2_set_overscan_menu_index(C2_MI_OV_OVERSCAN);
	  }
       }
    }
}

/* Callbacks for GUI part */

void
sisctrl_crt2_show_initial_crt2_type_menu(void)
{
   showcrt2typemenu(sc.detecteddevices, TRUE);
}

void
sisctrl_crt2_set_initial_crt2_type_history(void)
{
   setcrt2typehistory(sc.currentvbflags);
}


void
sisctrl_crt2_set_initial_tv_std(void)
{
   settvtypehistory(sc.currentvbflags);
}

void
sisctrl_crt2_set_initial_signal_type(void)
{
   settvsthistory(sc.currentvbflags);
}

void
sisctrl_crt2_set_initial_aspect_ratio(void)
{
   settvashistory(sc.currentvbflags);
}

/* End of callbacks */


/* update crt1 and crt2 pages */
static void
updatecrt1crt2(void)
{
    showcrt1tab();
    setcrt1history();
    hidecrt2typemenu();
    if((!(sc.sdflags & SiS_SD_ISDUALHEAD)) && (vbhavevb())) {
       if(gui_crt2_show_detected_devices_checked()) {
          showcrt2typemenu(sc.detecteddevices, TRUE);
       } else {
          showcrt2typemenu(sc.detecteddevices, FALSE);
       }
    }
    setcrt2typehistory(sc.currentvbflags);
    gui_crt2_show_tv_std_and_overscan_menus();
    settvtypehistory(sc.currentvbflags);
    gui_crt2_show_hide_signal_menu();
    gui_crt2_show_hide_aspect_menu();
    settvsthistory(sc.currentvbflags);
    settvashistory(sc.currentvbflags);
    gui_crt2_show_hide_overscan_menu();
    settvoverhistory();
    gui_show_lcd_menus(1);
    gui_show_lcd_menus(2);
    setlcdhistory();
    sc.currentmodeindex = setdmhistory();
#ifdef HAVE_RANDR
    if(sc.haverandr) {
       updateleaveunchanged();
    }
#endif
    gui_gamma_update_gamma_pattern_menu();
}

/****************** Gamma page ******************/

/* Set current gamma values in spinbuttons */
void
sisctrl_gamma_set_all_spinvalues(void)
{
    double s1, s2, s3, s4, s5, s6, s7 = 0.0, s8 = 0.0, s9 = 0.0;

    if(!sc.GammaFor2) {
       s1 = sc.currentgamma.red;
       s2 = sc.currentgamma.green;
       s3 = sc.currentgamma.blue;
       if(sc.usenewgamma) {
          s4 = sc.newgamma_bri[0];
          s5 = sc.newgamma_bri[1];
          s6 = sc.newgamma_bri[2];
          s7 = sc.newgamma_con[0];
          s8 = sc.newgamma_con[1];
          s9 = sc.newgamma_con[2];
       } else {
          s4 = sc.gamma_max[0];
          s5 = sc.gamma_max[1];
          s6 = sc.gamma_max[2];
       }
   } else {
       s1 = sc.gamma2r;
       s2 = sc.gamma2g;
       s3 = sc.gamma2b;
       if(sc.usenewgamma) {
          s4 = sc.newgamma_bri2[0];
          s5 = sc.newgamma_bri2[1];
          s6 = sc.newgamma_bri2[2];
          s7 = sc.newgamma_con2[0];
          s8 = sc.newgamma_con2[1];
          s9 = sc.newgamma_con2[2];
       } else {
          s4 = sc.gamma_bri2[0];
          s5 = sc.gamma_bri2[1];
          s6 = sc.gamma_bri2[2];
       }
   }
   gui_gamma_set_all_spin_values(s1, s2, s3, s4, s5, s6, s7, s8, s9);
}

static void
showhidesepgstuff(void)
{
    int index;

    sc.GammaFor2 = FALSE;

    if(!sc.HaveSepGC2) return;

    index = gui_gamma_get_active_separate_menu_item();

    if((sc.gammaflags & 0x03) == 0x03) {
       gui_gamma_show_separate_gamma_menu();
       sc.sepgammahidden = FALSE;
       if(sc.gammaflags & 0x08) {
          if(index == 0) {
             index = 1;
             gui_gamma_set_active_separate_menu_item(index);
          }
          if(index == 2) sc.GammaFor2 = TRUE;
          sisctrl_gamma_set_all_spinvalues();
       } else {
          gui_gamma_set_active_separate_menu_item(0);
       }
    } else {
       gui_gamma_hide_separate_gamma_menu();
       if((index) && (!sc.sepgammahidden) && (sc.gammaflags & 0x02)) {
	  sc.newgamma.red = sc.gamma2r;
	  sc.newgamma.green = sc.gamma2g;
	  sc.newgamma.blue = sc.gamma2b;
	  sc.newgamma_bri[0] = sc.newgamma_bri2[0];
	  sc.newgamma_bri[1] = sc.newgamma_bri2[1];
	  sc.newgamma_bri[2] = sc.newgamma_bri2[2];
	  sc.newgamma_con[0] = sc.newgamma_con2[0];
	  sc.newgamma_con[1] = sc.newgamma_con2[1];
	  sc.newgamma_con[2] = sc.newgamma_con2[2];
	  sc.gamma_max[0] = sc.gamma_bri2[0];
	  sc.gamma_max[1] = sc.gamma_bri2[1];
	  sc.gamma_max[2] = sc.gamma_bri2[2];
	  setgamma();
       }

       sisctrl_gamma_set_all_spinvalues();
       sc.sepgammahidden = TRUE;
    }
}

/* Set gamma-enabled button status */
void
sisctrl_gamma_set_status(void)
{
    if(sc.sdflags & SiS_SD_ISDEPTH8) {
       gui_gamma_set_gamma_enabled_check_box_stati(1, 1);
    } else {
       int b1 = (sc.gammaflags & 0x01) ? 1 : 0;
       int b2 = 0;
       if(vbsupportgamma2()) {
          if(sc.gammaflags & 0x02) b2 = 1;
       }
       gui_gamma_set_gamma_enabled_check_box_stati(b1, b2);
       if(sc.HaveSepGC2) {
          showhidesepgstuff();
       }
    }
}

void
sisctrl_gamma_set_sat_crt1(int value)
{
   newsetxvval(SDC_CMD_SETCRT1SATGAIN, 0, value);
}

int
sisctrl_gamma_get_sat_crt1(void)
{
   return newgetxvval(SDC_CMD_GETCRT1SATGAIN, 0);
}

void
sisctrl_gamma_set_sat_crt2(int value)
{
   newsetxvval(SDC_CMD_SETCRT2SATGAIN, 0, value);
}

int
sisctrl_gamma_get_sat_crt2(void)
{
   return newgetxvval(SDC_CMD_GETCRT2SATGAIN, 0);
}

void
sisctrl_gamma_get_monitor_gamma(int crtnum)
{
   float res = 0.0;
   if(sc.sdinterface) {
      if(sendsdcommand(crtnum ? SDC_CMD_GETMONGAMMACRT2 : SDC_CMD_GETMONGAMMACRT1)) {
         if(sc.sdstruct.sdc_result[0]) {
            res = (float)sc.sdstruct.sdc_result[0] / 1000.0;
	 }
      }
   }
   if(crtnum)
      gl.crt2mongamma = res;
   else
      gl.crt1mongamma = res;
}

/******************** TV page ********************/

void
sisctrl_tv_antiflicker_set_history(void)
{
   int index = 0, enable = 0;

   if(vbissisbridge()) {
      enable = 1;
      index = newgetxvval(SDC_CMD_GETTVANTIFLICKER, sc.atom_taf);
      /* default to "adaptive": */
      if(index < TV_AF_OFF || index > TV_AF_ADAPTIVE) index = TV_AF_ADAPTIVE;
      gui_tv_antiflicker_enable_and_set_menu_index(enable, index);
   }
}

static void
setsistvcolcalib(Atom myatom, Bool IsCoarse)
{
   int enable = 1;
   int val = 0;

   if(!(vbissisbridge())) return;

   val = sd_readcolorcalib(myatom, IsCoarse);

   if(IsCoarse) {
      gui_tv_colorcalib_c_enable_and_set_value(enable, (double)val);
   } else {
      gui_tv_colorcalib_f_enable_and_set_value(enable, (double)val);
   }

}

void
sisctrl_tv_col_calib_set_c_value(void)
{
   setsistvcolcalib(sc.atom_coc, TRUE);
}

void
sisctrl_tv_col_calib_set_f_value(void)
{
   setsistvcolcalib(sc.atom_cof, FALSE);
}

void
sisctrl_tv_saturation_set_value(void)
{
   int val = 0, enable = 1;

   if(!(vbsupporttvsaturation())) return;

   val = newgetxvval(SDC_CMD_GETTVSATURATION, sc.atom_tsa);
   if(val == -1) {
      enable = 0;
      val = 0;
   }

   gui_tv_saturation_enable_and_set_value(enable, (double)val);
}

void
sisctrl_tv_edgeenhance_set_value(void)
{
   int val = 0, enable = 1;

   if(!(vbsupporttvedgeenhance())) return;

   val = newgetxvval(SDC_CMD_SETTVEDGEENHANCE, sc.atom_tee);
   if(val == -1) {
      enable = 0;
      val = 0;
   }

   gui_tv_edgeenhance_enable_and_set_value(enable, (double)val);
}

void
sisctrl_tv_c_filter_set_value(void)
{
   int val = 1, enable = 1;

   if(!(vbissisbridge())) return;

   val = newgetxvval(SDC_CMD_GETTVCFILTER, sc.atom_cfi);
   if(val == -1) {
      enable = 0;
      val = 1;
   }
   gui_tv_filter_c_enable_and_set_value(enable, val);
}

void
sisctrl_tv_y_filter_set_value(void)
{
    int enable = 1, val = 1;

    if(!(vbissisbridge())) return;

    if(sc.currentvbflags & (TV_PALM | TV_PALN)) {
       enable = 0;
    } else {
       val = newgetxvval(SDC_CMD_GETTVYFILTER, sc.atom_yfi);
    }
    gui_tv_filter_y_enable_and_set_value(enable, val);
}

static void
setchtvvalue(int slider, unsigned int cmd, Atom myatom)
{
   int val = 0, enable = 1;

   if(vbischrontel()) {
      val = newgetxvval(cmd, myatom);
      if(val == -1) enable = val = 0;
      gui_tv_chrontel_slider_set_value(slider, enable, (double)val);
   }
}

void
sisctrl_tv_ch_contrast_set_value(void)
{
   setchtvvalue(TV_CH_SLI_CONTRAST, SDC_CMD_GETTVCHCONTRAST, sc.atom_tco);
}

void
sisctrl_tv_ch_textenhance_set_value(void)
{
   setchtvvalue(TV_CH_SLI_TEXTENHANCE, SDC_CMD_GETTVCHTEXTENHANCE, sc.atom_tte);
}

void
sisctrl_tv_ch_chromaff_set_value(void)
{
   setchtvvalue(TV_CH_SLI_CHROMAFF, SDC_CMD_GETTVCHCHROMAFLICKERFILTER, sc.atom_tcf);
}

void
sisctrl_tv_ch_lumaff_set_value(void)
{
   setchtvvalue(TV_CH_SLI_LUMAFF, SDC_CMD_GETTVCHLUMAFLICKERFILTER, sc.atom_tlf);
}

void
sisctrl_tv_ch_cvbscol_set_value(void)
{
   int val = 1, enable = 1;

   if(vbischrontel()) {
      val = newgetxvval(SDC_CMD_GETTVCHCVBSCOLOR, sc.atom_ccc);
      if(val == -1) {
         enable = 0;
         val = 1;
      }
      gui_tv_chrontel_cvbscolor_set_value(enable, val);
   }
}

/* Update the entire TV page */
static void
updatealltv(void)
{
   sisctrl_tv_antiflicker_set_history();
   sisctrl_tv_saturation_set_value();
   sisctrl_tv_edgeenhance_set_value();
   sisctrl_tv_col_calib_set_c_value();
   sisctrl_tv_col_calib_set_f_value();
   sisctrl_tv_c_filter_set_value();
   sisctrl_tv_y_filter_set_value();
   sisctrl_tv_ch_contrast_set_value();
   sisctrl_tv_ch_textenhance_set_value();
   sisctrl_tv_ch_chromaff_set_value();
   sisctrl_tv_ch_lumaff_set_value();
   sisctrl_tv_ch_cvbscol_set_value();
}

/* Video page */

static void
setvideorangevalue(int slider, unsigned int cmd, Atom viatom)
{
   int val = newgetrealxvval(cmd, viatom);

   gui_vi_slider_set_value(slider, (double)val);
}

void
sisctrl_vi_set_contrast_value(void)
{
   setvideorangevalue(VI_SLI_CONTRAST, sc.xvisglobal ? 0 : SDC_CMD_GETXVCONTRAST, sc.atom_con);
}

void
sisctrl_vi_set_brightness_value(void)
{
   setvideorangevalue(VI_SLI_BRIGHTNESS, sc.xvisglobal ? 0 : SDC_CMD_GETXVBRIGHTNESS, sc.atom_bri);
}

void
sisctrl_vi_set_saturation_value(void)
{
   setvideorangevalue(VI_SLI_SATURATION, sc.xvisglobal ? 0 : SDC_CMD_GETXVSATURATION, sc.atom_sat);
}

void
sisctrl_vi_set_hue_value(void)
{
   setvideorangevalue(VI_SLI_HUE, sc.xvisglobal ? 0 : SDC_CMD_GETXVHUE, sc.atom_hue);
}

void
sisctrl_vi_set_xv_gamma_status(void)
{
    int enable = 1, active = 0;

    if(!(sc.sdflags & SiS_SD_SUPPORTXVGAMMA1)) return;

    enable = ((sc.gammaflags & 0x01) && (!(sc.sdflags & SiS_SD_ISDEPTH8))) ? 1 : 0;
    active = ((sc.gammaflags & 0x04) && (!(sc.sdflags & SiS_SD_ISDEPTH8))) ? 1 : 0;

    gui_vi_enable_and_set_xv_gamma_check(enable, active);

    setxvgammasensitive();
}

void
sisctrl_vi_set_xvgamma_spin_values(void)
{
    if(!(sc.sdflags & SiS_SD_SUPPORTXVGAMMA1)) return;

    gui_vi_set_xvgamma_spins(sc.xvgammared, sc.xvgammagreen, sc.xvgammablue);
}

void
sisctrl_vi_set_switchcrt(void)
{
   int val;

   if( (!vbhavevb()) ||
       ((sc.sdflags & SiS_SD_SUPPORT2OVL) && (!(sc.sd2flags & SiS_SD2_SUPPORT760OO))) )
      return;

   val = newgetxvval(SDC_CMD_GETXVSWITCHCRT, sc.atom_swc);

   gui_vi_set_switchcrt_active_num(val ? 2 : 1);
}

static void
enableviswitchcrt(void)
{
   if((sc.havesdflags2) && (sc.sd2flags & SiS_SD2_SUPPORT760OO)) {
      sc.sd2flags = (unsigned int)newgetxvval(SDC_CMD_GETSD2FLAGS, sc.atom_gsf2);
      if(sc.sd2flags & SiS_SD2_SIS760ONEOVL) {
         gui_vi_enable_switchcrt(1);
      } else {
         gui_vi_enable_switchcrt(0);
      }
   }
}

unsigned long
sisctrl_vi_get_colorkey(void)
{
   unsigned long tmp;

   tmp = newgetxvval(SDC_CMD_GETXVCOLORKEY, sc.atom_col);
   return tmp;
}

/*******************************************/
/*        Signal handler callbacks         */
/*******************************************/

/* RandR handler */

#ifdef HAVE_RANDR
void
sisctrl_xrandreventhandler(void)
{
   updateleaveunchanged();
   /* Even when we ourselves change randr config,
    * we overwrite our backups. In order to have
    * "Reset" do something meaningful, we need to
    * overwrite the "backups" again after the switch.
    */
   sc.rrrotation_backup = sc.rrcurrent_rotation;
   sc.rrsize_backup = sc.rrcurrent_size;
   sc.rrrate_backup = sc.rrcurrent_rate;
}
#endif

/* Display mode page */

static Bool
getmfbdetails(int index)
{

   /* Do not need to unlock */

   sc.sdstruct.sdc_parm[0] = sc.vmmodelines[index]->dotclock;
   sc.sdstruct.sdc_parm[1] = sc.vmmodelines[index]->hdisplay;
   sc.sdstruct.sdc_parm[2] = sc.vmmodelines[index]->hsyncstart;
   sc.sdstruct.sdc_parm[3] = sc.vmmodelines[index]->hsyncend;
   sc.sdstruct.sdc_parm[4] = sc.vmmodelines[index]->htotal;
   sc.sdstruct.sdc_parm[5] = sc.vmmodelines[index]->vdisplay;
   sc.sdstruct.sdc_parm[6] = sc.vmmodelines[index]->vsyncstart;
   sc.sdstruct.sdc_parm[7] = sc.vmmodelines[index]->vsyncend;
   sc.sdstruct.sdc_parm[8] = sc.vmmodelines[index]->vtotal;

   if(!sendsdcommand(SDC_CMD_GETMERGEDMODEDETAILS)) {
      return FALSE;
   }

   return TRUE;
}

void
sisctrl_dm_selection_changed(int index, int rotindex,
		char *leftdev, char *leftcrt, char *leftcrtnum, char *leftres, char *lefthz,
		char *rightdev, char *rightcrt, char *rightcrtnum, char *rightres, char *righthz,
		int *left_crt_num, int *right_crt_num, char *lefthz2, char *righthz2,
		Bool *cloned)
{
   int devidx, test;
   char *crt1dev, *crt1crt, *crt1crtnum, *crt1res, *crt1hz, *crt1hz2;
   char *crt2dev, *crt2crt, *crt2crtnum, *crt2res, *crt2hz, *crt2hz2;
   Bool crt1right;
   const int temparray[] = { 1, 2, 3, 0, 4 };
   char *devstrings[] = {
       "VGA",
       "LCD",
       "TV",
       ""
   };

   *leftdev = *rightdev = *leftcrt = *rightcrt = *leftcrtnum = *rightcrtnum = 0;
   *leftres = *lefthz = *rightres = *righthz = *lefthz2 = *righthz2 = 0;
   *cloned = FALSE;

   /* Do not need to unlock */
   if(!getmfbdetails(index)) {
      fprintf(stderr, xverrorstring);
      strcpy(leftdev, "n/a");
      strcpy(rightdev, "n/a");
      return;
   }

   if(sc.sdstruct.sdc_result[0] == SDC_MMODE_POS_ERROR) {
      strcpy(leftdev, "ERROR");
      strcpy(rightdev, "ERROR");
      return;
   }

#ifdef HAVE_RANDR
   if(gl.supportedrots & 0x0e) {
      if(rotindex == 0) {
         rotindex = temparray[(sc.rrcurrent_rotation & 0x0f) >> 1];
      }
   } else
#endif
      rotindex = 1;

   crt1right = FALSE;

   test = sc.sdstruct.sdc_result[0];
   if(test == SDC_MMODE_POS_CLONE) {
      test = sc.sdstruct.sdc_result[7];
   }
   switch(test) {
   case SDC_MMODE_POS_LEFTOF:
      switch(rotindex) {
      case 2:
      case 3:
         break;
      default:
         crt1right = TRUE;
      }
      break;
   case SDC_MMODE_POS_ABOVE:
      switch(rotindex) {
      case 3:
      case 4:
         break;
      default:
         crt1right = TRUE;
      }
      break;
   case SDC_MMODE_POS_RIGHTOF:
      switch(rotindex) {
      case 2:
      case 3:
         crt1right = TRUE;
         break;
      }
      break;
   case SDC_MMODE_POS_BELOW:
      switch(rotindex) {
      case 3:
      case 4:
         crt1right = TRUE;
         break;
      }
      break;
   default:
      break;
   }

   if(crt1right) {
      crt1dev = rightdev;
      crt1crt = rightcrt;
      crt1crtnum = rightcrtnum;
      crt1res = rightres;
      crt1hz = righthz;
      crt1hz2 = righthz2;
      *right_crt_num = 1;
      crt2dev = leftdev;
      crt2crt = leftcrt;
      crt2crtnum = leftcrtnum;
      crt2res = leftres;
      crt2hz = lefthz;
      crt2hz2 = lefthz2;
      *left_crt_num = 2;
   } else {
      crt2dev = rightdev;
      crt2crt = rightcrt;
      crt2crtnum = rightcrtnum;
      crt2res = rightres;
      crt2hz = righthz;
      crt2hz2 = righthz2;
      *right_crt_num = 2;
      crt1dev = leftdev;
      crt1crt = leftcrt;
      crt1crtnum = leftcrtnum;
      crt1res = leftres;
      crt1hz = lefthz;
      crt1hz2 = lefthz2;
      *left_crt_num = 1;
   }

   strcpy(crt1crt, "CRT1");
   strcpy(crt1crtnum, "1");

   if(sc.CRT1isoff) devidx = 3;
   else if(sc.currentvbflags & CRT1_LCDA) devidx = 1;
   else devidx = 0;

   strcpy(crt1dev, devstrings[devidx]);

   sprintf(crt1res, "%dx%d", sc.sdstruct.sdc_result[1], sc.sdstruct.sdc_result[2]);

   if(devidx != 1) {
      sprintf(crt1hz, "%dHz", sc.sdstruct.sdc_result[3]);
   } else {
      sprintf(crt1hz2, "%dHz", sc.sdstruct.sdc_result[3]);
   }

   strcpy(crt2crt, "CRT2");
   strcpy(crt2crtnum, "2");

   devidx = 3;
   if(sc.currentvbflags & CRT2_LCD) devidx = 1;
   else if(sc.currentvbflags & CRT2_TV) devidx = 2;
   else if(sc.currentvbflags & CRT2_VGA) devidx = 0;

   strcpy(crt2dev, devstrings[devidx]);

   sprintf(crt2res, "%dx%d", sc.sdstruct.sdc_result[4], sc.sdstruct.sdc_result[5]);

   if(devidx == 0 || devidx == 3) {
      sprintf(crt2hz, "%dHz", sc.sdstruct.sdc_result[6]);
   } else {
      sprintf(crt2hz2, "%dHz", sc.sdstruct.sdc_result[6]);
   }

   *cloned = (sc.sdstruct.sdc_result[0] == SDC_MMODE_POS_CLONE) ? TRUE : FALSE;
}

/* CRT1 page - n/a */

void
sisctrl_crt1_redetect_devices(void)
{
    unsigned int xv_val;

    if(sc.sd3flags & SiS_SD3_REDETECTCRT1) {
       newsetxvval(SDC_CMD_REDETECTCRT1DEVICES, sc.atom_rdt, 0);
       if(!sd_readulong(SDC_CMD_GETSDFLAGS, sc.atom_gsf, &xv_val)) {
          fprintf(stderr, "Redetect: Couldn't read SiS Direct flags\n");
       }
       sc.sdflags = (unsigned int)xv_val;
       if(!sd_readulong(SDC_CMD_GETDETECTEDDEVICES, sc.atom_qdd, &xv_val)) {
          fprintf(stderr, "Redetect: Couldn't read detected devices\n");
       }
       sc.detecteddevices = (unsigned int)xv_val;
       rebuildmodefieldlist();
       sisctrl_set_crt2_types_for_modelist();
       updatecrt1crt2();
       sc.didcrt1redetection = TRUE;
    }
}

void
sisctrl_crt1_activate_redetection_result(void)
{

    if(UnlockSiSDirect(1)) {

      if(!sd_writeulong(SDC_CMD_RESETCRT1STATUS, sc.atom_ct1, 0)) {
         fprintf(stderr, xverrorstring);
      }
      XFlush(sc.dpy);

      UnlockSiSDirect(0);

      if(sc.sd3flags & SiS_SD3_DYNMODELISTS) {
         sisctrl_rebuild_mode_and_size_list();
      }

      updatevbandcrt1();

      sc.didcrt1redetection = FALSE;
   }
}

/* CRT2 page */

void
sisctrl_crt2_show_detected_only_changed(void)
{
    updatecrt1crt2();
}

void
sisctrl_crt2_redetect_devices(void)
{
    unsigned int xv_val;

    if(sc.HaveRedetect) {
       newsetxvval(SDC_CMD_REDETECTCRT2DEVICES, sc.atom_rdt, 0);
       if(!sd_readulong(SDC_CMD_GETSDFLAGS, sc.atom_gsf, &xv_val)) {
          fprintf(stderr, "Redetect: Couldn't read SiS Direct flags\n");
       }
       sc.sdflags = (unsigned int)xv_val;
       if(!sd_readulong(SDC_CMD_GETDETECTEDDEVICES, sc.atom_qdd, &xv_val)) {
          fprintf(stderr, "Redetect: Couldn't read detected devices\n");
       }
       sc.detecteddevices = (unsigned int)xv_val;
       rebuildmodefieldlist();
       sisctrl_set_crt2_types_for_modelist();
       updatecrt1crt2();
    }
}

/* Gamma page */

/* CRT1-gamma toggled */
void
sisctrl_gamma_enable_1(int checked)
{
    endisablegamma1(checked ? TRUE : FALSE);
    showhidesepgstuff();
}

/* CRT2-gamma toggled */
void
sisctrl_gamma_enable_2(int checked)
{
    endisablegamma2(checked ? TRUE : FALSE);
    showhidesepgstuff();
}

/* Gamma: Menu changed to/from "Both" (prev: "Adjust separately" toggled) */
void
sisctrl_gamma_separate_changed(int checked)
{
    endisablesepgamma2(checked ? TRUE : FALSE);
    showhidesepgstuff();
    sisctrl_gamma_set_all_spinvalues();
}

/* Gamma: Menu changed (prev: Radio "separate" CRT1 <> CRT2 toggled) */
void
sisctrl_gamma_separate_toggled(int checked)
{
    if(!sc.HaveSepGC2) return;
    sc.GammaFor2 = checked ? FALSE : TRUE;
    sisctrl_gamma_set_all_spinvalues();
}

void
sisctrl_gamma_spin_red_changed(double value)
{
   if(!sc.GammaFor2) {
      memcpy(&sc.newgamma, &sc.currentgamma, sizeof(XF86VidModeGamma));
      sc.newgamma.red = value;
      if(sc.newgamma.red >= 0.1 && sc.newgamma.red <= 10.0) setgamma();
   } else {
      sc.newgamma2r = value;
      if(sc.newgamma2r >= 0.1 && sc.newgamma2r <= 10.0) {
         if(sc.sdinterface) {
	    sd_writegamma2();
#ifdef USE_DEPRECATED_XV
	 } else {
      	    setxvval(sc.atom_g2r, multby1000(sc.newgamma2r));
#endif
	 }
	 setgamma2();
      }
   }
}

void
sisctrl_gamma_spin_green_changed(double value)
{
   if(!sc.GammaFor2) {
      memcpy(&sc.newgamma, &sc.currentgamma, sizeof(XF86VidModeGamma));
      sc.newgamma.green = value;
      if(sc.newgamma.green >= 0.1 && sc.newgamma.green <= 10.0) setgamma();
   } else {
      sc.newgamma2g = value;
      if(sc.newgamma2g >= 0.1 && sc.newgamma2g <= 10.0) {
         if(sc.sdinterface) {
	    sd_writegamma2();
#ifdef USE_DEPRECATED_XV
	 } else {
      	    setxvval(sc.atom_g2g, multby1000(sc.newgamma2g));
#endif
	 }
	 setgamma2();
      }
   }
}

void
sisctrl_gamma_spin_blue_changed(double value)
{
   if(!sc.GammaFor2) {
      memcpy(&sc.newgamma, &sc.currentgamma, sizeof(XF86VidModeGamma));
      sc.newgamma.blue = value;
      if(sc.newgamma.blue >= 0.1 && sc.newgamma.blue <= 10.0) setgamma();
   } else {
      sc.newgamma2b = value;
      if(sc.newgamma2b >= 0.1 && sc.newgamma2b <= 10.0) {
         if(sc.sdinterface) {
	    sd_writegamma2();
#ifdef USE_DEPRECATED_XV
	 } else {
      	    setxvval(sc.atom_g2b, multby1000(sc.newgamma2b));
#endif
	 }
	 setgamma2();
      }
   }
}

void
sisctrl_gamma_spin_all_changed(double value)
{
   if(!sc.GammaFor2) {
      memcpy(&sc.newgamma, &sc.currentgamma, sizeof(XF86VidModeGamma));
      sc.newgamma.red = sc.newgamma.green = sc.newgamma.blue = value;
      if(sc.newgamma.red >= 0.1 && sc.newgamma.red <= 10.0) setgamma();
   } else {
      sc.newgamma2r = sc.newgamma2g = sc.newgamma2b = value;
      if(sc.newgamma2r >= 0.1 && sc.newgamma2r <= 10.0) {
         if(sc.sdinterface) {
	    sd_writegamma2();
#ifdef USE_DEPRECATED_XV
	 } else {
      	    setxvval(sc.atom_g2r, multby1000(sc.newgamma2r));
      	    setxvval(sc.atom_g2g, multby1000(sc.newgamma2g));
      	    setxvval(sc.atom_g2b, multby1000(sc.newgamma2b));
#endif
	 }
	 setgamma2();
      }
   }
}

void
sisctrl_gamma_spin_bri_red_changed(double value)
{
   if(!sc.GammaFor2) {
      if(sc.usenewgamma) {
         sc.newgamma_bri[0] = value;
      } else {
         sc.gamma_max[0] = value;
      }
      setgamma();
   } else {
      if(sc.usenewgamma) {
         sc.newgamma_bri2[0] = value;
      } else {
         sc.gamma_bri2[0] = value;
      }
      if(sc.sdinterface) {
	 sd_writegamma2();
#ifdef USE_DEPRECATED_XV
      } else {
         setxvval(sc.atom_b2r, multby1000(sc.gamma_bri2[0]));
#endif
      }
      setgamma2();
   }
}

void
sisctrl_gamma_spin_bri_green_changed(double value)
{
   if(!sc.GammaFor2) {
      if(sc.usenewgamma) {
         sc.newgamma_bri[1] = value;
      } else {
         sc.gamma_max[1] = value;
      }
      setgamma();
   } else {
      if(sc.usenewgamma) {
         sc.newgamma_bri2[1] = value;
      } else {
         sc.gamma_bri2[1] = value;
      }
      if(sc.sdinterface) {
	 sd_writegamma2();
#ifdef USE_DEPRECATED_XV
      } else {
         setxvval(sc.atom_b2g, multby1000(sc.gamma_bri2[1]));
#endif
      }
      setgamma2();
   }
}

void
sisctrl_gamma_spin_bri_blue_changed(double value)
{
   if(!sc.GammaFor2) {
      if(sc.usenewgamma) {
         sc.newgamma_bri[2] = value;
      } else {
         sc.gamma_max[2] = value;
      }
      setgamma();
   } else {
      if(sc.usenewgamma) {
         sc.newgamma_bri2[2] = value;
      } else {
         sc.gamma_bri2[2] = value;
      }
      if(sc.sdinterface) {
	 sd_writegamma2();
#ifdef USE_DEPRECATED_XV
      } else {
         setxvval(sc.atom_b2b, multby1000(sc.gamma_bri2[2]));
#endif
      }
      setgamma2();
   }
}

void
sisctrl_gamma_spin_bri_all_changed(double value)
{
   if(!sc.GammaFor2) {
      if(sc.usenewgamma) {
         sc.newgamma_bri[0] = sc.newgamma_bri[1] = sc.newgamma_bri[2] = value;
      } else {
         sc.gamma_max[0] = sc.gamma_max[1] = sc.gamma_max[2] = value;
      }
      setgamma();
   } else {
      if(sc.usenewgamma) {
         sc.newgamma_bri2[0] = sc.newgamma_bri2[1] = sc.newgamma_bri2[2] = value;
      } else {
         sc.gamma_bri2[0] = sc.gamma_bri2[1] = sc.gamma_bri2[2] = value;
      }
      if(sc.sdinterface) {
	 sd_writegamma2();
#ifdef USE_DEPRECATED_XV
      } else {
         setxvval(sc.atom_b2b, multby1000(sc.gamma_bri2[0]));
         setxvval(sc.atom_b2g, multby1000(sc.gamma_bri2[1]));
         setxvval(sc.atom_b2b, multby1000(sc.gamma_bri2[2]));
#endif
      }
      setgamma2();
   }
}

void
sisctrl_gamma_spin_con_red_changed(double value)
{
   if(!sc.GammaFor2) {
      sc.newgamma_con[0] = value;
      setgamma();
   } else {
      sc.newgamma_con2[0] = value;
      sd_writegamma2();
      setgamma2();
   }
}

void
sisctrl_gamma_spin_con_green_changed(double value)
{
   if(!sc.GammaFor2) {
      sc.newgamma_con[1] = value;
      setgamma();
   } else {
      sc.newgamma_con2[1] = value;
      sd_writegamma2();
      setgamma2();
   }
}

void
sisctrl_gamma_spin_con_blue_changed(double value)
{
   if(!sc.GammaFor2) {
      sc.newgamma_con[2] = value;
      setgamma();
   } else {
      sc.newgamma_con2[2] = value;
      sd_writegamma2();
      setgamma2();
   }
}

void
sisctrl_gamma_spin_con_all_changed(double value)
{
   if(!sc.GammaFor2) {
      sc.newgamma_con[0] = sc.newgamma_con[1] = sc.newgamma_con[2] = value;
      setgamma();
   } else {
      sc.newgamma_con2[0] = sc.newgamma_con2[1] = sc.newgamma_con2[2] = value;
      sd_writegamma2();
      setgamma2();
   }
}

/* TV page */

void
sisctrl_tv_de_clicked(void)
{
   tvdefaultposscale();
}

void
sisctrl_tv_up_clicked(void)
{
   dotvposscale(SDC_CMD_GETTVYPOS, SDC_CMD_SETTVYPOS, sc.atom_tvy, FALSE, -32);
}

void
sisctrl_tv_dn_clicked(void)
{
   dotvposscale(SDC_CMD_GETTVYPOS, SDC_CMD_SETTVYPOS, sc.atom_tvy, TRUE, 32);
}

void
sisctrl_tv_le_clicked(void)
{
   dotvposscale(SDC_CMD_GETTVXPOS, SDC_CMD_SETTVXPOS, sc.atom_tvx, FALSE, -32);
}

void
sisctrl_tv_ri_clicked(void)
{
   dotvposscale(SDC_CMD_GETTVXPOS, SDC_CMD_SETTVXPOS, sc.atom_tvx, TRUE, 32);
}

void
sisctrl_tv_xm_clicked(void)
{
   dotvposscale(SDC_CMD_GETTVXSCALE, SDC_CMD_SETTVXSCALE, sc.atom_txs, FALSE, -16);
}

void
sisctrl_tv_xp_clicked(void)
{
   dotvposscale(SDC_CMD_GETTVXSCALE, SDC_CMD_SETTVXSCALE, sc.atom_txs, TRUE, 16);
}

void
sisctrl_tv_ym_clicked(void)
{
   dotvposscale(SDC_CMD_GETTVYSCALE, SDC_CMD_SETTVYSCALE, sc.atom_tys, FALSE, -4);
}

void
sisctrl_tv_yp_clicked(void)
{
   dotvposscale(SDC_CMD_GETTVYSCALE, SDC_CMD_SETTVYSCALE, sc.atom_tys, TRUE, 3);
}

void
sisctrl_tp_af_changed(int index)
{
   if(index >= TV_AF_OFF && index <= TV_AF_ADAPTIVE) {
      newsetxvval(SDC_CMD_SETTVANTIFLICKER, sc.atom_taf, index);
   }
}

static void
some_range_changed(double myval, unsigned int cmd, Atom myatom, double uplimit, double downlimit)
{
   if((myval >= downlimit) && (myval <= uplimit)) {
      newsetxvval(cmd, myatom, (int)myval);
   }
}

void
sisctrl_tp_sa_changed(double value)
{
   some_range_changed(value, SDC_CMD_SETTVSATURATION, sc.atom_tsa, 15.0, 0.0);
}

void
sisctrl_tp_ee_changed(double value)
{
   some_range_changed(value, SDC_CMD_SETTVEDGEENHANCE, sc.atom_tee, 15.0, 0.0);
}

void
sisctrl_tp_col_calib_changed(double valc, double valf)
{
   unsigned int val1, val2;

   if(valc >  120.0) valc =  120.0;
   if(valc < -120.0) valc = -120.0;
   if(valf >  127.0) valf =  127.0;
   if(valf < -128.0) valf = -128.0;

   if(sc.sdinterface) {
      val1 = (unsigned int)valc + 32768;
      val2 = (unsigned int)valf + 32768;
      sd_writecolorcalib(val1, val2);
   } else {
      some_range_changed(valc, 0, sc.atom_coc, 120.0, -120.0);
      some_range_changed(valf, 0, sc.atom_cof, 127.0, -128.0);
   }
}

void
sisctrl_tp_check_cfi_toggled(int val)
{
    int myval = val ? 1 : 0;

    newsetxvval(SDC_CMD_SETTVCFILTER, sc.atom_cfi, myval);
}

void
sisctrl_tp_yfi_changed(int index)
{
    if(index >= 0 && index <= 8) {
       newsetxvval(SDC_CMD_SETTVYFILTER, sc.atom_yfi, index);
    }
}

void
sisctrl_tp_co_changed(double value)
{
    some_range_changed(value, SDC_CMD_SETTVCHCONTRAST, sc.atom_tco, 15.0, 0.0);
}

void
sisctrl_tp_te_changed(double value)
{
    some_range_changed(value, SDC_CMD_SETTVCHTEXTENHANCE, sc.atom_tte, 15.0, 0.0);
}

void
sisctrl_tp_cf_changed(double value)
{
    some_range_changed(value, SDC_CMD_SETTVCHCHROMAFLICKERFILTER, sc.atom_tcf, 15.0, 0.0);
}

void
sisctrl_tp_lf_changed(double value)
{
    some_range_changed(value, SDC_CMD_SETTVCHLUMAFLICKERFILTER, sc.atom_tlf, 15.0, 0.0);
}

void
sisctrl_tp_check_col_toggled(int val)
{
    int myval = val ? 1 : 0;
    newsetxvval(SDC_CMD_SETTVCHCVBSCOLOR, sc.atom_ccc, myval);
}

/* Xv page */

static void
some_video_range_changed(double myval, unsigned int cmd, Atom myatom, double uplimit, double downlimit)
{
   int newval;
   if((myval >= downlimit) && (myval <= uplimit)) {
      newval = (int)myval;
      if((sc.sdinterface) && (cmd)) newval += 32768;
      newsetxvval(cmd, myatom, (int)newval);
   }
}

void
sisctrl_vi_co_changed(double value)
{
   some_video_range_changed(value, sc.xvisglobal ? 0 : SDC_CMD_SETXVCONTRAST, sc.atom_con, 7.0, 0.0);
}

void
sisctrl_vi_br_changed(double value)
{
   some_video_range_changed(value, sc.xvisglobal ? 0 : SDC_CMD_SETXVBRIGHTNESS, sc.atom_bri, 127.0, -128.0);
}

void
sisctrl_vi_hu_changed(double value)
{
   some_video_range_changed(value, sc.xvisglobal ? 0 : SDC_CMD_SETXVHUE, sc.atom_hue, 7.0, -8.0);
}

void
sisctrl_vi_sa_changed(double value)
{
   some_video_range_changed(value, sc.xvisglobal ? 0 : SDC_CMD_SETXVSATURATION, sc.atom_sat, 7.0, -7.0);
}

void
sisctrl_vi_radio_changed(int value)
{
   newsetxvval(SDC_CMD_SETXVSWITCHCRT, sc.atom_swc, value ? 1 : 0);
   sisctrl_vi_set_switchcrt();
}

void
sisctrl_vi_check_gamma_toggled(int val)
{
   endisablexvgamma(val ? TRUE : FALSE);
}

void
sisctrl_vi_spin1_changed(double value)
{
   sc.xvgammared = value;
   if(sc.xvgammared >= 0.1 && sc.xvgammared <= 10.0) setxvgammared();
}

void
sisctrl_vi_spin2_changed(double value)
{
   sc.xvgammagreen = value;
   if(sc.xvgammagreen >= 0.1 && sc.xvgammagreen <= 10.0) setxvgammagreen();
}

void
sisctrl_vi_spin3_changed(double value)
{
   sc.xvgammablue = value;
   if(sc.xvgammablue >= 0.1 && sc.xvgammablue <= 10.0) setxvgammablue();
}

static void
setxvgammasensitive(void)
{
   Bool enable = ((sc.gammaflags & 0x05) == 0x05) ? 1 : 0;

   if(!(sc.sdflags & SiS_SD_SUPPORTXVGAMMA1)) return;

   if(sc.sdflags & SiS_SD_ISDEPTH8) enable = 0;

   gui_vi_enable_xvgamma_spins(enable);
}

void
sisctrl_vi_check_video_only_toggled(int val)
{
   sc.backup_dgl = val ? 1 : 0;
   if(!gui_vi_get_xv_shown()) {
      if(sc.havedglgfx) {
         newsetxvval(SDC_CMD_SETXVDISABLEGFXLR, sc.atom_dgl, sc.backup_dgl);
      }
   }
}

void
sisctrl_vi_global_changed(int val)
{
   if(!sc.xinerama || !sc.sdinterface || !sc.havexv || sc.noxvglobal) return;

   sc.xvisglobal = val ? TRUE : FALSE;
}

/* Current page */

/* ... moved to sisctrl_conf.c */


/* Info page */

void
sisctrl_ab_generate_current(void)
{
   int val, i;
   char txtbuffer[64];
   char *chip;
   const char *vbtype;
   Bool IsSiS;

   chip = getchiptype(&IsSiS);
   vbtype = getvbtype();

   gui_ab_clear_text();

   gui_ab_append_text("GPU type: ", AB_TT_BLACK);
   if(sc.havenewhwinfo) {
      sprintf(txtbuffer, "%s %s rev %d%s\n",
			IsSiS ? "SiS" : "XGI", chip, sc.chipRev, vbtype);
      gui_ab_append_text(txtbuffer, AB_TT_BLACK_BOLD);
      gui_ab_append_text("Video RAM: ", AB_TT_BLACK);
      sprintf(txtbuffer, "%dMB", sc.videoRam/1024);
      gui_ab_append_text(txtbuffer, AB_TT_BLACK_BOLD);
      if(!sc.LFBsize) gui_ab_append_text(" (shared)\n", AB_TT_BLACK_BOLD);
      else if(!sc.UMAsize) gui_ab_append_text(" (local)\n", AB_TT_BLACK_BOLD);
      else {
         sprintf(txtbuffer, " (%dMB shared, %dMB local)\n", sc.UMAsize/1024, sc.LFBsize/1024);
	 gui_ab_append_text(txtbuffer, AB_TT_BLACK_BOLD);
      }
   } else {
      sprintf(txtbuffer, "%s %s rev %d%s\n",
			IsSiS ? "SiS" : "XGI", chip, (int)(sc.hwinfo >> 24), vbtype);
      gui_ab_append_text(txtbuffer, AB_TT_BLACK_BOLD);
   }

   if(sc.havenewhwinfo) {
      gui_ab_append_text("Bus type, location: ", AB_TT_BLACK);
      sprintf(txtbuffer, "%s",
			(sc.busType == SDC_BUS_TYPE_AGP) ? "AGP" :
			   ((sc.busType == SDC_BUS_TYPE_PCIE) ? "PCI-E" :
			      ((sc.busType == SDC_BUS_TYPE_PCI) ? "PCI" :
			        ((sc.busType == SDC_BUS_TYPE_USB) ? "USB" : "UNKNOWN"))));
      gui_ab_append_text(txtbuffer, AB_TT_BLACK_BOLD);
      if(sc.busType != SDC_BUS_TYPE_USB) {
         sprintf(txtbuffer, ", PCI:%d:%d.%d",
			sc.pciBus, sc.pciDev, sc.pciFunc);
         gui_ab_append_text(txtbuffer, AB_TT_BLACK_BOLD);
      }
      gui_ab_append_text("\n", AB_TT_BLACK_BOLD);
   }

   if(sc.havenewhwinfo) {
      gui_ab_append_text("BIOS version: ", AB_TT_BLACK);
      if(sc.biosversion[0]) {
         sprintf(txtbuffer, "%s, %s data layout\n", &sc.biosversion[0],
			(sc.NewROM == 1) ? "new SiS" :
				((sc.NewROM == 2) ? "XGI" : "old SiS"));
         gui_ab_append_text(txtbuffer, AB_TT_BLACK_BOLD);
      } else {
         gui_ab_append_text("n/a\n", AB_TT_BLACK_BOLD);
      }
   }

   gui_ab_append_text("Driver version: ", AB_TT_BLACK);
   if(sc.havenewhwinfo) {
      sprintf(txtbuffer, "%lu/%02lu/%02lu-%lu\n",
			sc.driverYear+ 2000, sc.driverMonth, sc.driverDay, sc.driverRev);
   } else {
      sprintf(txtbuffer, "%lu/%02lu/%02lu-%lu\n",
      			((sc.driverversion >> 16) & 0xff) + 2000,
			(sc.driverversion >> 8) & 0xff,
			sc.driverversion & 0xff,
			sc.driverversion >> 24);
   }
   gui_ab_append_text(txtbuffer, AB_TT_BLACK_BOLD);

   if(sc.sdinterface) {
      gui_ab_append_text("SiSCtrl extension version: ", AB_TT_BLACK);
      sprintf(txtbuffer, "%u.%u\n",
			sc.scextversion_major, sc.scextversion_minor);
      gui_ab_append_text(txtbuffer, AB_TT_BLACK_BOLD);
   } else {
      gui_ab_append_text("SiSCtrl extension not found, your X driver is outdated.\n", AB_TT_BLACK);
   }

   gui_ab_append_text("SiSCtrl version: ", AB_TT_BLACK);
   gui_ab_append_text(SISVERSION, AB_TT_BLACK_BOLD);
}

/* Backup/restore functions (for Reset and confirmation dialog) */

static void
sisctrl_backup_mode_data(void)
{
   sc.backupmodeindex = sc.currentmodeindex;
#ifdef HAVE_RANDR
   if(sc.haverandr) {
      sc.rrrotation_backup = sc.rrcurrent_rotation;
      sc.rrsize_backup = sc.rrcurrent_size;
      sc.rrrate_backup = sc.rrcurrent_rate;
   }
#endif
}

static void
sisctrl_backup2_mode_data(void)
{
   sc.backupmodeindex2 = sc.currentmodeindex;
   sc.backupmodeindex3 = sc.backupmodeindex;
#ifdef HAVE_RANDR
   if(sc.haverandr) {
      sc.rrrotation_backup2 = sc.rrcurrent_rotation;
      sc.rrsize_backup2 = sc.rrcurrent_size;
      sc.rrrate_backup2 = sc.rrcurrent_rate;
      sc.rrrotation_backup3 = sc.rrrotation_backup;
      sc.rrsize_backup3 = sc.rrsize_backup;
      sc.rrrate_backup3 = sc.rrrate_backup;
   }
#endif
}

static void
sisctrl_set_backup2_mode_data(void)
{
   sc.backupmodeindex = sc.backupmodeindex2;
#ifdef HAVE_RANDR
   if(sc.haverandr) {
      sc.rrrotation_backup = sc.rrrotation_backup2;
      sc.rrsize_backup = sc.rrsize_backup2;
      sc.rrrate_backup = sc.rrrate_backup2;
   }
#endif
}

static void
sisctrl_reset_backup_mode_data(void)
{
   sc.backupmodeindex = sc.backupmodeindex3;
#ifdef HAVE_RANDR
   if(sc.haverandr) {
      sc.rrrotation_backup = sc.rrrotation_backup3;
      sc.rrsize_backup = sc.rrsize_backup3;
      sc.rrrate_backup = sc.rrrate_backup3;
   }
#endif
}

static void
sisctrl_backup_crt1_data(void)
{
   sc.backupvbflags = sc.currentvbflags;
   sc.CRT1isoffbackup = sc.CRT1isoff;
   sc.backuplcdscaler = sc.lcdscaler;
}

static void
sisctrl_backup2_crt1_data(void)
{
   sc.backupvbflags2 = sc.currentvbflags;
   sc.CRT1isoffbackup2 = sc.CRT1isoff;
   sc.backuplcdscaler2 = sc.lcdscaler;

   sc.backupvbflags3 = sc.backupvbflags;
   sc.CRT1isoffbackup3 = sc.CRT1isoffbackup;
   sc.backuplcdscaler3 = sc.backuplcdscaler;
}

static void
sisctrl_set_backup2_crt1_data(void)
{
   sc.backupvbflags = sc.backupvbflags2;
   sc.CRT1isoffbackup = sc.CRT1isoffbackup2;
   sc.backuplcdscaler = sc.backuplcdscaler2;
}

static void
sisctrl_reset_backup_crt1_data(void)
{
   sc.backupvbflags = sc.backupvbflags3;
   sc.CRT1isoffbackup = sc.CRT1isoffbackup3;
   sc.backuplcdscaler = sc.backuplcdscaler3;
}

static void
sisctrl_backup_crt2_data(void)
{
   sc.backupvbflags = sc.currentvbflags;
   sc.backupoverscan = sc.choverscanstatus;
   sc.backuplcdscaler = sc.lcdscaler;
}

static void
sisctrl_backup2_crt2_data(void)
{
   sc.backupvbflags2 = sc.currentvbflags;
   sc.backupoverscan2 = sc.choverscanstatus;
   sc.backuplcdscaler2 = sc.lcdscaler;

   sc.backupvbflags3 = sc.backupvbflags;
   sc.backupoverscan3 = sc.backupoverscan;
   sc.backuplcdscaler3 = sc.backuplcdscaler;
}

static void
sisctrl_set_backup2_crt2_data(void)
{
   sc.backupvbflags = sc.backupvbflags2;
   sc.backupoverscan = sc.backupoverscan2;
   sc.backuplcdscaler = sc.backuplcdscaler2;
}

static void
sisctrl_reset_backup_crt2_data(void)
{
   sc.backupvbflags = sc.backupvbflags3;
   sc.backupoverscan = sc.backupoverscan3;
   sc.backuplcdscaler = sc.backuplcdscaler3;
}

/* Notebook page change handles */

void
sisctrl_gl_display_mode_page_selected(void)
{
   sisctrl_backup_mode_data();

   setdmhistory();
#ifdef HAVE_RANDR
   if(sc.haverandr) {
      updateleaveunchanged();
   }
#endif
}

void
sisctrl_gl_crt1_page_selected(void)
{
   sisctrl_backup_crt1_data();
}

void
sisctrl_gl_crt1_page_deselected(void)
{
    /* On page deselection, we restore it to the current setting */
    showcrt1tab();
    setcrt1history();
    gui_show_lcd_menus(1);
    setlcdhistory();
}

void
sisctrl_gl_crt2_page_selected(void)
{
   sisctrl_backup_crt2_data();
}

void
sisctrl_gl_crt2_page_deselected(void)
{
    /* On page deselection, we restore it to the current setting */
    hidecrt2typemenu();
    if((!(sc.sdflags & SiS_SD_ISDUALHEAD)) && (vbhavevb())) {
       if(gui_crt2_show_detected_devices_checked()) {
          showcrt2typemenu(sc.detecteddevices, TRUE);
       } else {
          showcrt2typemenu(sc.detecteddevices, FALSE);
       }
    }
    setcrt2typehistory(sc.currentvbflags);
    gui_crt2_show_tv_std_and_overscan_menus();
    settvtypehistory(sc.currentvbflags);
    gui_crt2_show_hide_signal_menu();
    gui_crt2_show_hide_aspect_menu();
    settvsthistory(sc.currentvbflags);
    settvashistory(sc.currentvbflags);
    gui_crt2_show_hide_overscan_menu();
    settvoverhistory();
    gui_show_lcd_menus(2);
    setlcdhistory();
}

void
sisctrl_gl_gamma_page_selected(void)
{
   int i;

   sc.backupgammaflags = sc.gammaflags;
   sc.backup_gred = sc.newgamma.red;
   sc.backup_ggreen = sc.newgamma.green;
   sc.backup_gblue = sc.newgamma.blue;
   sc.backup_g2r = sc.newgamma2r;
   sc.backup_g2g = sc.newgamma2g;
   sc.backup_g2b = sc.newgamma2b;
   if(sc.usenewgamma) {
      for(i=0; i<3; i++) {
         sc.newgamma_bri_b[i] = sc.newgamma_bri[i];
         sc.newgamma_bri2_b[i] = sc.newgamma_bri2[i];
         sc.newgamma_con_b[i] = sc.newgamma_con[i];
         sc.newgamma_con2_b[i] = sc.newgamma_con2[i];
      }
   } else {
      for(i=0; i<3; i++) {
         sc.gamma_max_b[i] = sc.gamma_max[i];
         sc.gamma_bri2_b[i] = sc.gamma_bri2[i];
      }
   }
}

void
sisctrl_gl_tv_page_selected(void)
{
}

void
sisctrl_gl_video_page_selected(void)
{
   enableviswitchcrt();
   gui_vi_set_video_only_value((sc.backup_dgl) ? 1 : 0);
}

void
sisctrl_gl_current_page_selected(int hints)
{
   cf_generate_current(hints);
}


/* "OK", "Apply" handlers */

static void
guinoguishowerror(const char *errorstring, Bool refertoxlog)
{
   if(sc.guimode) {
      gui_showerror((char *)errorstring);
   } else {
      fprintf(stderr, errorstring);
      if(refertoxlog) {
         fprintf(stderr, ", check X log for error messages");
      }
      fprintf(stderr, "\n");
   }
}

static int
handle_apply_mode(int dm_index, int randrindex, int rotindex)
{
   Bool dorandr = FALSE, matchsize = FALSE, dosizeset = FALSE;
   int  i;
   int  checkmask = 0xff;
   int  checkresult = 0;
   int  sizetoset = 0;
#ifdef HAVE_RANDR
   Rotation rottoset = 0;
#endif
   int  ret = TRUE;
   Bool havemfbdetails = FALSE;
   int c1x, c1y, c2x, c2y, pos;

   if(sc.sdflags & SiS_SD_ISMERGEDFB) {
      if(getmfbdetails(dm_index)) {
         c1x = sc.sdstruct.sdc_result[1];
         c1y = sc.sdstruct.sdc_result[2];
         c2x = sc.sdstruct.sdc_result[4];
         c2y = sc.sdstruct.sdc_result[5];
         pos = sc.sdstruct.sdc_result[0];
         havemfbdetails = TRUE;
      }
   }

#ifdef HAVE_RANDR
   if(sc.haverandr) {
      if(randrindex > 0) {
	 dorandr = TRUE;
	 checkmask = ~0x02;
	 if(randrindex == 1) matchsize = TRUE;
	 else {
	    sizetoset = randrindex - 2;
	    dosizeset = TRUE;
	 }
      }
      if(rotindex > 0) {
         dorandr = TRUE;
         rottoset = 1 << (rotindex - 1);
      }
   }
#endif

   if(sc.guimode) {
      if((sc.sdflags & SiS_SD_ISMERGEDFB) && (sc.sd3flags & SiS_SD3_MFBALLOWOFFCL)) {
         if((!(sc.currentvbflags & CRT2_ENABLE)) || sc.CRT1isoff) {
            if(havemfbdetails && (pos != SDC_MMODE_POS_CLONE)) {
               if(!(gui_showquestion("Currently, either CRT1 or CRT2 are off. You have chosen a "
               			"non-clone mode. This will lead to inaccessible screen areas.\n\n"
               			"Switch mode anyway?"))) {
                  sc.currentmodeindex = setdmhistory();
                  return FALSE;
               }
            }
         }
      }
   }

   if((checkresult = checkmodeforcrt2(dm_index)) & checkmask) {
      checkresult &= 0xff;
      if(checkresult & 0x01) {
	 guinoguishowerror("Display mode not supported for current CRT2 type", FALSE);
      } else if(checkresult & 0x02) {
	 guinoguishowerror("Display mode too large for current desktop size", FALSE);
      } else {
	 char buffer[64];
	 sprintf(buffer, "Failed to verify mode (error code %02x)", checkresult);
	 guinoguishowerror(buffer, FALSE);
      }
      if(sc.guimode) {
	 sc.currentmodeindex = setdmhistory();
      }
      return FALSE;
   }

   if(sc.haverandr) {
#ifdef HAVE_RANDR
      Bool sizefound = FALSE;
      updaterandrconfig();
      sc.rrcurrent_size = XRRConfigCurrentConfiguration(sc.sconf, &sc.rrcurrent_rotation);
      sc.rrcurrent_rate = XRRConfigCurrentRate(sc.sconf);
      if(!rottoset) rottoset = sc.rrcurrent_rotation;
      updaterandrsizes();
      if(dorandr) {
	 if(matchsize) {
	    for(sc.rrsize = 0; sc.rrsize < sc.rrnsize; sc.rrsize++) {
	       if( (sc.rrsizes[sc.rrsize].width == sc.vmmodelines[dm_index]->hdisplay) &&
		   (sc.rrsizes[sc.rrsize].height == sc.vmmodelines[dm_index]->vdisplay) ) {
		  sizefound = TRUE;
		  sc.rrrates = XRRConfigRates(sc.sconf, sc.rrsize, &sc.rrnrate);
		  sc.rrrate = searchrrrate(sc.moderate[dm_index]);
		  break;
	       }
	    }
	 } else {
	    if(!dosizeset) sizetoset = sc.rrcurrent_size;
	    if( (sc.rrsizes[sizetoset].width >= sc.vmmodelines[dm_index]->hdisplay) &&
		(sc.rrsizes[sizetoset].height >= sc.vmmodelines[dm_index]->vdisplay) ) {
	       sc.rrsize = sizetoset;
	       sc.rrrate = 0;
	       sizefound = TRUE;
#if 0 /* Does not work. VidmodeExtension call fails if modeline < virtual */
	    } else if((havemfbdetails) &&
	              (sc.rrsizes[sizetoset].width >= c1x)  &&
	              (sc.rrsizes[sizetoset].height >= c1y) &&
	              (sc.rrsizes[sizetoset].width >= c2x)  &&
	              (sc.rrsizes[sizetoset].height >= c2y)) {
	       sc.rrsize = sizetoset;
	       sc.rrrate = 0;
	       sizefound = TRUE;
#endif
	    } else {
	       guinoguishowerror("Display mode too large for desired desktop size", FALSE);
	       ret = FALSE;
	    }
	 }
      }

      if((sizefound) && ((sc.rrcurrent_size != sc.rrsize) || (sc.rrcurrent_rotation != rottoset))) {
	 Status status = RRSetConfigFailed;
	 XRRConfigRotations(sc.sconf, &sc.rrcurrent_rotation);
	 XSelectInput(sc.dpy, sc.root, StructureNotifyMask);
	 XRRSelectInput(sc.dpy, sc.root, RRScreenChangeNotifyMask);
	 status = XRRSetScreenConfigAndRate(sc.dpy, sc.sconf, DefaultRootWindow(sc.dpy),
		       (SizeID)sc.rrsize, (Rotation)rottoset, sc.rrrate, CurrentTime);
	 if(status != RRSetConfigSuccess) {
	    fprintf(stderr, "XRRSetScreenConfigAndRate failed (status %d)\n", status);
	    if(!checkmodeforscreen(dm_index, TRUE, c1x, c1y, c2x, c2y)) {
	       guinoguishowerror("Display mode too large for current desktop size", FALSE);
	       sc.currentmodeindex = setdmhistory();
	       ret = FALSE;
	    }
	 }
	 /* Rely on GDK to do XRRUpdateConfiguration() */
	 if(sc.guimode) {
	    updateleaveunchanged();
	 }
      }
#endif
   }

   if(ret) {
      if((!dorandr) || (!matchsize)) {
         if(!checkmodeforscreen(dm_index, TRUE, c1x, c1y, c2x, c2y)) {
            guinoguishowerror("Display mode too large for current desktop size", FALSE);
            if(sc.guimode) {
	       sc.currentmodeindex = setdmhistory();
	    }
	    return FALSE;
	 }
      }
   }

   if(ret) {
      XFlush(sc.dpy);

      if(!XF86VidModeSwitchToMode(sc.dpy, sc.myscreen, sc.vmmodelines[dm_index])) {
	 guinoguishowerror("Error calling vidmode extension", FALSE);
	 ret = FALSE;
      }
      XFlush(sc.dpy);

      updatevbandcrt1();

      if(sc.guimode) {
         updatecrt1crt2();
         updatealltv();
      }

#ifdef HAVE_RANDR
      if(sc.haverandr) {
	 updaterandrconfig();
	 sc.rrcurrent_size = XRRConfigCurrentConfiguration(sc.sconf, &sc.rrcurrent_rotation);
	 sc.rrcurrent_rate = XRRConfigCurrentRate(sc.sconf);
      }
#endif
   }

   return ret;
}

int
sisctrl_checkmodeforcrtx(int crt2, int devindex, int modeindex, int stdindex, int ypindex)
{
   unsigned short fieldmask, field = sisctrl_dm_get_modefield(modeindex);

   if(crt2) {

      switch(devindex) {
      case C2_MI_TV_LCD:
         fieldmask = SiS_CRT2_LCD;
         break;
      case C2_MI_TV_TV:
      case C2_MI_TV_SVHS:
      case C2_MI_TV_CVBS:
      case C2_MI_TV_CVSV:
         switch(stdindex) {
            case C2_MI_TV_PAL:   fieldmask = SiS_CRT2_TVPAL; break;
            case C2_MI_TV_NTSC:  fieldmask = SiS_CRT2_TVNTSC; break;
            case C2_MI_TV_PALM:  fieldmask = SiS_CRT2_TVPALM; break;
            case C2_MI_TV_PALN:  fieldmask = SiS_CRT2_TVPALN; break;
            case C2_MI_TV_NTSCJ: fieldmask = SiS_CRT2_TVNTSC; break;
         }
         break;
      case C2_MI_TV_YPBPR:
         switch(ypindex) {
            case C2_MI_ST_525I:  fieldmask = SiS_CRT2_YPBPR525I; break;
	    case C2_MI_ST_525P:  fieldmask = SiS_CRT2_YPBPR525P; break;
	    case C2_MI_ST_625I:  fieldmask = SiS_CRT2_YPBPR625I; break;
	    case C2_MI_ST_625P:  fieldmask = SiS_CRT2_YPBPR625P; break;
	    case C2_MI_ST_750P:  fieldmask = SiS_CRT2_YPBPR750P; break;
	    case C2_MI_ST_1080I: fieldmask = SiS_CRT2_YPBPR1080I; break;
         }
         break;
      case C2_MI_TV_SCART:
         fieldmask = SiS_CRT2_TVPAL;
         break;
      case C2_MI_TV_HIVI:
         fieldmask = SiS_CRT2_HIVISION;
         break;
      case C2_MI_TV_VGA:
         fieldmask = SiS_CRT2_VGA2;
         break;
      default:
         return TRUE;	/* For "off" */
      }

      if(!(field & fieldmask)) return FALSE;

   } else if(devindex == C1_MI_LCDA) {

      /* For CRT1, we only need to check for LCDA; VGA supports all modes */
      if(!(field & SiS_CRT2_CRT1LCDA)) return FALSE;

   }

   return TRUE;
}

static int
getcrt2lsmenuindex(int menunum)
{
   int mode=0, index;
   int mydefaultscaling = CO_MI_SC_DEFAULT;
   int mydefaultcenter = CO_MI_CE_DEFAULT;

   if(!(sc.sdflags & SiS_SD_SUPPORTSCALE)) return (0x01 | 0x04);

   if(sc.guimode) {

      if(menunum == 2) index = gui_crt2_get_scaling_menu_index(mydefaultscaling);
      else             index = gui_crt1_get_scaling_menu_index(mydefaultscaling);
      switch(index) {
      case CO_MI_SC_DEFAULT: mode |= 0x01; break;
      case CO_MI_SC_OFF:     mode |= 0x02; break;
      }
      if(sc.sdflags & SiS_SD_SUPPORTCENTER) {
         if(menunum == 2) index = gui_crt2_get_centering_menu_index(mydefaultcenter);
	 else             index = gui_crt1_get_centering_menu_index(mydefaultcenter);
         switch(index) {
         case CO_MI_CE_DEFAULT: mode |= 0x04; break;
         case CO_MI_CE_CENTER:  mode |= 0x08; break;
         }
      }
      return(mode);

   } else {

      /* Return value read previously */
      return(sc.givenlcdscale);

   }
}

static int
check_apply_CRT1(int c1_index, int modeindex, int iscurrentmode, int ignorevbflags)
{
   unsigned int val;

   /* Read CRT1 status here again; user might have changed the mode without us
    * (Slave mode modes switch CRT1 on!)
    */
   if(!sd_readulong(SDC_CMD_GETCRT1STATUS, sc.atom_ct1, &val)) {
      fprintf(stderr, "Failed to re-read CRT1 status\n");
   } else {
      sc.CRT1isoff = val ? 0 : 1;
   }

   if(c1_index == C1_MI_OFF) {

      if(sc.sdflags & SiS_SD_ISMERGEDFB) {
         if(!(sc.sd3flags & SiS_SD3_MFBALLOWOFFCL)) {
            guinoguishowerror("CRT1 can't be switched off in MergedFB mode", FALSE);
	    if(sc.guimode) {
	       setcrt1history();
	       sc.currentmodeindex = setdmhistory();
	    }
            return FALSE;
         } else if(getmfbdetails(modeindex) && (sc.sdstruct.sdc_result[0] != SDC_MMODE_POS_CLONE)) {
            guinoguishowerror("CRT1 can only be switched off while a clone mode is active", FALSE);
	    if(sc.guimode) {
	       setcrt1history();
	       sc.currentmodeindex = setdmhistory();
	    }
            return FALSE;
         }
      }

      if(!ignorevbflags && (!(sc.currentvbflags & CRT2_ENABLE))) {
	 guinoguishowerror("CRT1 can't be switched off, no CRT2 device enabled", FALSE);
	 if(sc.guimode) {
	    setcrt1history();
	    sc.currentmodeindex = setdmhistory();
	 }
         return FALSE;
      }

   } else if(c1_index == C1_MI_VGA) {

      if(!sisctrl_checkmodeforcrtx(0, c1_index, modeindex, 0, 0)) {
         if(iscurrentmode) {
            guinoguishowerror("Current display mode not suitable for this CRT1 device type", FALSE);
         } else {
            guinoguishowerror("Display mode not suitable for CRT1=VGA", FALSE);
         }
         return FALSE;
      }

   } else {   /* LCDA */

      if(!(sc.detecteddevices & CRT2_LCD)) {
	 guinoguishowerror("Cannot set CRT1 type to LCD; no LCD detected", FALSE);
         return FALSE;
      }

      if(!(sc.sdflags & SiS_SD_SUPPORTLCDA)) {
	 guinoguishowerror("Hardware does not support LCD-via-CRT1", FALSE);
	 return FALSE;
      }

      if(!ignorevbflags && (sc.sdflags & SiS_SD_ISMERGEDFB) && (sc.currentvbflags & CRT2_LCD)) {
	 guinoguishowerror("Can't switch to LCD-via-CRT1 while CRT2 is LCD", FALSE);
	 if(sc.guimode) {
	    setcrt1history();
	    sc.currentmodeindex = setdmhistory();
	    setlcdhistory();
	 }
         return FALSE;
      }

      if(!sisctrl_checkmodeforcrtx(0, c1_index, modeindex, 0, 0)) {
         if(iscurrentmode) {
            guinoguishowerror("Current display mode not suitable for this CRT1 device type", FALSE);
         } else {
            guinoguishowerror("Display mode not suitable for CRT1=LCD", FALSE);
         }
         return FALSE;
      }

   }

   return TRUE;

}

static int
handle_apply_CRT1(int c1_index, int modeindex, int iscurrentmode, int ignorevbflags)
{
   unsigned int val;
   int newc1, newpanelmode;
   unsigned int myvbflags = sc.currentvbflags;

   if(!check_apply_CRT1(c1_index, modeindex, iscurrentmode, ignorevbflags))
      return FALSE;

   if(c1_index == C1_MI_OFF) {

      newc1 = 0;

      if(sc.CRT1isoff) {
	 if(sc.guimode) {
 	    setcrt1history();
	    sc.currentmodeindex = setdmhistory();
	 }
	 return 2;  /* Already set */
      }

   } else if(c1_index == C1_MI_VGA) {

      newc1 = 1;

      if((!sc.CRT1isoff) && (!(sc.currentvbflags & CRT1_LCDA))) {
         if(sc.guimode) {
	    setcrt1history();
	    sc.currentmodeindex = setdmhistory();
	 }
	 return 2;  /* Already set */
      }

   } else {   /* LCDA */

      newc1 = 2;

      if(sc.sdflags & SiS_SD_SUPPORTSCALE) {
         newpanelmode = getcrt2lsmenuindex(1);
      }

      if( ((!sc.CRT1isoff) && (sc.currentvbflags & CRT1_LCDA)) &&
          ((!(sc.sdflags & SiS_SD_SUPPORTSCALE)) || (newpanelmode == sc.lcdscaler)) ) {
	 if(sc.guimode) {
	    setcrt1history();
	    sc.currentmodeindex = setdmhistory();
	    setlcdhistory();
	 }
	 return 2;  /* Already set */
      }

      if(sc.sdflags & SiS_SD_SUPPORTSCALE) {
         if(UnlockSiSDirect(1)) {
	    unsigned int val;
	    if(!sd_writeulong(SDC_CMD_SETPANELMODE, sc.atom_pmd, newpanelmode)) {
               fprintf(stderr, xverrorstring);
            }
	    if(!sd_readulong(SDC_CMD_GETPANELMODE, sc.atom_pmd, &val)) {
               fprintf(stderr, xverrorstring);
            } else {
	       sc.lcdscaler = val & 0x0f;
	    }
	    UnlockSiSDirect(0);
	 }
      }
   }

   if(UnlockSiSDirect(1)) {
      if(!sd_writeulong(SDC_CMD_SETCRT1STATUS, sc.atom_ct1, newc1)) {
         fprintf(stderr, xverrorstring);
      }
      XFlush(sc.dpy);
      UnlockSiSDirect(0);

      /* See comment far below for why only in guimode */
      if((sc.sd3flags & SiS_SD3_DYNMODELISTS) && sc.guimode) {
         sisctrl_rebuild_mode_and_size_list();
      }

      updatevbandcrt1();
      if( (myvbflags == sc.currentvbflags)            &&
          ( (!(sc.sdflags & SiS_SD_SUPPORTLCDA))  ||
	    (!(sc.sdflags & SiS_SD_SUPPORTSCALE)) ||
	    (newpanelmode != sc.lcdscaler) ) ) {
	 guinoguishowerror("Operation failed", TRUE);
	 return FALSE;
      }
   }

   return TRUE;
}

static int
getcrt2menuindex(void)
{
   if(sc.guimode)
      return(gui_crt2_get_type_menu_index());
   else
      return((int)sc.givencrt2index);
}

static int
getcrt2tvstdmenuindex(void)
{
   if(sc.guimode)
      return(gui_crt2_get_tvstd_menu_index());
   else
      return((int)sc.giventvstdindex);
}

static int
getcrt2tvstmenuindex(void)
{
   if(sc.guimode)
      return(gui_crt2_get_signal_menu_index());
   else
      return((int)sc.giventvstindex);
}

static int
getcrt2tvasmenuindex(void)
{
   if(sc.guimode)
      return(gui_crt2_get_aspect_menu_index());
   else
      return((int)sc.giventvasindex);
}

static int
getcrt2tvosmenuindex(void)
{
   if(sc.guimode)
      return(gui_crt2_get_overscan_menu_index());
   else
      return((int)sc.giventvosindex);
}

static int
check_apply_CRT2(int om_index, int modeindex, int crt1isoff, int hindex, int iscurrentmode)
{
   if(om_index == C2_MI_TV_OFF) {
      if(crt1isoff) {
         guinoguishowerror("CRT2 can't be switched off while CRT1 is off", FALSE);
	 return FALSE;
      }
      if(sc.sdflags & SiS_SD_ISMERGEDFB) {
         if(!(sc.sd3flags & SiS_SD3_MFBALLOWOFFCL)) {
	    guinoguishowerror("CRT2 can't be switched off in MergedFB mode", FALSE);
	    return FALSE;
	 } else if(getmfbdetails(modeindex) && (sc.sdstruct.sdc_result[0] != SDC_MMODE_POS_CLONE)) {
	    guinoguishowerror("CRT2 can only be switched off while a clone mode is active", FALSE);
	    return FALSE;
	 }
      }
   }

   if(om_index == C2_MI_TV_LCD) {
      if(!(sc.detecteddevices & CRT2_LCD)) {
	 guinoguishowerror("Cannot set CRT2 type to LCD; no LCD detected", FALSE);
         return FALSE;
      }
   }

   if((om_index == C2_MI_TV_SVHS) ||
      (om_index == C2_MI_TV_CVBS) ||
      (om_index == C2_MI_TV_SCART) ||
      (om_index == C2_MI_TV_TV)) {
      if(!vbsupportstv()) {
         guinoguishowerror("Hardware does not support TV output", FALSE);
         return FALSE;
      }
      if(om_index == C2_MI_TV_SCART) {
	 if(!(vbsupportsscart())) {
	    guinoguishowerror("Hardware does not support SCART output", FALSE);
	    return FALSE;
	 }
      }
   }

   if(om_index == C2_MI_TV_HIVI) {
      if(!(sc.sdflags & SiS_SD_SUPPORTHIVISION)) {
         guinoguishowerror("Hardware does not support HiVision output", FALSE);
         return FALSE;
      }
   }

   if(om_index == C2_MI_TV_YPBPR) {
      if(!(sc.sdflags & SiS_SD_SUPPORTYPBPR)) {
         guinoguishowerror("Hardware does not support YPbPr output", FALSE);
	 return FALSE;
      } else if((hindex == C2_MI_ST_625I) && (!(sc.sd2flags & SiS_SD2_SUPPORT625I))) {
         guinoguishowerror("Hardware or driver does not support YPbPr 576i output", FALSE);
	 return FALSE;
      } else if((hindex == C2_MI_ST_625P) && (!(sc.sd2flags & SiS_SD2_SUPPORT625P))) {
	 guinoguishowerror("Hardware or driver does not support YPbPr 576p output", FALSE);
	 return FALSE;
      }
   }

   if(om_index == C2_MI_TV_VGA) {
      if(!(vbsupportsvga2())) {
	 guinoguishowerror("Hardware does not support CRT2 VGA output", FALSE);
	 return FALSE;
      }
   }

   if(!sisctrl_checkmodeforcrtx(1, om_index, modeindex, hindex, hindex)) {
      if(iscurrentmode) {
         guinoguishowerror("Current display mode not supported for this CRT2 device type", FALSE);
      } else {
         guinoguishowerror("Display mode not supported for this CRT2 device type", FALSE);
      }
      return FALSE;
   }

   return TRUE;
}

static int
handle_apply_CRT2(int modeindex, int iscurrentmode)
{
   int newoverscan, newpanelmode = 0;
   int om_index = getcrt2menuindex();
   int hindex;

   switch(om_index) {
   case C2_MI_TV_SVHS:
   case C2_MI_TV_CVBS:
   case C2_MI_TV_CVSV:
   case C2_MI_TV_TV:
      hindex = getcrt2tvstdmenuindex();
      break;
   case C2_MI_TV_YPBPR:
      hindex = getcrt2tvstmenuindex();
      break;
   }

   if(!check_apply_CRT2(om_index, modeindex, sc.CRT1isoff, hindex, iscurrentmode))
      return FALSE;

   sc.newvbflags = sc.currentvbflags & ~(CRT2_ENABLE | TV_INTERFACE | TV_STANDARD);
   if(sc.sd2flags & SiS_SD2_VBINVB2ONLY) {
      sc.newvbflags &= ~(TV_YPBPR625I | TV_YPBPR625P);
   }

   switch(om_index) {
      case C2_MI_TV_LCD:   sc.newvbflags |= CRT2_LCD;
			   sc.newvbflags &= ~CRT1_LCDA;	      break;
      case C2_MI_TV_SVHS:  sc.newvbflags |= CRT2_TV | TV_SVIDEO; break;
      case C2_MI_TV_CVBS:  sc.newvbflags |= CRT2_TV | TV_AVIDEO; break;
      case C2_MI_TV_CVSV:  sc.newvbflags |= CRT2_TV | TV_SVIDEO | TV_AVIDEO; break;
      case C2_MI_TV_HIVI:  sc.newvbflags |= CRT2_TV | TV_HIVISION; break;
      case C2_MI_TV_YPBPR: sc.newvbflags |= CRT2_TV | TV_YPBPR;  break;
      case C2_MI_TV_TV:    sc.newvbflags |= CRT2_TV | TV_SVIDEO; break;
      case C2_MI_TV_VGA:   sc.newvbflags |= CRT2_VGA;            break;
      case C2_MI_TV_SCART: sc.newvbflags |= CRT2_TV | TV_SCART;  break;
   }

   if(sc.newvbflags & TV_YPBPR) {
      int indext = getcrt2tvstmenuindex();
      switch(indext) {
         case C2_MI_ST_525I:  sc.newvbflags |= TV_YPBPR525I;  break;
         case C2_MI_ST_525P:  sc.newvbflags |= TV_YPBPR525P;  break;
         case C2_MI_ST_625I:  sc.newvbflags |= TV_YPBPR625I;  break;
         case C2_MI_ST_625P:  sc.newvbflags |= TV_YPBPR625P;  break;
         case C2_MI_ST_750P:  sc.newvbflags |= TV_YPBPR750P;  break;
         case C2_MI_ST_1080I: sc.newvbflags |= TV_YPBPR1080I; break;
      }
      sc.newvbflags &= ~(TV_YPBPRAR);
      if(sc.sdflags & SiS_SD_SUPPORTYPBPRAR) {
         indext = getcrt2tvasmenuindex();
	 switch(indext) {
	    case C2_MI_AS_43:   sc.newvbflags |= TV_YPBPR43; break;
	    case C2_MI_AS_43LB: sc.newvbflags |= TV_YPBPR43LB; break;
	    case C2_MI_AS_169:  sc.newvbflags |= TV_YPBPR169; break;
	 }
      }
   }

   if((sc.sdflags & SiS_SD_SUPPORTSCALE) && (sc.newvbflags & CRT2_LCD)) {
      newpanelmode = getcrt2lsmenuindex(2);
   }

   if((sc.newvbflags & CRT2_TV) && (!(sc.newvbflags & TV_YPBPR))) {
      int indext = getcrt2tvstdmenuindex();
      sc.newvbflags &= ~(TV_PAL | TV_NTSC | TV_PALM | TV_PALN | TV_NTSCJ);
      if(om_index == C2_MI_TV_SCART)     sc.newvbflags |= TV_PAL;
      else if(om_index == C2_MI_TV_HIVI) sc.newvbflags |= TV_PAL;
      else {
         switch(indext) {
            case C2_MI_TV_PAL:   sc.newvbflags |= TV_PAL;  break;
	    case C2_MI_TV_NTSC:  sc.newvbflags |= TV_NTSC; break;
	    case C2_MI_TV_PALM:  sc.newvbflags |= (TV_PAL | TV_PALM); break;
	    case C2_MI_TV_PALN:  sc.newvbflags |= (TV_PAL | TV_PALN); break;
	    case C2_MI_TV_NTSCJ: sc.newvbflags |= (TV_NTSC | TV_NTSCJ); break;
         }
      }
      if(vbsupportsoverscan()) {
	 newoverscan = getcrt2tvosmenuindex();
      }
   }

   if( (sc.newvbflags != sc.currentvbflags)     ||
       ( (vbsupportsoverscan())             &&
         (sc.newvbflags & CRT2_TV)          &&
	 (newoverscan != sc.choverscanstatus) ) ||
       ( (sc.sdflags & SiS_SD_SUPPORTSCALE) &&
         (sc.newvbflags & CRT2_LCD)         &&
	 (newpanelmode != sc.lcdscaler) ) ) {
      if(UnlockSiSDirect(1)) {
	 if((vbsupportsoverscan()) && (sc.newvbflags & CRT2_TV)) {
	    unsigned int ulval;
	    if(!sd_writeulong(SDC_CMD_SETCHTVOVERSCAN, sc.atom_ovr, newoverscan)) {
               fprintf(stderr, xverrorstring);
            }
	    if(!sd_readulong(SDC_CMD_GETCHTVOVERSCAN, sc.atom_ovr, &ulval)) {
               fprintf(stderr, xverrorstring);
            } else {
	       sc.choverscanstatus = ulval;
	    }
	 }
	 if((sc.sdflags & SiS_SD_SUPPORTSCALE) && (sc.newvbflags & CRT2_LCD)) {
	    unsigned int val;
	    if(!sd_writeulong(SDC_CMD_SETPANELMODE, sc.atom_pmd, newpanelmode)) {
               fprintf(stderr, xverrorstring);
            }
	    if(!sd_readulong(SDC_CMD_GETPANELMODE, sc.atom_pmd, &val)) {
               fprintf(stderr, xverrorstring);
            } else {
	       sc.lcdscaler = val & 0x0f;
	    }
	 }
	 if(!sd_writeulong(SDC_CMD_SETVBFLAGS, sc.atom_svf, sc.newvbflags)) {
            fprintf(stderr, xverrorstring);
         }
	 UnlockSiSDirect(0);

	 /* See comment far below for why only in guimode */
	 if((sc.sd3flags & SiS_SD3_DYNMODELISTS) && sc.guimode) {
            sisctrl_rebuild_mode_and_size_list();
         }

	 updatevbandcrt1();

	 if(sc.guimode) {
	    updatecrt1crt2();
	    updatealltv();
	 }

	 if((sc.sdflags & SiS_SD_SUPPORTSCALE) && (sc.newvbflags & CRT2_LCD)) {
	    if(sc.lcdscaler != newpanelmode) {
	       guinoguishowerror("Failed to change LCD scaling mode", FALSE);
	    }
	 }

	 if((sc.currentvbflags & ~DISPLAY_MODE) != (sc.newvbflags & ~DISPLAY_MODE)) {
	    if(!(sc.newvbflags & CRT2_ENABLE)) {
	       guinoguishowerror("CRT2 cannot be switched off", TRUE);
	    } else {
	       guinoguishowerror("Current display mode not supported for this CRT2 type", FALSE);
	    }
	    return FALSE;
	 }
      } else {
         return FALSE;
      }

      return TRUE;

   } else {

      return 2; /* Already set */

   }

}

/* Bottom buttons */

int
sisctrl_do_apply(int what)
{
   int ret = TRUE;

   switch(what) {

   case SIS_PAGE_DM:    /* apply display mode */
      {
      int  dm_index = -1;
      int  randrindex = 0;
      int  rotindex = 0;
      Bool didreset = FALSE;

      sisctrl_backup2_mode_data();

      dm_index = gui_dm_get_current_selected_mode();

      if(dm_index == -1) {
	 fprintf(stderr, "Internal error: List iterator not found\n");
	 ret = FALSE;
	 break;
      }

#ifdef HAVE_RANDR
      if(sc.haverandr) {
         randrindex = gui_dm_get_current_randr_index();
         if(sc.supprotations & 0x0e) {
            rotindex = gui_dm_get_current_randr_rot_index();
         }
      }
#endif

      if(!(handle_apply_mode(dm_index, randrindex, rotindex)))
         return FALSE;

      if(sc.doconfirm) {
         if(!gui_confirmation_popup()) {
            sisctrl_set_backup2_mode_data();
            sisctrl_revert_clicked(0);
            didreset = TRUE;
         }
      }


      /* Do this regardless of confirmation: Our
       * randr event handler overwrites our backup
       * data even we ourselves set the randr config.
       * This way, the "backup"s from before the switch
       * are restored as the real "backups".
       */
      sisctrl_reset_backup_mode_data();

      /* Set display size menu to "keep" if a specific
       * size was chosen; set rotation to "keep" in any
       * case.
       */
#ifdef HAVE_RANDR
      if(gl.do_randr && !didreset) {
         if(gui_dm_get_current_randr_index() >= 2) {
            gui_dm_set_current_randr_index(0);
         }
         if(sc.supprotations & 0x0e) {
            gui_dm_set_current_randr_rot_index(0);
         }
      }
#endif

      gui_enable_revert_button(1);

      break;
      }

   case SIS_PAGE_CRT1:    /* apply CRT1 settings */
      {
      int c1_index, result;
      Bool dontenablerevert = FALSE;

      if(sc.sdflags & SiS_SD_ISDUALHEAD)
         break;

      /* VB-less system: Use "Apply" to make redetection results effective */
      if(gl.crt1_page_type == CRT1_PAGE_TYPE_REDETECT_ONLY) {
         sisctrl_crt1_activate_redetection_result();
         break;
      }

      /* Otherwise, if no video bridge: Bail out */
      if(!(vbhavevb()))
         break;

      sisctrl_backup2_crt1_data();

      c1_index = gui_crt1_get_type_menu_index();

      if(!(result = handle_apply_CRT1(c1_index, sisctrl_dm_get_current_mode_index(), 1, 0)))
         return FALSE;

      /* if current settings already were set, check if we have
       * done a redetection and in case we did, make the results
       * effective. Since the redetection can't be undone, don't
       * enable the Revert button (Note: It could have been
       * enabled previously, but in that case, we have sufficient
       * info to actually revert.)
       */
      if(result == 2 && sc.didcrt1redetection) {
         sisctrl_crt1_activate_redetection_result();
         dontenablerevert = TRUE;
      }

      updatecrt1crt2();

      if(result != 2) { /* No confirmation, no backup if mode was already set */
         if(sc.doconfirm) {
            if(!gui_confirmation_popup()) {
	       sisctrl_set_backup2_crt1_data();
	       sisctrl_revert_clicked(1);
	       sisctrl_reset_backup_crt1_data();
	    }
	 }
      }

      if(!dontenablerevert) {
         gui_enable_revert_button(1);
      }

      break;
      }

   case SIS_PAGE_CRT2:    /* apply CRT2 settings */
      {
      int result;

      if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!(vbhavevb()))) break;

      sisctrl_backup2_crt2_data();

      if(!(result = handle_apply_CRT2(sisctrl_dm_get_current_mode_index(), 1)))
         return FALSE;

      if(result != 2) { /* No confirmation, no backup if mode was already set */
         if(sc.doconfirm) {
            if(!gui_confirmation_popup()) {
	       sisctrl_set_backup2_crt2_data();
	       sisctrl_revert_clicked(2);
	       sisctrl_reset_backup_crt2_data();
	    }
	 }
      }

      gui_enable_revert_button(1);

      break;
      }

   case SIS_PAGE_TV:      /* apply TV settings - done directly */
   case SIS_PAGE_COLOR:   /* apply Gamma settings - done directly */
   case SIS_PAGE_VIDEO:   /* apply Video settings - done directly */
   default:

     break;

   }
   return ret;
}

void
sisctrl_ok_clicked(int what)
{
   if(sisctrl_do_apply(what)) {
      cleanupandexit(0);
      gui_main_quit();
   }
}

void
sisctrl_close_clicked(void)
{
   sisctrl_restore_disable_gfx();
}

void
sisctrl_revert_clicked(int what)
{
   int i;
   Bool didit = FALSE;

   switch(what) {

   case SIS_PAGE_DM:    /* revert display mode */
      {
      int checkmask = 0xff;
      if((sc.backupmodeindex != sc.currentmodeindex)
#ifdef HAVE_RANDR
         						 ||
         (sc.rrrotation_backup != sc.rrcurrent_rotation) ||
	 (sc.rrsize_backup != sc.rrcurrent_size)	 ||
	 (sc.rrrate_backup != sc.rrcurrent_rate)
#endif
						) {
#ifdef HAVE_RANDR
         if((sc.rrrotation_backup != sc.rrcurrent_rotation) ||
	    (sc.rrsize_backup != sc.rrcurrent_size) ||
	    (sc.rrrate_backup != sc.rrcurrent_rate)) {
	    updaterandrconfig();
	    XSelectInput(sc.dpy, sc.root, StructureNotifyMask);
   	    XRRSelectInput(sc.dpy, sc.root, RRScreenChangeNotifyMask);
   	    XRRSetScreenConfigAndRate(sc.dpy, sc.sconf, DefaultRootWindow(sc.dpy),
	    	       (SizeID)sc.rrsize_backup, (Rotation)sc.rrrotation_backup,
		       sc.rrrate_backup, CurrentTime);
	    updaterandrconfig();
	    sc.rrcurrent_size = XRRConfigCurrentConfiguration(sc.sconf, &sc.rrcurrent_rotation);
	    sc.rrcurrent_rate = XRRConfigCurrentRate(sc.sconf);
	    checkmask = ~0x02;
	 }
#endif
         if(checkmodeforcrt2(sc.backupmodeindex) & checkmask) {
	    XFlush(sc.dpy);
	    sc.currentvbflags &= ~BMASK;
	    sc.currentvbflags |= (sc.backupvbflags & BMASK);

	    if(UnlockSiSDirect(1)) {
	       if(!sd_writeulong(SDC_CMD_SETVBFLAGS, sc.atom_svf, sc.currentvbflags)) {
                  fprintf(stderr, xverrorstring);
               }
	       UnlockSiSDirect(0);

	       if(sc.sd3flags & SiS_SD3_DYNMODELISTS) {
		  sisctrl_rebuild_mode_and_size_list();
	       }

	       updatevbandcrt1();
	    }
         }

	 XFlush(sc.dpy);

	 if(checkmodeforscreen(sc.backupmodeindex, FALSE, 0, 0, 0, 0)) {

            if(!XF86VidModeSwitchToMode(sc.dpy, sc.myscreen, sc.vmmodelines[sc.backupmodeindex])) {
               fprintf(stderr, "Failed to set the video mode\n");
            }
            XFlush(sc.dpy);
	    gui_enable_revert_button(0);

	 } else {
	    gui_showerror("Original mode too large for current root window size");
	 }

#ifdef HAVE_RANDR
 	 if(sc.haverandr) {
	    updateleaveunchanged();
	 }
#endif

         updatevbandcrt1();

	 updatecrt1crt2();
      }

      break;
      }

   case SIS_PAGE_CRT1:    /* revert CRT1 settings */

      if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!(vbhavevb()))) break;

      if(sc.sdflags & SiS_SD_SUPPORTSCALE) {
         if(sc.lcdscaler != sc.backuplcdscaler) {
	    if(UnlockSiSDirect(1)) {
	       if(!sd_writeulong(SDC_CMD_SETPANELMODE, sc.atom_pmd, sc.backuplcdscaler)) {
                  fprintf(stderr, xverrorstring);
               }
	       sc.lcdscaler = sc.backuplcdscaler;
	       UnlockSiSDirect(0);
	       didit = TRUE;
	    }
	 }
      }

      if( (sc.CRT1isoff != sc.CRT1isoffbackup) ||
          ((sc.currentvbflags & CRT1_LCDA) != (sc.backupvbflags & CRT1_LCDA)) ||
	  (didit) ) {
         if(UnlockSiSDirect(1)) {
	    if(sc.sdflags & SiS_SD_SUPPORTLCDA) {
	       int val = 1;
	       if(sc.CRT1isoffbackup) val = 0;
	       if((sc.currentvbflags & CRT1_LCDA) != (sc.backupvbflags & CRT1_LCDA)) {
	          sc.currentvbflags &= ~BMASK;
	 	  sc.currentvbflags |= (sc.backupvbflags & BMASK);
		  if(!sd_writeulong(SDC_CMD_SETVBFLAGS, sc.atom_svf, sc.currentvbflags)) {
                     fprintf(stderr, xverrorstring);
                  }
	       }
	       if(sc.currentvbflags & CRT1_LCDA) val = 2;
	       if(!sd_writeulong(SDC_CMD_SETCRT1STATUS, sc.atom_ct1, val)) {
                  fprintf(stderr, xverrorstring);
               }
	    } else {
	       if(!sd_writeulong(SDC_CMD_SETCRT1STATUS, sc.atom_ct1, sc.CRT1isoffbackup ? 0 : 1)) {
                  fprintf(stderr, xverrorstring);
               }
	    }
	    UnlockSiSDirect(0);
	    didit = TRUE;
         }
      }

      if(didit) {
         if(sc.sd3flags & SiS_SD3_DYNMODELISTS) {
	    sisctrl_rebuild_mode_and_size_list();
	 }
         updatevbandcrt1();
	 updatecrt1crt2();
	 gui_enable_revert_button(0);
      }
      break;

   case SIS_PAGE_CRT2:    /* revert CRT2 type */

      if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!(vbhavevb()))) break;

      if(sc.sdflags & SiS_SD_SUPPORTSCALE) {
         if(sc.lcdscaler != sc.backuplcdscaler) {
	    if(UnlockSiSDirect(1)) {
	       if(!sd_writeulong(SDC_CMD_SETPANELMODE, sc.atom_pmd, sc.backuplcdscaler)) {
                  fprintf(stderr, xverrorstring);
               }
	       sc.lcdscaler = sc.backuplcdscaler;
	       UnlockSiSDirect(0);
	       didit = TRUE;
	    }
	 }
      }

      if( ((sc.currentvbflags & BMASK) != (sc.backupvbflags & BMASK)) ||
          ((vbsupportsoverscan()) && (sc.choverscanstatus != sc.backupoverscan)) ||
	  (didit) ) {

         sc.currentvbflags &= ~BMASK;
	 sc.currentvbflags |= (sc.backupvbflags & BMASK);

	 if(UnlockSiSDirect(1)) {
	    if(vbsupportsoverscan()) {
	       if(!sd_writeulong(SDC_CMD_SETCHTVOVERSCAN, sc.atom_ovr, sc.backupoverscan)) {
                  fprintf(stderr, xverrorstring);
               }
  	    }
	    if(!sd_writeulong(SDC_CMD_SETVBFLAGS, sc.atom_svf, sc.currentvbflags)) {
               fprintf(stderr, xverrorstring);
            }
	    UnlockSiSDirect(0);
	    updatevbandcrt1();
	    didit = TRUE;
         }
      }

      if(didit) {
         if(sc.sd3flags & SiS_SD3_DYNMODELISTS) {
	    sisctrl_rebuild_mode_and_size_list();
	 }
	 updatecrt1crt2();
	 updatealltv();
	 gui_enable_revert_button(0);
      }
      break;

   case SIS_PAGE_COLOR:   /* Revert gamma */

      memcpy(&sc.newgamma, &sc.currentgamma, sizeof(XF86VidModeGamma));
      sc.newgamma.red = sc.backup_gred;
      sc.newgamma.green = sc.backup_ggreen;
      sc.newgamma.blue = sc.backup_gblue;
      if(sc.usenewgamma) {
         for(i=0; i<3; i++) {
            sc.newgamma_bri[i] = sc.newgamma_bri_b[i];
            sc.newgamma_con[i] = sc.newgamma_con_b[i];
         }
      } else {
         for(i=0; i<3; i++) {
            sc.gamma_max[i] = sc.gamma_max_b[i];
         }
      }

      if(sc.HaveSepGC2) {
         Bool changed = FALSE;
         if(sc.newgamma2r != sc.backup_g2r) {
            sc.newgamma2r = sc.backup_g2r;
	    changed = TRUE;
#ifdef USE_DEPRECATED_XV
	    if(!sc.sdinterface) setxvval(sc.atom_g2r, multby1000(sc.newgamma2r));
#endif
         }
         if(sc.newgamma2g != sc.backup_g2g) {
            sc.newgamma2g = sc.backup_g2g;
	    changed = TRUE;
#ifdef USE_DEPRECATED_XV
	    if(!sc.sdinterface) setxvval(sc.atom_g2g, multby1000(sc.newgamma2g));
#endif
         }
         if(sc.newgamma2b != sc.backup_g2b) {
            sc.newgamma2b = sc.backup_g2b;
	    changed = TRUE;
#ifdef USE_DEPRECATED_XV
	    if(!sc.sdinterface) setxvval(sc.atom_g2b, multby1000(sc.newgamma2b));
#endif
         }
         for(i=0; i<3; i++) {
            if(sc.usenewgamma) {
               if((sc.newgamma_bri2[i] != sc.newgamma_bri2_b[i]) ||
                  (sc.newgamma_con2[i] != sc.newgamma_con2_b[i])) {
                  sc.newgamma_bri2[i] = sc.newgamma_bri2_b[i];
                  sc.newgamma_con2[i] = sc.newgamma_con2_b[i];
                  changed = TRUE;
               }
            } else {
#ifdef USE_DEPRECATED_XV
               Atom temp;
#endif
	       if(sc.gamma_bri2[i] != sc.gamma_bri2_b[i]) {
  	          sc.gamma_bri2[i] = sc.gamma_bri2_b[i];
	          changed = TRUE;
#ifdef USE_DEPRECATED_XV
	          if(!sc.sdinterface) {
	             switch(i) {
	             case 0: temp = sc.atom_b2r; break;
	             case 1: temp = sc.atom_b2g; break;
	             default:
	             case 2: temp = sc.atom_b2b;
	             }
	             setxvval(temp, multby1000(sc.gamma_bri2[i]));
	          }
#endif
	       }
	    }
	 }
	 if(sc.sdinterface && changed) sd_writegamma2();
      }

      if((!sc.HaveSepGC2) || (!(sc.backupgammaflags & 0x08))) {
         sc.GammaFor2 = FALSE;
      }

      if(sc.HaveSepGC2) {
         if(sc.GammaFor2) {
	    setgamma();
            setgamma2();
	 } else {
	    setgamma2();
            setgamma();
	 }
      } else {
         setgamma();
      }

      sisctrl_gamma_set_all_spinvalues();
      if(!sc.sdinterface && (!(sc.sdflags & SiS_SD_ISDHXINERAMA))) {
         if((!(sc.sdflags & SiS_SD_ISDUALHEAD)) || (sc.sdflags & SiS_SD_ISDHSECONDHEAD)) {
            endisablegamma1((sc.backupgammaflags & 0x01));
	 }
	 if((!(sc.sdflags & SiS_SD_ISDUALHEAD)) || (!(sc.sdflags & SiS_SD_ISDHSECONDHEAD))) {
            endisablegamma2((sc.backupgammaflags & 0x02));
	 }
	 if(sc.HaveSepGC2) {
	    endisablesepgamma2((sc.backupgammaflags & 0x08));
	 }
         sisctrl_gamma_set_status();
      }
      break;

   default:
      break;

   }
}

void
sisctrl_delete_event(void)
{
   cleanupandexit(0);
}

void
sisctrl_destroy_event(void)
{
   cleanupandexit(0);
}


/**********************/
/*      Build GUI     */
/**********************/

static void
buildgui(void)
{
    char wtitlebuffer[256];
#ifdef USE_STRAY
    char straytooltipt[256];
#endif

    /* Copy "display" and xv stuff */
    gl.dpy = sc.dpy;
    gl.xv_port_nr = sc.xv_port_nr;

    gl.xv_req_base = sc.xv_req_base;
    gl.xv_event_base = sc.xv_event_base;
    gl.xv_error_base = sc.xv_error_base;

    /* set window title */
    if(sc.sdinterface) {
       if(sc.xinerama) {
          sprintf(wtitlebuffer, "SiS/XGI display control panel [%s screen %d]", XDisplayString(sc.dpy),
				sc.myscreen);
#ifdef USE_STRAY
	  sprintf(straytooltipt, "SiSCtrl [%s screen %d]", XDisplayString(sc.dpy), sc.myscreen);
#endif
       } else {
          sprintf(wtitlebuffer, "SiS/XGI display control panel [%s]", XDisplayString(sc.dpy));
#ifdef USE_STRAY
	  sprintf(straytooltipt, "SiSCtrl [%s]", XDisplayString(sc.dpy));
#endif
       }
       gl.windowtitle = wtitlebuffer;
#ifdef USE_STRAY
       gl.straytooltip = straytooltipt;
#endif
    } else {
       gl.windowtitle = "SiSCtrl - outdated X driver found";
#ifdef USE_STRAY
       gl.straytooltip = "SiSCtrl";
#endif
    }

    /* fill "gui-look" structure */

    gl.do_randr = (sc.haverandr) ? 1 : 0;

    gl.screennum = sc.myscreen;

    gl.xineramaeb = sc.xi_event_base;

    gl.havevb = vbhavevb() ? 1 : 0;

    gl.mergedfb = (sc.sdflags & SiS_SD_ISMERGEDFB) ? 1 : 0;
    gl.dualhead = (sc.sdflags & SiS_SD_ISDUALHEAD) ? 1 : 0;

    gl.do_mergedfb_details = (sc.sdinterface && (sc.sdflags & SiS_SD_ISMERGEDFB)) ? 1 : 0;
    gl.do_mergedfb_gfx = 0;
    if(gl.do_mergedfb_details) {
       if((sc.sdinterface) && getmfbdetails(0)) {
          if(sc.sdstruct.sdc_result[7] != SDC_MMODE_POS_CLONE) {
             gl.do_mergedfb_gfx = 1;
             gl.globalmfbpos = (sc.sdstruct.sdc_result[7] == SDC_MMODE_POS_LEFTOF ||
				sc.sdstruct.sdc_result[7] == SDC_MMODE_POS_RIGHTOF) ?
             		SIS_MFB_HORIZ : SIS_MFB_VERT;
          }
       }
    }

#ifdef HAVE_RANDR
    gl.leaveunchanged_text = &sc.leaveunchangedtext[0];

    gl.keeprot_text = &sc.currentrottext[0];

    if(sc.haverandr) {
       build_randr_sizes();
       gl.numrandrsizes = sc.rrnsize;
       gl.supportedrots = sc.supprotations;
       updateleaveunchanged();
    }
#endif

    sc.backupmodeindex = sc.currentmodeindex = gl.startmodeindex = finddmhistory();

    /* LCD related */
    gl.supportlcda = (sc.sdflags & SiS_SD_SUPPORTLCDA) ? 1 : 0;
    gl.supportdvi = (vbsupportstmds()) ? 1 : 0;
    gl.supportscale = (sc.sdflags & SiS_SD_SUPPORTSCALE) ? 1 : 0;
    gl.supportcenter = (sc.sdflags & SiS_SD_SUPPORTCENTER) ? 1 : 0;

    /* Laptop? (Used for graphics only) */
    if(sc.sd2flags & SiS_SD2_SUPPLTFLAG) {
       gl.islaptop = (sc.sd2flags & SiS_SD2_ISLAPTOP) ? 1 : 0;
       /* If laptop, certainly no DVI */
       if(gl.islaptop) gl.supportdvi = 0;
    } else {
       /* Hm. If we don't have the flag, let's take
        * a guess. If the machine has DVI output,
	* it's hardly a laptop... Unfortunately,
	* this assumption is wrong for eg. Averatec
	* machines with a 301C plus TMDS panel...
	*/
       gl.islaptop = (gl.supportdvi) ? 0 : 1;
    }

    gl.supportvga2 = vbsupportsvga2()? 1 : 0;

    /* TV related (part 1) */
    gl.supportpalmn = (sc.sdflags & SiS_SD_SUPPORTPALMN) ? 1 : 0;
    gl.supportntscj = (sc.sdflags & SiS_SD_SUPPORTNTSCJ) ? 1 : 0;
    gl.supportypbpr = (sc.sdflags & SiS_SD_SUPPORTYPBPR) ? 1 : 0;
    gl.supportypbpr625i = (sc.sd2flags & SiS_SD2_SUPPORT625I) ? 1 : 0;
    gl.supportypbpr625p = (sc.sd2flags & SiS_SD2_SUPPORT625P) ? 1 : 0;
    gl.supporthivision = (sc.sdflags & SiS_SD_SUPPORTHIVISION) ? 1 : 0;
    gl.supportoverscan = (vbsupportsoverscan()) ? 1 : 0;
    gl.supportsoverscan = (vbsupportsoverscan() && (sc.sdflags & SiS_SD_SUPPORTSOVER)) ? 1 : 0;
    gl.supportypbprar = (sc.sdflags & SiS_SD_SUPPORTYPBPRAR) ? 1 : 0;

    /* Common */
    gl.supportcrt1redetect = (sc.sd3flags & SiS_SD3_REDETECTCRT1) ? 1 : 0;
    gl.supportcrt2redetect = (sc.HaveRedetect) ? 1 : 0;

    /* Gamma related */
    gl.usenewgamma = sc.usenewgamma;
    gl.havesepgamma2 = (sc.HaveSepGC2) ? 1 : 0;

    gl.gammaenablecrt1 = 1;
    gl.gammaenablecrt2 = 1;
    gl.supportsatcrt1 = (sc.sd3flags & SiS_SD3_CRT1SATGAIN) ? 1 : 0;
    gl.supportsatcrt2 = (sc.sd3flags & SiS_SD3_CRT2SATGAIN) ? 1 : 0;
    if(sc.sdflags & SiS_SD_ISDUALHEAD) {
       if(sc.sdflags & SiS_SD_ISDHSECONDHEAD) {
          gl.gammaenablecrt2 = gl.supportsatcrt2 = 0;
       } else {
          gl.gammaenablecrt1 = gl.supportsatcrt1 = 0;
       }
    }
    if(!(vbsupportgamma2())) {
       gl.gammaenablecrt2 = 0;
    }
    if(sc.sdflags & SiS_SD_ISDEPTH8) {
       gl.gammaenablecrt1 = 0;
       gl.gammaenablecrt2 = 0;
    }

    if(!sc.sdinterface) {
       gl.supportsatcrt1 = gl.supportsatcrt2 = 0;
    }

    /* TV related (part 2) */

    gl.supporttvsaturation = (vbsupporttvsaturation()) ? 1 : 0;
    gl.supporttvedgeenhance = (vbsupporttvedgeenhance()) ? 1 : 0;
    gl.supporttvpos = (sc.sdflags & SiS_SD_SUPPORTTVPOS) ? 1 : 0;
    gl.supporttvsize = vbsupporttvsize() ? 1 : 0;

    /* Video related */

    gl.supportxvgamma1 = (sc.sdflags & SiS_SD_SUPPORTXVGAMMA1) ? 1 : 0;
    gl.supporthuesat = supportvihuesat()? 1 : 0;

    gl.needswitchcrt = ( (vbhavevb()) &&
			 /* (!(sc.sdflags & SiS_SD_ISMERGEDFB)) && */
			 ( (!(sc.sdflags & SiS_SD_SUPPORT2OVL)) || (sc.sd2flags & SiS_SD2_SUPPORT760OO) ) ) ?
			 	1 : 0;

    gl.noxvdemo = (sc.noxvdemo) ? 1 : 0;

#ifdef USE_STRAY
    gl.dontusesystemtray = sc.dontusesystemtray;
#endif
    gl.iconify = sc.iconify;

    /* Set up page types */
    if(!(sc.sdflags & SiS_SD_ISDUALHEAD)) {
       if(vbhavevb()) {
          gl.crt1_page_type = CRT1_PAGE_TYPE_NORMAL;
	  gl.crt2_page_type = CRT2_PAGE_TYPE_NORMAL;
       } else {
          if(sc.sd3flags & SiS_SD3_REDETECTCRT1) {
             gl.crt1_page_type = CRT1_PAGE_TYPE_REDETECT_ONLY;
          } else {
             gl.crt1_page_type = CRT1_PAGE_TYPE_NA_NOVB;
          }
	  gl.crt2_page_type = CRT2_PAGE_TYPE_NA_NOVB;
       }
    } else {
       gl.crt1_page_type = CRT1_PAGE_TYPE_NA_DHM;
       gl.crt2_page_type = CRT2_PAGE_TYPE_NA_DHM;
    }

    if(!sc.sdinterface && (sc.sdflags & SiS_SD_ISDHXINERAMA)) {
       gl.gamma_page_type = GAMMA_PAGE_TYPE_XINERAMA;
    } else {
       gl.gamma_page_type = GAMMA_PAGE_TYPE_NORMAL;
    }

    if(vbsupportstv()) {
       gl.tv_page_type = TV_PAGE_TYPE_NORMAL;
       if(vbissisbridge()) {
          gl.tv_page_subtype = TV_PAGE_SUBTYPE_SIS;
       } else if(vbischrontel()) {
          gl.tv_page_subtype = TV_PAGE_SUBTYPE_CHRONTEL;
       } else {
          gl.tv_page_subtype = TV_PAGE_SUBTYPE_UNDEFINED;
       }
    } else {
       gl.tv_page_type = TV_PAGE_TYPE_NA_NOTV;
       gl.tv_page_subtype = TV_PAGE_SUBTYPE_UNDEFINED;
    }

    if(!(sc.sd2flags & SiS_SD2_NOOVERLAY)) {
       gl.video_page_type = VIDEO_PAGE_TYPE_NORMAL;
       if(sc.xinerama && sc.sdinterface && sc.havexv && !sc.noxvglobal) {
          gl.video_page_subtype = VIDEO_PAGE_SUBTYPE_WITH_GLOBAL;
       } else {
          gl.video_page_subtype = VIDEO_PAGE_SUBTYPE_NORMAL;
       }
    } else {
       gl.video_page_type = VIDEO_PAGE_TYPE_NA_NOOVL;
       gl.video_page_subtype = VIDEO_PAGE_SUBTYPE_UNDEFINED;
    }

    gl.config_page_type = CONFIG_PAGE_TYPE_HINTS;
#if 0
    if( vbhavevb() &&
        (!(sc.sdflags & SiS_SD_ISDUALHEAD)) &&
        (!(sc.sdflags & SiS_SD_ISMERGEDFB)) ) {
       gl.config_page_type = CONFIG_PAGE_TYPE_HINTS;
    } else {
       gl.config_page_type = CONFIG_PAGE_TYPE_NO_HINTS;
    }
#endif

    /* Now build the GUI */
    gui_build_gui();
}

static Bool
sisctrl_xinerama_chooser(void)
{
    int screennum;

    Bool ret = gui_xinerama_chooser(sc.sdmaxscreens, &screennum) ? TRUE : FALSE;

    if(ret) {
       if(screennum != -1) sc.myscreen = screennum;
    }

    return ret;
}

Bool
sisctrl_xinerama_query_screen(int screennum, char **chip, int *crtnum, Bool *IsSiS)
{
    int scrbackup = sc.myscreen;
    Bool ret = FALSE;

    sc.myscreen = screennum;

    if(sendsdcommand(SDC_CMD_GETHWINFO)) {
       sc.chipFlags = sc.sdstruct.sdc_result[5];
       sc.chipType = sc.sdstruct.sdc_result[6];
       sc.havenewhwinfo = TRUE;
       *chip = getchiptype(IsSiS);
       *crtnum = -1;
       if(sendsdcommand(SDC_CMD_GETSDFLAGS)) {
          if(sc.sdstruct.sdc_result[0] & SiS_SD_ISDUALHEAD) {
	     if(sc.sdstruct.sdc_result[0] & SiS_SD_ISDHSECONDHEAD)
	        *crtnum = 1;
	     else
	        *crtnum = 2;
	  }
       }
       ret = TRUE;
    }

    sc.myscreen = scrbackup;

    return ret;
}

static void
writeoutstatus(void)
{
    fprintf(stdout, "SISCTRL VERSION: " SISVERSION "\n");
    if(sc.sdinterface) {
       fprintf(stdout, "SIS DRIVER VERSION: %lu/%02lu/%02lu-%lu\n",
		sc.driverYear + 2000, sc.driverMonth, sc.driverDay, sc.driverRev);
       fprintf(stdout, "SISCTRL EXTENSION VERSION: %d.%d\n",
       		sc.scextversion_major, sc.scextversion_minor);
    } else {
       fprintf(stdout, "SIS DRIVER VERSION: %lu/%02lu/%02lu-%lu\n",
		((sc.driverversion >> 16) & 0xff) + 2000,
		(sc.driverversion >> 8) & 0xff,
		sc.driverversion & 0xff,
		sc.driverversion >> 24);
    }
    fprintf(stdout, "MERGEDFB MODE: %s\n", (sc.sdflags & SiS_SD_ISMERGEDFB) ? "ON" : "OFF");
    fprintf(stdout, "DUALHEAD MODE: %s\n", (sc.sdflags & SiS_SD_ISDUALHEAD) ? "ON" : "OFF");
    if(sc.sdflags & SiS_SD_ISDUALHEAD) {
       fprintf(stdout, "XINERAMA: %s\n", (sc.xinerama) ? "ON" : "OFF");
       if(!sc.xinerama) {
          fprintf(stdout, "THIS HEAD: %s\n", (sc.sdflags & SiS_SD_ISDHSECONDHEAD) ? "CRT1" : "CRT2");
       }
    }

    fprintf(stdout, "CRT1: %s\n", (sc.currentvbflags & CRT1_LCDA) ?
       			"LCD" : (sc.CRT1isoff ? "OFF" : "ON"));

    fprintf(stdout, "CRT1 SUPPORTS LCD: %s\n", (sc.sdflags & SiS_SD_SUPPORTLCDA) ? "YES" : "NO");

    fprintf(stdout, "CRT2 CURRENT: ");
    switch(sc.currentvbflags & CRT2_ENABLE) {
       case CRT2_LCD: fprintf(stdout, "LCD\n"); break;
       case CRT2_VGA: fprintf(stdout, "VGA\n"); break;
       case CRT2_TV:
	  if(vbsupportsettvtype()) {
	     if((sc.currentvbflags & (TV_SVIDEO | TV_AVIDEO)) == (TV_SVIDEO | TV_AVIDEO))
	        fprintf(stdout, "SVIDEO+COMPOSITE\n");
	     else if(sc.currentvbflags & TV_SVIDEO)
	        fprintf(stdout, "SVIDEO\n");
	     else if(sc.currentvbflags & TV_AVIDEO)
		fprintf(stdout, "COMPOSITE\n");
	     else if(sc.currentvbflags & TV_SCART)
		fprintf(stdout, "SCART\n");
	     else if(sc.currentvbflags & TV_HIVISION)
		fprintf(stdout, "HIVISION\n");
	     else if(sc.currentvbflags & TV_YPBPR) {
		fprintf(stdout, "YPBPR");
		if(sc.currentvbflags & TV_YPBPR525I)
		   fprintf(stdout, "480I [");
		else if(sc.currentvbflags & TV_YPBPR525P)
		   fprintf(stdout, "480P [");
		else if(sc.currentvbflags & TV_YPBPR750P)
		   fprintf(stdout, "720P [");
		else if(sc.currentvbflags & TV_YPBPR1080I)
		   fprintf(stdout, "1080I [");
		else if((sc.currentvbflags & TV_YPBPR625I) && (sc.sd2flags & SiS_SD2_SUPPORT625I))
		   fprintf(stdout, "576I [");
		else if((sc.currentvbflags & TV_YPBPR625P) && (sc.sd2flags & SiS_SD2_SUPPORT625P))
		   fprintf(stdout, "576P [");
		else
		   fprintf(stdout, "??? [");
		if(sc.sdflags & SiS_SD_SUPPORTYPBPRAR) {
		   if((sc.currentvbflags & TV_YPBPRAR) == TV_YPBPR43)
		      fprintf(stdout, "4:3]\n");
		   else if((sc.currentvbflags & TV_YPBPRAR) == TV_YPBPR43LB)
		      fprintf(stdout, "4:3LB]\n");
		   else if((sc.currentvbflags & TV_YPBPRAR) == TV_YPBPR169)
		      fprintf(stdout, "16:9]\n");
		   else
		      fprintf(stdout, "UNKNOWN ASPECT]\n");
		} else fprintf(stdout, "UNKNOWN ASPECT]\n");
	     } else
		fprintf(stdout, "UNKNOWN\n");
	  } else {
	     fprintf(stdout, "TV\n");
	  }
	  break;
       default: fprintf(stdout, "OFF\n");
    }
    fprintf(stdout, "CRT2 DETECTED: ");
    if(sc.detecteddevices & CRT2_LCD) fprintf(stdout, "LCD ");
    if(sc.detecteddevices & CRT2_VGA) fprintf(stdout, "VGA ");
    if(sc.detecteddevices & CRT2_TV) {
       if(vbsupportsettvtype()) {
	  if(sc.detecteddevices & TV_SVIDEO) fprintf(stdout, "TV-SVIDEO ");
	  if(sc.detecteddevices & TV_AVIDEO) fprintf(stdout, "TV-COMPOSITE ");
	  if(sc.detecteddevices & TV_SCART) fprintf(stdout, "TV-SCART ");
	  if(sc.detecteddevices & TV_HIVISION) fprintf(stdout, "TV-HIVISION ");
	  if(sc.detecteddevices & TV_YPBPR) fprintf(stdout, "TV-YPBPR ");
       } else {
          fprintf(stdout, "TV");
       }
    }
    fprintf(stdout, "\n");

    if((sc.currentvbflags & CRT2_TV) && (!(sc.currentvbflags & (TV_YPBPR | TV_HIVISION)))) {
       fprintf(stdout, "TVSTANDARD: ");
       if(sc.currentvbflags & TV_NTSC) {
	  if(sc.currentvbflags & TV_NTSCJ)
	     fprintf(stdout, "NTSC-J\n");
	  else
	     fprintf(stdout, "NTSC\n");
       } else {
	  if(sc.currentvbflags & TV_PALM)
	     fprintf(stdout, "PAL-M\n");
          if(sc.currentvbflags & TV_PALN)
	     fprintf(stdout, "PAL-N\n");
	  else
	     fprintf(stdout, "PAL\n");
       }
       if(vbsupportsoverscan()) {
	  fprintf(stdout, "TVOVERSCAN: ");
	  switch(sc.choverscanstatus) {
	     case 0: fprintf(stdout, "BIOS\n"); break;
	     case 1: fprintf(stdout, "UNDERSCAN\n"); break;
	     case 2: fprintf(stdout, "OVERSCAN\n"); break;
	     case 3: fprintf(stdout, "SUPEROVERSCAN\n"); break;
	     default: fprintf(stdout, "UNKNOWN\n");
	  }
       }
    }

    fprintf(stdout, "GAMMA CORRECTION: ");
    if(!sc.sdinterface && (sc.sdflags & SiS_SD_ISDHXINERAMA)) {
       fprintf(stdout, "UNKNOWN");
    } else if(sc.sdflags & SiS_SD_ISDUALHEAD) {
       if(sc.sdflags & SiS_SD_ISDHSECONDHEAD) {
	  if(sc.gammaflags & 0x01) fprintf(stdout, "CRT1");
       } else {
	  if(sc.gammaflags & 0x02) fprintf(stdout, "CRT2");
       }
    } else {
       if(sc.gammaflags & 0x01) fprintf(stdout, "CRT1 ");
       if(vbsupportgamma2()) {
          if(sc.gammaflags & 0x02) fprintf(stdout, "CRT2");
	  if((sc.HaveSepGC2) && (sc.gammaflags & 0x08)) {
	     fprintf(stdout, " (SEPARATE)");
	  }
       }
    }
    fprintf(stdout, "\n");
    fprintf(stdout, "GAMMA CURRENT: %.3f %.3f %.3f\n",
      	    sc.currentgamma.red, sc.currentgamma.green, sc.currentgamma.blue);
    if(!sc.usenewgamma) {
       fprintf(stdout, "USING NEW BRIGHTNESS: NO\n");
       fprintf(stdout, "GAMMA BRIGHTNESS CURRENT: %.3f %.3f %.3f\n",
            sc.gamma_max[0], sc.gamma_max[1], sc.gamma_max[2]);
    } else {
       fprintf(stdout, "USING NEW BRIGHTNESS: YES\n");
       fprintf(stdout, "BRIGHTNESS CURRENT: %.3f %.3f %.3f\n",
            sc.newgamma_bri[0], sc.newgamma_bri[1], sc.newgamma_bri[2]);
       fprintf(stdout, "CONTRAST CURRENT: %.3f %.3f %.3f\n",
            sc.newgamma_con[0], sc.newgamma_con[1], sc.newgamma_con[2]);
    }

    if((sc.HaveSepGC2) && (sc.gammaflags & 0x08)) {
       fprintf(stdout, "GAMMA CRT2 CURRENT: %.3f %.3f %.3f\n",
	  	sc.gamma2r, sc.gamma2g, sc.gamma2b);
       if(!sc.usenewgamma) {
          fprintf(stdout, "GAMMA CRT2 BRIGHTNESS CURRENT: %.3f %.3f %.3f\n",
            	sc.gamma_bri2[0], sc.gamma_bri2[1], sc.gamma_bri2[2]);
       } else {
          fprintf(stdout, "BRIGHTNESS CURRENT: %.3f %.3f %.3f\n",
               sc.newgamma_bri2[0], sc.newgamma_bri2[1], sc.newgamma_bri2[2]);
          fprintf(stdout, "CONTRAST CURRENT: %.3f %.3f %.3f\n",
               sc.newgamma_con2[0], sc.newgamma_con2[1], sc.newgamma_con2[2]);
       }
    }

    if(sc.currentvbflags & CRT2_TV) {
       if(vbissisbridge()) {
          int temp = newgetxvval(SDC_CMD_SETTVANTIFLICKER, sc.atom_taf);
          fprintf(stdout, "TV ANTIFLICKER: ");
	  switch(temp) {
	     case 0: fprintf(stdout, "OFF\n"); break;
	     case 1: fprintf(stdout, "LOW\n"); break;
	     case 2: fprintf(stdout, "MED\n"); break;
	     case 3: fprintf(stdout, "HIGH\n"); break;
	     case 4: fprintf(stdout, "ADAPTIVE\n"); break;
	  }
	  if(vbsupporttvsaturation()) {
	     fprintf(stdout, "TV SATURATION: %d\n", newgetxvval(SDC_CMD_GETTVSATURATION, sc.atom_tsa));
	  }
	  if(vbsupporttvedgeenhance()) {
	     fprintf(stdout, "TV EDGEENHANCE: %d\n", newgetxvval(SDC_CMD_GETTVEDGEENHANCE, sc.atom_tee));
	  }
	  fprintf(stdout, "TV COLOR CALIBRATION COARSE: %d\n", sd_readcolorcalib(sc.atom_coc, TRUE));
	  fprintf(stdout, "TV COLOR CALIBRATION FINE: %d\n", sd_readcolorcalib(sc.atom_cof, FALSE));
	  fprintf(stdout, "TV C-FILTER: %s\n", newgetxvval(SDC_CMD_GETTVCFILTER, sc.atom_cfi) ? "ON" : "OFF");
	  fprintf(stdout, "TV Y-FILTER: %d\n", newgetxvval(SDC_CMD_GETTVYFILTER, sc.atom_yfi));
       } else if(vbischrontel()) {
	  fprintf(stdout, "TV CONTRAST: %d\n", newgetxvval(SDC_CMD_GETTVCHCONTRAST, sc.atom_tco));
	  fprintf(stdout, "TV EDGEENHANCE: %d\n", newgetxvval(SDC_CMD_GETTVCHTEXTENHANCE, sc.atom_tte));
	  fprintf(stdout, "TV CHROMAFLICKERFILTER: %d\n", newgetxvval(SDC_CMD_GETTVCHCHROMAFLICKERFILTER, sc.atom_tcf));
	  fprintf(stdout, "TV LUMAFLICKERFILTER: %d\n", newgetxvval(SDC_CMD_GETTVCHLUMAFLICKERFILTER, sc.atom_tlf));
	  fprintf(stdout, "TV CVBS COLOR: %s\n", newgetxvval(SDC_CMD_GETTVCHCVBSCOLOR, sc.atom_ccc) ? "ON" : "OFF");
       }
       if(sc.sdflags & SiS_SD_SUPPORTTVPOS) {
	  fprintf(stdout, "TV X POSITION: %d\n", sd_gettvposscale(SDC_CMD_GETTVXPOS, sc.atom_tvx));
	  fprintf(stdout, "TV Y POSITION: %d\n", sd_gettvposscale(SDC_CMD_GETTVYPOS, sc.atom_tvy));
       }
       if(vbsupporttvsize()) {
	  fprintf(stdout, "TV X SCALE: %d\n", sd_gettvposscale(SDC_CMD_GETTVXSCALE, sc.atom_txs));
	  fprintf(stdout, "TV Y SCALE: %d\n", sd_gettvposscale(SDC_CMD_GETTVYSCALE, sc.atom_tys));
       }
    }

    fprintf(stdout, "VIDEO BRIGHTNESS: %d\n", newgetrealxvval(SDC_CMD_GETXVBRIGHTNESS, sc.atom_bri));
    fprintf(stdout, "VIDEO CONTRAST: %d\n", newgetrealxvval(SDC_CMD_GETXVCONTRAST, sc.atom_con));
    if(supportvihuesat()) {
       fprintf(stdout, "VIDEO SATURATION: %d\n", newgetrealxvval(SDC_CMD_GETXVSATURATION, sc.atom_sat));
       fprintf(stdout, "VIDEO HUE: %d\n", newgetrealxvval(SDC_CMD_GETXVHUE, sc.atom_hue));
    }
    if(sc.sdflags & SiS_SD_SUPPORTXVGAMMA1) {
       fprintf(stdout, "VIDEO CRT1 GAMMA ENABLE: %s\n", (sc.gammaflags & 0x04) ? "ON" : "OFF");
       if(sc.gammaflags & 0x04) {
          fprintf(stdout, "VIDEO CRT1 GAMMA: %.3f %.3f %.3f\n", sc.xvgammared, sc.xvgammagreen, sc.xvgammablue);
       }
    }
    if((!(sc.sdflags & SiS_SD_ISMERGEDFB)) &&
       ((!(sc.sdflags & SiS_SD_SUPPORT2OVL)) || (sc.sd2flags & SiS_SD2_SUPPORT760OO))) {
       fprintf(stdout, "VIDEO ON CRT2: %s\n", newgetxvval(SDC_CMD_GETXVSWITCHCRT, sc.atom_swc) ? "YES" : "NO");
    }
}

static int
calcrefresh(XF86VidModeModeInfo *modeinfo)
{
   float hsync, refresh = 0.0;

   if(modeinfo->htotal > 0)
       	hsync = (float)modeinfo->dotclock / (float)modeinfo->htotal;
   else return 0;

   if(modeinfo->vtotal > 0)
       	refresh = hsync * 1000.0 / modeinfo->vtotal;
   else return 0;

   if(modeinfo->flags & V_INTERLACE)
       	refresh *= 2.0;

   if(modeinfo->flags & V_DBLSCAN)
       	refresh /= 2.0;

   return(int)(refresh + 0.5);
}

/***********************************************/
/*                   main()                    */
/***********************************************/

int
main(int argc, char *argv[] )
{
    char *displayname = NULL;
    int i, force = 0;
    int MajorVersion, MinorVersion;
    int EventBase, ErrorBase;
    int MIErrorBase, MIEventBase;
    int rr_event_base, rr_error_base;
    Bool screengiven = FALSE;
    Bool printstatus = FALSE;
    Bool setgammafromdriver = FALSE;
    Bool listmodes = FALSE, setmode = FALSE, resizetomode = FALSE, setlcdscale = FALSE, setlcdcenter = FALSE;
    Bool setcrt1 = FALSE, setcrt2 = FALSE, settvstd = FALSE, settvos = FALSE;
    Bool setxvcrt = FALSE, settvxpos = FALSE, settvypos = FALSE, settvas = FALSE;
    Bool settvxscale = FALSE, settvyscale = FALSE, doexit = FALSE;
    Bool xinewarn = FALSE;
    Bool parmgiven = FALSE;
    int modetoset;
    int newlcdscaling, newlcdcenter;
    int tvxposvalue, tvyposvalue, tvxscalevalue, tvyscalevalue;
    int setcrt1index, xvcrtindex;
    char *missingparm = "Missing -%s argument, try -help\n";
    char *illegalparm = "Illegal -%s argument, try -help\n";

    char buffer[256];

    unsigned int num_adaptors = 0;
    int num_attr = 0;
    XvAdaptorInfo *adaptor_info = NULL;
    XvAttribute *attributes = NULL;
    unsigned int xv_major, xv_minor, xv_ulval;
    int xv_val;

    int sc_event_base, sc_error_base, sc_version_major, sc_version_minor;

    sc.myscreen = -1;
    sc.lockfile = -1;
    sc.xv_port_nr = -1;
    sc.attr_name = NULL;
    sc.vmmodecount = 0;
    sc.guimode = FALSE;
    sc.xlogquiet = FALSE;
#ifdef HAVE_RANDR
    sc.sconf = NULL;
#endif
#ifdef USEXV
    sc.noxvdemo = FALSE;
#endif
#ifdef USE_STRAY
    sc.dontusesystemtray = 0;
#endif
    sc.iconify = 0;
    sc.doconfirm = 1;

    sc.havedglgfx = FALSE;

    sc.gamma_max[0] = sc.gamma_max[1] = sc.gamma_max[2] = 1.0;
    sc.sepgammahidden = TRUE;

    sc.havesdflags2 = FALSE;

    sc.sdinterface = sc.havenewhwinfo = FALSE;

    sc.havemonname1 = sc.havemonname2 = sc.havedevname1 = sc.havedevname2 = FALSE;

    sc.sd2flags = sc.vbflags2 = 0;

    sc.didcrt1redetection = FALSE;

    /* handle gui-specific command line options */
    gui_init(&argc, &argv);

    /* handle our command line options */

    for(i = 1; i < argc; i++) {
	char *arg = argv[i];

	if(arg[0] == '-') {
	   if(strcmp(arg, "-display") == 0) {
		if(++i >= argc) {
		   fprintf(stderr, missingparm, "display");
		   exit(1);
		}
		displayname = argv[i];
		continue;
	    } else if(strcmp(arg, "-screen") == 0) {
		if(++i >= argc) {
		   fprintf(stderr, missingparm, "screen");
		   exit(1);
		}
		sc.myscreen = atoi(argv[i]);
		screengiven = TRUE;
		continue;
	    } else if(strcmp(arg, "-force") == 0) {
	        force = 1;
	    } else if((strncmp(arg, "-h", 2) == 0) || (strcmp(arg, "-?") == 0)) {
	        fprintf(stdout, "%s [-display host:dpy] [-screen screen] [-force] [-status]\n\t"
				"[-setgammabrightness|-sg] [-listmodes|-lm] [-setmode mode] "
#ifdef HAVE_RANDR
				"[-resize] "
#endif
				"\n\t"
				"[-setcrt1 device] [-setcrt2 device] [-setlcdscaling mode]\n\t"
				"[-setlcdcentering mode] [-settvstandard standard]\n\t"
				"[-settvoverscan mode] [settvsapect mode] [-setxvcrt num]\n\t"
				"[-settvxpos x] [-settvypos y] [-settvxscale x] [-settvyscale y]\n\t"
#ifdef USEXV
				"[-noxvdemo] "
#endif
#ifdef USE_STRAY
				"[-nosystemtray] "
#endif
				"[-noconfirm]\n\t"
				"[-v|-version] [-h|-help|-?]\n", argv[0]);
		fprintf(stdout, "display: target display in host:dpy format\n");
		fprintf(stdout, "screen: screen number (used only if Xinerama is detected)\n");
		fprintf(stdout, "force: to delete lock file (and ignore other eventually running sessions)\n");
#ifdef USEXV
		fprintf(stdout, "noxvdemo: disable xv demo graphics in Video settings tab\n");
#endif
#ifdef USE_STRAY
		fprintf(stderr, "nosystemtray: disable system tray icon\n");
#endif
		fprintf(stderr, "iconify: start with iconified window\n");
		fprintf(stderr, "noconfirm: disable mode change confirmation window\n");
		fprintf(stdout, "-- Specifying one of the following options will disable the GUI:\n");
		fprintf(stdout, "version: print version and quit\n");
		fprintf(stdout, "setgammabrightness: set stored gamma brightness from driver and quit\n");
		fprintf(stdout, "status: print current display status and quit\n");
		fprintf(stdout, "listmodes: list all supported display modes and quit\n");
		fprintf(stdout, "setmode: set display mode, use -listmodes to find out about mode number\n");
#ifdef HAVE_RANDR
		fprintf(stdout, "resize: resize desktop to display mode\n");
#endif
		fprintf(stdout, "setcrt1: set CRT1 device (\"on\"=\"vga\", \"off\", \"dvi-d\"=\"lcd\")\n");
		fprintf(stdout, "setcrt2: set CRT2 device (\"dvi-d\"=\"lcd\", \"tv\", \"svideo\", \"cvbs\", \"svideo+cvbs\",\n\t\"hivision\", \"ypbpr1080i\", \"ypbpr720p\", \"ypbpr576p\", \"ypbpr576i\",\n\t\"ypbpr480p\", \"ypbpr480i\", \"dvi-a\"=\"vga\", \"off\")\n");
		fprintf(stdout, "setlcdscaling: set LCD scaling mode (\"default\", \"on\", \"off\")\n");
		fprintf(stdout, "setlcdcenter: set LCD centering mode (\"default\", \"on\", \"off\")\n");
		fprintf(stdout, "settvstandard: set TV standard (\"pal\", \"ntsc\", \"palm\", \"paln\", \"ntscj\")\n");
		fprintf(stdout, "settvoverscan: set TV overscan (\"bios\", \"overscan\", \"underscan\", \"soverscan\")\n");
		fprintf(stdout, "settvaspect: set YPbPr TV aspect ratio (\"4:3\", \"4:3lb\", \"16:9\")\n");
		fprintf(stdout, "setxvcrt: set video overlay CRT number (1 or 2)\n");
		fprintf(stdout, "settvxpos: set TV x position (-32 <= x <= 32)\n");
		fprintf(stdout, "settvypos: set TV y position (-32 <= y <= 32)\n");
		fprintf(stdout, "settvxscale: set TV x scale (-16 <= x <= 16)\n");
		fprintf(stdout, "settvyscale: set TV y scale (-4 <= y <= 3)\n");
		exit(0);
	    } else if(strncmp(arg, "-v", 2) == 0) {
	        fprintf(stdout, "sisctrl version " SISVERSION "\n");
		exit(0);
	    } else if(strcmp(arg, "-status") == 0) {
	        printstatus = TRUE;
		parmgiven = TRUE;
	    } else if((strcmp(arg, "-setgammabrightness") == 0) || (strncmp(arg, "-sg", 3) == 0)) {
	        setgammafromdriver = TRUE;
		parmgiven = TRUE;
	    } else if(strcmp(arg, "-setmode") == 0) {
	        if(++i >= argc) {
		   fprintf(stderr, missingparm, "setmode");
		   exit(1);
		}
		modetoset = atoi(argv[i]);
		setmode = TRUE;
		parmgiven = TRUE;
#ifdef HAVE_RANDR
	    } else if(strcmp(arg, "-resize") == 0) {
	        resizetomode = TRUE;
	        parmgiven = TRUE;
#endif
	    } else if((strcmp(arg, "-listmodes") == 0) ||
	              (strcmp(arg, "-lm") == 0)) {
	        listmodes = TRUE;
	        parmgiven = TRUE;
	    } else if(strcmp(arg, "-setcrt1") == 0) {
	        if(++i >= argc) {
		   fprintf(stderr, missingparm, "setcrt1");
		   exit(1);
		}
		if((strcmp(argv[i], "on") == 0) ||
		   (strcmp(argv[i], "vga") == 0) ||
		   (strcmp(argv[i], "true") == 0) ||
		   (strcmp(argv[i], "yes") == 0)) {
		   setcrt1index = C1_MI_VGA;
		   setcrt1 = TRUE;
		} else if((strcmp(argv[i], "off") == 0) ||
		          (strcmp(argv[i], "false") == 0) ||
		          (strcmp(argv[i], "none") == 0) ||
		          (strcmp(argv[i], "no") == 0)) {
		   setcrt1index = C1_MI_OFF;
		   setcrt1 = TRUE;
		} else if((strcmp(argv[i], "lcd") == 0) ||
		          (strcmp(argv[i], "lcda") == 0) ||
			  (strcmp(argv[i], "lcd-a") == 0) ||
		          (strcmp(argv[i], "dvi-d") == 0) ||
			  (strcmp(argv[i], "dvi") == 0)) {
		   setcrt1index = C1_MI_LCDA;
		   setcrt1 = TRUE;
		} else {
		   fprintf(stderr, illegalparm, "setcrt1");
		   exit(1);
		}
		parmgiven = TRUE;
	    } else if(strcmp(arg, "-setcrt2") == 0) {
	        if(++i >= argc) {
		   fprintf(stderr, missingparm, "setcrt2");
		   exit(1);
		}
		if((strcmp(argv[i], "lcd") == 0) ||
		   (strcmp(argv[i], "plasma") == 0) ||
		   (strcmp(argv[i], "dvi") == 0) ||
		   (strcmp(argv[i], "dvi-d") == 0)) {
		   sc.givencrt2index = C2_MI_TV_LCD;
		   setcrt2 = TRUE;
		} else if((strcmp(argv[i], "svideo") == 0) ||
		          (strcmp(argv[i], "svhs") == 0)) {
		   sc.givencrt2index = C2_MI_TV_SVHS;
		   setcrt2 = TRUE;
		} else if((strcmp(argv[i], "composite") == 0) ||
		          (strcmp(argv[i], "cvbs") == 0)) {
		   sc.givencrt2index = C2_MI_TV_CVBS;
		   setcrt2 = TRUE;
		} else if((strcmp(argv[i], "svideo+composite") == 0) ||
			  (strcmp(argv[i], "composite+svideo") == 0) ||
			  (strcmp(argv[i], "svideo+cvbs") == 0) ||
			  (strcmp(argv[i], "cvbs+svideo") == 0)) {
		   sc.givencrt2index = C2_MI_TV_CVSV;
		   setcrt2 = TRUE;
		} else if((strcmp(argv[i], "hivision") == 0)) {
		   sc.givencrt2index = C2_MI_TV_HIVI;
		   setcrt2 = TRUE;
		} else if((strcmp(argv[i], "ypbpr480i") == 0)) {
		   sc.givencrt2index = C2_MI_TV_YPBPR;
		   sc.giventvstindex = C2_MI_ST_525I;
		   setcrt2 = TRUE;
		} else if((strcmp(argv[i], "ypbpr480p") == 0)) {
		   sc.givencrt2index = C2_MI_TV_YPBPR;
		   sc.giventvstindex = C2_MI_ST_525P;
		   setcrt2 = TRUE;
		} else if((strcmp(argv[i], "ypbpr576i") == 0)) {
		   sc.givencrt2index = C2_MI_TV_YPBPR;
		   sc.giventvstindex = C2_MI_ST_625I;
		   setcrt2 = TRUE;
		} else if((strcmp(argv[i], "ypbpr576p") == 0)) {
		   sc.givencrt2index = C2_MI_TV_YPBPR;
		   sc.giventvstindex = C2_MI_ST_625P;
		   setcrt2 = TRUE;
		} else if((strcmp(argv[i], "ypbpr720p") == 0)) {
		   sc.givencrt2index = C2_MI_TV_YPBPR;
		   sc.giventvstindex = C2_MI_ST_750P;
		   setcrt2 = TRUE;
		} else if((strcmp(argv[i], "ypbpr1080i") == 0)) {
		   sc.givencrt2index = C2_MI_TV_YPBPR;
		   sc.giventvstindex = C2_MI_ST_1080I;
		   setcrt2 = TRUE;
		} else if((strcmp(argv[i], "tv") == 0)) {
		   sc.givencrt2index = C2_MI_TV_TV;
		   setcrt2 = TRUE;
		} else if((strcmp(argv[i], "vga") == 0) ||
			  (strcmp(argv[i], "dvi-a") == 0) ||
		          (strcmp(argv[i], "vga2") == 0)) {
		   sc.givencrt2index = C2_MI_TV_VGA;
		   setcrt2 = TRUE;
		} else if((strcmp(argv[i], "scart") == 0)) {
		   sc.givencrt2index = C2_MI_TV_SCART;
		   setcrt2 = TRUE;
		} else if((strcmp(argv[i], "off") == 0) ||
		          (strcmp(argv[i], "none") == 0)) {
		   sc.givencrt2index = C2_MI_TV_OFF;
		   setcrt2 = TRUE;
		} else {
		   fprintf(stderr, illegalparm, "setcrt2");
		   exit(1);
		}
		parmgiven = TRUE;
#if 0
	     } else if(strcmp(arg, "-setlcdscaling") == 0) {
	        if(++i >= argc) {
		   fprintf(stderr, missingparm, "setlcdscaling");
		   exit(1);
		}
		if(strcmp(argv[i], "default") == 0) {
		   newlcdscaling = 0x01;
		   setlcdscale = TRUE;
		} else if(strcmp(argv[i], "on") == 0) {
		   newlcdscaling = 0x00;
		   setlcdscale = TRUE;
		} else if(strcmp(argv[i], "off") == 0) {
		   newlcdscaling = 0x02;
		   setlcdscale = TRUE;
		} else {
		   fprintf(stderr, illegalparm, "setlcdscaling");
		   exit(1);
		}
		parmgiven = TRUE;
	     } else if(strcmp(arg, "-setlcdcenter") == 0) {
	        if(++i >= argc) {
		   fprintf(stderr, missingparm, "setlcdcenter");
		   exit(1);
		}
		if(strcmp(argv[i], "default") == 0) {
		   newlcdcenter = 0x04;
		   setlcdcenter = TRUE;
		} else if(strcmp(argv[i], "on") == 0) {
		   newlcdcenter = 0x08;
		   setlcdcenter = TRUE;
		} else if(strcmp(argv[i], "off") == 0) {
		   newlcdcenter = 0x00;
		   setlcdcenter = TRUE;
		} else {
		   fprintf(stderr, illegalparm, "setlcdcenter");
		   exit(1);
		}
		parmgiven = TRUE;
#endif
  	     } else if((strcmp(arg, "-settvstd") == 0) ||
	               (strcmp(arg, "-settvstandard") == 0)) {
	        if(++i >= argc) {
		   fprintf(stderr, missingparm, "settvstandard");
		   exit(1);
		}
		if(strcmp(argv[i], "pal") == 0) {
		   sc.giventvstdindex = C2_MI_TV_PAL;
		   settvstd = TRUE;
		} else if(strcmp(argv[i], "ntsc") == 0) {
		   sc.giventvstdindex = C2_MI_TV_NTSC;
		   settvstd = TRUE;
		} else if((strcmp(argv[i], "pal-m") == 0) ||
		          (strcmp(argv[i], "palm") == 0)) {
		   sc.giventvstdindex = C2_MI_TV_PALM;
		   settvstd = TRUE;
		} else if((strcmp(argv[i], "pal-n") == 0) ||
		          (strcmp(argv[i], "paln") == 0)) {
		   sc.giventvstdindex = C2_MI_TV_PALN;
		   settvstd = TRUE;
		} else if((strcmp(argv[i], "ntsc-j") == 0) ||
		          (strcmp(argv[i], "ntscj") == 0)) {
		   sc.giventvstdindex = C2_MI_TV_NTSCJ;
		   settvstd = TRUE;
		} else {
		   fprintf(stderr, illegalparm, "settvstandard");
		   exit(1);
		}
		parmgiven = TRUE;
            } else if(strcmp(arg, "-settvoverscan") == 0) {
	        if(++i >= argc) {
		   fprintf(stderr, missingparm, "settvoverscan");
		   exit(1);
		}
		if(strcmp(argv[i], "bios") == 0) {
		   sc.giventvosindex = C2_MI_OV_BIOS;
		   settvos = TRUE;
		} else if((strcmp(argv[i], "underscan") == 0) ||
		          (strcmp(argv[i], "under") == 0)) {
		   sc.giventvosindex = C2_MI_OV_UNDERSCAN;
		   settvos = TRUE;
		} else if((strcmp(argv[i], "overscan") == 0) ||
		          (strcmp(argv[i], "over") == 0)) {
		   sc.giventvosindex = C2_MI_OV_OVERSCAN;
		   settvos = TRUE;
		} else if((strcmp(argv[i], "soverscan") == 0) ||
		          (strcmp(argv[i], "sover") == 0)) {
		   sc.giventvosindex = C2_MI_OV_SUPEROVERSCAN;
		   settvos = TRUE;
		} else {
		   fprintf(stderr, illegalparm, "settvoverscan");
		   exit(1);
		}
		parmgiven = TRUE;
	    } else if(strcmp(arg, "-settvaspect") == 0) {
	        if(++i >= argc) {
		   fprintf(stderr, missingparm, "settvaspect");
		   exit(1);
		}
		if(strcmp(argv[i], "4:3") == 0) {
		   sc.giventvasindex = C2_MI_AS_43;
		   settvas = TRUE;
		} else if(strcmp(argv[i], "4:3lb") == 0) {
		   sc.giventvasindex = C2_MI_AS_43LB;
		   settvas = TRUE;
		} else if(strcmp(argv[i], "16:9") == 0) {
		   sc.giventvasindex = C2_MI_AS_169;
		   settvas = TRUE;
		} else {
		   fprintf(stderr, illegalparm, "settvaspect");
		   exit(1);
		}
		parmgiven = TRUE;
	    } else if(strcmp(arg, "-setxvcrt") == 0) {
	        if(++i >= argc) {
		   fprintf(stderr, missingparm, "setxvcrt");
		   exit(1);
		}
		xvcrtindex = atoi(argv[i]);
		if((xvcrtindex != 1) && (xvcrtindex != 2)) {
		   fprintf(stderr, illegalparm, "setxvcrt");
		   exit(1);
		} else {
		   setxvcrt = TRUE;
		   parmgiven = TRUE;
		}
	    } else if((strcmp(arg, "-settvxposition") == 0) ||
	              (strcmp(arg, "-settvxpos") == 0)) {
	    	if(++i >= argc) {
		   fprintf(stderr, missingparm, "settvxpos");
		   exit(1);
		}
		tvxposvalue = atoi(argv[i]);
		if(tvxposvalue >= -32 && tvxposvalue <= 32) {
	           settvxpos = TRUE;
		   parmgiven = TRUE;
		} else {
		   fprintf(stderr, illegalparm, "settvxpos");
		   exit(1);
		}
	    } else if((strcmp(arg, "-settvyposition") == 0) ||
	              (strcmp(arg, "-settvypos") == 0)) {
	        if(++i >= argc) {
		   fprintf(stderr, missingparm, "settvypos");
		   exit(1);
		}
		tvyposvalue = atoi(argv[i]);
		if(tvyposvalue >= -32 && tvyposvalue <= 32) {
	           settvypos = TRUE;
		   parmgiven = TRUE;
		} else {
		   fprintf(stderr, illegalparm, "settvypos");
		   exit(1);
		}
	    } else if(strcmp(arg, "-settvxscale") == 0) {
	        if(++i >= argc) {
		   fprintf(stderr, missingparm, "settvxscale");
		   exit(1);
		}
		tvxscalevalue = atoi(argv[i]);
		if(tvxscalevalue >= -16 && tvxscalevalue <= 16) {
	           settvxscale = TRUE;
		   parmgiven = TRUE;
		} else {
		   fprintf(stderr, illegalparm, "settvxscale");
		   exit(1);
		}
	    } else if(strcmp(arg, "-settvyscale") == 0) {
	        if(++i >= argc) {
		   fprintf(stderr, missingparm, "settvyscale");
		   exit(1);
		}
		tvyscalevalue = atoi(argv[i]);
		if(tvyscalevalue >= -4 && tvyscalevalue <= 3) {
	           settvyscale = TRUE;
		   parmgiven = TRUE;
		} else {
		   fprintf(stderr, illegalparm, "settvyscale");
		   exit(1);
		}
#ifdef USEXV
	    } else if(strcmp(arg, "-noxvdemo") == 0) {
	        sc.noxvdemo = TRUE;
#endif
#ifdef USE_STRAY
	    } else if(strcmp(arg, "-nosystemtray") == 0) {
	        sc.dontusesystemtray = 1;
#endif
	    } else if(strcmp(arg, "-iconify") == 0) {
	           sc.iconify = 1;
	     } else if(strcmp(arg, "-noconfirm") == 0) {
	           sc.doconfirm = 0;
	    } else {
	        fprintf(stderr, "Unknown parameter: %s, try -help\n", arg);
	    }
        }
    }

    if(!(sc.dpy = XOpenDisplay(displayname))) {
       fprintf(stderr, "%s: failed to open display \"%s\"\n",
		argv[0], XDisplayName(displayname));
       exit(1);
    }

    {
       char *tptr;
       strcpy(sc.filenamebuffer, LOCKFILENAME);
       strncat(sc.filenamebuffer, XDisplayName(displayname), 32);
       if((tptr = strpbrk(sc.filenamebuffer, ":"))) {
          if((tptr = strpbrk(tptr, "."))) *tptr = 0;
       }
    }

    if(sc.myscreen == -1) sc.myscreen = DefaultScreen(sc.dpy);

    for(i = 0; i < SISCTRL_MAXMODES; i++) {
       gl.modetextptr[i] = NULL;
       gl.sizetextptr[i] = NULL;
    }

    if(!printstatus) {

       if((sc.lockfile = open(sc.filenamebuffer, O_RDONLY)) == -1) {
          if(errno != ENOENT) {
             fprintf(stderr, "Error accessing lock file %s\n", sc.filenamebuffer);
	     sc.lockfile = -1;
	     cleanupandexit(3);
          }
       } else {
          close(sc.lockfile);
          if(force) {
             remove(sc.filenamebuffer);
          } else {
             sc.lockfile = -1;
	     if(!parmgiven) {
	        if(gui_showerror_pregui("SiSCtrl found a lockfile which indicates that another session "
	                            "is already running. SiSCtrl should only be run once at a time.\n\n"
				    "If no other session is running and the lockfile is left from "
				    "a previous session which had crashed, click YES to start it anyway, "
				    "or use the -force parameter\n\n"
				    "Start sisctrl anyway?", TRUE, TRUE)) {
	           remove(sc.filenamebuffer);
	        } else {
	           cleanupandexit(3);
	        }
	     } else {
                fprintf(stderr, "Another session is already running. Exiting...\n");
	        fprintf(stderr, "(If no other session is running, try starting with -force\n");
	        fprintf(stderr, "or remove the lockfile %s)\n", sc.filenamebuffer);
                cleanupandexit(3);
	     }
          }
       }

       if((sc.lockfile = open(sc.filenamebuffer, O_RDWR|O_CREAT|O_EXCL|O_TRUNC, S_IRWXU)) == -1) {
	   fprintf(stderr, "Warning: Could not create lock file; another session might be active\n");
	   fprintf(stderr, "         If no other session is running, try starting with -force\n");
	   fprintf(stderr, "         or remove the lockfile %s\n", sc.filenamebuffer);
	   sc.lockfile = -1;
       }

       signal(SIGTERM, signal_handler);
       signal(SIGHUP, signal_handler);
       signal(SIGINT, signal_handler);

    }

    /* About Xinerama:
     * We need to find out whether it's real Xinerama or the SiS pseudo-Xinerama.
     * 1. Check Xinerama extension if Xinerama is active.
     * 2. Check if ISDHXINERAMA is set. This means we can't switch outputs.
     *    If this is not set, we CAN switch outputs REGARDLESS whether or
     *    not Xinerama is detected. (ISDHXINERAMA is only set if the driver
     *    is in Dual Head mode.)
     * 3. As regards sc.myscreen: We need this if Xinerama is detected and
     *    regardless of ISDHXINERAMA: The SISCTRL extension needs this
     *    as an argument, and we need it for switching the mode and changing
     *    Gamma.
     */

    /* Query the Xinerama extension */
    sc.xinerama = FALSE;
    sc.xinescrcnt = 0;
#ifdef HAVE_XINERAMA
    {
       int xi_event_base, xi_error_base;
       if(XineramaQueryExtension(sc.dpy, &sc.xi_event_base, &xi_error_base)) {
          if(XineramaIsActive(sc.dpy)) {
             XineramaScreenInfo *myxi;
	     int xinum;
	     int vmaj, vmin;
             if((sc.xineinfo = XineramaQueryScreens(sc.dpy, &xinum))) {
		sc.xinerama = TRUE;
		sc.xinescrcnt = xinum;
	     }
	     sc.havexineramaevent = FALSE;
	     if(XineramaQueryVersion(sc.dpy, &vmaj, &vmin)) {
	        if(vmaj > 1 || vmin >= 2) {
	           sc.havexineramaevent = TRUE;
	        }
	     }
          }
       }
    }
#endif

    xinewarn = FALSE;
    if(sc.xinerama && (sc.xinescrcnt > 1)) {
       if(!screengiven) xinewarn = TRUE;
    } else {
       sc.myscreen = DefaultScreen(sc.dpy);
    }

    /***********************************************/

    /* At this point, we need a valid sc.myscreen! */

    /***********************************************/

    /* Query SiSCtrl extension */
    if(!SiSCtrlQueryExtension(sc.dpy, &sc_event_base, &sc_error_base)) {
       fprintf(stderr, "Failed to find SISCTRL extension, your X driver is out-dated\n");
#ifndef USE_DEPRECATED_XV
       cleanupandexit(3);
#endif
    } else {
       if(!SiSCtrlQueryVersion(sc.dpy, &sc.scextversion_major, &sc.scextversion_minor)) {
	  fprintf(stderr, "Failed to read SISCTRL extension version information\n");
	  cleanupandexit(3);
       } else {
          sc.sdinterface = TRUE;
	  sc.sdmaxscreens = SiSCtrlGetMaxScreens();

	  /* First thing to do: Find out whether Xinerama is real or pseudo.
	   * We do this by checking all screens until we hit one with
	   * PSEUDOXINERAMA set. PSEUDOXINERAMA and Xinerama are mutually
	   * exclusive, so if it set somewhere, we know Xinerama isn't real.
	   */
	  if(sc.xinerama) {
	     int i, scrbackup = sc.myscreen;
	     for(i = 0; i < sc.sdmaxscreens; i++) {
		sc.myscreen = i;
		if(sendsdcommand(SDC_CMD_GETSDFLAGS)) {
		   if(sc.sdstruct.sdc_result[0] & SiS_SD_PSEUDOXINERAMA) {
		      sc.xinerama = FALSE;
		      scrbackup = DefaultScreen(sc.dpy);
		   }
		   break;
		}
	     }
	     if(i == sc.sdmaxscreens) {
		fprintf(stderr, "No screens registered with SISCTRL extension\n");
		sc.sdinterface = FALSE;
		cleanupandexit(3);
	     }
	     sc.myscreen = scrbackup;
	  }

	  /* If we found real Xinerama, present the user with a chooser if he
	   * did not specify a screen.
	   */
	  if(sc.xinerama && xinewarn) {
	     if(parmgiven) {
	        int i, scnbackup = sc.myscreen;
		char *form1 = ", %d";
	        char *form2 = "%d";
	        Bool firstdone = FALSE;
	        fprintf(stdout, "Xinerama detected. You might want to use the -screen parameter\n");
                fprintf(stdout, "to choose for which screen you want the settings to apply.\n");
	        fprintf(stdout, "SiSCtrl can access screens ");
	        for(i = 0; i < sc.sdmaxscreens; i++) {
	           sc.myscreen = i;
	           if(sendsdcommand(SDC_CMD_GETVERSION)) {
	              fprintf(stdout, firstdone ? form1 : form2, i);
		      firstdone = TRUE;
	           }
	        }
	        sc.myscreen = scnbackup;
	        fprintf(stdout, "\n");
	     } else {
	        if(!sisctrl_xinerama_chooser()) {
		   cleanupandexit(3);
		}
	     }
	  }

	  /* Read SISCTRL version (and enable X logging) */
	  if(!sendsdcommand(SDC_CMD_GETVERSION)) {
	     if(screengiven) {
	        fprintf(stderr, "Failed to read SISCTRL extension info for screen %d\n", sc.myscreen);
	     }
             sc.sdinterface = FALSE;
	     cleanupandexit(3);
          } else {
	     sc.sdversion = sc.sdstruct.sdc_result[0];
	     sc.sdmaxcommand = sc.sdstruct.sdc_result[1];

	     /* Enable X logging for device/mode changes */
	     sc.sdstruct.sdc_parm[0] = 0;
	     sendsdcommand(SDC_CMD_LOGQUIET);
          }
       }
    }

    /* Query the Xv extension */
    sc.havexv = TRUE;
    if(XvQueryExtension(sc.dpy, &xv_major, &xv_minor, &sc.xv_req_base,
					&sc.xv_event_base, &sc.xv_error_base) != Success) {
       fprintf(stderr, "Failed to query xv extension\n");
       if(!sc.sdinterface) {
	  /* This is fatal if the SISCTRL extension is not available */
          cleanupandexit(3);
       } else {
          /* Otherwise we just don't use it. */
          sc.havexv = FALSE;
       }
    }

    if(sc.havexv) {
       Bool found = FALSE;
       if(XvQueryAdaptors(sc.dpy, DefaultRootWindow(sc.dpy), &num_adaptors, &adaptor_info) == Success) {
          if(num_adaptors >= 1) {
             for(i = 0; i < num_adaptors; i++) {
	        if(!strcmp(adaptor_info[i].name, "SIS 300/315/330 series Video Overlay") ||
	           !strcmp(adaptor_info[i].name, "SIS 300/315/330/340/350 series Video Overlay")) {
	           sc.xv_port_nr = adaptor_info[i].base_id;
		   found = TRUE;
                }
	     }
          }
          XvFreeAdaptorInfo(adaptor_info);
          if((!found) || (sc.xv_port_nr == -1)) {
             fprintf(stderr, "SiS 3xx video adaptor not found\n");
	     if(!sc.sdinterface) {
	        fprintf(stderr, "This program is for the SiS 300, 315, 330, 340 and 350 series only\n");
	        fprintf(stderr, "and requires the Xv (XVideo) extension.\n");
                cleanupandexit(3);
	     } else {
	        sc.havexv = FALSE;
	     }
          }
       }
    }

    /* Now get some info from the driver/hardware */
    if(sc.sdinterface) {
       if(!sendsdcommand(SDC_CMD_GETHWINFO)) {
	  sc.sdinterface = FALSE;
	  fprintf(stderr, "Failed to read SISCTRL hardware information\n");
#ifndef USE_DEPRECATED_XV
	  XFree(attributes);
	  cleanupandexit(3);
#endif
       } else {
	  sc.busType = sc.sdstruct.sdc_result[0];
	  sc.pciBus = sc.sdstruct.sdc_result[1];
	  sc.pciDev = sc.sdstruct.sdc_result[2];
	  sc.pciFunc = sc.sdstruct.sdc_result[3];
	  sc.NewROM = sc.sdstruct.sdc_result[4];
	  sc.chipFlags = sc.sdstruct.sdc_result[5];
	  sc.chipType = sc.sdstruct.sdc_result[6];
	  sc.chipRev = sc.sdstruct.sdc_result[7];
	  sc.driverYear = sc.sdstruct.sdc_result[8];
	  sc.driverMonth = sc.sdstruct.sdc_result[9];
	  sc.driverDay = sc.sdstruct.sdc_result[10];
	  sc.driverRev = sc.sdstruct.sdc_result[11];
	  sc.videoRam = sc.sdstruct.sdc_result[12];
	  sc.UMAsize = sc.sdstruct.sdc_result[13];
	  sc.LFBsize = sc.sdstruct.sdc_result[14];
	  strncpy(&sc.biosversion[0], &sc.sdstruct.sdc_buffer[0], 31);
	  sc.havenewhwinfo = TRUE;
	  sc.busisusb = isusb() ? TRUE : FALSE;
       }

       if(!sendsdcommand(SDC_CMD_GETVBFLAGSVERSION)) {
	  fprintf(stderr, "Failed to read vbflags version information\n");
	  if(attributes) XFree(attributes);
	  cleanupandexit(3);
       } else {
          sc.vbflagsvers = sc.sdstruct.sdc_result[0];
	  if(sc.vbflagsvers != 1) {
	     fprintf(stderr, "Unsupported vbflags version (%d)\n", sc.vbflagsvers);
	     if(attributes) XFree(attributes);
	     cleanupandexit(3);
	  }
       }
    }

    if(sc.havexv) {
       attributes = XvQueryPortAttributes(sc.dpy, sc.xv_port_nr, &num_attr);
       if((!attributes) || num_attr == 0) {
          fprintf(stderr, "No Xv attributes found\n");
          if(attributes) XFree(attributes);
	  if(!sc.sdinterface) {
             cleanupandexit(3);
	  } else {
	     sc.havexv = FALSE;
	  }
       }
    }

    /* Check existance of eventually required Xv attributes */

    sc.noxvglobal = FALSE;

    if(sc.sdinterface) {
       /* We only need those attributes for "global" Xinerama video adjustments */
       /* (We can here rely on that fact that a SiS adaptor is the "Xinerama adaptor"
        * because havexv is only set if this is the case.)
	*/
       if(sc.xinerama && sc.havexv) {
          if((!(sc.atom_con = searchandmakeatom("XV_CONTRAST", num_attr, attributes, TRUE)))   ||
             (!(sc.atom_bri = searchandmakeatom("XV_BRIGHTNESS", num_attr, attributes, TRUE))) ||
             (!(sc.atom_col = searchandmakeatom("XV_COLORKEY", num_attr, attributes, TRUE)))) {
             sc.noxvglobal = TRUE;
	  }
       } else {
	  sc.noxvglobal = TRUE;
       }
#ifdef USE_DEPRECATED_XV
    } else {
       /* No need to check havexv - if Xv is not found, we wouldn't be here */
       if((!(sc.atom_con = searchandmakeatom("XV_CONTRAST", num_attr, attributes, TRUE)))            ||
          (!(sc.atom_bri = searchandmakeatom("XV_BRIGHTNESS", num_attr, attributes, TRUE)))          ||
          (!(sc.atom_col = searchandmakeatom("XV_COLORKEY", num_attr, attributes, TRUE)))            ||
          (!(sc.atom_qvf = searchandmakeatom("XV_QUERYVBFLAGS", num_attr, attributes, TRUE)))        ||
          (!(sc.atom_tvx = searchandmakeatom("XV_TVXPOSITION", num_attr, attributes, TRUE)))         ||
          (!(sc.atom_tvy = searchandmakeatom("XV_TVYPOSITION", num_attr, attributes, TRUE)))         ||
          (!(sc.atom_usd = searchandmakeatom("XV_SD_UNLOCKSISDIRECT", num_attr, attributes, TRUE)))  ||
          (!(sc.atom_qvv = searchandmakeatom("XV_SD_QUERYVBFLAGSVERSION", num_attr, attributes, TRUE)))   ||
          (!(sc.atom_gsf = searchandmakeatom("XV_SD_GETSDFLAGS", num_attr, attributes, TRUE)))       ||
          (!(sc.atom_qdd = searchandmakeatom("XV_SD_QUERYDETECTEDDEVICES", num_attr, attributes, TRUE)))  ||
          (!(sc.atom_ct1 = searchandmakeatom("XV_SD_CRT1STATUS", num_attr, attributes, TRUE)))       ||
          (!(sc.atom_cmd = searchandmakeatom("XV_SD_CHECKMODEINDEXFORCRT2", num_attr, attributes, TRUE))) ||
          (!(sc.atom_cmdr= searchandmakeatom("XV_SD_RESULTCHECKMODEINDEXFORCRT2", num_attr, attributes, TRUE))) ||
          (!(sc.atom_svf = searchandmakeatom("XV_SD_SETVBFLAGS", num_attr, attributes, TRUE)))       ||
          (!(sc.atom_taf = searchandmakeatom("XV_SD_SISANTIFLICKER", num_attr, attributes, TRUE)))   ||
          (!(sc.atom_tsa = searchandmakeatom("XV_SD_SISSATURATION", num_attr, attributes, TRUE)))    ||
          (!(sc.atom_tee = searchandmakeatom("XV_SD_SISEDGEENHANCE", num_attr, attributes, TRUE)))   ||
          (!(sc.atom_coc = searchandmakeatom("XV_SD_SISCOLCALIBC", num_attr, attributes, TRUE)))     ||
          (!(sc.atom_cof = searchandmakeatom("XV_SD_SISCOLCALIBF", num_attr, attributes, TRUE)))     ||
          (!(sc.atom_cfi = searchandmakeatom("XV_SD_SISCFILTER", num_attr, attributes, TRUE)))       ||
          (!(sc.atom_yfi = searchandmakeatom("XV_SD_SISYFILTER", num_attr, attributes, TRUE)))       ||
          (!(sc.atom_tco = searchandmakeatom("XV_SD_CHCONTRAST", num_attr, attributes, TRUE)))       ||
          (!(sc.atom_tte = searchandmakeatom("XV_SD_CHTEXTENHANCE", num_attr, attributes, TRUE)))    ||
          (!(sc.atom_tcf = searchandmakeatom("XV_SD_CHCHROMAFLICKERFILTER", num_attr, attributes, TRUE))) ||
          (!(sc.atom_tlf = searchandmakeatom("XV_SD_CHLUMAFLICKERFILTER", num_attr, attributes, TRUE)))   ||
          (!(sc.atom_ccc = searchandmakeatom("XV_SD_CHCVBSCOLOR", num_attr, attributes, TRUE)))      ||
          (!(sc.atom_ovr = searchandmakeatom("XV_SD_CHOVERSCAN", num_attr, attributes, TRUE)))       ||
          (!(sc.atom_sga = searchandmakeatom("XV_SD_ENABLEGAMMA", num_attr, attributes, TRUE)))      ||
          (!(sc.atom_gdv = searchandmakeatom("XV_SD_GETDRIVERVERSION", num_attr, attributes, TRUE))) ||
          (!(sc.atom_ghi = searchandmakeatom("XV_SD_GETHARDWAREINFO", num_attr, attributes, TRUE)))  ||
          (!(sc.atom_gbi = searchandmakeatom("XV_SD_GETBUSID", num_attr, attributes, TRUE)))         ||
          (!(sc.atom_txs = searchandmakeatom("XV_SD_TVXSCALE", num_attr, attributes, TRUE)))         ||
          (!(sc.atom_tys = searchandmakeatom("XV_SD_TVYSCALE", num_attr, attributes, TRUE)))	     ||
          (!(sc.atom_dgl = searchandmakeatom("XV_DISABLE_GRAPHICS_LR", num_attr, attributes, TRUE))) ||
          (!(sc.atom_gfx = searchandmakeatom("XV_DISABLE_GRAPHICS", num_attr, attributes, TRUE)))) {
          XFree(attributes);
          cleanupandexit(3);
       }
#endif
    }

    if(!sd_readulong(SDC_CMD_GETLOCKSTATUS, sc.atom_usd, &xv_ulval)) {
       fprintf(stderr, "Fatal: Failed to read SiSCtrl interface lock status\n");
       if(attributes) XFree(attributes);
       cleanupandexit(3);
    } else {
       if(xv_ulval) {
	  if(!gui_showerror_pregui("SiSCtrl found the SiS Direct Interface in unlocked state.\n\n"
				"This might mean that another application is currently using "
				"it, possibly leading to interface congestions.\n\n"
				"Start sisctrl anyway?", TRUE, TRUE)) {
	     if(attributes) XFree(attributes);
	     cleanupandexit(3);
	  }
       }
    }

    if(sc.sdinterface) {

       if(!sendsdcommand(SDC_CMD_GETALLFLAGS)) {
          fprintf(stderr, "Fatal: Couldn't read SiS Direct flags\n");
	  if(attributes) XFree(attributes);
          cleanupandexit(3);
       }

       if(sc.sdstruct.sdc_result[0] != 1) {
          fprintf(stderr, "Fatal: Invalid vbflags version (got %u, expected 1)\n", sc.sdstruct.sdc_result[0]);
	  if(attributes) XFree(attributes);
          cleanupandexit(3);
       }

       sc.currentvbflags = sc.backupvbflags = sc.sdstruct.sdc_result[1];

       sc.sdflags = sc.sdstruct.sdc_result[2];
       sc.havesdflags2 = TRUE;
       sc.sd2flags = sc.sdstruct.sdc_result[3];
#ifdef USEXV
       if((sc.sd2flags & SiS_SD2_NOOVERLAY) || !sc.havexv) {
          sc.noxvdemo = TRUE;
       }
#endif

       if(sc.sd2flags & SiS_SD2_USEVBFLAGS2) {
          sc.vbflags2 = sc.sdstruct.sdc_result[5];
       }

       if(sc.sd2flags & SiS_SD2_HAVESD34) {
          sc.sd3flags = sc.sdstruct.sdc_result[6];
          sc.sd4flags = sc.sdstruct.sdc_result[7];
       } else {
          sc.sd3flags = sc.sd4flags = 0;
       }

       if( ((unsigned int)sc.sdmaxcommand > (unsigned int)SDC_MAXCOMMAND) ||
           (sc.sd3flags & SiS_SD3_UNUSED)      ||
           (sc.sd4flags & SiS_SD4_UNUSED)      ||
           (sc.vbflags2 & VB2_UNUSED) ) {
          if(!sisctrl_warn_version()) {
             if(attributes) XFree(attributes);
             cleanupandexit(3);
          }
       }

       sc.HaveGss = TRUE;

       sc.HaveBri = TRUE;

       /* No need to check for Xinerama. We can now get directly
        * through to our device (as opposed to the Xv interface)
	*/
       sc.cmd_get_br = SDC_CMD_GETGAMMABRIGHTNESS;
       sc.cmd_set_br = SDC_CMD_SETGAMMABRIGHTNESS;
       sc.cmd_get_newbr = SDC_CMD_GETNEWGAMMABRICON;
       sc.cmd_set_newbr = SDC_CMD_SETNEWGAMMABRICON;

       sc.usenewgamma = FALSE;
       if(sc.sd2flags & SiS_SD2_NEWGAMMABRICON) {
          if(!(sc.sd3flags & SiS_SD3_OLDGAMMAINUSE)) {
             sc.usenewgamma = TRUE;
          }
       }

       sc.HaveRedetect = FALSE;
       if(sc.sdflags & SiS_SD_SUPPORTREDETECT) {
          if(!(sc.sdflags & SiS_SD_ISDUALHEAD)) {
             sc.HaveRedetect = TRUE;
          }
       }

       sc.HaveSepGC2 = FALSE;
       if(sc.sdflags & SiS_SD_SUPPORTSGRCRT2) sc.HaveSepGC2 = TRUE;

       if(sc.xinerama && sc.havexv && !sc.noxvglobal) {
          if(supportvihuesat()) {
             if((!(sc.atom_sat = searchandmakeatom("XV_SATURATION", num_attr, attributes, TRUE))) ||
                (!(sc.atom_hue = searchandmakeatom("XV_HUE", num_attr, attributes, TRUE)))) {
		sc.noxvglobal = TRUE;
             }
          }
       }

       if(sendsdcommand(SDC_CMD_GETDEVICENAME)) {
          int slen = sc.sdstruct.sdc_result[0];
	  if(slen) {
	     strncpy(&sc.devname1[0], &sc.sdstruct.sdc_buffer[0], slen);
	     sc.havedevname1 = TRUE;
	  }
	  sc.devname1[slen] = 0;
       }

       if(sendsdcommand(SDC_CMD_GETMONITORNAME)) {
          int slen = sc.sdstruct.sdc_result[0];
	  if(slen) {
	     strncpy(&sc.monname1[0], &sc.sdstruct.sdc_buffer[0], slen);
	     sc.havemonname1 = TRUE;
	  }
	  sc.monname1[slen] = 0;
       }

       if(sc.sdflags & SiS_SD_ISDUALHEAD) {
          if(sendsdcommand(SDC_CMD_GETDEVICENAME2)) {
             int slen = sc.sdstruct.sdc_result[0];
	     if(slen) {
	        strncpy(&sc.devname2[0], &sc.sdstruct.sdc_buffer[0], slen);
		sc.havedevname2 = TRUE;
	     }
	     sc.devname2[slen] = 0;
          }
          if(sendsdcommand(SDC_CMD_GETMONITORNAME2)) {
             int slen = sc.sdstruct.sdc_result[0];
	     if(slen) {
	        strncpy(&sc.monname2[0], &sc.sdstruct.sdc_buffer[0], slen);
		sc.havemonname2 = TRUE;
	     }
	     sc.monname2[slen] = 0;
          }
       }

#ifdef USE_DEPRECATED_XV

    } else {

       if(XvGetPortAttribute(sc.dpy, sc.xv_port_nr, sc.atom_gsf, &xv_val) != Success) {
          fprintf(stderr, "Fatal: Couldn't read SiS Direct flags\n");
          XFree(attributes);
          cleanupandexit(3);
       }
       sc.sdflags = (unsigned int)xv_val;

       /* First thing to do: Find out whether Xinerama is real or pseudo */
       if(sc.xinerama && (sc.sdflags & SiS_SD_PSEUDOXINERAMA)) {
          sc.xinerama = FALSE;
	  sc.myscreen = DefaultScreen(sc.dpy);
       }

       sc.sd2flags = 0;
       if(sc.atom_gsf2 = searchandmakeatom("XV_SD_GETSDFLAGS2", num_attr, attributes, FALSE)) {
          if(XvGetPortAttribute(sc.dpy, sc.xv_port_nr, sc.atom_gsf2, &xv_val) == Success) {
             sc.sd2flags = (unsigned int)xv_val;
	     sc.havesdflags2 = TRUE;
#ifdef USEXV
	     if(sc.sd2flags & SiS_SD2_NOOVERLAY) {
	        sc.noxvdemo = TRUE;
	     }
#endif
          }
       }

       sc.HaveGss = FALSE;
       if(!(sc.sdflags & SiS_SD_ISDHXINERAMA)) {
          if((sc.atom_gss = searchandmakeatom("XV_SD_GETSCREENSIZE", num_attr, attributes, TRUE))) {
             sc.HaveGss = TRUE;
          }
       }

       sc.HaveBri = FALSE;	/* FIXME: Need a more elegant way to find out */
       if((sc.sdflags & SiS_SD_ISDHXINERAMA) && (sc.myscreen != DefaultScreen(sc.dpy))) {
          if(((sc.atom_brr = searchandmakeatom("XV_SD_STOREDGAMMABRIR2", num_attr, attributes, TRUE))) &&
             ((sc.atom_brg = searchandmakeatom("XV_SD_STOREDGAMMABRIG2", num_attr, attributes, TRUE))) &&
             ((sc.atom_brb = searchandmakeatom("XV_SD_STOREDGAMMABRIB2", num_attr, attributes, TRUE)))) {
             sc.HaveBri = TRUE;
          }
       } else {
          if(((sc.atom_brr = searchandmakeatom("XV_SD_STOREDGAMMABRIR", num_attr, attributes, TRUE))) &&
             ((sc.atom_brg = searchandmakeatom("XV_SD_STOREDGAMMABRIG", num_attr, attributes, TRUE))) &&
             ((sc.atom_brb = searchandmakeatom("XV_SD_STOREDGAMMABRIB", num_attr, attributes, TRUE)))) {
             sc.HaveBri = TRUE;
          }
       }

       sc.usenewgamma = FALSE;

       if(sc.sdflags & SiS_SD_SUPPORTSCALE) {
          if(!(sc.atom_pmd = searchandmakeatom("XV_SD_PANELMODE", num_attr, attributes, TRUE))) {
             sc.sdflags &= ~SiS_SD_SUPPORTSCALE;
          }
       }

       sc.HaveRedetect = FALSE;
       if(sc.sdflags & SiS_SD_SUPPORTREDETECT) {
          if(sc.atom_rdt = searchandmakeatom("XV_SD_REDETECTCRT2DEVICES", num_attr, attributes, TRUE)) {
             if(!(sc.sdflags & SiS_SD_ISDUALHEAD)) {
                sc.HaveRedetect = TRUE;
             }
          }
       }

       sc.HaveSepGC2 = FALSE;
       if(sc.sdflags & SiS_SD_SUPPORTSGRCRT2) {
          if(((sc.atom_g2r = searchandmakeatom("XV_SD_GAMMACRT2R", num_attr, attributes, TRUE))) &&
             ((sc.atom_g2g = searchandmakeatom("XV_SD_GAMMACRT2G", num_attr, attributes, TRUE))) &&
             ((sc.atom_g2b = searchandmakeatom("XV_SD_GAMMACRT2B", num_attr, attributes, TRUE))) &&
	     ((sc.atom_b2r = searchandmakeatom("XV_SD_STOREDGAMMABRIRC2", num_attr, attributes, TRUE))) &&
             ((sc.atom_b2g = searchandmakeatom("XV_SD_STOREDGAMMABRIGC2", num_attr, attributes, TRUE))) &&
             ((sc.atom_b2b = searchandmakeatom("XV_SD_STOREDGAMMABRIBC2", num_attr, attributes, TRUE)))) {
             sc.HaveSepGC2 = TRUE;
          }
       }

       /* Check for hardware specific Xv attributes */
       if(supportvihuesat()) {
          if((!(sc.atom_sat = searchandmakeatom("XV_SATURATION", num_attr, attributes, TRUE))) ||
             (!(sc.atom_hue = searchandmakeatom("XV_HUE", num_attr, attributes, TRUE)))) {
	     XFree(attributes);
	     cleanupandexit(3);
          }
       }

       /* Check for SWITCHCRT Xv attribute */
       if( (!(sc.sdflags & SiS_SD_SUPPORT2OVL)) || (sc.sd2flags & SiS_SD2_SUPPORT760OO) ) {
          if(!(sc.atom_swc = searchandmakeatom("XV_SWITCHCRT", num_attr, attributes, TRUE))) {
             XFree(attributes);
             cleanupandexit(3);
          }
       }

       /* Check for Xv gamma correction */
       if(sc.sdflags & SiS_SD_SUPPORTXVGAMMA1) {
          if( (!(sc.atom_vgr = searchandmakeatom("XV_GAMMA_RED", num_attr, attributes, TRUE)))   ||
              (!(sc.atom_vgg = searchandmakeatom("XV_GAMMA_GREEN", num_attr, attributes, TRUE))) ||
	      (!(sc.atom_vgb = searchandmakeatom("XV_GAMMA_BLUE", num_attr, attributes, TRUE))) ) {
             XFree(attributes);
             cleanupandexit(3);
          }
       }
#endif /* DEPRECATED */
    }

    if(attributes) XFree(attributes);
    attributes = NULL;

#ifdef USE_DEPRECATED_XV
    if(!sc.sdinterface && sc.xinerama && xinewarn) {
       fprintf(stdout, "Xinerama detected. You might want to use the -screen parameter\n");
       fprintf(stdout, "to choose for which screen you want the settings to apply.\n");
    }
#endif

    /* For security reasons, we only allow us to run if the X server allows it */
    if(!(sc.sdflags & SiS_SD_ENABLED)) {
       if(argc <= 1) {
          gui_showerror_pregui("SiSCtrl interface disabled.\n\n"
				"To enable the SiSCtrl interface, place\n\n"
				"\tOption \"EnableSiSCtrl\" \"yes\"\n\n"
				" in the \"Device\" section of your config file.\n", FALSE, FALSE);
       } else {
          fprintf(stderr, "Fatal: SiSCtrl interface is disabled\n");
       }
       cleanupandexit(3);
    }

#ifdef USE_DEPRECATED_XV
    if(!sc.sdinterface) {
       if(XvGetPortAttribute(sc.dpy, sc.xv_port_nr, sc.atom_qvv, &xv_val) != Success) {
          fprintf(stderr, "Fatal: Couldn't read vbflags version\n");
          cleanupandexit(3);
       }

       sc.vbflagsvers = xv_val;

       if(xv_val != 1){
          fprintf(stderr, "Fatal: Unsupported vbflags version (%d)\n", xv_val);
          cleanupandexit(3);
       }

       if(XvGetPortAttribute(sc.dpy, sc.xv_port_nr, sc.atom_qvf, &xv_val) != Success) {
          fprintf(stderr, "Fatal: Couldn't read vbflags\n");
          cleanupandexit(3);
       }
       sc.currentvbflags = sc.backupvbflags = (unsigned int)xv_val;

       if(XvGetPortAttribute(sc.dpy, sc.xv_port_nr, sc.atom_ghi, &xv_val) != Success) {
          fprintf(stderr, "Fatal: Couldn't read SiS hardware info\n");
          cleanupandexit(3);
       }
       sc.hwinfo = (unsigned int)xv_val;
       sc.busisusb = isusb() ? TRUE : FALSE;

       if(XvGetPortAttribute(sc.dpy, sc.xv_port_nr, sc.atom_gbi, &xv_val) != Success) {
          fprintf(stderr, "Fatal: Couldn't read busID info\n");
          cleanupandexit(3);
       }
       sc.busid = (unsigned int)xv_val;

       if(XvGetPortAttribute(sc.dpy, sc.xv_port_nr, sc.atom_gdv, &xv_val) != Success) {
          fprintf(stderr, "Fatal: Couldn't read SiS driver version\n");
          cleanupandexit(3);
       }
       sc.driverversion = (unsigned int)xv_val;

       if((sc.driverversion & 0x00ffffff) < 0x030c04) {  /* 2003/12/04 */
          fprintf(stderr, "Fatal: This version of SiSCtrl is too new for this version of the SiS X driver\n");
          cleanupandexit(3);
       }
    }
#endif

    if(!sd_readulong(SDC_CMD_GETDETECTEDDEVICES, sc.atom_qdd, &xv_ulval)) {
       fprintf(stderr, "Fatal: Couldn't read detected devices\n");
       cleanupandexit(3);
    }
    sc.detecteddevices = xv_ulval;

    if(!sd_readulong(SDC_CMD_GETCRT1STATUS, sc.atom_ct1, &xv_ulval)) {
       fprintf(stderr, "Fatal: Couldn't read CRT1 status\n");
       cleanupandexit(3);
    }
    sc.CRT1isoff = sc.CRT1isoffbackup = xv_ulval ? 0 : 1;

    if(!sd_readulong(SDC_CMD_GETGAMMASTATUS, sc.atom_sga, &xv_ulval)) {
       fprintf(stderr, "Fatal: Couldn't read gamma enable/disable\n");
       cleanupandexit(3);
    }
    xv_ulval &= 0x0f;
    if(!(sc.sdflags & SiS_SD_SUPPORTSGRCRT2)) xv_ulval &= 0x07;
    if(!(sc.sdflags & SiS_SD_SUPPORTXVGAMMA1)) xv_ulval &= 0x0b;
    sc.gammaflags = sc.backupgammaflags = xv_ulval;

    if(!sd_readulong(SDC_CMD_GETCHTVOVERSCAN, sc.atom_ovr, &xv_ulval)) {
       fprintf(stderr, "Fatal: Couldn't read overscan setting\n");
       cleanupandexit(3);
    }
    sc.choverscanstatus = sc.backupoverscan = xv_ulval & 0x03;

    sc.lcdscaler = sc.backuplcdscaler = sc.givenlcdscale = 0;
    if(sc.sdflags & SiS_SD_SUPPORTSCALE) {
       if(!sd_readulong(SDC_CMD_GETPANELMODE, sc.atom_pmd, &xv_ulval)) {
          fprintf(stderr, "Fatal: Couldn't read panel mode setting\n");
          cleanupandexit(3);
       }
       sc.lcdscaler = sc.backuplcdscaler = sc.givenlcdscale = xv_ulval & 0x0f;
    }

    if(sc.havexv && (!(sc.sd2flags & SiS_SD2_NOOVERLAY))) {
       sc.backup_dgl = newgetxvval(SDC_CMD_GETXVDISABLEGFXLR, sc.atom_dgl);
       sc.backup_gfx = newgetxvval(SDC_CMD_GETXVDISABLEGFX, sc.atom_gfx);
       sc.havedglgfx = TRUE;
    } else {
       sc.havedglgfx = FALSE;
    }

    /* Query the XVidModExtension */
    if(!XF86VidModeQueryExtension(sc.dpy, &EventBase, &ErrorBase)) {
       fprintf(stderr, "Fatal: Unable to query vidmode extension information\n");
       cleanupandexit(2);
    }

    if(!XF86VidModeQueryVersion(sc.dpy, &MajorVersion, &MinorVersion)) {
       fprintf(stderr, "Fatal: Unable to query vidmode extension version\n");
       cleanupandexit(2);
    }

    if(MajorVersion < VMMINMAJOR || (MajorVersion == VMMINMAJOR && MinorVersion < VMMINMINOR)) {
       fprintf(stderr, "Fatal: Found VidModeExtension version (%d.%d)\n", MajorVersion, MinorVersion);
       fprintf(stderr, "Minimum required version for this program is %d.%d\n", VMMINMAJOR, VMMINMINOR);
       cleanupandexit(2);
    }

    {
       int perms = 0;
       if(!XF86VidModeGetPermissions(sc.dpy, sc.myscreen, &perms)) {
	  fprintf(stderr, "Warning: Failed to query permissions from vidmode extension\n");
	  fprintf(stderr, "Will try to continue...\n");
       } else if(!(perms & XF86VM_WRITE_PERMISSION)) {
	  fprintf(stderr, "Fatal: No write permissions to vidmode extension.\n");
	  fprintf(stderr, "If SiSCtrl connected to a remote display server, add\n");
	  fprintf(stderr, "\tAllowNonLocalXvidtune\n");
	  fprintf(stderr, "to the \"ServerFlags\" section of the remote server's config file.\n");
	  cleanupandexit(2);
       }
    }

    /* Read all needed data from VidMode extension */
    if(!XF86VidModeGetAllModeLines(sc.dpy, sc.myscreen, &sc.vmmodecount, &sc.vmmodelines)) {
       fprintf(stderr, "Fatal: Failed to query modelines from vidmode extension\n");
       cleanupandexit(2);
    }

    if(!XF86VidModeGetGamma(sc.dpy, sc.myscreen, &sc.currentgamma)) {
       fprintf(stderr, "Fatal: Failed to query gamma correction from vidmode extension\n");
       cleanupandexit (2);
    }
    memcpy(&sc.newgamma, &sc.currentgamma, sizeof(XF86VidModeGamma));
    sc.backup_gred = sc.currentgamma.red;
    sc.backup_ggreen = sc.currentgamma.green;
    sc.backup_gblue = sc.currentgamma.blue;

    if(sc.HaveBri) {
       if(sc.sdinterface) {
          if(sc.usenewgamma) {
             sendsdcommand(sc.cmd_get_newbr);
             sc.newgamma_bri[0] = sc.newgamma_bri_b[0] = (float)((int)sc.sdstruct.sdc_result[0] - 1000) / 1000.0;
             sc.newgamma_bri[1] = sc.newgamma_bri_b[1] = (float)((int)sc.sdstruct.sdc_result[1] - 1000) / 1000.0;
             sc.newgamma_bri[2] = sc.newgamma_bri_b[2] = (float)((int)sc.sdstruct.sdc_result[2] - 1000) / 1000.0;
             sc.newgamma_con[0] = sc.newgamma_con_b[0] = (float)((int)sc.sdstruct.sdc_result[3] - 1000) / 1000.0;
             sc.newgamma_con[1] = sc.newgamma_con_b[1] = (float)((int)sc.sdstruct.sdc_result[4] - 1000) / 1000.0;
             sc.newgamma_con[2] = sc.newgamma_con_b[2] = (float)((int)sc.sdstruct.sdc_result[5] - 1000) / 1000.0;
          } else {
             sendsdcommand(sc.cmd_get_br);
	     sc.gamma_max[0] = sc.gamma_max_b[0] = (float)sc.sdstruct.sdc_result[0] / 1000;
             sc.gamma_max[1] = sc.gamma_max_b[1] = (float)sc.sdstruct.sdc_result[1] / 1000;
             sc.gamma_max[2] = sc.gamma_max_b[2] = (float)sc.sdstruct.sdc_result[2] / 1000;
          }
#ifdef USE_DEPRECATED_XV
       } else {
          sc.gamma_max[0] = sc.gamma_max_b[0] = (float)getxvval(sc.atom_brr) / 1000;
          sc.gamma_max[1] = sc.gamma_max_b[1] = (float)getxvval(sc.atom_brg) / 1000;
          sc.gamma_max[2] = sc.gamma_max_b[2] = (float)getxvval(sc.atom_brb) / 1000;
#endif
       }
    }

    if(sc.sdflags & SiS_SD_SUPPORTXVGAMMA1) {
       if(sc.sdinterface) {
          sendsdcommand(SDC_CMD_GETXVGAMMA);
	  sc.xvgammared   = (float)sc.sdstruct.sdc_result[0] / 1000;
          sc.xvgammagreen = (float)sc.sdstruct.sdc_result[1] / 1000;
          sc.xvgammablue  = (float)sc.sdstruct.sdc_result[2] / 1000;
#ifdef USE_DEPRECATED_XV
       } else {
          sc.xvgammared   = (float)getxvval(sc.atom_vgr) / 1000;
          sc.xvgammagreen = (float)getxvval(sc.atom_vgg) / 1000;
          sc.xvgammablue  = (float)getxvval(sc.atom_vgb) / 1000;
#endif
       }
    }

    if(sc.HaveSepGC2) {
       if(sc.sdinterface) {
          if(sc.usenewgamma) {
             sendsdcommand(SDC_CMD_GETGETNEWGAMMACRT2);
             sc.gamma2r = sc.newgamma2r = sc.backup_g2r = (float)sc.sdstruct.sdc_result[0] / 1000;
             sc.gamma2g = sc.newgamma2g = sc.backup_g2g = (float)sc.sdstruct.sdc_result[1] / 1000;
             sc.gamma2b = sc.newgamma2b = sc.backup_g2b = (float)sc.sdstruct.sdc_result[2] / 1000;
             sc.newgamma_bri2[0] = sc.newgamma_bri2_b[0] = (float)((int)sc.sdstruct.sdc_result[3] - 1000) / 1000.0;
             sc.newgamma_bri2[1] = sc.newgamma_bri2_b[1] = (float)((int)sc.sdstruct.sdc_result[4] - 1000) / 1000.0;
             sc.newgamma_bri2[2] = sc.newgamma_bri2_b[2] = (float)((int)sc.sdstruct.sdc_result[5] - 1000) / 1000.0;
             sc.newgamma_con2[0] = sc.newgamma_con2_b[0] = (float)((int)sc.sdstruct.sdc_result[6] - 1000) / 1000.0;
             sc.newgamma_con2[1] = sc.newgamma_con2_b[1] = (float)((int)sc.sdstruct.sdc_result[7] - 1000) / 1000.0;
             sc.newgamma_con2[2] = sc.newgamma_con2_b[2] = (float)((int)sc.sdstruct.sdc_result[8] - 1000) / 1000.0;
          } else {
             sendsdcommand(SDC_CMD_GETGETGAMMACRT2);
	     sc.gamma2r = sc.newgamma2r = sc.backup_g2r = (float)sc.sdstruct.sdc_result[0] / 1000;
             sc.gamma2g = sc.newgamma2g = sc.backup_g2g = (float)sc.sdstruct.sdc_result[1] / 1000;
             sc.gamma2b = sc.newgamma2b = sc.backup_g2b = (float)sc.sdstruct.sdc_result[2] / 1000;
             sc.gamma_bri2[0] = sc.gamma_bri2_b[0] = (float)sc.sdstruct.sdc_result[3] / 1000;
             sc.gamma_bri2[1] = sc.gamma_bri2_b[1] = (float)sc.sdstruct.sdc_result[4] / 1000;
             sc.gamma_bri2[2] = sc.gamma_bri2_b[2] = (float)sc.sdstruct.sdc_result[5] / 1000;
          }
#ifdef USE_DEPRECATED_XV
       } else {
          sc.gamma2r = sc.newgamma2r = sc.backup_g2r = (float)getxvval(sc.atom_g2r) / 1000;
          sc.gamma2g = sc.newgamma2g = sc.backup_g2g = (float)getxvval(sc.atom_g2g) / 1000;
          sc.gamma2b = sc.newgamma2b = sc.backup_g2b = (float)getxvval(sc.atom_g2b) / 1000;
          sc.gamma_bri2[0] = sc.gamma_bri2_b[0] = (float)getxvval(sc.atom_b2r) / 1000;
          sc.gamma_bri2[1] = sc.gamma_bri2_b[1] = (float)getxvval(sc.atom_b2g) / 1000;
          sc.gamma_bri2[2] = sc.gamma_bri2_b[2] = (float)getxvval(sc.atom_b2b) / 1000;
#endif
       }
    }

    /* Now deal with the RandR extension */
    sc.haverandr = FALSE;
#ifdef HAVE_RANDR
    if(XRRQueryExtension(sc.dpy, &rr_event_base, &rr_error_base)) sc.haverandr = TRUE;
    if(sc.haverandr) {
       XRRQueryVersion(sc.dpy, &MajorVersion, &MinorVersion);
       if((MajorVersion < RRMINMAJOR) ||
         ((MajorVersion == RRMINMAJOR) && (MinorVersion < RRMINMINOR))) {
          fprintf(stderr, "Fatal: Found RANDR Extension version (%d.%d)\n", MajorVersion, MinorVersion);
          fprintf(stderr, "Minimum required version for this program is %d.%d\n", RRMINMAJOR, RRMINMINOR);
          sc.haverandr = FALSE;
       } else {
          /* Need to catch system error here due to bug in RandR <= 4.3.0 */
          XErrorHandler oldErrHandler = XSetErrorHandler(myerrorhandler);
	  sc.xerror = FALSE;
          sc.root = RootWindow(sc.dpy, sc.myscreen);
          sc.sconf = XRRGetScreenInfo(sc.dpy, sc.root);
	  XSync(sc.dpy, False);
	  XSetErrorHandler(oldErrHandler);
          if((!sc.sconf) || (sc.xerror)) {
             fprintf(stderr, "Failed to get screen info from RandR\n");
             sc.haverandr = FALSE;
          } else {
             Rotation temp;
             sc.rrcurrent_size = sc.rrsize_backup =
	         XRRConfigCurrentConfiguration(sc.sconf, &sc.rrcurrent_rotation);
	     sc.rrrotation_backup = sc.rrcurrent_rotation;
	     sc.rrcurrent_rate = sc.rrrate_backup = XRRConfigCurrentRate(sc.sconf);
	     sc.supprotations = XRRConfigRotations(sc.sconf, &temp);
          }
       }
    }
#endif

    if(printstatus) {
       writeoutstatus();
       cleanupandexit(0);
       exit(0);
    }

    if(listmodes) {
       for(i = 0; (i < sc.vmmodecount) && (i < SISCTRL_MAXMODES); i++) {
          if(sc.sdflags & SiS_SD_ISMERGEDFB) {
             fprintf(stdout, "%d: %dx%d\n",
                i,
		sc.vmmodelines[i]->hdisplay,
		sc.vmmodelines[i]->vdisplay);
          } else {
             int refresh = calcrefresh(sc.vmmodelines[i]);
             fprintf(stdout, "%d: %dx%d (%d Hz%s)\n",
                i,
                sc.vmmodelines[i]->hdisplay,
		sc.vmmodelines[i]->vdisplay,
		refresh,
		(sc.vmmodelines[i]->flags & V_INTERLACE) ? " I" : "");
	  }
       }
       cleanupandexit(0);
       exit(0);
    }

    if(sc.HaveBri) {
       memcpy(&sc.newgamma, &sc.currentgamma, sizeof(XF86VidModeGamma));
       setgamma();
    } else {
       fprintf(stderr, "This version of the SiS X driver does not support storing gamma brightness\n");
    }

    /*****************************************/
    /*             COMMAND LINE              */
    /*****************************************/

    /* Check lcd scaling/centering */

    if(!(sc.sdflags & SiS_SD_SUPPORTSCALE)) {
       if(setlcdscale) {
          fprintf(stderr, "Adjusting LCD scaling mode not supported\n");
          setlcdscale = FALSE;
          doexit = TRUE;
       }
       if(!(sc.sdflags & SiS_SD_SUPPORTCENTER)) {
          if(setlcdcenter) {
             fprintf(stderr, "Adjusting LCD centering mode not supported\n");
             setlcdcenter = FALSE;
             doexit = TRUE;
          }
       }
    }

    if(setlcdscale || setlcdcenter) {
       doexit = TRUE;
       if(setlcdscale) {
          sc.givenlcdscale &= 0xfc;
          sc.givenlcdscale |= (newlcdscaling & 0x03);
       }
       if(setlcdcenter) {
          sc.givenlcdscale &= 0xf3;
          sc.givenlcdscale |= (newlcdcenter & 0x0c);
       }

       if(sc.givenlcdscale != sc.lcdscaler) {
          if(UnlockSiSDirect(1)) {
	     unsigned int val;
	     if(!sd_writeulong(SDC_CMD_SETPANELMODE, sc.atom_pmd, sc.givenlcdscale)) {
                fprintf(stderr, xverrorstring);
             }
	     if(!sd_readulong(SDC_CMD_GETPANELMODE, sc.atom_pmd, &val)) {
                fprintf(stderr, xverrorstring);
             } else {
	        sc.lcdscaler = val & 0x0f;
	     }
	     UnlockSiSDirect(0);
	  }
          if(!setcrt1 && !setcrt2 && !setmode && !resizetomode) {
             setmode = TRUE;
             modetoset = finddmhistory();
          }
       }
    }

#define SISNEWCMDLINE

    /* New: Cumulative handling of -setmode, -setcrt1, -setcrt2 */
#ifdef SISNEWCMDLINE
    if(setmode || resizetomode || setcrt1 || setcrt2 || settvstd || settvos || settvas) {

       int dm_index, dm_current, iscurrentmode, randrindex = 0;
       int newcrt1, crt1isoff, hindex;
       int currentcrt1, currentcrt2, currenttvstd, currenttvst, currenttvas, currenttvos;
       int quickcheck = 0, defersetcrt1;

       doexit = TRUE;

#ifdef HAVE_RANDR
       if(sc.haverandr && resizetomode) randrindex = 1;
#endif

       dm_index = dm_current = finddmhistory();
       iscurrentmode = 1;

       if(setmode) {
          if(modetoset < 0 || modetoset >= sc.vmmodecount || modetoset >= SISCTRL_MAXMODES) {
             fprintf(stderr, "Bad mode number\n");
             setmode = resizetomode = FALSE;
          } else {
             dm_index = modetoset;
             iscurrentmode = 0;
          }
          if(dm_index == dm_current) {
             setmode = FALSE;
          }
          if(setmode && !resizetomode) {
	     if(!checkmodeforscreen(dm_index, FALSE, 0, 0, 0, 0)) {
	        fprintf(stderr, "Display mode too large for current desktop size\n");
	        setmode = FALSE;
	     }
          }
       }

       if(!setmode) {
          dm_index = dm_current;
	  iscurrentmode = 1;
       }

       /* setcrt1, setcrt2 not supported in dhm or if no video bridge present */
       if((sc.sdflags & SiS_SD_ISDUALHEAD) || (!vbhavevb())) {
          quickcheck = TRUE;
          if(setcrt1) {
             fprintf(stderr, "-setcrt1 not supported in dual head mode or if no video bridge present\n");
             setcrt1 = FALSE;
          }
          if(setcrt2 || settvstd || settvos || settvas) {
              fprintf(stderr, "-set[crt2|tvstandard|tvoverscan|tvaspect] not supported in dual head mode "
				"or if no video bridge present\n");
	      setcrt2 = settvstd = settvos = settvas = FALSE;
          }
          if(setlcdscale) {
             fprintf(stderr, "-setlcd[scaling|centering] not supported in dual head mode or if no video bridge present\n");
             setlcdscale = setlcdcenter = FALSE;
          }
       }

       /* Build our bitfield of supported output devices */
       sc.modecheckfield[dm_index] = rebuildmodefield(dm_index);
       if(dm_index != dm_current) {
          sc.modecheckfield[dm_current] = rebuildmodefield(dm_current);
       }

       /* Check (desired | current) display mode for (desired | current) crt1 */

       newcrt1 = currentcrt1 = (sc.currentvbflags & CRT1_LCDA) ? C1_MI_LCDA :
						   (sc.CRT1isoff ? C1_MI_OFF : C1_MI_VGA);

       if(!quickcheck && (setmode || setcrt1)) {

          if(setcrt1) {
             newcrt1 = setcrt1index;
          }

          if(!check_apply_CRT1(newcrt1, dm_index, iscurrentmode, 1))
             goto bailout;
       }

       /* Check (desired | current) display mode for (desired | current) crt2/tvstd/etc */

       currentcrt2 = getcrt2typeindex(sc.currentvbflags);

       if(!quickcheck && (setmode || setcrt2 || settvstd || settvos || settvas)) {

          currenttvstd = gettvtypeindex(sc.currentvbflags);
	  currenttvst = gettvstindex(sc.currentvbflags);
          currenttvas = gettvasindex(sc.currentvbflags);
          currenttvos = sc.choverscanstatus;

          if(!vbsupportstv()) {
             settvstd = settvos = settvas = FALSE;
          } else if(setcrt2) {
             if(vbsupportsettvtype()) {
	        if(sc.givencrt2index == C2_MI_TV_TV) {
	           if(sc.detecteddevices & TV_AVIDEO)
	              sc.givencrt2index = C2_MI_TV_CVBS;
	           else
	              sc.givencrt2index = C2_MI_TV_SVHS;
	           fprintf(stderr, "TV plug type not specified, default is detected or svideo\n");
	        }
	        if(sc.givencrt2index == C2_MI_TV_SCART) {
	           sc.giventvstdindex = C2_MI_TV_PAL;
	           if(settvstd && (sc.giventvstdindex != C2_MI_TV_PAL)) {
	              fprintf(stderr, "SCART only supported for PAL\n");
	           }
		   settvstd = TRUE;
	        }
	     } else if((sc.givencrt2index == C2_MI_TV_SVHS) ||
	               (sc.givencrt2index == C2_MI_TV_CVBS) ||
	               (sc.givencrt2index == C2_MI_TV_CVSV) ||
		       (sc.givencrt2index == C2_MI_TV_SCART)) {
	        sc.givencrt2index = C2_MI_TV_TV;
	     }
          }

          /* Check settvstd validity */
          if(settvstd) {
	     if(!(sc.sdflags & SiS_SD_SUPPORTPALMN)) {
	        if((sc.giventvstdindex == C2_MI_TV_PALM) ||
		   (sc.giventvstdindex == C2_MI_TV_PALN)) {
		   fprintf(stderr, "Hardware does not support PALM/PALN, using PAL\n");
		   sc.giventvstdindex = C2_MI_TV_PAL;
	        }
	     }
	     if(!(sc.sdflags & SiS_SD_SUPPORTNTSCJ)) {
	        if(sc.giventvstdindex == C2_MI_TV_NTSCJ) {
		   fprintf(stderr, "Hardware does not support NTSC-J, using NTSC\n");
		   sc.giventvstdindex = C2_MI_TV_NTSC;
	        }
	     }
	     if(!setcrt2 && (currentcrt2 == C2_MI_TV_SCART)) {
	        if(sc.giventvstdindex != C2_MI_TV_PAL) {
	           fprintf(stderr, "SCART only supported for PAL\n");
	           sc.giventvstdindex = C2_MI_TV_PAL;
	        }
	     }
          }

          if(!setcrt2) {
	     sc.givencrt2index = currentcrt2;
	     sc.giventvstindex = currenttvst;
          }
          if(!settvstd) {
	     sc.giventvstdindex = currenttvstd;
          }
          if(!settvas) {
 	     sc.giventvasindex = currenttvas;
          }

          /* Check settvos validity */
          if(settvos) {
	     if(!vbsupportsoverscan()) {
	        fprintf(stderr, "Overscan only supported on Chrontel TV encoders\n");
	        settvos = FALSE;
	     } else {
	        if(sc.giventvosindex == C2_MI_OV_SUPEROVERSCAN) {
		   if(!(sc.sdflags & SiS_SD_SUPPORTSOVER)) {
		      fprintf(stderr, "soverscan not supported on this hardware\n");
		      sc.giventvosindex = C2_MI_OV_OVERSCAN;
		   } else if(sc.giventvstdindex != C2_MI_TV_PAL) {
		      fprintf(stderr, "soverscan only supported for PAL\n");
		      sc.giventvosindex = C2_MI_OV_OVERSCAN;
		   }
	        }
	     }
          }

          if(!settvos) {
	     sc.giventvosindex = currenttvos;
          }

          hindex = (sc.givencrt2index == C2_MI_TV_YPBPR) ? sc.giventvstindex : sc.giventvstdindex;

          crt1isoff = (newcrt1 == C1_MI_OFF) ? 1 : 0;

          if(vbhavevb()) {
             if(!check_apply_CRT2(sc.givencrt2index, dm_index, crt1isoff, hindex, iscurrentmode))
                goto bailout;
          }

       }

       if( ((setcrt1 && (newcrt1 == C1_MI_OFF)) ||
            (!setcrt1 && (currentcrt1 == C1_MI_OFF))) &&
           ((setcrt2 && (sc.givencrt2index == C2_MI_TV_OFF)) ||
            (!setcrt2 && (currentcrt2 == C2_MI_TV_OFF))) ) {
          fprintf(stderr, "Can't switch off both CRT1 and CRT2\n");
          goto bailout;
       }

       /* Now for the real thing: */

       /* A note on dynamic modelists (X driver feature since 2005/10/09):
        * Switching output devices will make the driver redo its modelists.
        * In gui mode, we update the modelist and randr sizelist.
        * In non-gui mode, we leave our lists alone, because we work with
        * indices here and the desired mode is sent to the driver in form
        * of a modeline. The desired and current modelines shouldn't have
        * changed during an output device switch.
        * There is just one problem: Wide/Non-wide modes. If the driver
        * operated with non-wide modes before the switch, and with wide
        * after the switch, our current and desired modelines might have
        * vanished. However, this is on the TODO list.
        * The sizelist doesn't need to be rebuilt; the sizes are re-read
        * from RandR in handle_apply_mode and the desired size is searched
        * then.
        */

       defersetcrt1 = FALSE;

       if(setmode && setcrt1 && (currentcrt1 != newcrt1)) {
          if((currentcrt1 != C1_MI_VGA) &&
             (!sisctrl_checkmodeforcrtx(0, currentcrt1, dm_index, 0, 0))) {
             handle_apply_CRT1(C1_MI_VGA, dm_index, 0, 1);
             /* fprintf(stderr, "CRT1 -> VGA\n"); */
          }
       }
       if(setmode && (setcrt2 || settvstd || settvos) &&
                   ( (currentcrt2 != sc.givencrt2index)   ||
                     (currenttvstd != sc.giventvstdindex) ||
                     (currenttvos != sc.giventvosindex) ) ) {
          int hindex = (currentcrt2 == C2_MI_TV_YPBPR) ? currenttvst : currenttvstd;
          if((currentcrt2 != C2_MI_TV_OFF) &&
             (!sisctrl_checkmodeforcrtx(1, currentcrt2, dm_index, hindex, hindex))) {
             int backup = sc.givencrt2index;
             sc.givencrt2index = currentcrt2 = C2_MI_TV_OFF;
             handle_apply_CRT2(dm_index, 0);
             sc.givencrt2index = backup;
             /* fprintf(stderr, "CRT2 -> OFF\n"); */
          }
       }
       if(setmode || resizetomode) {
          if(!(handle_apply_mode(dm_index, randrindex, 0))) {
	     fprintf(stderr, "-setmode or -resize failed\n");
          }
       }
       if(setcrt1) {
          if((currentcrt2 == C2_MI_TV_OFF) && (newcrt1 == C1_MI_OFF)) {
             defersetcrt1 = TRUE;
          } else if(!(handle_apply_CRT1(newcrt1, dm_index, 1, 1))) {
             fprintf(stderr, "-setcrt1 failed (1)\n");
          }
       }
       if(setcrt2 || settvstd || settvos || settvas) {
          if(!(handle_apply_CRT2(dm_index, 1))) {
             fprintf(stderr, "-set[crt2|tvstd|tvaspect|tvoverscan] failed\n");
          }
       }
       if(defersetcrt1) {
          if(!(handle_apply_CRT1(newcrt1, dm_index, 1, 1))) {
             fprintf(stderr, "-setcrt1 failed (2)\n");
          }
       }

    }

bailout:

#endif	/* // End of new */

#ifndef SISNEWCMDLINE
    if(setmode || resizetomode) {

       int dm_index, randrindex = 0;
#ifdef HAVE_RANDR
       if(sc.haverandr && resizetomode) randrindex = 1;
#endif
       if(setmode) {
          if(modetoset < 0 || modetoset >= sc.vmmodecount || modetoset >= SISCTRL_MAXMODES) {
             fprintf(stderr, "bad mode number\n");
          } else {
             dm_index = modetoset;
          }
       } else {
          dm_index = finddmhistory();
       }
       if(!(handle_apply_mode(dm_index, randrindex, 0))) {
	  fprintf(stderr, "setmode or resize operation failed\n");
       }
       if(setcrt1 || setcrt2 || settvstd || settvos || settvas) {
	  fprintf(stderr, "-setmode|-resize and -set[crt1|crt2|tvstandard|tvoverscan|tvaspect] cannot be handled at the same time\n");
       }

    } else if(setcrt1) {

       if((!(sc.sdflags & SiS_SD_ISDUALHEAD)) && (vbhavevb())) {
          if(!(handle_apply_CRT1(setcrt1index, sisctrl_dm_get_current_mode_index(), 1, 0))) {
	     fprintf(stderr, "setcrt1 operation failed\n");
	  }
       } else {
          fprintf(stderr, "setcrt1 not allowed in dual head mode or if no video bridge present\n");
       }
       if(setmode || resizetomode || setcrt2 || settvstd || settvos || settvas) {
          fprintf(stderr, "-setcrt1 and -set[mode|crt2|tvstandard|tvoverscan|tvaspect]|-resize cannot be handled at the same time\n");
       }

    } else if(setcrt2 || settvstd || settvos || settvas) {

       if((!(sc.sdflags & SiS_SD_ISDUALHEAD)) && (vbhavevb())) {
          if(setcrt2) {
             /* Cannot set LCD if no LCD detected */
             if(sc.givencrt2index == C2_MI_TV_LCD) {
	        if(!(sc.detecteddevices & CRT2_LCD)) {
	           fprintf(stderr, "Cannot set CRT2 type to LCD; no LCD detected\n");
	           setcrt2 = FALSE;
		   doexit = TRUE;
	        }
	     }
	     /* VGA only on 301/301B/301C/302B */
	     if(sc.givencrt2index == C2_MI_TV_VGA) {
	        if(!(vbsupportsvga2())) {
		   fprintf(stderr, "Hardware does not support CRT2 VGA output\n");
		   setcrt2 = FALSE;
		   doexit = TRUE;
	        }
	     }
	     if(sc.givencrt2index == C2_MI_TV_HIVI) {
	        if(!(sc.sdflags & SiS_SD_SUPPORTHIVISION)) {
		   fprintf(stderr, "Hardware does not support HiVision output\n");
		   setcrt2 = FALSE;
		   doexit = TRUE;
		} else settvstd = FALSE;
	     }
	     if(sc.givencrt2index == C2_MI_TV_YPBPR) {
		if(!(sc.sdflags & SiS_SD_SUPPORTYPBPR)) {
		   fprintf(stderr, "Hardware does not support YPbPr output\n");
		   setcrt2 = FALSE;
		   doexit = TRUE;
		} else if((sc.giventvstindex == C2_MI_ST_625I) && (!(sc.sd2flags & SiS_SD2_SUPPORT625I))) {
		   fprintf(stderr, "Hardware or driver does not support YPbPr 576i output\n");
		   setcrt2 = FALSE;
		   doexit = TRUE;
		} else if((sc.giventvstindex == C2_MI_ST_625P) && (!(sc.sd2flags & SiS_SD2_SUPPORT625P))) {
		   fprintf(stderr, "Hardware or driver does not support YPbPr 576p output\n");
		   setcrt2 = FALSE;
		   doexit = TRUE;
		} else settvstd = FALSE;
	     }

	     if(vbsupportstv()) {
	        if(vbsupportsettvtype()) {
		   if(sc.givencrt2index == C2_MI_TV_TV) {
	              if(sc.detecteddevices & TV_AVIDEO)
		         sc.givencrt2index = C2_MI_TV_CVBS;
		      else
	                 sc.givencrt2index = C2_MI_TV_SVHS;
		      fprintf(stderr, "TV plug type not specified, default is detected or svideo\n");
	           }
	           if(sc.givencrt2index == C2_MI_TV_SCART) {
	              if(!(vbsupportsscart())) {
		         fprintf(stderr, "This video bridge does not support SCART output\n");
		         setcrt2 = FALSE;
		         doexit = TRUE;
		      } else {
		         sc.giventvstdindex = C2_MI_TV_PAL;
		         settvstd = TRUE;
		      }
	           }
	        } else if(vbsupportstv()) {
	           if((sc.givencrt2index == C2_MI_TV_SVHS) ||
	              (sc.givencrt2index == C2_MI_TV_CVBS) ||
		      (sc.givencrt2index == C2_MI_TV_SCART)) {
		      sc.givencrt2index = C2_MI_TV_TV;
	           }
		}
	     } else if((sc.givencrt2index == C2_MI_TV_SVHS) ||
	               (sc.givencrt2index == C2_MI_TV_CVBS) ||
		       (sc.givencrt2index == C2_MI_TV_SCART) ||
		       (sc.givencrt2index == C2_MI_TV_TV)) {
		fprintf(stderr, "This hardware does not support TV output\n");
		setcrt2 = FALSE;
		doexit = TRUE;
	     }
          }
	  if(settvstd) {
	     if(vbsupportstv()) {
	        if(!(sc.sdflags & SiS_SD_SUPPORTPALMN)) {
	           if((sc.giventvstdindex == C2_MI_TV_PALM) ||
		      (sc.giventvstdindex == C2_MI_TV_PALN)) {
		      fprintf(stderr, "Hardware does not support PALM/PALN, using PAL\n");
		      sc.giventvstdindex = C2_MI_TV_PAL;
		   }
	        }
	        if(!(sc.sdflags & SiS_SD_SUPPORTNTSCJ)) {
	           if(sc.giventvstdindex == C2_MI_TV_NTSCJ) {
		      fprintf(stderr, "Hardware does not support NTSC-J, using NTSC\n");
		      sc.giventvstdindex = C2_MI_TV_NTSC;
		   }
	        }
	     } else {
	        fprintf(stderr, "This hardware does not support TV output\n");
	        settvstd = FALSE;
		doexit = TRUE;
	     }
	  }
	  if(!setcrt2) {
	     sc.givencrt2index = getcrt2typeindex(sc.currentvbflags);
	     if(sc.givencrt2index == C2_MI_TV_YPBPR) {
	        sc.giventvstindex = gettvstindex(sc.currentvbflags);
	     }
          }
          if(!settvstd) {
	     sc.giventvstdindex = gettvtypeindex(sc.currentvbflags);
          }
	  if(!settvas) {
	     sc.giventvasindex = gettvasindex(sc.currentvbflags);
	  }

	  if(settvos) {
	     if(!vbsupportsoverscan()) {
	        fprintf(stderr, "Overscan only supported on Chrontel TV encoders\n");
		settvos = FALSE;
		doexit = TRUE;
	     } else {
	        if(sc.giventvosindex == C2_MI_OV_SUPEROVERSCAN) {
		   if(!(sc.sdflags & SiS_SD_SUPPORTSOVER)) {
		      fprintf(stderr, "soverscan not supported on this hardware\n");
		      sc.giventvosindex = C2_MI_OV_OVERSCAN;
		   } else {
		      if(sc.giventvstdindex != C2_MI_TV_PAL) {
		         fprintf(stderr, "soverscan only supported for PAL\n");
			 sc.giventvosindex = C2_MI_OV_OVERSCAN;
		      }
		   }
		}
	     }
	  }

          if(!settvos) {
	     sc.giventvosindex = sc.choverscanstatus;
          }

	  if(setcrt2 || settvstd || settvos || settvas) {
             if(!(handle_apply_CRT2(sisctrl_dm_get_current_mode_index(), 1))) {
	        fprintf(stderr, "set[crt2|tvstandard|tvoverscan|tvaspect] failed, check X log for error messages\n");
             }
	  }

	  if(setcrt1 || setmode || resizetomode) {
	     fprintf(stderr, "-set[crt2|tvstandard|tvoverscan|tvaspect] and -set[crt1|mode]|-resize cannot be handled at the same time\n");
	  }
       } else {
          fprintf(stderr, "set[crt2|tvstandard|tvoverscan|tvaspect] not supported in dual head mode "
				"or if no video bridge present\n");
       }
    }
#endif

    if(setxvcrt) {
       if((sc.sdflags & SiS_SD_SUPPORT2OVL) && (!(sc.sd2flags & SiS_SD2_SUPPORT760OO))) {
          fprintf(stderr, "setxvcrt not supported on this hardware\n");
	  doexit = TRUE;
       } else {
          newsetxvval(SDC_CMD_SETXVSWITCHCRT, sc.atom_swc, xvcrtindex - 1);
       }
    }

    if((settvxpos || settvypos) && (!(sc.sdflags & SiS_SD_SUPPORTTVPOS))) {
       fprintf(stderr, "Hardware does not support setting the TV position\n");
       settvxpos = settvypos = FALSE;
       doexit = TRUE;
    }

    if((settvxscale || settvyscale) && (!(vbsupporttvsize()))) {
       fprintf(stderr, "Hardware does not support setting the TV scale\n");
       settvxscale = settvyscale = FALSE;
       doexit = TRUE;
    }

    if(settvxpos) {
       sd_settvposscale(SDC_CMD_SETTVXPOS, sc.atom_tvx, tvxposvalue);
    }

    if(settvypos) {
       sd_settvposscale(SDC_CMD_SETTVYPOS, sc.atom_tvy, tvyposvalue);
    }

    if(settvxscale) {
       sd_settvposscale(SDC_CMD_SETTVXSCALE, sc.atom_txs, tvxscalevalue);
    }

    if(settvyscale) {
       sd_settvposscale(SDC_CMD_SETTVYSCALE, sc.atom_tys, tvyscalevalue);
    }

    if(setgammafromdriver || setmode || resizetomode || setcrt1 || setcrt2 || settvstd ||
       settvos || setxvcrt || settvxpos || settvypos ||
       settvxscale || settvyscale || doexit) {
       cleanupandexit(0);
       exit(0);
    }

    /* GUI mode : */

    /* Silence X log in GUI mode */
    if(sc.sdinterface) {
       sc.sdstruct.sdc_parm[0] = 1;
       sendsdcommand(SDC_CMD_LOGQUIET);
       sc.xlogquiet = TRUE;
    }

    /* Build the mode list */
    rebuildmodestrings();

    rebuildmodefieldlist();

    /* Now init gtk and build the gui */

    sc.guimode = TRUE;

    buildgui();

    gui_main();

    return 0;
}

