/*

Copyright (C) 2000, 2001 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <nd.h>
#include <nd_gui.h>

#include <nd_snap.h>
#include <nd_snap_callbacks.h>

static ND_ProtoField snap_fields[] = {
  { ND_VAL_FIELD, N_("DSAP (0x%.2x)"),     N_("Destination SAP"),                   8,   nd_snap_dsap_cb  },
  { ND_VAL_FIELD, N_("SSAP (0x%.2x)"),     N_("Source SAP"),                        8,   nd_snap_ssap_cb  },
  { ND_VAL_FIELD, N_("Control (0x%.2x)"),  N_("Control Field"),                     8,   nd_snap_ctrl_cb  },
  { ND_VAL_FIELD, N_("OUI (%s)"),          N_("Organizational Unique Identifier"),  24,  nd_snap_oui_cb   },
  { ND_VAL_FIELD, N_("Proto (%s)"),        N_("Protocol Type"),                     16,  nd_snap_proto_cb },
  { 0, NULL, NULL, 0, NULL }
};

ND_MenuData snap_menu_type_data[] = {
  { "IP",         N_("IP (0x0800)"),              ETHERTYPE_IP,     nd_snap_proto_value_cb },
  { "ARP",        N_("ARP (0x0806)"),             ETHERTYPE_ARP,    nd_snap_proto_value_cb },
  { "RARP",       N_("RARP (0x8035)"),            ETHERTYPE_REVARP, nd_snap_proto_value_cb },
  { "Custom",     N_("Custom type value"),        -1,               nd_snap_proto_custom_cb },
  { NULL, NULL, 0, NULL}
};

static LND_Protocol *snap;
static ND_Protocol  *snap_gui;


/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return ("SNAP Plugin");
}


const char *
description(void)
{
  return ("A small plugin providing support for "
	  "LLC encapsulation and SNAP (Sub-Network "
	  "Access Protocol).");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return VERSION_MAJOR;
}


LND_Protocol *
init(void)
{
  if (! (snap = libnd_proto_registry_find(LND_PROTO_LAYER_LINK, LND_PROTO_SNAP)))
    return NULL;
  
  snap_gui = nd_proto_new(snap);
  
  snap_gui->create_gui      = nd_snap_create_gui;
  snap_gui->set_gui         = nd_snap_set_gui;

  /* We're using a button table to display the protocol content,
     so we need to hook it in here: */
  snap_gui->fields = snap_fields;
  snap_gui->header_width = 64;

  /* That's all -- we don't need checksums or state maintenance
     for simple Ethernet. The other methods got initialized
     to dummy null operations in the constructor call above.
     
     We also don't need a special menu to appear in the Protocols
     menu for this plugin.
  */
	     
  return snap;
}


/* Protocol method implementations: ------------------------------------ */

GtkWidget *
nd_snap_create_gui(LND_Trace *trace, LND_ProtoInfo *pinf)
{
  GtkWidget *table;

  table = nd_gui_proto_table_create(trace, pinf);
  return table;
}


void       
nd_snap_set_gui(const LND_Packet *packet, LND_ProtoInfo *pinf)
{
  struct lnd_snap_header *sh;
  
  sh = (struct lnd_snap_header*) libnd_packet_get_data(packet, snap, pinf->inst.nesting);

  nd_snap_set_gui_ssap(pinf, sh);
  nd_snap_set_gui_dsap(pinf, sh);
  nd_snap_set_gui_ctrl(pinf, sh);
  nd_snap_set_gui_oui(pinf, sh);
  nd_snap_set_gui_proto(pinf, sh);
}


/* Misc helper stuff below --------------------------------------------- */

void
nd_snap_set_gui_dsap(LND_ProtoInfo *pinf, struct lnd_snap_header *sh)
{
  nd_proto_field_set(pinf, &snap_fields[0], DATA_TO_PTR(sh->dsap));
}


void
nd_snap_set_gui_ssap(LND_ProtoInfo *pinf, struct lnd_snap_header *sh)
{
  nd_proto_field_set(pinf, &snap_fields[1], DATA_TO_PTR(sh->ssap));
}


void
nd_snap_set_gui_ctrl(LND_ProtoInfo *pinf, struct lnd_snap_header *sh)
{
  nd_proto_field_set(pinf, &snap_fields[2], DATA_TO_PTR(sh->ctrl));
}


void
nd_snap_set_gui_oui(LND_ProtoInfo *pinf, struct lnd_snap_header *sh)
{
  GtkWidget             *button;
  char                   s[MAXPATHLEN];
  guint32                val;
  guchar                *val_p;

  if (sh->ssap == 0xaa && sh->dsap == 0xaa)
    {
      /* Regular SNAP. Bare it all :) */

#ifdef WORDS_BIGENDIAN
      val = (sh->oui[2] << 16 | sh->oui[1] << 8 | sh->oui[0]);
#else
      val = (sh->oui[0] << 16 | sh->oui[1] << 8 | sh->oui[2]);
#endif
      val_p = (guchar *) &val;

      g_snprintf(s, MAXPATHLEN, "%.2x:%.2x:%.2x", val_p[0], val_p[1], val_p[2]);
      nd_proto_field_set(pinf, &snap_fields[3], s);
    }
  else
    {
      /* Mhm. Who knows -- hide it. */

      button = libnd_reg_get_data(pinf->registry, nd_proto_field_to_string(&snap_fields[3]));
      D_ASSERT_PTR(button);
      gtk_widget_hide(button);
    }
}


void
nd_snap_set_gui_proto(LND_ProtoInfo *pinf, struct lnd_snap_header *sh)
{
  GtkWidget             *button;
  
  if (sh->ssap != 0xaa || sh->dsap != 0xaa)
    {
      /* Not SNAP? Hide this. */

      button = libnd_reg_get_data(pinf->registry, nd_proto_field_to_string(&snap_fields[4]));
      gtk_widget_hide(button);
      return;
    }

  nd_proto_field_set_for_menu(pinf, &snap_fields[4], DATA_TO_PTR(ntohs(sh->proto)),
			      snap_menu_type_data, "0x%.4x");
}


LND_Protocol *
nd_snap_get(void)
{
  return snap;
}


ND_Protocol *
nd_snap_get_gui(void)
{
  return snap_gui;
}

