/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: main.cc,v 1.20.2.2 2003/09/30 14:04:51 dheck Exp $
 */
#include "config.h"

#include "enigma.hh"
#include "display.hh"
#include "game.hh"
#include "lua.hh"
#include "menus.hh"
#include "options.hh"
#include "oxyd.hh"
#include "sound.hh"
#include "video.hh"
#include "px/argp.hh"

#include "SDL.h"

#include <cstdio>
#include <cstdlib>
#include <fstream>
#include <iostream>

using namespace std;
using namespace px;
using namespace enigma;

namespace
{
    class Nulbuf : public std::streambuf {};
    Nulbuf* nullbuffer = new Nulbuf;
}



//======================================================================
// APPLICATION DATA
//======================================================================
namespace enigma
{
    Application app;
}

/* The stream object that is used for logging messages.  As defined
   here, it is not connected to any file or buffer.

   (Note: I think writing to a stream without a streambuffer *should*
   be a no-op, but it leads to crashes with g++ 2.95.  to circumvent
   this, Log is initialized with a dummy streambuf in init(). )
*/
std::ostream enigma::Log(0);

/* This is the stream object that is used when logging to a file.  In
   this case, enigma::Log uses this object's streambuffer. */
static std::fstream logfile;

//----------------------------------------------------------------------
// Startup
//----------------------------------------------------------------------

static void usage()
{
    printf("Available command line options for Enigma:\n\n"
           "    --nosound      Disable music and sound\n"
           "    --nomusic      Disable music\n"
           "    --window       Run in a window; do not enter fullscreen mode\n"
           "    --help -h      Show this help\n"
           "    --version      Print the executable's version number\n"
           "    --nograb       Do not use exclusive mouse/keyboard access\n"
           "    --zoom         Automatically create missing previews\n"
           "\n");
}

namespace
{
    struct AP : public argp::ArgParser {
    public:
        enum {
            OPT_NOSOUND, OPT_NOMUSIC, OPT_VERSION, OPT_HELP,
            OPT_WINDOW, OPT_WIZARD, OPT_NOGRAB, OPT_ZOOM, OPT_LOG,
            OPT_8BPP, OPT_GAME,
        };

        AP() : ArgParser (app.args.begin(), app.args.end())
        {
            nosound  = nomusic = show_help = show_version = do_log = false;
            gamename = "";

            def (OPT_NOSOUND,   0,   "nosound");
            def (OPT_NOMUSIC,   0,   "nomusic");
            def (OPT_VERSION,   0,  "version");
            def (OPT_HELP,      'h', "help");
            def (OPT_WINDOW,    'w', "window");
            def (OPT_WIZARD,    0,   "wizard");
            def (OPT_NOGRAB,    0,   "nograb");
            def (OPT_ZOOM,      0,   "zoom");
            def (OPT_LOG,       'l', "log");
            def (OPT_8BPP,      '8', "8bpp");
            def (OPT_GAME,      'g', "game", true);
        }

        // ArgParser interface.
        void on_error (ErrorType t, const string &option) {
            cout << errormsg(t, option) << endl;
            show_help = true;
        }

        void on_option (int id, const string &param) {
            switch (id) {
            case OPT_NOSOUND: nosound=true; break;
            case OPT_NOMUSIC: nomusic=true; break;
            case OPT_VERSION: show_version=true; break;
            case OPT_HELP:    show_help=true; break;
            case OPT_WINDOW:  options::FullScreen=false; break;
            case OPT_WIZARD:  options::WizardMode=true; break;
            case OPT_NOGRAB:  options::Nograb=true; break;
            case OPT_ZOOM:    options::Nozoom=false; break;
            case OPT_8BPP:    options::BitsPerPixel = 8; break;
            case OPT_GAME:    gamename = param; break;
            case OPT_LOG:     do_log = true; break;
            }
        }

        void on_argument (const string &/*arg*/) {
        }

        // Variables.
        bool nosound, nomusic, show_help, show_version, do_log;
        string gamename;
    };
}


static void
init()
{
    enigma::Log.rdbuf(::nullbuffer);

    lua::Init();

    // Run initialization scripts
    if (lua::Dofile("startup.lua") != 0) {
        fprintf(stderr, "There was an error loading 'startup.lua'.\n");
        fprintf(stderr, "Your installation may be incomplete or invalid.\n");
        exit (1);
    }

    lua::Dofile("levels/index.lua");
    lua::DoSubfolderfile("levels", "index.lua");

    // Load preferences
    if (!options::Load())
    {
        fprintf(stderr, "Error in configuration file.\n");
    }

    /*
    ** Evaluate command line arguments.
    */
    AP ap;
    ap.parse();

    if (ap.show_help || ap.show_version) {
        printf("Enigma %s\n",PACKAGE_VERSION);
        if (ap.show_help) usage();
        exit(0);
    }

    if (ap.do_log) {
        logfile.open("enigma.log", ios::out);
        if (logfile)
            enigma::Log.rdbuf(logfile.rdbuf());
        else
            fprintf(stderr,"Could not open log file \"enigma.log\" for writing.\n");
    }

    int sdl_flags = SDL_INIT_VIDEO | SDL_INIT_NOPARACHUTE;
    if (!ap.nosound)
        sdl_flags |= SDL_INIT_AUDIO;

    if (SDL_Init(sdl_flags) < 0) {
        fprintf(stderr, "Couldn't init SDL: %s\n", SDL_GetError());
        exit(1);
    }
    atexit(SDL_Quit);

    if (ap.nosound)
        sound::DisableSound();
    else if (ap.nomusic)
        sound::DisableMusic();

    enigma::Randomize();

    video::Init();
    video::SetPalette("enigma.pal");
    display::Init();
    sound::Init();
    world::Init();

    video::SetMouseCursor(enigma::LoadImage("cur-magic"), 4, 4);
    video::ShowMouse();

    oxyd::Init();
    oxyd::ChangeSoundset(options::SoundSet, 1);
}

static void
shutdown()
{
    oxyd::Shutdown();
    world::Shutdown();
    video::Shutdown();
    display::Shutdown();
    sound::Shutdown();
    options::Save();
    delete_sequence(enigma::LevelPacks.begin(),
                    enigma::LevelPacks.end());
    delete ::nullbuffer;
}

int
main(int argc, char** argv)
{
#ifdef MACOSX
    // In Mac OS X, applications are self-contained bundles,
    // i.e., directories like "Enigma.app". Resources are
    // placed in those bundles under "Enigma.app/Contents/Resources",
    // the main executable would be "Enigma.app/Contents/MacOS/enigma".
    // Here, we get the executable name, clip off the last bit, chdir into it,
    // then chdir to ../Resources. The original SDL implementation chdirs to
    // "../../..", i.e. the directory the bundle is placed in. This break
    // the self-containedness.
    char parentdir[1024];
    char *c;

    strncpy ( parentdir, argv[0], sizeof(parentdir) );
    c = (char*) parentdir;

    while (*c != '\0')     /* go to end */
        c++;

    while (*c != '/')      /* back up to parent */
        c--;

    *c++ = '\0';           /* cut off last part (binary name) */

    chdir (parentdir);  /* chdir to the binary app's parent */
    chdir ("../Resources/"); /* chdir to the .app's parent */
#endif //MACOSX

    try
    {
        app.init(argc,argv);

        init();

        GUI_MainMenu(LevelPacks[0], 0);
        shutdown();
    }
    catch (px::XGeneric &e) {
        cerr << "Error: " << e.what() << endl;
    }
    catch (std::exception &e) {
        cerr << "Error: " << e.what() << endl;
    }
    return 0;
}
