%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\chapter{Web interface}
\label{web_interface}

The \verb!darcs! web interface allows you to conveniently browse the
information stored in a repository.  You can view the repo by file and see
the history of that file as various patches were applied, or you can browse
in patch view mode, seeing which files were modified in each patch.

\begin{code}
module Main (main) where

import System
import CGI
import IO
import Maybe ( fromJust )
import Monad ( liftM, liftM2 )
import FastPackedString

import Text.Html hiding ( name )
import Patch
import PatchInfo
import Repository
import Directory
import RegexString
import SlurpDirectory
import Autoconf
import PopulationData
\end{code}

\begin{code}
main = do
  my_reposdir <- read_conf "reposdir"
  setCurrentDirectory my_reposdir
  wrapper $ cache_page make_page
\end{code}

\begin{code}
cache_page :: ([(String,String)] -> IO Html) -> [(String,String)] -> IO Html
cache_page mp a = do
  cachedir <- read_conf "cachedir"
  have_cachedir <- doesDirectoryExist cachedir
  if not have_cachedir
   then mp a -- if there's no cachedir, just don't use a cache!  :)
   else
    case takeWhile (/='*') $ get_query a of
    "" -> mp a
    rn ->do cached <- (++"/"++qn) `liftM` read_conf "cachedir"
            repoinv <- (++"/"++rn++"/_darcs/inventory")
                       `liftM` read_conf "reposdir"
            is_cached <- doesFileExist cached
            is_changed <- is_newer repoinv cached
            if is_cached && not is_changed
               then do liftM primHtml $ readFile cached
               else do page <- mp a
                       writeFile cached (renderHtml page)
                       return page
    where qn = clean_query $ get_query a
clean_query ('/':s) = 'l': clean_query s
clean_query ('*':s) = '.': clean_query s
clean_query (c:s) = c : clean_query s
clean_query [] = []
is_newer fa fb = do
  fa_exists <- doesFileExist fa
  fb_exists <- doesFileExist fb
  if fa_exists && fb_exists
     then liftM2 (>) (getModificationTime fa) (getModificationTime fb)
     else return False
\end{code}

The \verb!darcs_cgi! cgi script allows you to browse changes made in your
darcs repository via the web.  To use it with apache, you can install it
using \verb!make installserver!, and create a cache directory at
\verb!/var/cache/darcs_cgi!.  This cache directory must be writeable by the
cgi script, which for me means \verb!chown!ing it to the user and group
\verb!www-data!  Finally, you should create a directory named {\tt repos}
in {\tt /var/www}, in which you will place symlinks to the repos
themselves.  Once all this is done, the user can the browse the repos at
{\tt http://your.site/cgi-bin/darcs}.

The repos directory is configurable via the configuration file
\verb!/etc/darcs/cgi.conf!.  This file can contain comments (any line
starting with a `\#' char) and key value pairs with an equal sign in
between.  For example:
\example{cgi.conf}

\begin{code}
read_conf :: String -> IO String
read_conf var = do
  ls <- (map (takeWhile (/='#')).lines) `liftM` readFile (darcsconfdir ++ "/cgi.conf")
        `catch` (\_->return [])
  case filter ((== var).fst) $ map (break (== '=') . filter (/=' ')) ls of
      [(_,val)] -> return $ tail val
      _ -> return ""
\end{code}

\begin{code}
make_page :: [(String,String)] -> IO Html
make_page a =
  case get_query a of
  "" -> home_page
  _ -> case get_reponame a of
       ("Error",msg) -> return $ htmlError $ "Bad URL:\n"++msg
       (repo,fnpn) -> do
         setCurrentDirectory repo
         patches <- read_repo_patches "."
         case fnpn of
           "" -> repo_page repo
           "*" -> patch_page repo
           _ ->
            case get_filename_and_patchname fnpn of
             ["Error",msg] -> return $ htmlError $ "Bad URL:\n"++msg
             [filename,""] ->
                 return $ htmlError $ "Haven't yet implemented file view."
             ["",pn] ->
                 case get_patchinfo_from_name pn patches of
                 Nothing -> return $ htmlError $ "Patch "++pn++" does not exist."
                 Just pi -> one_patch_page repo pi
             [filename,pn,createdname,cn] ->
                 case (get_patchinfo_from_name pn patches,
                       get_patchinfo_from_name cn patches) of
                 (Just pi, Just ci) -> do
                     mf <- get_markedup_file ci createdname
                     return $ make_markedup_page repo filename pi
                              createdname ci (map fst patches) mf
                 _ -> return $ htmlError $
                      "Patch "++pn++" or maybe "++cn++" does not exist."
\end{code}

The page, {\tt http://your.site/cgi-bin/darcs}, displays a listing of all
repos available on the server.  From this page, the user can get to any
available repository.

\begin{code}
home_page :: IO Html
home_page = do
  repos <- getDirectoryContents "."
  return $ header << thetitle << "Darcs repositories:" +++
         body ! [bgcolor "#ffffff"] <<
                  (h1 << toHtml "Darcs repositories:")
                  +++ h3 << (link_repos $ filter (\s->head s /= '.') repos)

link_repos repos = foldl (+++) br $
                   map (\r-> hotlink ("darcs?"++r++"*") [toHtml r] +++ br) repos
\end{code}

Clicking on a given repository will take you to the file view page of that
repository, which shows a listing of all the files in the repo.  This page
also has a link to the patch view page, which is a bit more interesting.

\begin{code}
read_repo_patches d = (reverse . concat) `liftM` read_repo d
repo_page :: String -> IO Html
repo_page repo = do
  patches <- read_repo_patches "."
  case fst$head$reverse$patches of
    pi -> do
      mur <- get_markedup_repo pi
      return $ header << thetitle << repotitle repo +++
             body ! [bgcolor "#ffffff"] <<
                      (h1 << (toHtml $ repotitle repo) +++
                       h3 << (hotlink ("darcs?"++repo++"**")
                              [toHtml "Switch to patch view"]) +++
                       mur_to_html (repo++"*") (make_filename pi) mur)
repotitle repo = "The darcs '"++repo++"' repository"
\end{code}

In patch view mode, the web interface displays a listing of all the patches
in the repo.  Clicking on a patch gives a listing of all files that were in
the repo at the time that patch was applied.

\begin{code}
patch_page :: String -> IO Html
patch_page repo = do
  patches <- read_repo_patches "."
  return $ header << thetitle << repotitle repo +++
               body ! [bgcolor "#ffffff"] <<
                        (h1 << (toHtml $ repotitle repo) +++
                         h3 << (hotlink ("darcs?"++repo++"*")
                                [toHtml "Switch to file view"]) +++
                         (table ! [border 0] <<
                          (foldl (above) (cell $ h3 << "Patches")
                           (map (toHtml.(repopatchinfo repo).fst) $ reverse patches))))
replace_repo :: String -> String -> String
replace_repo repo ('d':'a':'r':'c':'s':'?':r) =
    "darcs?"++repo++"*"++replace_repo repo r
replace_repo repo (c:cs) = c: replace_repo repo cs
replace_repo _ "" = ""
\end{code}

Clicking on one of the files shows the file contents, with added lines
shown in green, and removed ones in red.  To the left of each line is a
small `+' and `-'.  These are links to the patch which added or removed
that line.

\begin{code}
one_patch_page :: String -> PatchInfo -> IO Html
one_patch_page repo pi = do
  mur <- get_markedup_repo pi
  return $ header << thetitle << ("Patch:  "++just_name pi) +++
         body ! [bgcolor "#ffffff"] <<
                  (h3 << (hotlink ("darcs?"++repo++"**")
                          [toHtml $ "Patch:  "++just_name pi]) +++
                   mur_to_html (repo++"*") (make_filename pi) mur)

\end{code}

\begin{code}
make_markedup_page :: String -> FilePath -> PatchInfo
                   -> FilePath -> PatchInfo -> [PatchInfo]
                   -> MarkedUpFile -> Html
make_markedup_page repo f pi cfn ci ps mk =
  header << thetitle << (f++" ** "++just_name pi) +++
    body ! [bgcolor "#ffffff"] <<
             (nav repo f pi cfn ci ps +++
              h1 << font! [color blue] << f +++
              h2 << ("Patch:  "++just_name pi) +++
              font! [face "Courier"] <<
              markup_html (repo++"*"++f) pi cfn ci ps mk)

markup_html :: FilePath -> PatchInfo -> FilePath
            -> PatchInfo -> [PatchInfo] -> MarkedUpFile -> Html
markup_html f pi cfn ci ps [] = p << ""
markup_html f pi cfn ci ps ((l,None):mk) =
    cl +++ font! [color black] << line_to_html (unpackPS l)
           +++ (markup_html f pi cfn ci ps mk)
    where cl = changelink f cfn ci $ None
markup_html f pi cfn ci ps ((l,RemovedLine thei):mk) =
    if pi == thei
    then cl +++ font! [color red] << line_to_html (unpackPS l)
             +++ (markup_html f pi cfn ci ps mk)
    else if is_old_patch pi ps thei
    then markup_html f pi cfn ci ps mk
    else markup_html f pi cfn ci ps mk
    where cl = changelink f cfn ci $ RemovedLine thei
markup_html f pi cfn ci ps ((l,AddedLine thei):mk) =
    if pi == thei
    then cl +++ font! [color green] << line_to_html (unpackPS l)
             +++ (markup_html f pi cfn ci ps mk)
    else if is_old_patch pi ps thei
    then cl +++ font! [color black] << line_to_html (unpackPS l)
             +++ (markup_html f pi cfn ci ps mk)
    else markup_html f pi cfn ci ps mk
    where cl = changelink f cfn ci $ AddedLine thei
markup_html f pi cfn ci ps ((l,AddedRemovedLine add rem):mk) =
    if pi == rem
    then cl +++ font! [color red] << line_to_html (unpackPS l)
             +++ (markup_html f pi cfn ci ps mk)
    else if pi == add
    then cl +++ font! [color green] << line_to_html (unpackPS l)
             +++ (markup_html f pi cfn ci ps mk)
    else if is_old_patch pi ps add && (not $ is_old_patch pi ps rem)
    then cl +++ font! [color black] << line_to_html (unpackPS l)
             +++ markup_html f pi cfn ci ps mk
    else markup_html f pi cfn ci ps mk
    where cl = changelink f cfn ci $ AddedRemovedLine add rem

changelink :: FilePath -> FilePath -> PatchInfo -> LineMark -> Html
changelink f cfn ci (RemovedLine pi) =
    font ! [color black] << "+" +++
             (((toHtml $ hotlink ("darcs?"++f++"*"++make_filename pi
                                  ++"*"++cfn++"*"++make_filename ci)
                [toHtml "-"])
               ! [thestyle "text-decoration:none"]) +++
              spaceHtml)
changelink f cfn ci (AddedLine pi) =
    font ! [color black] <<
             (((toHtml $ hotlink ("darcs?"++f++"*"++make_filename pi
                                  ++"*"++cfn++"*"++make_filename ci) [toHtml "+"])
               ! [thestyle "text-decoration:none"]) +++
              "-" +++ spaceHtml)
changelink f cfn ci (AddedRemovedLine add rem) =
    font ! [color black] <<
             (((toHtml $ hotlink ("darcs?"++f++"*"++make_filename add
                                  ++"*"++cfn++"*"++make_filename ci)
                [toHtml "+"])
               ! [thestyle "text-decoration:none"]) +++
              ((toHtml $ hotlink ("darcs?"++f++"*"++make_filename rem
                                  ++"*"++cfn++"*"++make_filename ci)
                [toHtml "-"])
               ! [thestyle "text-decoration:none"]) +++
              spaceHtml)
changelink _ _ _ _ =
    font ! [color black] << "+-" +++ spaceHtml

is_old_patch :: PatchInfo -> [PatchInfo] -> PatchInfo -> Bool
is_old_patch now (i:is) this =
    if this == now then False
    else if i == now then False
    else if i == this then True
    else is_old_patch now is this

line_to_html :: String -> Html
line_to_html "" = br
line_to_html (' ':s) = spaceHtml +++ line_to_html s
line_to_html ('\t':s) = spaceHtml +++ spaceHtml +++ line_to_html s
line_to_html (c:s) = c +++ line_to_html s
\end{code}

\begin{code}
htmlError e
      = header
        << thetitle
           << ("Error: "++e)
   +++ body ! [bgcolor "#aaff88"] << (h1 << "Error! "+++br+++e)
\end{code}

\begin{code}
nav :: String -> FilePath -> PatchInfo -> FilePath -> PatchInfo -> [PatchInfo] -> Html
nav repo f pi cfn ci ps =
    nav_prev_nex repo f pi cfn ci (before_pi pi ps) (after_pi pi ps)

nav_prev_nex repo f pi cfn ci mb ma =
    let
      b = case mb of
          Nothing -> "darcs?"++repo++"*"
          Just bi -> "darcs?"++repo++"*"++f++"*"++make_filename bi++
                     "*"++cfn++"*"++make_filename ci
      a = case ma of
          Nothing -> "darcs?"++repo++"*"
          Just ai -> "darcs?"++repo++"*"++f++"*"++make_filename ai++
                     "*"++cfn++"*"++make_filename ci
    in
    table ! [border 0] << ((td ! [align "left"] <<
                            hotlink b [h4 << "Previous patch"])
                           `beside`
                           (td ! [align "center"] <<
                            hotlink ("darcs?"++repo++"*") [h4 << "Home"])
                           `beside`
                           (td ! [align "right"] <<
                            hotlink a [h4 << "Next patch"]))

before_pi pi [] = Nothing
before_pi pi [_] = Nothing
before_pi pi (b:it:ps)
    | it == pi = Just b
    | otherwise = before_pi pi (it:ps)
after_pi pi [] = Nothing
after_pi pi [p] = Nothing
after_pi pi (it:a:ps)
    | it == pi = Just a
    | otherwise = after_pi pi (a:ps)
\end{code}

\begin{code}
get_reponame :: [(String,String)] -> (String,String)
get_reponame env =
    case matchRegex (mkRegex "([^\\*^/^\\.]*)\\*(.*)") $ get_query env of
    Just (repo:rest:_) -> (repo,rest)
    _ -> ("Error","Bad filename and patchname: "++get_query env)
\end{code}

\begin{code}
get_filename_and_patchname :: String -> [String]
get_filename_and_patchname query =
    case matchRegex (mkRegex "(.*)\\*(.*)\\*(.*)\\*(.*)") $ query of
    Just [f,pn,cfn,cn] -> [f,pn,cfn,cn]
    _ -> case matchRegex (mkRegex "(.*)\\*(.*)") $ query of
         Just (f:pn:_) -> [f,pn]
         _ -> ["Error","Bad filename and patchname: "++query]
\end{code}

\begin{code}
pop_to_html :: String -> Population -> Html
pop_to_html before (Pop pi (PopDir _ pops)) =
    foldl (+++) noh $ map (p2h before pi "./" noh) pops
    where noh = toHtml ""
p2h :: String -> PatchInfo -> FilePath -> Html -> PopTree -> Html
p2h before pi dir indent (PopFile info) =
    indent +++
    hotlink ("darcs?"++before++fullf++"*"++make_filename pi++"*"++cn++"*"++cp) [toHtml f] +++
    file_change+++br
        where f = unpackPS $ name info
              fullf = dir ++ f
              cn = unpackPS $ fromJust $ creationName info
              cp = make_filename $ fromJust $ createdBy info
              file_change = if modifiedBy info == pi
                            then toHtml $ " "++show (modifiedHow info)
                            else toHtml ""
p2h before pi dir indent (PopDir info pops) =
    foldl (+++) (indent +++ toHtml fulldir +++ dir_change +++ br) $
          map (p2h before pi fulldir (indent+++indmore)) pops
    where fulldir = dir ++ (unpackPS $ name info)
          indmore = spaceHtml +++ spaceHtml +++ spaceHtml +++ spaceHtml
          dir_change = if modifiedBy info == pi
                       then toHtml $ " "++show (modifiedHow info)
                       else toHtml ""

indent_fn ('/':s) = spaceHtml +++ spaceHtml +++ spaceHtml +++ spaceHtml +++ indent_fn s
indent_fn (_:s) = indent_fn s
indent_fn [] = spaceHtml
get_fn f = toHtml $ reverse $ takeWhile (/='/') $ reverse f

mur_to_html :: String -> String -> MarkedUpRepo -> Html

mur_to_html before pn ((f,cpi,cn,m):mk)
    | mur_is_dir m = indent_fn f +++ get_fn f +++ "/" +++
                     describe_dir_change m +++ br +++
                     mur_to_html before pn mk
    | mur_is_file m =
        indent_fn f +++
        hotlink ("darcs?"++before++f++"*"++pn++"*"++cn++"*"++make_filename cpi) [get_fn f] +++
        describe_file_change m +++
        br +++ mur_to_html before pn mk
mur_to_html before pn [] = br

mur_is_dir RemovedDir = True
mur_is_dir (MovedDir _) = True
mur_is_dir AddedDir = True
mur_is_dir DullDir = True
mur_is_dir _ = False
mur_is_file = not . mur_is_dir

describe_dir_change RemovedDir = " removed"
describe_dir_change (MovedDir _) = " moved"
describe_dir_change AddedDir = " added"
describe_dir_change DullDir = ""

describe_file_change RemovedFile = " removed"
describe_file_change ModifiedFile = " changed"
describe_file_change (MovedFile _) = " moved"
describe_file_change AddedFile = " added"
describe_file_change DullFile = ""
\end{code}

\begin{code}
get_patchinfo_from_name :: String -> [(PatchInfo,Maybe Patch)] -> Maybe PatchInfo
get_patchinfo_from_name n [] = Nothing
get_patchinfo_from_name n ((pi,_):pps) =
  if n == make_filename pi then Just pi
  else get_patchinfo_from_name n pps
\end{code}

\begin{code}
get_query :: [(String,String)] -> String
get_query [] = ""
get_query (("QUERY_STRING",query):_) = query
get_query (a:as) = get_query as
\end{code}


