%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs unrecord}
\begin{code}
module Unrecord ( unrecord, unpull ) where
import Directory
import IO
import System
import Char ( toUpper )

import DarcsCommands
import Repository
import Patch
import PatchInfo
import SlurpDirectory
import DarcsArguments
import Lock ( withLock )
\end{code}
\begin{code}
unrecord_description =
 "Unrecord a named patch."
\end{code}

\options{unrecord}

\haskell{unrecord_help} Note that unrecord doesn't affect your working copy
of the tree at all, so if you want to completely undo the change, you'll
also need to \verb!darcs revert!, or do an unpull.

If you don't revert after unrecording, then the changes made by the
unrecorded patches are left in your working tree.  If these patches are
actually from another repository, interaction (either pushes or pulls) with
that repository may be massively slowed down, as darcs tries to cope with
the fact that you appear to have made a large number of changes that
conflict with those present on the other repository.  So if you really want
to undo the result of a \emph{pull} operation, use unpull! Unrecord is
primarily intended for when you record a patch, realize it needs just one
more change, but would rather not have a separate patch for just that one
change.

\begin{code}
unrecord_help =
 "Unrecord is used to undo a single recorded patch.  It will prompt you
for which patch to unrecord, and then will undo that patch.
"
\end{code}
\begin{code}
unrecord = DarcsCommand {command_name = "unrecord",
                         command_help = unrecord_help,
                         command_description = unrecord_description,
                         command_extra_args = 0,
                         command_command = unrecord_cmd,
                         command_prereq = am_in_repo,
                         command_get_arg_possibilities = return [],
                         command_argdefaults = nodefaults,
                         command_darcsoptions = [verbose]}
\end{code}
\begin{code}
unrecord_cmd opts args = withLock "./_darcs/lock" $ do
  patches <- read_repo "."
  recorded <- slurp_recorded "."
  mps <- choose_patch "unrecord" patches
  case mps of
    Nothing -> putStr "Unable to unrecord patch.\n"
    Just (p,p_after_pending) ->
      case apply_to_slurpy (invert p) recorded of
        Nothing -> putStr "Unable to apply inverse patch!\n"
        Just working -> do
          patches' <- read_repo "."
          write_inventory "." $ rempatch p patches'
          former_dir <- getCurrentDirectory
          setCurrentDirectory "_darcs/current"
          slurp_write_dirty working
          setCurrentDirectory former_dir
          putStr $ "Finished unrecording.\n"

rempatch :: Patch -> PatchSet -> PatchSet
rempatch p (pps:ppss) =
    case patchname p of
    Nothing -> pps:ppss
    Just pn -> (filter (\pp-> (make_filename $ fst pp) /= pn) pps):ppss
\end{code}

\begin{code}
choose_patch :: String -> PatchSet -> IO (Maybe (Patch,Patch))
choose_patch v [] = return Nothing
choose_patch v (pps:ppss) = choose_patch_helper v [] pps ppss

capitalize (c:s) = toUpper c : s
choose_patch_helper :: String
                    -> [(PatchInfo,Maybe Patch)]
                    -> [(PatchInfo,Maybe Patch)]
                    -> PatchSet
                    -> IO (Maybe (Patch,Patch))
choose_patch_helper v mps ((pi, mp):pps) ppss = do
  putStr $ "Considering "++v++"ing patch:\n"++show pi
  putStr $ "\n"++capitalize v++" this patch? "
  hFlush stdout
  yorn <- getLine
  if (length yorn >= 1 && head yorn /= 'y') || length yorn < 1
     then choose_patch_helper v ((pi,mp):mps) pps ppss
     else do
       mp <- commute_back mps (pi,mp) pps ppss
       case mp of
         Nothing -> return Nothing
         Just p -> do
           pend <- read_pending
           case pend of
                Nothing -> return $ Just (p,p)
                Just pending ->
                    case commute (pending, p) of
                      Nothing -> return Nothing
                      Just (p', pending') ->
                        case patchname p' of
                        Nothing -> return Nothing
                        Just np' -> do
                          write_pending pending'
                          write_patch ("_darcs/patches/"++np') p'
                          return $ Just (p,p')
ignore_error e = return ()
commute_back :: PatchSequence -> (PatchInfo,Maybe Patch) -> PatchSequence
             -> PatchSet -> IO (Maybe Patch)
commute_back [] (tpi,mtp) rest _ = return mtp
commute_back ((pi,mp):mps) (tpi,mtp) rest ppss =
  case mtp of
  Nothing -> return Nothing
  Just tp ->
    case mp of
    Nothing -> return Nothing
    Just p ->
      case commute (p,tp) of
      Nothing -> return Nothing
      Just (tp',p') -> do
        write_patch ("_darcs/patches/"++make_filename pi) p'
        write_patch ("_darcs/patches/"++make_filename tpi) tp'
-- Note that the patches were already reversed once, then mps got reversed again.
        write_inventory "." $ (reverse $ reverse rest++
                               [(pi,Nothing),(tpi,Nothing)]++mps) : ppss
        commute_back mps (tpi, Just tp') ((pi,mp):rest) ppss
write_patch pfn p = do
    renameFile pfn (pfn++".backup") `catch` ignore_error
    writePatch pfn p
    removeFile (pfn++".backup") `catch` ignore_error
\end{code}

FIXME: The one major bug remaining is that unrecording a `darcs add' causes
that add to be lost, while it really should just be stuck pack in `pending'.


\section{darcs unpull}

\begin{code}
unpull_description =
 "Unpull a named patch."
\end{code}

\options{unpull}

\haskell{unpull_help}

\begin{code}
unpull_help =
 "Unpull is used to undo a single patch that has been pulled from another
repository.  It will prompt you for which patch to unpull, and then will
undo that patch.  Beware that unpull undoes the patch both from the repo
records AND from the current working directory, and does NOT check that
the patch originated with a pull.  In otherwords, you could lose precious
code by unpulling!
"
\end{code}
\begin{code}
unpull = DarcsCommand {command_name = "unpull",
                       command_help = unpull_help,
                       command_description = unpull_description,
                       command_extra_args = 0,
                       command_command = unpull_cmd,
                       command_prereq = am_in_repo,
                       command_get_arg_possibilities = return [],
                       command_argdefaults = nodefaults,
                       command_darcsoptions = [verbose]}
\end{code}
\begin{code}
unpull_cmd opts args = withLock "./_darcs/lock" $ do
  patches <- read_repo "."
  recorded <- slurp_recorded "."
  work <- slurp "."
  pend <- read_pending
  mps <- choose_patch "unpull" patches
  case mps of
    Nothing -> putStr "Unable to unpull patch.\n"
    Just (p,p_after_pending) ->
      case apply_to_slurpy (invert p) recorded of
      Nothing -> putStr "Unable to apply inverse patch!\n"
      Just rec' ->
          case apply_to_slurpy (invert p_after_pending) work of
          Nothing -> putStr "Couldn't undo patch in working dir.\n"
          Just work' -> do
                patches' <- read_repo "."
                write_inventory "." $ rempatch p patches'
                former_dir <- getCurrentDirectory
                setCurrentDirectory "_darcs/current"
                slurp_write_dirty rec'
                setCurrentDirectory former_dir
                slurp_write_dirty work'
                putStr $ "Finished unpulling.\n"
\end{code}
