%  Copyright (C) 2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs tag}
\begin{code}
module Tag ( tag ) where
import IO
import System ( getEnv )
import Time ( toCalendarTime, getClockTime, calendarTimeToString )
import Monad ( liftM )

import DarcsCommands
import DarcsArguments
import Repository
import Patch
import PatchInfo
import Depends
\end{code}
\begin{code}
tag_description =
 "Tag the contents of a repo with a given version name."
\end{code}

\options{tag}

\haskell{tag_help} Tag differs from record in that it doesn't record any
new changes, and it always depends on all patches residing in the
repository when it is tagged.  This means that one can later reproduce this
version of the repository by calling, for example:
\begin{verbatim}
% darcs get --tag-name "darcs 3.14" REPOLOCATION
\end{verbatim}

\begin{code}
tag_help =
 "Tag is used to name a version of the tree.
"
\end{code}
\begin{code}
tag = DarcsCommand {command_name = "tag",
                    command_help = tag_help,
                    command_description = tag_description,
                    command_extra_args = 0,
                    command_command = tag_cmd,
                    command_prereq = am_in_repo,
                    command_get_arg_possibilities = return [],
                    command_argdefaults = nodefaults,
                    command_darcsoptions = [patchname_option, author,
                                       verbose]}
\end{code}
\begin{code}
tag_cmd opts args = do
    date <- get_date opts
    author <- get_author opts

    recorded <- slurp_recorded "."
    deps <- liftM get_tags_right $ read_repo "."
    name <- get_patchname opts
    let mypatch = namepatch date name author [] $ join_patches []
        myinfo = patchinfo date name author []
        myfn = "_darcs/patches/"++make_filename myinfo
       in do
       writePatch myfn $ adddeps mypatch deps
       add_to_inventory "." myinfo
       putStr $ "Finished tagging patch '"++name++"'\n"
\end{code}
Each tagged version has a version name.
\begin{code}
get_patchname :: [DarcsFlag] -> IO String
get_patchname (PatchName n:_) = return n
get_patchname (_:flags) = get_patchname flags
get_patchname [] = do
    putStr "What is the version name? "
    hFlush stdout
    liftM ("TAG "++) getLine
\end{code}
The version is also flagged with the person who tagged it (taken by default
from the `DARCS\_EMAIL' or `EMAIL' environment variable.  The date is also
included in the version information.
\begin{code}
get_date :: [DarcsFlag] -> IO String -- No, get_date doesn't ask a girl out for you!
get_date _ = do
    clocktime <- getClockTime
    caltime <- toCalendarTime clocktime
    return $ calendarTimeToString caltime
\end{code}

A tagged version automatically depends on all patches in the repo.  This
allows you to later reproduce precisely that version.  The tag does this by
depending on all patches in the repo, except for those which are depended
upon by other tags already in the repo.  In the common case of a sequential
series of tags, this means that the tag depends on all patches since the
last tag, plut that tag itself.
