%  Copyright (C) 2002-2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs pull}
\begin{code}
module Pull ( pull, merge_with_us_and_pending, save_patches, select_patches
            ) where
import IO
import System
import Monad ( when )
import List ( intersect, elem, nub, sort )
import RegexString

import DarcsCommands
import DarcsArguments
import Directory
import Repository
import Patch
import PatchInfo
import SlurpDirectory
import RepoPrefs
import Depends ( get_common_and_uncommon )
import Resolution ( standard_resolution )
import Lock ( withLock )
import PrintPatch ( printPatch )
\end{code}
\begin{code}
pull_description =
 "Pull patches from another repo."
\end{code}

\options{pull}

\haskell{pull_help}
\begin{code}
pull_help =
 "Pull is used to bring changes made in another repo into the current repo
(that is, the one that is the current directory).  Pull allows you to bring
over all or some of the patches that are in that repo but not in the
current one.  Pull accepts an argument, which is the URL from which to pull,
and when called without an argument, pull will use the repository from which
you have most recently either pushed or pulled.
"
\end{code}
\begin{code}
pull = DarcsCommand {command_name = "pull",
                     command_help = pull_help,
                     command_description = pull_description,
                     command_extra_args = 1,
                     command_command = pull_cmd,
                     command_prereq = am_in_repo,
                     command_get_arg_possibilities = get_preflist "repos",
                     command_argdefaults = lastrepo,
                     command_darcsoptions = [verbose,patchname_option,all_patches,
                                             ignoretimes, no_deps]}
\end{code}
\begin{code}
pull_cmd opts [repodir] = withLock "./_darcs/lock" $ do
  am_verbose <- return $ Verbose `elem` opts
  former_dir <- getCurrentDirectory
  repovalid <- is_repo repodir
  if not repovalid then do putStr $ "Bad repo directory: "++repodir++"\n"
                           exitWith $ ExitFailure 1
     else return ()
  set_lastrepo repodir
  aminrepo <- is_repo "."
  if not aminrepo then do putStr $ "Aaack, I'm not in a repo!\n"
                          exitWith $ ExitFailure 1
     else return ()
  when am_verbose $ putStr "About to read in the inventories...\n"
  them <- read_repo repodir
  us <- read_repo "."
  when am_verbose $ putStr "Finished reading in the inventories...\n"
  case get_common_and_uncommon (us, them) of
    (common, us', them') -> do
      when am_verbose $ putStr $
          "We the following new (to them) patches:\n"++
          format_inventory (head us')
      when am_verbose $ putStr $
          "They have the following new (to us) patches:\n"++
          format_inventory (head them')
      when (them' == [[]]) $ do
          putStr "No remote changes to pull in!\n"
          exitWith ExitSuccess
      to_be_pulled <- select_patches "pull" opts $ reverse $ head them'
      when (to_be_pulled == []) $ do
          putStr "You don't want to pull any patches, and that's fine with me!\n"
          exitWith ExitSuccess
      when am_verbose $ putStr "Getting and merging patches the following:\n"
      when am_verbose $ putStr $ format_inventory to_be_pulled
      (pc,pw) <- merge_with_us_and_pending opts
                 (reverse $ head us', to_be_pulled)
      case sort $ nub $ list_conflicted_files pc++list_conflicted_files pw of
        [] -> return ()
        cfs -> do putStr $ "We have conflicts in the following files:\n"
                  putStrLn $ unwords cfs
      recorded <- slurp_recorded "."
      working <- slurp "."
      when am_verbose $ putStr "Applying patches to the local directories...\n"
      case apply_to_slurpy pc recorded of
        Nothing -> do putStr "Error applying patch to recorded!\n"
                      putStr $ "The patch was:\n"++ show pc
                      exitWith $ ExitFailure 1
        Just rec' ->
          case apply_to_slurpy (standard_resolution pw) working of
          Nothing -> do putStr "Error applying patch to working dir.\n"
                        putStrLn $ show pc
                        putStrLn $ show (standard_resolution pw)
                        exitWith $ ExitFailure 1
          Just work' -> do
              save_patches $ unjoin_patches pc
              setCurrentDirectory "_darcs/current"
              slurp_write_dirty rec'
              wait_a_moment
              setCurrentDirectory former_dir
              sequence $ map (add_to_inventory "." . fst) to_be_pulled
              slurp_write_dirty work'
              putStr $ "Finished pulling.\n"
\end{code}

The \verb!--patch-name! argument can be used to specify a regexp, which
should be of the extended type used by \verb!egrep!.  If this option is
used, only patches which match this regexp (along with their dependencies)
are considered.

If you give a \verb!--patch-name! argument, darcs will silently pull along
any other patches upon which the patches which match the patch-name depend.
So \verb!--patch-name bugfix! mean ``pull all the patches with `bugfix' in
their name, along with any patches they require.''  If you really only want
the patches with `bugfix' in their name, you should use the
\verb!--no-deps! option, which is only useful in combination with
\verb!--patch-name!, and makes darcs only pull in those matching patches
which have no dependencies (apart from other matching patches).

\begin{code}
mymatch ('^':r) s = matchRegex (mkRegex ('^':r)) s /= Nothing
mymatch r s = matchRegex (mkRegex (".*"++r)) s /= Nothing
get_patchname :: [DarcsFlag] -> (String -> Bool)
get_patchname (PatchName n:_) = mymatch n
get_patchname (_:flags) = get_patchname flags
get_patchname [] = \_ -> True
\end{code}

\begin{code}
save_patches :: Maybe [Patch] -> IO ()
save_patches (Just []) = return ()
save_patches Nothing = return ()
save_patches (Just (p:ps)) =
  case patchname p of
  Just pn -> do
    writePatch ("_darcs/patches/"++pn) p
    save_patches $ Just ps
  Nothing -> do
    putStr "Aaack! An anonymous patch!\n"
    exitWith $ ExitFailure 1
\end{code}

\begin{code}
merge_with_us_and_pending :: [DarcsFlag] -> (PatchSequence,PatchSequence) ->
                             IO (Patch, Patch)
merge_with_us_and_pending opts (us,them) =
  case (mkpfs us, mkpfs them) of
  (Just usp, Just themp) ->
      case merge (themp, usp) of
      Nothing -> pperr "There was a conflict in merging... giving up!\n"
      Just (themp',usp') -> do
         putStr "So far so good... the merge succeeded.\n"
         past_pending <- merge_with_pending opts themp'
         case past_pending of
              Nothing -> pperr "Can't commute by pending!\n"
              Just pp -> return (themp', pp)
  _ -> pperr "Some sort of problem reading patches.\n"
merge_with_pending opts p = do
  pend <- get_unrecorded (AnyOrder:opts) -- we don't care if it looks pretty...
  case pend of
    Nothing -> return $ Just p
    Just pendp ->
      case merge (p,pendp) of
      Nothing -> return Nothing
      Just (p',_) -> return $ Just p'
pperr :: String -> IO (Patch,Patch)
pperr s = do
  putStr s
  exitWith $ ExitFailure 1
\end{code}

\begin{code}
mkpfs :: PatchSequence -> Maybe Patch
mkpfs pps = case sequence $ map snd pps of
            Just ps -> Just $ join_patches ps
            Nothing -> Nothing
\end{code}

\begin{code}
select_patches :: String -> [DarcsFlag] -> PatchSequence -> IO PatchSequence
select_patches cn opts pps =
    sp_helper opts cn [] (get_patchname opts) pps

move_p_after p ((pi,Just p2):pps) =
    case commute (p2,p) of
    Nothing -> move_p_after p $ move_p_after p2 pps
    Just (p',p2') -> (pi,Just p2') : move_p_after p' pps
move_p_after _ [] = []

sp_help cn = "y: "++cn++" this patch
n: don't "++cn++" this patch or any that depend on it
v: let me view this patch
a: "++cn++" all the rest of the patches
s: skip (and don't "++cn++") the rest of the patches
w: wait on this patch to see if it is needed by another patch I'm
   interested in.
? or h: show this help
"

commuteM (Just p, Just p') = commute (p,p')
commuteM _ = Nothing
addtrue (a,b) = (a,b,True)
filter_out_unneeded_waiting :: [(PatchInfo,Maybe Patch,Bool)] -> PatchSequence
filter_out_unneeded_waiting [] = []
filter_out_unneeded_waiting ((pi,mp,True):pps) =
    (pi,mp): filter_out_unneeded_waiting pps
filter_out_unneeded_waiting ((pi,mp,False):(pi',mp',True):pps) =
    case commuteM (mp',mp) of
    Nothing -> (pi,mp) : (pi',mp') : (filter_out_unneeded_waiting pps)
    Just (p,p') -> (pi', Just p') :
                   filter_out_unneeded_waiting ((pi,Just p,False):pps)
filter_out_unneeded_waiting [(pi,mp,False)] = []
filter_out_unneeded_waiting ((pi,mp,False):pps) =
    filter_out_unneeded_waiting ((pi,mp,False) :
                                 map addtrue (filter_out_unneeded_waiting pps))

sp_helper :: [DarcsFlag] -> String -> [(PatchInfo,Maybe Patch,Bool)]
          -> (String -> Bool) -> PatchSequence -> IO PatchSequence
sp_helper opts cn already pn [] =
    return $ filter_out_unneeded_waiting $ reverse already
sp_helper opts cn already pn ((pi,mp):pps) =
  case mp of
  Nothing -> do putStr "Error reading patch!\n"
                return []
  Just p -> do
    yorn <- if pn $ just_name pi
            then if All `elem` opts then return "y"
                 else do putStr $ show pi
                         putStr $ "\nShall I "++cn++" this patch? [ynvaswh] "
                         hFlush stdout
                         getLine
            else if DontGrabDeps `elem` opts then return "n" else return "w"
    case yorn of
      ('y':_) -> sp_helper opts cn ((pi,mp,True):already) pn pps
      ('w':_) -> sp_helper opts cn ((pi,mp,False):already) pn pps
      ('v':_) -> do printPatch p
                    sp_helper opts cn already pn ((pi,mp):pps)
      ('h':_) -> do putStr $ sp_help cn
                    sp_helper opts cn already pn ((pi,mp):pps)
      ('?':_) -> do putStr $ sp_help cn
                    sp_helper opts cn already pn ((pi,mp):pps)
      ('a':_) -> return $ sp_all ((pi,mp,True):already) pn pps
      ('s':_) -> return $ filter_out_unneeded_waiting $ reverse already
      otherwise -> sp_helper opts cn already pn $ move_p_after p pps

sp_all already pn [] = filter_out_unneeded_waiting $ reverse already
sp_all already pn ((pi,mp):pps) =
  case mp of
  Nothing -> filter_out_unneeded_waiting $ reverse already
  Just p -> if pn $ just_name pi
            then sp_all ((pi,mp,True):already) pn pps
            else sp_all ((pi,mp,False):already) pn pps
\end{code}
