%  Copyright (C) 2003 Jan Scheffczyk and David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

\section{Population}
NOTE: this documentation belongs in a ``libdarcs API'' chapter, which
currently doesn't exist.
\begin{code}
module Population ( Population, patchChanges, adjustPopStates, applyToPop,
                    applyPatchesPop, applyPatchSetPop, getPopFrom,
                    popUnfold, popUnfoldDirty,
                    initPop, cleanPop, setPopState,
                    DirMark(..),
                    getRepoPop, getRepoPopVersion,
                  ) where
import IO
import FastPackedString
import Monad ( liftM )
import List ( nub )
import Maybe ( fromJust, catMaybes )
import Directory ( getCurrentDirectory, setCurrentDirectory,
                   doesDirectoryExist, getDirectoryContents )

import PatchInfo ( PatchInfo, patchinfo )
import Patch ( Patch, applyToPop, patchChanges )
import Repository ( read_repo )
import PopulationData
\end{code}

a dummy PatchInfo

\begin{code}
nullPI = patchinfo [] [] [] []
\end{code}

population of an empty repository

\begin{code}
initPop = Pop nullPI (PopDir i [])
 where i = Info {name = packString ".",
                 modifiedBy = nullPI,
                 modifiedHow = DullDir,
                 createdBy = Nothing,
                 creationName = Just (packString ".")}
\end{code}

get the actual documents and folders (with full name)
from a population tree (and their modification times)
(take care for ``deleted'' files and dirs)

\begin{code}
popUnfold :: Population -> ([(PackedString,PatchInfo)],
                            [(PackedString,PatchInfo)])
popUnfold (Pop _ s) = popUnfold' nilPS s
 where popUnfold' pre (PopDir f ss)
        | modifiedHow f == RemovedDir = (ds,fs)
        | otherwise = (ds,f':fs)
         where f' = (newname, modifiedBy f)
               newname = if pre == nilPS
                            then name f
                            else pre `appendPS` (packString "/") `appendPS` name f
               (dss,fss) = (unzip . map (popUnfold' newname)) ss
               ds = concat dss
               fs = concat fss
       popUnfold' pre (PopFile d)
        | modifiedHow d == RemovedFile = ([],[])
        | pre == nilPS = ([(name d,modifiedBy d)],[])
        | otherwise = ([(pre `appendPS` (packString "/") `appendPS`
                         name d,
                         modifiedBy d)],[])
\end{code}

get the changed ``dirty'' documents and folders (with full name)
from a population tree (and their modification times)
(take care for ``deleted'' files and dirs)

\begin{code}
popUnfoldDirty :: Population -> ([(PackedString,PatchInfo)],
                                 [(PackedString,PatchInfo)])
popUnfoldDirty (Pop _ s) = popUnfold' nilPS s
 where popUnfold' pre (PopDir f ss)
        | notModified f || modifiedHow f == RemovedDir = (ds,fs)
        | otherwise = (ds,f':fs)
         where f' = (newname, modifiedBy f)
               newname = if pre == nilPS
                            then name f
                            else pre `appendPS` (packString "/") `appendPS` name f
               (dss,fss) = (unzip . map (popUnfold' newname)) ss
               ds = concat dss
               fs = concat fss
       popUnfold' pre (PopFile d)
        | notModified d || modifiedHow d == RemovedFile = ([],[])
        | pre == nilPS = ([(name d,modifiedBy d)],[])
        | otherwise = ([(pre `appendPS` (packString "/") `appendPS`
                         name d,
                         modifiedBy d)],[])
\end{code}

read the population from a given directory ``dirname''
all folders and documents get the given time ``t''

\begin{code}
getPopFrom :: FilePath -> PatchInfo -> IO Population
getPopFrom dirname pi
  = do former_dir <- getCurrentDirectory
       setCurrentDirectory dirname
       popT <- getPopFrom_helper "."
       setCurrentDirectory former_dir
       return (Pop pi popT)
 where getPopFrom_helper :: FilePath -> IO PopTree
       getPopFrom_helper dirname = do
        isdir <- doesDirectoryExist dirname
        let n = packString dirname
        if isdir
          then do
           former_dir <- getCurrentDirectory
           fnames <- getDirectoryContents dirname
           setCurrentDirectory dirname
           sl <- sequence $ map getPopFrom_helper $ filter not_hidden fnames
           setCurrentDirectory former_dir
           let i = Info {name = n,
                         modifiedBy = pi,
                         modifiedHow = DullDir,
                         createdBy = Just pi,
                         creationName = Just n}
           return $ PopDir i sl
          else do let i = Info {name = n,
                                modifiedBy = pi,
                                modifiedHow = DullFile,
                                createdBy = Just pi,
                                creationName = Just n}
                  return $ PopFile i

not_hidden :: FilePath -> Bool
not_hidden ('.':_) = False
not_hidden ('_':_) = False
not_hidden _ = True
\end{code}

apply a patchset to a population
[[(PatchInfo, Maybe Patch)]] is actually a PatchSet but this provokes cycles in import hierarchy

\begin{code}
applyPatchSetPop :: [[(PatchInfo, Maybe Patch)]] -> Population -> Population
applyPatchSetPop ps pop = applyPatchesPop (catMaybes' $ concat ps) pop
 where catMaybes' [] = []
       catMaybes' ((_,Nothing):ps) = catMaybes' ps
       catMaybes' ((pi,Just p):ps) = (pi,p) : catMaybes' ps
\end{code}

apply Patches to a population

\begin{code}
applyPatchesPop :: [(PatchInfo,Patch)] -> Population -> Population
applyPatchesPop pips pop = foldl (\pop (pi,p) -> applyToPop pi p pop) pop pips
\end{code}

adjust the ``modifiedBy'' fields of a population
this is necessary for backward restoring a population
Patches must be in !reverse! order and already inversed!
Usually they go from the populations PatchInfo back to the initial PatchInfo.
Note that backward restoring invalidates (must!) the fields createdBy and creationName.
This is necessary because while backward creation of a population we use inverted patches.
So RmFile becomes an AddFile etc. which means that the file ``would be created after it has been changed''!

\begin{code}
adjustPopStates :: [(PatchInfo,Patch)] -> Population -> Population
adjustPopStates  pips (Pop _ tree)
 = Pop (fst (pips!!i)) tree'
 where (tree',i) = adjustTimesPTree tree
       -- snd component: greatest number of PI which changed the population (0 based)
       -- that is the youngest given patchinfo (relied on the order of the given list)
       adjustTimesPTree :: PopTree -> (PopTree,Int)
       adjustTimesPTree tr@(PopDir f trs)
        | modifiedHow f == RemovedDir = (tr,0) -- for removed dirs there if no previous modifying patch!
        | otherwise
        = let (pi,dm,i) = lastChange 0 (name f) changes
              (trs',is) = unzip (map adjustTimesPTree trs)
              i' = max (maximum is) i
          in (PopDir (Info {name = (name f),
                           modifiedBy = pi,
                           modifiedHow = dm,
                           createdBy = Nothing,
                           creationName = Nothing}) trs'
             ,i')
       adjustTimesPTree tr@(PopFile f)
        | modifiedHow f == RemovedFile = (tr,0) -- for removed files there if no previous modifying patch!
        | otherwise
        = let (pi,dm,i) = lastChange 0 (name f) changes
          in (PopFile (Info {name = (name f),
                            modifiedBy = pi,
                            modifiedHow = dm,
                            createdBy = Nothing,
                            creationName = Nothing})
             ,i)
       lastChange :: Int -> PackedString -> [(PatchInfo,[(PackedString,DirMark)])]
                                         -> (PatchInfo,DirMark,Int)
       lastChange _ n [] = error ("lastChange internal error: No modifying patch for " ++ show n)
       lastChange i n ((pi,ssdm):sss)
         = case lookupBy (\ss -> n `elem` (splitPS '/' ss)) ssdm of
            Just dm -> (pi,dm,i) -- pi changes n somehow (indicated by the DirMark dm)
            _ -> lastChange (i+1) n sss
       changes = map (\ (pi,p) -> (pi,nub $ map (\ (s,d) -> (packString s,d)) $ patchChanges p)) pips

lookupBy :: (a -> Bool) -> [(a,b)] -> Maybe b
lookupBy f ((a,b):abs)
 | f a = Just b
 | otherwise = lookupBy f abs
lookupBy _ [] = Nothing
\end{code}

clean up a population
remove any change markers, delete ``deleted'' files and dirs

\begin{code}
cleanPop :: Population -> Population
cleanPop (Pop t tr) = Pop t (fromJust (cleanPopTr tr))
 where cleanPopTr (PopDir i trs)
        | modifiedHow i == RemovedDir = Nothing
        | otherwise = Just $ PopDir (modInfo DullDir i)
                                    (catMaybes (map cleanPopTr trs))
       cleanPopTr (PopFile i)
        | modifiedHow i == RemovedFile = Nothing
        | otherwise = Just $ PopFile (modInfo DullFile i)
       modInfo s i = Info {name = name i,
                           modifiedBy = nullPI,
                           modifiedHow = s,
                           createdBy = createdBy i,
                           creationName = creationName i}
\end{code}


get the current(!) population from a repo

\begin{code}
getRepoPop :: FilePath -> IO Population
getRepoPop repobasedir
 = do pi <- liftM (fst . head . concat) (read_repo repobasedir)
      -- pi is the latest patchinfo
      getPopFrom (repobasedir ++ "_darcs/current") pi
\end{code}

\begin{code}
getRepoPopVersion :: FilePath -> PatchInfo -> IO Population
getRepoPopVersion repobasedir pi
 = do pips <- read_repo repobasedir
      let (pips',rest) = span (\ps -> pi `notElem` (map fst ps)) pips
          (rest',rest'') = span (\ (pi',_) -> pi'/=pi) (head rest)
          pips'' = pips' ++ [rest' ++ [head rest'']]
      -- pi is the latest patchinfo
      return (applyPatchSetPop pips'' initPop)
\end{code}
