%  Copyright (C) 2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

\begin{code}
module Lock ( withLock,
              withTemp, withOpenTemp,
              withTempDir,
            ) where

import Prelude hiding ( catch )
import Monad ( liftM )
import System ( exitWith, ExitCode(..), system )
import IO hiding ( catch )
import qualified IO ( catch )
import Control.Concurrent ( myThreadId )
import Control.Exception ( catch, catchJust, throw, throwTo, ioErrors,
                           Exception( ExitException ) )
import GHC.Handle ( openFd )
import Directory ( setCurrentDirectory, getCurrentDirectory, removeFile,
                   createDirectory )
import Posix ( sleep,
               installHandler, Handler(..), raiseSignal,
               sigINT, sigQUIT, sigKILL, sigHUP )
import Foreign
import Foreign.C

throwIO e = return $ throw e

withLock :: String -> IO a -> IO a
takeLock :: String -> IO Bool
releaseLock :: String -> IO ()

smartbracket :: IO a -> (a -> IO b) -> (a -> IO c) -> IO c
smartbracket init clean job = do
  r <- init
  id <- myThreadId
  installHandler sigINT -- Make sure to clean up if user hits ctrl-C
       (Catch $ do putStrLn "Interrupted!"
                   throwTo id $ ExitException $ ExitFailure 3)
       Nothing
  -- FIXME: I really should remove this handler when I clean up, but there
  -- doesn't seem to be any way to know what the old handler was!
  out <- job r `catch` (cleanup r)
  clean r
  return out
      where cleanup r e@(ExitException _) = do { clean r; throwIO e }
            cleanup _ e = throwIO e

withLock s job = smartbracket (getlock s 30) releaseLock (\_ -> job)
getlock :: String -> Int -> IO String
getlock l 0 = do putStrLn $ "Couldn't get lock "++l
                 exitWith $ ExitFailure 1
getlock lbad tl = do l <- canonFilename lbad
                     gotit <- takeLock l
                     if gotit then return l
                              else do putStrLn $ "Waiting for lock "++l
                                      sleep 5
                                      getlock l (tl - 1)

releaseLock s = catchJust ioErrors (removeFile s) $
                \e -> if isDoesNotExistError e then return ()
                                               else ioError e
takeLock s = withCString s $ \cstr -> do
    fd <- c_creat cstr 0
    if fd == -1 then return False
                else do c_close fd
                        return True

foreign import ccall unsafe "static fcntl.h creat" c_creat
    :: CString -> Int -> IO Int
foreign import ccall unsafe "static unistd.h unlink" c_unlink
    :: CString -> IO Int
foreign import ccall unsafe "static unistd.h close" c_close
    :: Int -> IO Int

canonFilename f@('/':_) = return f
canonFilename ('.':'/':f) = do cd <- getCurrentDirectory
                               return $ cd ++ "/" ++ f
canonFilename f = case reverse $ dropWhile (/='/') $ reverse f of
                  "" -> liftM (++('/':f)) getCurrentDirectory
                  rd -> do cd <- getCurrentDirectory
                           setCurrentDirectory rd
                           fd <- getCurrentDirectory
                           setCurrentDirectory cd
                           return $ fd ++ "/" ++ simplefilename f
    where
    simplefilename f = reverse $ takeWhile (/='/') $ reverse f
\end{code}

\verb!withTemp! safely creates an empty file (not open for writing) and
returns its name.  \verb!withOpenTemp! creates an already open temporary
file.  Both of them run their argument and then delete the file.  Also,
both of them (to my knowledge) are not susceptible to race conditions on
the temporary file (as long as you never delete the temporary file--that
would reintroduce a race condition).

The temp file operations are rather similar to the locking operations, in
that they both should always try to clean up, so exitWith causes trouble.

\begin{code}
withTemp :: (String -> IO a) -> IO a
withTemp = smartbracket get_empty_file removeFile
    where get_empty_file = do (h,f) <- mkstemp "darcs"
                              hClose h
                              return f

withOpenTemp :: ((Handle, String) -> IO a) -> IO a
withOpenTemp = smartbracket (mkstemp "darcs") cleanup
    where cleanup (h,f) = do try $ hClose h
                             removeFile f

mkstemp :: String -> IO (Handle, String)
mkstemp str = withCString (str++"XXXXXX") $
    \cstr -> do fd <- c_mkstemp cstr
                if fd < 0
                  then throwErrno $ "Failed to create temporary file "++str
                  else do str' <- peekCString cstr
                          fname <- canonFilename str'
                          h <- openFd fd Nothing fname ReadWriteMode True False
                          return (h, fname)

foreign import ccall unsafe "static stdlib.h mkstemp" c_mkstemp :: CString -> IO Int
\end{code}

\verb!withTempDir! creates an empty directory and then removes it when it
is no longer needed.  Unlike the temp file routines, withTemp creates the
directory in the current directory, which is assumed to have write
permission only for the current user (so that race conditions aren't a
problem, unlike in /tmp, which is world writeable).

\begin{code}
withTempDir :: String -> (String -> IO a) -> IO a
withTempDir name job = do
  formerdir <- getCurrentDirectory
  smartbracket (create_directory 0) (remove_directory formerdir) job
    where newname 0 = name
          newname n = name ++ "-" ++ show n
          create_directory n
              = do createDirectory (newname n)
                   setCurrentDirectory (newname n)
                   getCurrentDirectory
                `IO.catch` (\e -> if isAlreadyExistsError e
                            then create_directory (n+1)
                            else ioError e)
          remove_directory f d = do setCurrentDirectory f
                                    system $ "rm -rf "++d
\end{code}
