%  Copyright (C) 2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs diff}
\begin{code}
module DiffCommand ( diff_command ) where
import Directory
import IO
import System
import RegexString ( mkRegex, matchRegex )
import Monad ( when )
import List ( (\\) )

import DarcsCommands
import DarcsArguments
import Repository
import Patch
import PatchInfo
import SlurpDirectory
import External
import Lock ( withTempDir )
\end{code}

\options{diff}
\begin{code}
diff_description = "Create a diff between two versions of the repository.
"
\end{code}
\haskell{diff_help}
\begin{code}
diff_help = "Diff can be used to create a diff between two versions which
are in your repository.
"

diff_command = DarcsCommand {command_name = "diff",
                             command_help = diff_help,
                             command_description = diff_description,
                             command_extra_args = -1,
                             command_command = diff_cmd,
                             command_prereq = am_in_repo,
                             command_get_arg_possibilities = return ["-u","-c"],
                             command_argdefaults = nodefaults,
                             command_darcsoptions = [tagname,
                                                     patchname_option]}
\end{code}
Diff calls an external ``diff'' command to do the actual work, and passes
any unrecognized flags to this diff command.  Thus you can call
\begin{verbatim}
% darcs diff -t 0.9.8 -t 0.9.10 -- -u
\end{verbatim}
to get a diff in the unified format.  Actually, thanks to the wonders of
getopt you need the ``\verb!--!'' shown above before any arguments to diff.
FIXME: I probably should just bite the bullet and figure out how to get
diff flags passed through properly.  Any hints from getopt gurus would be
appreciated.

FIXME: I should allow the user to specify the external diff command.
Currently it is hardwired to /usr/bin/diff.

\begin{code}
diff_cmd opts args = do
  when (not $ chose_one opts) $
       do  putStr "You must specify at least one version to get a diff.\n"
           exitWith $ ExitFailure 1
  
  formerdir <- getCurrentDirectory
  thename <- return $reverse (takeWhile (/='/') $ reverse formerdir)
  withTempDir (formerdir++"/_darcs/"++thename++"-old") $ \odir ->
    withTempDir (formerdir++"/_darcs/"++thename++"-new") $ \ndir -> do
    setCurrentDirectory odir
    patches <- read_repo formerdir
    sequence_ $ map (apply_patch_here False) $
              reverse $ concat $ pick_patches opts patches
    setCurrentDirectory ndir
    if chose_two opts
      then do ps <- read_repo formerdir
              sequence_ $ map (apply_patch_here False) $
                        reverse $ concat $ pick_second_patches opts ps
      else do recorded <- slurp_recorded formerdir
              slurp_write recorded
    thediff <- runPipe (["/usr/bin/diff","-rN"]++args++[odir,ndir]) ""
    morepatches <- read_repo formerdir
    putStrLn $ changelog $ get_diff_info opts morepatches
    putStr thediff
\end{code}

\begin{code}
chose_two :: [DarcsFlag] -> Bool
chose_two (TagName _:fs) = chose_one fs
chose_two (PatchName _:fs) = chose_one fs
chose_two (_:as) = chose_two as
chose_two [] = False
chose_one :: [DarcsFlag] -> Bool
chose_one (TagName _:_) = True
chose_one (PatchName _:_) = True
chose_one (_:as) = chose_one as
chose_one [] = False

pick_second_patches :: [DarcsFlag] -> PatchSet -> PatchSet
pick_second_patches (TagName _:ps) = pick_patches ps
pick_second_patches (PatchName _:ps) = pick_patches ps
pick_second_patches (_:fs) = pick_second_patches fs

pick_patches :: [DarcsFlag] -> PatchSet -> PatchSet
pick_patches (TagName t:_) = pictag (mymatch t)
pick_patches (PatchName pn:_) = picpatch (mymatch pn)
pick_patches (_:fs) = pick_patches fs
pick_patches [] = \ ps -> ps

pictag :: (String -> Bool) -> PatchSet -> PatchSet
pictag t (((pi,mp):ps):r)
    | take 4 (just_name pi) == "TAG " && t (drop 4 (just_name pi))
        = ((pi,mp):ps):r
    | otherwise = pictag t (ps:r)
pictag t ([]:r) = pictag t r
pictag t [] = error "Couldn't find matching tag!\n"

picpatch :: (String -> Bool) -> PatchSet -> PatchSet
picpatch t (((pi,mp):ps):r) =
    if t (just_name pi) then ((pi,mp):ps):r
                        else picpatch t (ps:r)
picpatch t ([]:r) = picpatch t r
picpatch t [] = error "Couldn't find matching patch!\n"

mymatch ('^':r) s = matchRegex (mkRegex ('^':r)) s /= Nothing
mymatch r s = matchRegex (mkRegex (".*"++r)) s /= Nothing
\end{code}

\begin{code}
apply_patch_here :: Bool -> (PatchInfo, Maybe Patch) -> IO ()
apply_patch_here verb (pi,mp) = do
  s <- slurp "."
  when verb $ putStr $ "\nApplying patch:\n"++show pi
  case mp of
    Nothing -> do putStr "There seems to be an error in the repo!\n"
                  putStr "I'm having trouble reading a patch.  :(\n"
                  exitWith $ ExitFailure 1
    Just p ->
        case apply_to_slurpy p s of
        Just s' -> slurp_write_dirty s'
        Nothing -> do putStr "There seems to be an error in the repo!\n"
                      putStr "Error in patch:\n"
                      putStr $ show p
                      exitWith $ ExitFailure 1
\end{code}

\begin{code}
get_diff_info :: [DarcsFlag] -> PatchSet -> [PatchInfo]
get_diff_info opts ps =
    case map fst $ concat $ pick_patches opts ps of
    pi1s ->
        case (if chose_two opts
              then map fst $ concat $ pick_second_patches opts ps
              else map fst $ concat ps) of
        pi2s -> pi2s \\ pi1s

changelog :: [PatchInfo] -> String
changelog pis = unlines $ map human_friendly pis
\end{code}
