%  Copyright (C) 2003 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\section{darcs apply}
\begin{code}
module Apply ( apply ) where
import IO
import System
import Monad ( when, unless, liftM )
import List ( intersect, elem )
import Maybe ( fromJust )

import DarcsCommands
import DarcsArguments
import Directory
import Repository
import Patch
import PatchInfo
import SlurpDirectory
import DarcsArguments
import RegexString
import RepoPrefs
import Lock ( withLock )
import Pull ( merge_with_us_and_pending, save_patches )
import Depends ( get_common_and_uncommon )
import Resolution ( standard_resolution, no_resolution )
import Test ( test_slurpy )
\end{code}
\begin{code}
apply_description =
 "Apply patches to a repo."
\end{code}

\options{apply}

\haskell{apply_help}
\begin{code}
apply_help =
 "Apply is used to apply a bundle of patches to this repository.
Such a bundle may be created using push.
"
\end{code}
\begin{code}
apply = DarcsCommand {command_name = "apply",
                      command_help = apply_help,
                      command_description = apply_description,
                      command_extra_args = 1,
                      command_command = apply_cmd,
                      command_prereq = am_in_repo,
                      command_get_arg_possibilities = list_files,
                      command_argdefaults = nodefaults,
                      command_darcsoptions = [verify, verbose, ignoretimes,
                                              no_resolve_conflicts, test]}
\end{code}
\begin{code}
apply_cmd opts [patchesfile] = withLock "./_darcs/lock" $ do
  am_verbose <- return $ Verbose `elem` opts
  resolve <- if NoResolve `elem` opts then return no_resolution
                                      else return standard_resolution
  former_dir <- getCurrentDirectory
  aminrepo <- is_repo "."
  unless aminrepo $ do putStr $ "Aaack, I'm not in a repo!\n"
                       exitWith $ ExitFailure 1
  us <- read_repo "."
  them <- get_patch_bundle opts patchesfile
  (comm, us', them') <- return $ get_common_and_uncommon (us, them)
  when am_verbose $ putStr "We have the following extra patches:\n"
  when am_verbose $ putStr $ format_inventory $ head us'
  when am_verbose $ putStr "Will apply the following patches:\n"
  when am_verbose $ putStr $ format_inventory $ head them'
  (us_patch, work_patch) <- merge_with_us_and_pending opts
                            (reverse $ head us', reverse $ head them')
  recorded <- slurp_recorded "."
  working <- slurp "."
  when am_verbose $ putStr "Applying patches to the local directories...\n"
  case apply_to_slurpy us_patch recorded of
      Nothing -> do putStr "Error applying patch to recorded!\n"
                    exitWith $ ExitFailure 1
      Just rec' ->
          case apply_to_slurpy (resolve work_patch) working of
          Nothing -> do putStr "Error applying patch to working dir.\n"
                        exitWith $ ExitFailure 1
          Just work' -> do
              when (Test `elem` opts) $
                do recb <- slurp_recorded "."
                   testproblem <- test_slurpy $
                                  fromJust $ apply_to_slurpy us_patch recb
                   when (testproblem /= ExitSuccess) $
                        exitWith $ ExitFailure 1
              save_patches $ unjoin_patches us_patch
              setCurrentDirectory "_darcs/current"
              slurp_write_dirty rec'
              wait_a_moment
              setCurrentDirectory former_dir
              sequence $ map (add_to_inventory "." . fst) $ reverse $ head them'
              slurp_write_dirty work'
              putStr $ "Finished applying.\n"
\end{code}

If you specify the \verb!--verify PUBRING! option, darcs will check that
the patch was gpg-signed by a key which is in \verb!PUBRING!.

\begin{code}
get_patch_bundle :: [DarcsFlag] -> FilePath -> IO PatchSet
get_patch_bundle [] f = liftM scan_patch_bundle $ readFile f
get_patch_bundle (Verify pr:_) f = verify_signed_patch_bundle pr f
get_patch_bundle (_:os) f = get_patch_bundle os f
\end{code}

\begin{code}
verify_signed_patch_bundle :: FilePath -> FilePath
                           -> IO PatchSet
verify_signed_patch_bundle goodkeys f = do
    rval <- system ("gpg --batch --no-default-keyring --keyring "
                    ++ goodkeys ++ " --verify " ++ f)
    case rval of
      ExitSuccess ->
          liftM (scan_patch_bundle . unlines .
                 (dropWhile (/= "-----BEGIN PGP SIGNED MESSAGE-----"))
                 . lines) $ readFile f
      _ -> do putStr "Bad signature on patch (or gpg failure)!\n"
              exitWith $ ExitFailure 1
\end{code}

\begin{code}
scan_patch_bundle str =
    case silly_lex str of
      ("Context:",rest) ->
        case get_context rest of
        (cont, rest') ->
            case silly_lex rest' of
            ("New patches:", rest'') -> [reverse (parse_patches rest'') ++
                                         reverse (zip cont nothings)]
      ("-----BEGIN PGP SIGNED MESSAGE-----",rest) ->
            scan_patch_bundle $ filter_gpg_dashes rest
      (_,rest) -> scan_patch_bundle rest
filter_gpg_dashes s = unlines $ map drop_dashes $
                      takeWhile (/= "-----END PGP SIGNED MESSAGE-----") $
                      dropWhile (/= "Context:") $ lines s
drop_dashes ('-':' ':rest) = rest
drop_dashes r = r
nothings = Nothing : nothings
get_context :: String -> ([PatchInfo],String)
get_context s =
    case reads s of
    [(pi,r')] ->
        case get_context r' of
        (pis,r'') -> (pi:pis, r'')
    [] -> ([],s)
parse_patches :: String -> PatchSequence
parse_patches s =
  case reads s of
  [] -> []
  [(pi,_)] ->
    case readPatch s of
    [] -> []
    [(p, r)] -> (pi,Just p) : parse_patches r
silly_lex ('\n':s) = silly_lex s
silly_lex (' ':s) = silly_lex s
silly_lex s = (takeWhile (/='\n') s, dropWhile (/='\n') s)
\end{code}
