;;
;; The contents of this file are subject to the Mozilla Public
;; License Version 1.1 (the "License"); you may not use this file
;; except in compliance with the License. You may obtain a copy of
;; the License at http://www.mozilla.org/MPL/
;; 
;; Software distributed under the License is distributed on an "AS
;; IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
;; implied. See the License for the specific language governing
;; rights and limitations under the License.
;; 
;; The Original Code is the Sablotron XSLT Processor.
;; 
;; The Initial Developer of the Original Code is Ginger Alliance Ltd.
;; Portions created by Ginger Alliance are Copyright (C) 2002 Ginger
;; Alliance Ltd. All Rights Reserved.
;; 
;; Contributor(s):
;; 
;; Alternatively, the contents of this file may be used under the
;; terms of the GNU General Public License Version 2 or later (the
;; "GPL"), in which case the provisions of the GPL are applicable 
;; instead of those above.  If you wish to allow use of your 
;; version of this file only under the terms of the GPL and not to
;; allow others to use your version of this file under the MPL,
;; indicate your decision by deleting the provisions above and
;; replace them with the notice and other provisions required by
;; the GPL.  If you do not delete the provisions above, a recipient
;; may use your version of this file under either the MPL or the
;; GPL.


(require 'comint)

;;we need to steal the current buffer, when the comint hooks are executed
(defadvice comint-output-filter (around sabdbg-steal-buffer activate)
  (let ((sabdbg-command-origin (current-buffer)))
    ad-do-it))

(defun sabdbg-debug-stylesheet(sheet data)
  "Run the Sablotron XSLT debugger"
  (interactive (list (read-file-name "Stylesheet: ")
		     (read-file-name "Data file: ")))
  (let ((buff (get-buffer "*sabdbg*"))
	(proc nil)
	(isnew nil))
    (unless buff
      (setq buff (make-comint "sabdbg" "sabcmd" nil "--debugger")
	    isnew t))
    ;;show buffer
    (switch-to-buffer-other-window buff)
    (when isnew 
      (sabdbg-set-keys)
      (add-hook 'comint-output-filter-functions 'sabdbg-output-hook))
    ;;set globals
    (setq sabdbg-buffer buff
	  sabdbg-proc (get-buffer-process buff))
    ;;set stylesheet
    (comint-simple-send proc (concat "sheet " (expand-file-name sheet)))
    (if data
	(comint-simple-send proc (concat "data " (expand-file-name data))))))
  
(defun sabdbg-output-hook(str)
  (let ((lines (split-string str "\n"))
	(re-step (eval-when-compile 
		   (concat "^"
			   (regexp-opt 
			    '("Template" "Step" "Breakpoint met") 
			   t))))
	(re-break (eval-when-compile 
		    (concat "^"
			    (regexp-opt 
			     '("breakpoint set" "breakpoint deleted" 
			       "all breakpoints" "breakpoint modified") 
			     t))))
	(re-finish "^debugger finished")
	(re-start "^debugger started")
	(re-quit "^debugger quited"))
    ;;function body
    (mapcar '(lambda(x)
	       (when (and (not (string= x ""))
			  (not (string-match "^[[:digit:]]+\t" x)))
		 ;;we met some instruction moving the IP
		 (cond ((string-match re-finish x)
			(sabdbg-hide-arrow t))
		       ((string-match re-step x)
			(when (and (boundp 'sabdbg-just-ran) sabdbg-just-ran)
			  (setq sabdbg-just-ran nil)
			  (sabdbg-update-breakpoints))
			(sabdbg-sync-to-trace x))
		       ((string-match re-break x)
			(sabdbg-update-breakpoints))
		       ((string-match re-start x)
			(setq sabdbg-just-ran t))
		       ((string-match re-quit x)
			(sabdbg-terminate))
		       (t nil))))
	    lines)))

(defun sabdbg-discard-output-hook(str)
  (let ((lines (split-string str "\n")))
    (catch 'prompt-met
      (mapcar '(lambda(x)
		 (if (not (string-match "^sablot>" x))
		     (setq sabdbg-last-output
			   (concat sabdbg-last-output "\n" x))
		   (setq sabdbg-output-done t)
		   (throw 'prompt-met nil))) 
	      lines))
    (if sabdbg-output-done 
	"sablot> "
      "")))

(defun sabdbg-read-output-discard(cmd)
  (let ((proc sabdbg-proc) (buff sabdbg-buffer))
    (unless proc
      (error "sabdbg is not running"))
    (save-current-buffer
      (set-buffer buff)
      (accept-process-output proc t 200) ;discard pending output
      (add-hook 'comint-preoutput-filter-functions 
		'sabdbg-discard-output-hook)
      (setq sabdbg-last-output "") ; here the output goes
      (let ((sabdbg-output-done nil))
	(comint-simple-send proc cmd)
	(while (not sabdbg-output-done)
	  (accept-process-output proc t 100000)))
      (remove-hook 'comint-preoutput-filter-functions 
		   'sabdbg-discard-output-hook))
    sabdbg-last-output))

(defun sabdbg-hide-arrow(force)
  (if (and (boundp 'sabdbg-arrow-marker) 
	   (marker-buffer sabdbg-arrow-marker)
	   (or force (not (eq (current-buffer) 
			      (marker-buffer sabdbg-arrow-marker)))))
      (save-current-buffer
	(set-buffer (marker-buffer sabdbg-arrow-marker))
	(setq overlay-arrow-position nil))))

(defun sabdbg-show-arrow (buffer line)
  (let* ((window (get-buffer-window buffer))
	 (pos))
    (save-excursion
      (set-buffer buffer)
      (save-restriction
	(widen)
	(goto-line line)
	(setq pos (point))
	(setq overlay-arrow-string "=>")
	(or overlay-arrow-position
	    (setq overlay-arrow-position (make-marker)))
	;;clear arrow at old position
	(sabdbg-hide-arrow nil)
	;;set a new arrow marker
	(set-marker overlay-arrow-position (point) (current-buffer))
	;;save, where the arrow marker is
	(setq sabdbg-arrow-marker overlay-arrow-position))
      (cond ((or (< pos (point-min)) (> pos (point-max)))
	     (widen)
	     (goto-char pos))))
    (set-window-point window overlay-arrow-position)))

;;we're stealing the sabdbg-command-origin in the advice for 
;;comint-output-filter, it's a bit dangersou, but just a bit
(defun sabdbg-sync-to-trace(str)
  (let ((mdata) (file) (line) (currbuf) 
	(sabdbg-command-origin (or (and (boundp 'sabdbg-command-origin)
					sabdbg-command-origin)
				   (current-buffer))))
    ;;body
    (save-selected-window
      (save-current-buffer
	(catch 'no-sync
	  (when (string-match "^error" str)
	    (ding)
	    (throw 'no-sync nil))
	  (if (string-match "file:\\([^:]+\\):\\([[:digit:]]*\\)" str)
	      (progn
		(setq mdata (match-data))
		(setq file (substring str (nth 2 mdata) (nth 3 mdata)))
		(setq line (substring str (nth 4 mdata) (nth 5 mdata)))
		(setq currbuf (if (eq sabdbg-command-origin sabdbg-buffer)
				  (find-file-other-window file)
				(find-file file)))
		(sabdbg-show-arrow currbuf (string-to-int line)))
	    (ding)))))))

(defun sabdbg-get-breakpoints()
  (let ((str) (bpoints) (ret) (one) (aux)
	(file) (line) (mdata) (num) (status))
    (setq str (sabdbg-read-output-discard "B"))
    (if (string-match "^no break" str)
	nil
      ;;the salt
      (setq bpoints (split-string str "\n"))
      (mapcar '(lambda(x)
		 (when (string-match "^#\\([[:digit:]]+\\):\\(.*?\\)file:\\([^:]+\\):\\([[:digit:]]+\\)" x)
		   (setq mdata (match-data))
		   (setq num  (substring x (nth 2 mdata) (nth 3 mdata)))
		   ;;(setq invalid (> (- (nth 5 mdata) (nth 4 mdata)) 2))
		   (setq aux (substring x (nth 4 mdata) (nth 5 mdata)))
		   (cond ((string-match "disabled" aux) 
			  (setq status 'disabled))
			 ((string-match "invalid" aux)
			  (setq status 'invalid))
			 (t (setq status 'ok)))
		   (setq file (substring x (nth 6 mdata) (nth 7 mdata)))
		   (setq line (substring x (nth 8 mdata) (nth 9 mdata)))
		   (setq one (list
			      ;;(cons 'invalid invalid)
			      (cons 'status status)
			      (cons 'number num)
			      (cons 'line (string-to-int line))
			      (cons 'file file)
			      (cons 'props (substring x (+ 2 (nth 3 mdata))))))
		   (if ret
		       (nconc ret (cons one nil))
		     (setq ret (cons one nil)))))
	      bpoints))
    ret))

(defun sabdbg-apply-breakpoints(filename)
  (let ((buff) (over) (beg) (file) (line) (status) (bface))
    (mapcar '(lambda(x)
	       (setq file (cdr (assq 'file x)))
	       (setq line (cdr (assq 'line x)))
	       (if (and (or (not filename) (string= filename file))
			(setq buff (get-file-buffer file)))
		   (with-current-buffer buff
		     (save-excursion
		       (goto-line line)
		       (beginning-of-line)
		       (setq beg (point))
		       (setq over (make-overlay beg (1+ beg)))
		       (overlay-put over 'intangible t)
		       (overlay-put over 'before-string (cdr (assq 'number x)))
		       (setq status (cdr (assq 'status x)))
		       (cond ((eq status 'invalid)
			      (setq bface '(:background "lightgray" 
					    :foreground "black")))
			     ((eq status 'disabled)
			      (setq bface '(:background "black" 
					    :foreground "white")))
			     (t (setq bface '(:background "red" 
					      :foreground "white"))))
		       (overlay-put over 'face bface)
		       (overlay-put over 'help-echo (cdr (assq 'props x)))
		       (move-overlay over beg (1+ beg))
		       (nconc x (cons (cons 'overlay over) nil))))))
	    sabdbg-breakpoints)))

(defun sabdbg-update-breakpoints()
  (let ((over))
  ;; clear old breakpoints
    (if (boundp 'sabdbg-breakpoints)
	(mapcar '(lambda(x)
		   (setq over (cdr (assq 'overlay x)))
		   (if over (delete-overlay over)))
		sabdbg-breakpoints))
    ;;get new breakpoints
    (if (and (boundp 'sabdbg-proc) sabdbg-proc)
	(setq sabdbg-breakpoints (sabdbg-get-breakpoints))
      (setq sabdbg-breakpoints nil))
    ;;apply new breakpoints
    (sabdbg-apply-breakpoints nil)))

(defun sabdbg-terminate()
  (setq sabdbg-proc nil)
  (sabdbg-update-breakpoints))

(defun sabdbg-show-point()
  "Show the current execution point"
  (interactive)
  (let ((str nil))
    (save-current-buffer
      (setq str (sabdbg-read-output-discard "point"))
      (if (string-match "processor is NOT running" str)
	  (error "debugger is in an idle mode")))
    (sabdbg-sync-to-trace str)))

;;; functions for xsl mode

(defun sabdbg-check-debugger()
  (let* ((buff (get-buffer "*sabdbg*"))
	 (proc (get-buffer-process buff)))
    (unless (and buff proc)
      (error "debugger is not running"))))

(defun sabdbg-debug-current()
  "Run current stylesheet under debugger"
  (interactive)
  (save-selected-window
    (setq data (read-file-name "Data file: "))
    (sabdbg-debug-stylesheet buffer-file-name data)))

(defun sabdbg-do-command(cmd)
  (sabdbg-check-debugger)
  (comint-simple-send sabdbg-proc cmd))

(defun sabdbg-step()
  "Steps the debugger"
  (interactive)
  (sabdbg-do-command "step"))

(defun sabdbg-next()
  "Next in the debugger"
  (interactive)
  (sabdbg-do-command "next"))

(defun sabdbg-step-template()
  "Steps to template in the debugger"
  (interactive)
  (sabdbg-do-command "templ"))

(defun sabdbg-continue()
  "Continues the debugger"
  (interactive)
  (sabdbg-do-command "cont"))

(defun sabdbg-run()
  "Runs the debugger"
  (interactive)
  (sabdbg-do-command "run"))

(defun sabdbg-set-breakpoint()
  "Set breakpoin at the current line"
  (interactive)
  (let ((file buffer-file-name) (line))
    (setq line (1+ (count-lines 1 (point))))
    (sabdbg-do-command (concat "b " file ":" (int-to-string line)))))

(defun sabdbg-set-keys()
  "Set keys for debugger"
  (interactive)
  (local-set-key [(control c)(control g)(control x)] 'sabdbg-debug-current)
  (local-set-key [(control c)(control g)(control s)] 'sabdbg-step)
  (local-set-key [(control c)(control g)(control n)] 'sabdbg-next)
  (local-set-key [(control c)(control g)(control t)] 'sabdbg-step-template)
  (local-set-key [(control c)(control g)(control c)] 'sabdbg-continue)
  (local-set-key [(control c)(control g)(control p)] 'sabdbg-show-point)
  (local-set-key [(control c)(control g)(control b)] 'sabdbg-set-breakpoint)
  (local-set-key [(control c)(control g)(control r)] 'sabdbg-run)
)

(defun sabdbg-init()
  (sabdbg-set-keys)
  (make-local-variable 'overlay-arrow-position)
  (put 'overlay-arrow-position 'permanent-local t)
  (if (and (boundp 'sabdbg-proc)
	   (boundp 'sabdbg-breakpoints))
      (sabdbg-apply-breakpoints buffer-file-name)
    ))

