/********************************************************************************
*                                                                               *
*                             R a n g e    C l a s s                            *
*                                                                               *
*********************************************************************************
* Copyright (C) 1994,2003 by Jeroen van der Zijp.   All Rights Reserved.        *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************
* $Id: FXRange.cpp,v 1.14 2003/08/26 23:08:50 fox Exp $                         *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "FXVec2f.h"
#include "FXVec3f.h"
#include "FXRange.h"


/*
  Notes:
  - Need to redo range as a pair of FX3Vec's, upper and lower.
  - You can also use FX3Vec::lo(a,b) and FX3Vec::hi() in that case.
*/


using namespace FX;

/**************************  R a n g e   C l a s s   *************************/

namespace FX {

// Test if empty
FXbool FXRange::empty() const {
  return d[0][1]<=d[0][0] || d[1][1]<=d[1][0] || d[2][1]<=d[2][0];
  }

// Test if overlap
FXbool FXRange::overlap(const FXRange& box) const {
  return d[0][1]>box.d[0][0] && d[0][0]<box.d[0][1] &&
         d[1][1]>box.d[1][0] && d[1][0]<box.d[1][1] &&
         d[2][1]>box.d[2][0] && d[2][0]<box.d[2][1];
  }


// Test if box contains point i,j,k
FXbool FXRange::contains(FXfloat x,FXfloat y,FXfloat z) const {
  return d[0][0]<=x && x<=d[0][1] && d[1][0]<=y && y<=d[1][1] && d[2][0]<=z && z<=d[2][1];
  }


// Longest side
FXfloat FXRange::longest() const {
  register FXfloat l,len;
  len=d[0][1]-d[0][0];
  if((l=d[1][1]-d[1][0])>len) len=l;
  if((l=d[2][1]-d[2][0])>len) len=l;
  return len;
  }


// shortest side
FXfloat FXRange::shortest() const {
  register FXfloat l,len;
  len=d[0][1]-d[0][0];
  if((l=d[1][1]-d[1][0])<len) len=l;
  if((l=d[2][1]-d[2][0])<len) len=l;
  return len;
  }


// Include given box into box's range
FXRange& FXRange::include(const FXRange& box){
  if(box.d[0][0]<d[0][0]) d[0][0]=box.d[0][0];
  if(box.d[1][0]<d[1][0]) d[1][0]=box.d[1][0];
  if(box.d[2][0]<d[2][0]) d[2][0]=box.d[2][0];
  if(box.d[0][1]>d[0][1]) d[0][1]=box.d[0][1];
  if(box.d[1][1]>d[1][1]) d[1][1]=box.d[1][1];
  if(box.d[2][1]>d[2][1]) d[2][1]=box.d[2][1];
  return *this;
  }


// Include point into range
FXRange& FXRange::include(FXfloat x,FXfloat y,FXfloat z){
  if(x<d[0][0]) d[0][0]=x; if(x>d[0][1]) d[0][1]=x;
  if(y<d[1][0]) d[1][0]=y; if(y>d[1][1]) d[1][1]=y;
  if(z<d[2][0]) d[2][0]=z; if(z>d[2][1]) d[2][1]=z;
  return *this;
  }


// Include point into range
FXRange& FXRange::include(const FXVec3f& v){
  if(v.x<d[0][0]) d[0][0]=v.x; if(v.x>d[0][1]) d[0][1]=v.x;
  if(v.y<d[1][0]) d[1][0]=v.y; if(v.y>d[1][1]) d[1][1]=v.y;
  if(v.z<d[2][0]) d[2][0]=v.z; if(v.z>d[2][1]) d[2][1]=v.z;
  return *this;
  }


// Clip domain against another
FXRange& FXRange::clipTo(const FXRange& box){
  if(d[0][0]<box.d[0][0]) d[0][0]=box.d[0][0];
  if(d[1][0]<box.d[1][0]) d[1][0]=box.d[1][0];
  if(d[2][0]<box.d[2][0]) d[2][0]=box.d[2][0];
  if(d[0][1]>box.d[0][1]) d[0][1]=box.d[0][1];
  if(d[1][1]>box.d[1][1]) d[1][1]=box.d[1][1];
  if(d[2][1]>box.d[2][1]) d[2][1]=box.d[2][1];
  return *this;
  }


// Get corners of box
void boxCorners(FXVec3f* points,const FXRange& box){
  points[0].x = points[2].x = points[4].x = points[6].x = box[0][0];
  points[1].x = points[3].x = points[5].x = points[7].x = box[0][1];
  points[0].y = points[1].y = points[4].y = points[5].y = box[1][0];
  points[2].y = points[3].y = points[6].y = points[7].y = box[1][1];
  points[0].z = points[1].z = points[2].z = points[3].z = box[2][0];
  points[4].z = points[5].z = points[6].z = points[7].z = box[2][1];
  }


// Ray intersection test
FXbool boxIntersect(const FXRange& box,const FXVec3f& u,const FXVec3f& v){
  FXfloat f,n,d,ni,fi,t;
  FXVec3f dir = v-u;
  int c;
  f = FLT_MAX;
  n = -FLT_MAX;
  for(c=0; c<3; c++){
    d = dir[c];
    if(d==0.0){
      if((box[c][1]<u[c])||(u[c]<box[c][0])) return FALSE;
      }
    else{
      ni = (box[c][0]-u[c])/d;
      fi = (box[c][1]-u[c])/d;
      if(ni>fi) FXSWAP(ni,fi,t);
      if(ni>n) n=ni;
      if(fi<f) f=fi;
      if(n>f) return FALSE;
      }
    }
  return TRUE;
  }


// Get center of box
FXVec3f boxCenter(const FXRange& box){
  return FXVec3f(0.5f*(box[0][0]+box[0][1]),0.5f*(box[1][0]+box[1][1]),0.5f*(box[2][0]+box[2][1]));
  }


// Get diagonal of box
FXfloat boxDiagonal(const FXRange& box){
  float dx=box.width();
  float dy=box.height();
  float dz=box.depth();
  return (FXfloat)sqrt(dx*dx+dy*dy+dz*dz);
  }


// Saving
FXStream& operator<<(FXStream& store,const FXRange& box){
  store << box.d[0][0] << box.d[0][1];
  store << box.d[1][0] << box.d[1][1];
  store << box.d[2][0] << box.d[2][1];
  return store;
  }


// Loading
FXStream& operator>>(FXStream& store,FXRange& box){
  store >> box.d[0][0] >> box.d[0][1];
  store >> box.d[1][0] >> box.d[1][1];
  store >> box.d[2][0] >> box.d[2][1];
  return store;
  }

}
