/*
 * GQview
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "gqview.h"
#include "view_file_list.h"

#include "cache_maint.h"
#include "dnd.h"
#include "img-view.h"
#include "info.h"
#include "layout.h"
#include "layout_image.h"
#include "menu.h"
#include "thumb.h"
#include "utilops.h"
#include "ui_clist_edit.h"
#include "ui_fileops.h"
#include "ui_menu.h"

/*
 *-----------------------------------------------------------------------------
 * signals
 *-----------------------------------------------------------------------------
 */

static void vflist_send_update(ViewFileList *vfl)
{
	if (vfl->func_status) vfl->func_status(vfl, vfl->data_status);
}

/*
 *-----------------------------------------------------------------------------
 * dnd
 *-----------------------------------------------------------------------------
 */

static void vflist_dnd_get(GtkWidget *widget, GdkDragContext *context,
			   GtkSelectionData *selection_data, guint info,
			   guint time, gpointer data)
{
	ViewFileList *vfl = data;
	gchar *uri_text = NULL;
	gint total;

	if (vflist_index_is_selected(vfl, vfl->click_row))
		{
		GList *list;

		list = vflist_selection_get_list(vfl);
		if (!list) return;

		uri_text = make_uri_file_list(list, (info == TARGET_TEXT_PLAIN), &total);
		path_list_free(list);
		}
	else
		{
		const gchar *path = vflist_index_get_path(vfl, vfl->click_row);

		if (!path) return;

		switch (info)
			{
			case TARGET_URI_LIST:
				uri_text = g_strconcat("file:", path, "\r\n", NULL);
				break;
			case TARGET_TEXT_PLAIN:
				uri_text = g_strdup(path);
				break;
			}
		total = strlen(uri_text);
		}

	if (debug) printf(uri_text);

	gtk_selection_data_set(selection_data, selection_data->target, 8, uri_text, total);
	g_free(uri_text);
}

static void vflist_dnd_begin(GtkWidget *widget, GdkDragContext *context, gpointer data)
{
	ViewFileList *vfl = data;

	clist_edit_set_highlight(vfl->clist, vfl->click_row, TRUE);
}

static void vflist_dnd_end(GtkWidget *widget, GdkDragContext *context, gpointer data)
{
	ViewFileList *vfl = data;

	clist_edit_set_highlight(vfl->clist, vfl->click_row, FALSE);

	if (context->action == GDK_ACTION_MOVE)
		{
		vflist_refresh(vfl);
		}
}

static void vflist_dnd_init(ViewFileList *vfl)
{
	gtk_drag_source_set(vfl->clist, GDK_BUTTON2_MASK,
			    dnd_file_drag_types, dnd_file_drag_types_count,
			    GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK);
	gtk_signal_connect(GTK_OBJECT(vfl->clist), "drag_data_get",
			   GTK_SIGNAL_FUNC(vflist_dnd_get), vfl);
	gtk_signal_connect(GTK_OBJECT(vfl->clist), "drag_begin",
			   GTK_SIGNAL_FUNC(vflist_dnd_begin), vfl);
	gtk_signal_connect(GTK_OBJECT(vfl->clist), "drag_end",
			   GTK_SIGNAL_FUNC(vflist_dnd_end), vfl);
}

/*
 *-----------------------------------------------------------------------------
 * pop-up menu
 *-----------------------------------------------------------------------------
 */

static GList *vflist_pop_menu_file_list(ViewFileList *vfl)
{
	if (vflist_index_is_selected(vfl, vfl->click_row))
		{
		return vflist_selection_get_list(vfl);
		}

	return g_list_append(NULL, g_strdup(vflist_index_get_path(vfl, vfl->click_row)));
}

static void vflist_pop_menu_edit_cb(GtkWidget *widget, gpointer data)
{
	ViewFileList *vfl;
	gint n;
	GList *list;

	vfl = submenu_item_get_data(widget);
	n = GPOINTER_TO_INT(data);

	if (!vfl) return;

	list = vflist_pop_menu_file_list(vfl);
	start_editor_from_path_list(n, list);
	path_list_free(list);
}

static void vflist_pop_menu_info_cb(GtkWidget *widget, gpointer data)
{
	ViewFileList *vfl = data;

	info_window_new(NULL, vflist_pop_menu_file_list(vfl));
}

static void vflist_pop_menu_view_cb(GtkWidget *widget, gpointer data)
{
	ViewFileList *vfl = data;
	const gchar *path;

	path = vflist_index_get_path(vfl, vfl->click_row);
	view_window_new(path);
}

static void vflist_pop_menu_copy_cb(GtkWidget *widget, gpointer data)
{
	ViewFileList *vfl = data;

	file_util_copy(NULL, vflist_pop_menu_file_list(vfl), NULL);
}

static void vflist_pop_menu_move_cb(GtkWidget *widget, gpointer data)
{
	ViewFileList *vfl = data;

	clist_edit_set_highlight(vfl->clist, vfl->click_row, FALSE);

	file_util_move(NULL, vflist_pop_menu_file_list(vfl), NULL);
}

static void vflist_pop_menu_rename_cb(GtkWidget *widget, gpointer data)
{
	ViewFileList *vfl = data;

	clist_edit_set_highlight(vfl->clist, vfl->click_row, FALSE);

	file_util_rename(NULL, vflist_pop_menu_file_list(vfl));
}

static void vflist_pop_menu_delete_cb(GtkWidget *widget, gpointer data)
{
	ViewFileList *vfl = data;

	file_util_delete(NULL, vflist_pop_menu_file_list(vfl));
}

static void vflist_pop_menu_sort_cb(GtkWidget *widget, gpointer data)
{
	ViewFileList *vfl;
	SortType type;
	
	vfl = submenu_item_get_data(widget);
	if (!vfl) return;

	type = (SortType)GPOINTER_TO_INT(data);

	clist_edit_set_highlight(vfl->clist, vfl->click_row, FALSE);

	if (vfl->layout)
		{
		layout_sort_set(vfl->layout, type, vfl->sort_ascend);
		}
	else
		{
		vflist_sort_set(vfl, type, vfl->sort_ascend);
		}
}

static void vflist_pop_menu_sort_ascend_cb(GtkWidget *widget, gpointer data)
{
	ViewFileList *vfl = data;

	clist_edit_set_highlight(vfl->clist, vfl->click_row, FALSE);

	if (vfl->layout)
		{
		layout_sort_set(vfl->layout, vfl->sort_method, !vfl->sort_ascend);
		}
	else
		{
		vflist_sort_set(vfl, vfl->sort_method, !vfl->sort_ascend);
		}
}

static void vflist_pop_menu_icons_cb(GtkWidget *widget, gpointer data)
{
	ViewFileList *vfl = data;

	if (vfl->layout) layout_views_set(vfl->layout, vfl->layout->tree_view, TRUE);
}

static void vflist_pop_menu_refresh_cb(GtkWidget *widget, gpointer data)
{
	ViewFileList *vfl = data;

	clist_edit_set_highlight(vfl->clist, vfl->click_row, FALSE);

	vflist_refresh(vfl);
}

static GtkWidget *vflist_pop_menu(ViewFileList *vfl, gint active)
{
	GtkWidget *menu;
	GtkWidget *item;
	GtkWidget *submenu;

	menu = popup_menu_short_lived();

	submenu_add_edit(menu, &item, vflist_pop_menu_edit_cb, vfl);
	gtk_widget_set_sensitive(item, active);

	item = menu_item_add(menu, _("Properties"), GTK_SIGNAL_FUNC(vflist_pop_menu_info_cb), vfl);
	gtk_widget_set_sensitive(item, active);

	item = menu_item_add(menu, _("View in new window"), GTK_SIGNAL_FUNC(vflist_pop_menu_view_cb), vfl);
	gtk_widget_set_sensitive(item, active);

	menu_item_add_divider(menu);
	item = menu_item_add(menu, _("Copy..."), GTK_SIGNAL_FUNC(vflist_pop_menu_copy_cb), vfl);
	gtk_widget_set_sensitive(item, active);
	item = menu_item_add(menu, _("Move..."), GTK_SIGNAL_FUNC(vflist_pop_menu_move_cb), vfl);
	gtk_widget_set_sensitive(item, active);
	item = menu_item_add(menu, _("Rename..."), GTK_SIGNAL_FUNC(vflist_pop_menu_rename_cb), vfl);
	gtk_widget_set_sensitive(item, active);
	item = menu_item_add(menu, _("Delete..."), GTK_SIGNAL_FUNC(vflist_pop_menu_delete_cb), vfl);
	gtk_widget_set_sensitive(item, active);

	menu_item_add_divider(menu);

	submenu = submenu_add_sort(NULL, GTK_SIGNAL_FUNC(vflist_pop_menu_sort_cb), vfl,
				   FALSE, FALSE, TRUE, vfl->sort_method);
	menu_item_add_divider(submenu);
	menu_item_add_check(submenu, _("Ascending"), vfl->sort_ascend,
			    GTK_SIGNAL_FUNC(vflist_pop_menu_sort_ascend_cb), vfl);

	item = menu_item_add(menu, _("Sort"), NULL, NULL);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), submenu);

	menu_item_add_check(menu, _("View as icons"), FALSE,
			    GTK_SIGNAL_FUNC(vflist_pop_menu_icons_cb), vfl);
	menu_item_add(menu, _("Refresh"), GTK_SIGNAL_FUNC(vflist_pop_menu_refresh_cb), vfl);

	return menu;
}

/*
 *-----------------------------------------------------------------------------
 * callbacks
 *-----------------------------------------------------------------------------
 */

static gint vflist_row_rename_cb(ClistEditData *ced, const gchar *old, const gchar *new, gpointer data)
{
	ViewFileList *vfl = data;
	gchar *old_path;
	gchar *new_path;

	old_path = concat_dir_and_file(vfl->path, old);
	new_path = concat_dir_and_file(vfl->path, new);

	if (strlen(new) == 0 || strchr(new, '/') != NULL)
		{
		gchar *text = g_strdup_printf(_("Invalid file name:\n%s"), new);
		file_util_warning_dialog(_("Error renaming file"), text);
		g_free(text);
		}
	else if (isfile(new_path))
		{
		gchar *text = g_strdup_printf(_("File named %s already exists."), new);
		file_util_warning_dialog(_("Error renaming file"), text);
		g_free(text);
		}
	else if (rename (old_path, new_path) < 0)
		{
		gchar *text = g_strdup_printf(_("Unable to rename file:\n%s\nto:\n%s"), old, new);
		file_util_warning_dialog(_("Error renaming file"), text);
		g_free(text);
		}
	else
		{
		file_maint_renamed(old_path, new_path);
		}

	g_free(old_path);
	g_free(new_path);

	return FALSE;
}

static void vflist_popup_destroy_cb(GtkWidget *widget, gpointer data)
{
	ViewFileList *vfl = data;
	clist_edit_set_highlight(vfl->clist, vfl->click_row, FALSE);
	vfl->click_row = -1;
	vfl->popup = NULL;
}

static gint vflist_press_cb(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	ViewFileList *vfl = data;
	gint row = -1;
	gint col = -1;

	gtk_clist_get_selection_info(GTK_CLIST(widget), bevent->x, bevent->y, &row, &col);

	vfl->click_row = row;

	if (bevent->button == 3)
		{
		clist_edit_set_highlight(vfl->clist, row, TRUE);

		vfl->popup = vflist_pop_menu(vfl, (row >= 0));

		gtk_signal_connect(GTK_OBJECT(vfl->popup), "destroy",
				   GTK_SIGNAL_FUNC(vflist_popup_destroy_cb), vfl);

		gtk_menu_popup(GTK_MENU(vfl->popup), NULL, NULL, NULL, NULL,
				bevent->button, bevent->time);
		return FALSE;
		}

	if (row == -1 || col == -1) return FALSE;

	if (bevent->button == 1 && bevent->type == GDK_BUTTON_PRESS)
		{
		if (bevent->state & GDK_SHIFT_MASK || bevent->state & GDK_CONTROL_MASK)
			{
			vfl->rename_row = -1;
			}
		else
			{
			if (bevent->time - vfl->click_time > 333)	/* 1/3 sec */
				{
				if (enable_in_place_rename &&
				    vfl->click_row == row && col == 0 &&
				    vflist_index_is_selected(vfl, row))
					{
					/* set up rename window */
					clist_edit_by_row(GTK_CLIST(vfl->clist), row, 0,
							  vflist_row_rename_cb, vfl);
					return TRUE;
					}
				}
			else
				{
				vfl->rename_row = row;
				}
			}
		vfl->click_time = bevent->time;
		}

	return FALSE;
}

static void vflist_select_image(ViewFileList *vfl, gint row)
{
	const gchar *path;
	const gchar *read_ahead_path = NULL;

	path = vflist_index_get_path(vfl, row);

	if (path && enable_read_ahead)
		{
		FileData *fd;
		if (row > vflist_index_by_path(vfl, layout_image_get_path(vfl->layout)) &&
		    row + 1 < vflist_count(vfl, NULL))
			{
			fd = vflist_index_get_data(vfl, row + 1);
			}
		else if (row > 0)
			{
			fd = vflist_index_get_data(vfl, row - 1);
			}
		else
			{
			fd = NULL;
			}
		if (fd) read_ahead_path = fd->path;
		}

	layout_image_set_with_ahead(vfl->layout, path, read_ahead_path);
}

static gint vflist_select_idle_cb(gpointer data)
{
	ViewFileList *vfl = data;
	GList *list;

	if (!vfl->layout)
		{
		vfl->select_idle_id = -1;
		return FALSE;
		}

	list = g_list_last(GTK_CLIST(vfl->clist)->selection);
	if (list) vflist_select_image(vfl, GPOINTER_TO_INT(list->data));

	vfl->select_idle_id = -1;
	return FALSE;
}

static void vflist_select_idle_cancel(ViewFileList *vfl)
{
	if (vfl->select_idle_id != -1) gtk_idle_remove(vfl->select_idle_id);
	vfl->select_idle_id = -1;
}

static void vflist_select_cb(GtkWidget *widget, gint row, gint col,
			     GdkEvent *event, gpointer data)
{
	ViewFileList *vfl = data;

	vflist_send_update(vfl);

	if (!vfl->layout) return;

	if (vfl->select_idle_id == -1) vfl->select_idle_id = gtk_idle_add(vflist_select_idle_cb, vfl);
}

static void vflist_unselect_cb(GtkWidget *widget, gint row, gint col,
			       GdkEvent *event, gpointer data)
{
	ViewFileList *vfl = data;
	vflist_send_update(vfl);
}

/*
 *-----------------------------------------------------------------------------
 * misc
 *-----------------------------------------------------------------------------
 */

void vflist_sort_set(ViewFileList *vfl, SortType type, gint ascend)
{
	GtkCList *clist;
	GList *work;
	gint row;
	gint count;

	if (vfl->sort_method == type && vfl->sort_ascend == ascend) return;

	vfl->sort_method = type;
	vfl->sort_ascend = ascend;

	if (!vfl->list) return;

	vfl->list = filelist_sort(vfl->list, vfl->sort_method, vfl->sort_ascend);

	clist = GTK_CLIST(vfl->clist);

	gtk_clist_freeze(clist);

	count = 0;
	work = vfl->list;
	while (work)
		{
		row = gtk_clist_find_row_from_data(clist, work->data);
		gtk_clist_row_move(clist, row, count);
		work = work->next;
		count++;
		}

	gtk_clist_thaw(clist);
}

/*
 *-----------------------------------------------------------------------------
 * thumb updates
 *-----------------------------------------------------------------------------
 */

static gint vflist_thumb_next(ViewFileList *vfl);

static void vflist_thumb_status(ViewFileList *vfl, gfloat val, const gchar *text)
{
	if (vfl->func_thumb_status)
		{
		vfl->func_thumb_status(vfl, val, text, vfl->data_thumb_status);
		}
}

static void vflist_thumb_cleanup(ViewFileList *vfl)
{
	vflist_thumb_status(vfl, 0.0, NULL);

	g_list_free(vfl->thumbs_list);
	vfl->thumbs_list = NULL;
	vfl->thumbs_count = 0;
	vfl->thumbs_running = FALSE;

	thumb_loader_free(vfl->thumbs_loader);
	vfl->thumbs_loader = NULL;

	vfl->thumbs_filedata = NULL;
}

static void vflist_thumb_stop(ViewFileList *vfl)
{
        if (vfl->thumbs_running) vflist_thumb_cleanup(vfl);
}

static void vflist_thumb_do(ViewFileList *vfl, ThumbLoader *tl, gint p, FileData *fd)
{
	GtkCList *clist;
	GdkPixmap *pixmap = NULL;
	GdkBitmap *mask = NULL;
	gint spacing = -1;

	if (!fd) return;

	clist = GTK_CLIST(vfl->clist);

	if (p < 0)
		{
		p = gtk_clist_find_row_from_data(clist, fd);
		if (p < 0) return;
		}

	if (tl) spacing = thumb_loader_to_pixmap(tl, &pixmap, &mask);

	if (spacing < 0)
		{
		/* unknown */
		spacing = thumb_max_width - thumb_from_xpm_d(img_unknown, thumb_max_width, thumb_max_height, &pixmap, &mask);
		}

	gtk_clist_set_pixtext(clist, p, 0, fd->name, (guint8)(spacing + 5), pixmap, mask);
	gtk_clist_set_shift(clist, p, 0, 0, 0);

	if (pixmap) gdk_pixmap_unref(pixmap);
	if (mask) gdk_bitmap_unref(mask);

	vflist_thumb_status(vfl, (gfloat)(vfl->thumbs_count) / clist->rows, _("Loading thumbs..."));
}

static void vflist_thumb_error_cb(ThumbLoader *tl, gpointer data)
{
	ViewFileList *vfl = data;

	if (vfl->thumbs_filedata && vfl->thumbs_loader == tl)
		{
		vflist_thumb_do(vfl, NULL, -1, vfl->thumbs_filedata);
		}

	while (vflist_thumb_next(vfl));
}

static void vflist_thumb_done_cb(ThumbLoader *tl, gpointer data)
{
	ViewFileList *vfl = data;

	if (vfl->thumbs_filedata && vfl->thumbs_loader == tl)
		{
		vflist_thumb_do(vfl, tl, -1, vfl->thumbs_filedata);
		}

	while (vflist_thumb_next(vfl));
}

static gint vflist_thumb_next(ViewFileList *vfl)
{
	GtkCList *clist;
	gint p = -1;
	gint r = -1;
	gint c = -1;

	clist = GTK_CLIST(vfl->clist);

	gtk_clist_get_selection_info(clist, 1, 1, &r, &c);

	/* find first visible undone */

	if (r != -1)
		{
		GList *work = g_list_nth(vfl->thumbs_list, r);
		while (work)
			{
			if (gtk_clist_row_is_visible(clist, r))
				{
				if (!GPOINTER_TO_INT(work->data))
					{
					work->data = GINT_TO_POINTER(TRUE);
					p = r;
					work = NULL;
					}
				else
					{
					r++;
					work = work->next;
					}
				}
			else
				{
				work = NULL;
				}
			}
		}

	/* then find first undone */

	if (p == -1)
		{
		GList *work = vfl->thumbs_list;
		r = 0;
		while (work && p == -1)
			{
			if (!GPOINTER_TO_INT(work->data))
				{
				p = r;
				work->data = GINT_TO_POINTER(TRUE);
				}
			if (p == -1)
				{
				r++;
				work = work->next;
				if (!work)
					{
					vflist_thumb_cleanup(vfl);
					return FALSE;
					}
				}
			}
		}

	vfl->thumbs_count++;

	if (gtk_clist_get_cell_type(clist, p, 0) != GTK_CELL_PIXTEXT)
		{
		FileData *fd;

		fd = gtk_clist_get_row_data(clist, p);
		if (fd)
			{
			vfl->thumbs_filedata = fd;

			thumb_loader_free(vfl->thumbs_loader);

			vfl->thumbs_loader = thumb_loader_new(fd->path, thumb_max_width, thumb_max_height);
			thumb_loader_set_error_func(vfl->thumbs_loader, vflist_thumb_error_cb, vfl);

			if (!thumb_loader_start(vfl->thumbs_loader, vflist_thumb_done_cb, vfl))
				{
				/* set icon to unknown, continue */
				if (debug) printf("thumb loader start failed %s\n", vfl->thumbs_loader->path);
				vflist_thumb_do(vfl, NULL, p, fd);
				return TRUE;
				}
			return FALSE;
			}
		else
			{
			/* well, this should not happen, but handle it */
			if (debug) printf("thumb walked past end\n");
			vflist_thumb_cleanup(vfl);
			return FALSE;
			}
		}

	return TRUE;
}

static void vflist_thumb_update(ViewFileList *vfl)
{
	gint i;
	GtkCList *clist;

	vflist_thumb_stop(vfl);
	if (!vfl->thumbs_enabled) return;

	clist = GTK_CLIST(vfl->clist);

	vflist_thumb_status(vfl, 0.0, _("Loading thumbs..."));

	for (i = 0; i < clist->rows; i++)
		{
		vfl->thumbs_list = g_list_prepend(vfl->thumbs_list, GINT_TO_POINTER(FALSE));
		}

	vfl->thumbs_running = TRUE;

	while (vflist_thumb_next(vfl));
}

/*
 *-----------------------------------------------------------------------------
 * row stuff
 *-----------------------------------------------------------------------------
 */

FileData *vflist_index_get_data(ViewFileList *vfl, gint row)
{
	return g_list_nth_data(vfl->list, row);
}

gchar *vflist_index_get_path(ViewFileList *vfl, gint row)
{
	FileData *fd;

	fd = g_list_nth_data(vfl->list, row);

	return (fd ? fd->path : NULL);
}

gint vflist_index_by_path(ViewFileList *vfl, const gchar *path)
{
	gint p = 0;
	GList *work;

	if (!path) return -1;

	work = vfl->list;
	while (work)
		{
		FileData *fd = work->data;
		if (strcmp(path, fd->path) == 0) return p;	
		work = work->next;
		p++;
		}

	return -1;
}

gint vflist_count(ViewFileList *vfl, gint *bytes)
{
	if (bytes)
		{
		gint b = 0;
		GList *work;

		work = vfl->list;
		while (work)
			{
			FileData *fd = work->data;
			work = work->next;
			b += fd->size;
			}

		*bytes = b;
		}

	return g_list_length(vfl->list);
}

GList *vflist_get_list(ViewFileList *vfl)
{
	GList *list = NULL;
	GList *work;

	work = vfl->list;
	while (work)
		{
		FileData *fd = work->data;
		work = work->next;

		list = g_list_prepend(list, g_strdup(fd->path));
		}

	return g_list_reverse(list);
}

/*
 *-----------------------------------------------------------------------------
 * selections
 *-----------------------------------------------------------------------------
 */

gint vflist_index_is_selected(ViewFileList *vfl, gint row)
{
	GList *work;

	work = GTK_CLIST(vfl->clist)->selection;
	while (work)
		{
		if (row == GPOINTER_TO_INT(work->data)) return TRUE;
		work = work->next;
		}

	return FALSE;
}

gint vflist_selection_count(ViewFileList *vfl, gint *bytes)
{
	if (bytes)
		{
		gint b = 0;
		GList *work;

		work = GTK_CLIST(vfl->clist)->selection;
		while (work)
			{
			gint row;
			FileData *fd;

			row = GPOINTER_TO_INT(work->data);
			work = work->next;
			
			fd = vflist_index_get_data(vfl, row);
			b += fd->size;
			}

		*bytes = b;
		}

	return g_list_length(GTK_CLIST(vfl->clist)->selection);
}

GList *vflist_selection_get_list(ViewFileList *vfl)
{
	GList *list = NULL;
	GList *work;

	work = GTK_CLIST(vfl->clist)->selection;
	while (work)
		{
		gint row = GPOINTER_TO_INT(work->data);
		work = work->next;

		list = g_list_prepend(list, g_strdup(vflist_index_get_path(vfl, row)));
		}

	return g_list_reverse(list);
}

GList *vflist_selection_get_list_by_index(ViewFileList *vfl)
{
	return g_list_copy(GTK_CLIST(vfl->clist)->selection);
}

void vflist_select_all(ViewFileList *vfl)
{
	gtk_clist_select_all(GTK_CLIST(vfl->clist));
}

void vflist_select_none(ViewFileList *vfl)
{
	gtk_clist_unselect_all(GTK_CLIST(vfl->clist));
}

void vflist_select_by_path(ViewFileList *vfl, const gchar *path)
{
	GtkCList *clist;
	gint row;

	row = vflist_index_by_path(vfl, path);
	if (row < 0) return;

	clist = GTK_CLIST(vfl->clist);

	if (!vflist_index_is_selected(vfl, row))
		{
		gtk_clist_unselect_all(clist);
		gtk_clist_select_row(clist, row, -1);
		}
	if (gtk_clist_row_is_visible(clist, row) != GTK_VISIBILITY_FULL)
		{
		gtk_clist_moveto(clist, row, -1, 0.5, 0.0);
		}
}

/*
 *-----------------------------------------------------------------------------
 * core (population)
 *-----------------------------------------------------------------------------
 */

static void vflist_populate_view(ViewFileList *vfl)
{
	GtkCList *clist;
	gint thumbs;

	GList *work;
	gint width;
	gint tmp_width;
	gint row;
	const gchar *image_name = NULL;
	gint image_row = -1;
	gint row_p = 0;


	clist = GTK_CLIST(vfl->clist);
	thumbs = vfl->thumbs_enabled;

	vflist_thumb_stop(vfl);

	if (!vfl->list)
		{
		gtk_clist_clear(clist);
		vflist_send_update(vfl);
		return;
		}

	if (vfl->layout)
		{
		gchar *buf;

		buf = remove_level_from_path(layout_image_get_path(vfl->layout));
		if (buf && strcmp(buf, vfl->path) == 0)
			{
			image_name = layout_image_get_name(vfl->layout);
			}
		g_free(buf);
		}
	else

	gtk_clist_freeze(clist);

	if (thumbs)
		{
		gtk_clist_set_row_height(clist, thumb_max_height + 2);
		cache_maintain_home_dir(vfl->path, FALSE, FALSE);
		cache_maintain_dir(vfl->path, FALSE, FALSE);
		}
	else
		{
		gtk_clist_set_row_height(clist,
			GTK_WIDGET(clist)->style->font->ascent +
			GTK_WIDGET(clist)->style->font->descent + 1);
		}

	width = 0;
	work = vfl->list;

	while (work)
		{
		gint match;
		FileData *fd = work->data;
		gint done = FALSE;

		while (!done)
			{
			FileData *old_fd = NULL;

			if (clist->rows > row_p)
				{
				old_fd = gtk_clist_get_row_data(clist, row_p);
				match = strcmp(fd->name, old_fd->name);
				}
			else
				{
				match = -1;
				}

			if (match < 0)
				{
				gchar *buf[4];

				buf[0] = (gchar *)fd->name;
				buf[1] = text_from_size(fd->size);
				buf[2] = (gchar *)text_from_time(fd->date);
				buf[3] = NULL;
				row = gtk_clist_insert(clist, row_p, buf);
				g_free(buf[1]);

				gtk_clist_set_row_data(clist, row, fd);
				if (vfl->thumbs_enabled)
					{
					gtk_clist_set_shift(clist, row, 0, 0, 5 + thumb_max_width);
					}
				done = TRUE;
				if (image_name && strcmp(fd->name, image_name) == 0)
					{
					gtk_clist_select_row(clist, row, 0);
					image_row = row;
					}
				}
			else if (match > 0)
				{
				gtk_clist_remove(clist, row_p);
				}
			else
				{
				gchar *buf;
				gint pixtext;
				gint updated;

				pixtext = (gtk_clist_get_cell_type(clist, row_p, 0) == GTK_CELL_PIXTEXT);
				updated = (fd->date != old_fd->date);

				if (vfl->thumbs_enabled)
					{
					if (!pixtext || updated)
						{
						gtk_clist_set_text(clist, row_p, 0, fd->name);
						gtk_clist_set_shift(clist, row_p, 0, 0, 5 + thumb_max_width);
						}
					}
				else
					{
					gtk_clist_set_text(clist, row_p, 0, fd->name);
					gtk_clist_set_shift(clist, row_p, 0, 0, 0);
					}

				if (updated)
					{
					buf = text_from_size(fd->size);
					gtk_clist_set_text(clist, row_p, 1, buf);
					g_free(buf);

					gtk_clist_set_text(clist, row_p, 2, text_from_time(fd->date));
					}

				gtk_clist_set_row_data(clist, row_p, fd);
				done = TRUE;
				}
			}
		row_p++;

		tmp_width = gdk_string_width(GTK_WIDGET(clist)->style->font, fd->name);
		if (thumbs) tmp_width += thumb_max_width + 5;
		if (tmp_width > width) width = tmp_width;

		work = work->next;
		}

	while (clist->rows > row_p)
		{
		gtk_clist_remove(clist, row_p);
		}

	gtk_clist_set_column_width(clist, 0, width);

	if (image_row != -1 && gtk_clist_row_is_visible(clist, image_row) == GTK_VISIBILITY_NONE)
		{
		gtk_clist_moveto(clist, image_row, 0, 0.50, 0.0);
		}

	gtk_clist_thaw(clist);

	vflist_send_update(vfl);
	vflist_thumb_update(vfl);
}

gint vflist_refresh(ViewFileList *vfl)
{
	GList *old_list;
	gint ret = TRUE;

	old_list = vfl->list;
	vfl->list = NULL;

	if (vfl->path)
		{
		ret = filelist_read(vfl->path, &vfl->list, NULL);
		}

	vfl->list = filelist_sort(vfl->list, vfl->sort_method, vfl->sort_ascend);
	vflist_populate_view(vfl);

	filelist_free(old_list);

	return ret;
}

/*
 *-----------------------------------------------------------------------------
 * base
 *-----------------------------------------------------------------------------
 */

gint vflist_set_path(ViewFileList *vfl, const gchar *path)
{
	if (!path) return FALSE;
	if (vfl->path && strcmp(path, vfl->path) == 0) return TRUE;

	g_free(vfl->path);
	vfl->path = g_strdup(path);

	/* force complete reload */
	gtk_clist_clear(GTK_CLIST(vfl->clist));
	filelist_free(vfl->list);
	vfl->list = NULL;

	return vflist_refresh(vfl);
}

static void vflist_destroy_cb(GtkWidget *widget, gpointer data)
{
	ViewFileList *vfl = data;

	if (vfl->popup)
		{
		gtk_signal_disconnect_by_data(GTK_OBJECT(vfl->popup), vfl);
		}

	vflist_select_idle_cancel(vfl);
	vflist_thumb_stop(vfl);

	g_free(vfl->path);
	filelist_free(vfl->list);
	g_free(vfl);
}

ViewFileList *vflist_new(const gchar *path, gint thumbs)
{
	ViewFileList *vfl;

	vfl = g_new0(ViewFileList, 1);

	vfl->path = NULL;
	vfl->list = NULL;
	vfl->click_row = -1;
	vfl->click_time = 0;
	vfl->rename_row = -1;
	vfl->sort_method = SORT_NAME;
	vfl->sort_ascend = TRUE;
	vfl->thumbs_enabled = thumbs;

	vfl->thumbs_running = FALSE;
	vfl->thumbs_list = NULL;
	vfl->thumbs_count = 0;
	vfl->thumbs_loader = NULL;
	vfl->thumbs_filedata = NULL;

	vfl->select_idle_id = -1;

	vfl->popup = NULL;

	vfl->widget = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(vfl->widget),
				       GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_signal_connect(GTK_OBJECT(vfl->widget), "destroy",
			   GTK_SIGNAL_FUNC(vflist_destroy_cb), vfl);

	vfl->clist = gtk_clist_new(3);

	gtk_clist_set_column_auto_resize(GTK_CLIST(vfl->clist), 1, TRUE);
	gtk_clist_set_column_justification(GTK_CLIST(vfl->clist), 1, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_auto_resize(GTK_CLIST(vfl->clist), 2, TRUE);
        gtk_clist_set_column_justification(GTK_CLIST(vfl->clist), 2, GTK_JUSTIFY_RIGHT);
	gtk_clist_column_titles_passive(GTK_CLIST(vfl->clist)); 

        gtk_clist_set_selection_mode(GTK_CLIST(vfl->clist), GTK_SELECTION_EXTENDED);

        gtk_signal_connect(GTK_OBJECT(vfl->clist), "button_press_event",
			   GTK_SIGNAL_FUNC(vflist_press_cb), vfl);
	gtk_signal_connect(GTK_OBJECT(vfl->clist), "select_row",
			   GTK_SIGNAL_FUNC(vflist_select_cb), vfl);
	gtk_signal_connect_after(GTK_OBJECT(vfl->clist), "unselect_row",
			         GTK_SIGNAL_FUNC(vflist_unselect_cb), vfl);
	
	gtk_container_add (GTK_CONTAINER(vfl->widget), vfl->clist);
	gtk_widget_show(vfl->clist);

	vflist_dnd_init(vfl);

	if (path) vflist_set_path(vfl, path);

	return vfl;
}

void vflist_set_status_func(ViewFileList *vfl,
			    void (*func)(ViewFileList *vfl, gpointer data), gpointer data)
{
	vfl->func_status = func;
	vfl->data_status = data;
}

void vflist_set_thumb_status_func(ViewFileList *vfl,
				  void (*func)(ViewFileList *vfl, gfloat val, const gchar *text, gpointer data),
				  gpointer data)
{
	vfl->func_thumb_status = func;
	vfl->data_thumb_status = data;
}

void vflist_thumb_set(ViewFileList *vfl, gint enable)
{
	if (vfl->thumbs_enabled == enable) return;

	vfl->thumbs_enabled = enable;
	vflist_refresh(vfl);
}

void vflist_set_layout(ViewFileList *vfl, LayoutWindow *layout)
{
	vfl->layout = layout;
}

/*
 *-----------------------------------------------------------------------------
 * maintenance (for rename, move, remove)
 *-----------------------------------------------------------------------------
 */

static gint vflist_maint_find_closest(ViewFileList *vfl, gint row, gint count, GList *ignore_list)
{
	GList *list = NULL;
	GList *work;
	gint rev = row - 1;
	row ++;

	work = ignore_list;
	while (work)
		{
		gint f = vflist_index_by_path(vfl, work->data);
		if (f >= 0) list = g_list_prepend(list, GINT_TO_POINTER(f));
		work = work->next;
		}

	while (list)
		{
		gint c = TRUE;
		work = list;
		while (work && c)
			{
			gpointer p = work->data;
			work = work->next;
			if (row == GPOINTER_TO_INT(p))
				{
				row++;
				c = FALSE;
				}
			if (rev == GPOINTER_TO_INT(p))
				{
				rev--;
				c = FALSE;
				}
			if (!c) list = g_list_remove(list, p);
			}
		if (c && list)
			{
			g_list_free(list);
			list = NULL;
			}
		}
	if (row > count - 1)
		{
		if (rev < 0)
			return -1;
		else
			return rev;
		}
	else
		{
		return row;
		}
}

void vflist_maint_renamed(ViewFileList *vfl, const gchar *source, const gchar *dest)
{
	gint row;
	gchar *source_base;
	gchar *dest_base;
	GList *work;
	FileData *fd;

	row = vflist_index_by_path(vfl, source);
	if (row < 0) return;

	source_base = remove_level_from_path(source);
	dest_base = remove_level_from_path(dest);

	work = g_list_nth(vfl->list, row);
	fd = work->data;

	if (strcmp(source_base, dest_base) == 0)
		{
		GtkCList *clist;
		gint n;

		vfl->list = g_list_remove(vfl->list, fd);
		g_free(fd->path);

		fd->path = g_strdup(dest);
		fd->name = filename_from_path(fd->path);

		vfl->list = filelist_insert_sort(vfl->list, fd, vfl->sort_method, vfl->sort_ascend);
		n = g_list_index(vfl->list, fd);

		clist = GTK_CLIST(vfl->clist);

		if (gtk_clist_get_cell_type(clist, row, 0) != GTK_CELL_PIXTEXT)
			{
			gtk_clist_set_text(clist, row, 0, fd->name);
			}
		else
			{
			guint8 spacing = 0;
			GdkPixmap *pixmap = NULL;
			GdkBitmap *mask = NULL;
			gtk_clist_get_pixtext(clist, row, 0, NULL, &spacing, &pixmap, &mask);
			gtk_clist_set_pixtext(clist, row, 0, fd->name, spacing, pixmap, mask);
			}

		gtk_clist_set_row_data(clist, row, fd);
		gtk_clist_row_move(clist, row, n);
		}
	else
		{
		vflist_maint_removed(vfl, source, NULL);
		}

	g_free(source_base);
	g_free(dest_base);
}

void vflist_maint_removed(ViewFileList *vfl, const gchar *path, GList *ignore_list)
{
	GList *list;
	FileData *fd;
	gint row;
	gint new_row = -1;

	row = vflist_index_by_path(vfl, path);
	if (row < 0) return;

	if (vflist_index_is_selected(vfl, row))
		{
		gint n;

		n = vflist_count(vfl, NULL);
		if (ignore_list)
			{
			new_row = vflist_maint_find_closest(vfl, row, n, ignore_list);
			if (debug) printf("row = %d, closest is %d\n", row, new_row);
			}
		else
			{
			if (row + 1 < n)
				{
				new_row = row + 1;
				}
			else if (row > 0)
				{
				new_row = row - 1;
				}
			}
		vflist_select_none(vfl);
		if (new_row >= 0)
			{
			gtk_clist_select_row(GTK_CLIST(vfl->clist), new_row, -1);
			}
		}

	gtk_clist_remove(GTK_CLIST(vfl->clist), row);
	list = g_list_nth(vfl->list, row);
	fd = list->data;
	vfl->list = g_list_remove(vfl->list, fd);
	file_data_free(fd);

	vflist_send_update(vfl);
}

void vflist_maint_moved(ViewFileList *vfl, const gchar *source, const gchar *dest, GList *ignore_list)
{
	gchar *buf;

	if (!source || !vfl->path) return;

	buf = remove_level_from_path(source);

	if (strcmp(buf, vfl->path) == 0)
		{
		vflist_maint_removed(vfl, source, ignore_list);
		}

	g_free(buf);
}

