/*
 * GQview
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "gqview.h"
#include "view_dir_tree.h"


#include "dnd.h"
#include "dupe.h"
#include "filelist.h"
#include "layout.h"
#include "layout_image.h"
#include "layout_util.h"
#include "utilops.h"
#include "ui_clist_edit.h"
#include "ui_fileops.h"
#include "ui_menu.h"


/*
 * tree core code based on dirtree.c of the bitgrazer file manager
 * bitgrazer never advanced beyond 0.2.0 - I lost interest :/
 */


#define VDTREE_INDENT 14
#define VDTREE_PAD 4


typedef struct _PathData PathData;
struct _PathData
{
	gchar *name;
	GtkCTreeNode *node;
};

typedef struct _NodeData NodeData;
struct _NodeData
{
	FileData *fd;
	gint expanded;
	time_t last_update;
};


static const gchar *vdtree_node_get_path(ViewDirTree *vdt, GtkCTreeNode *node);
static gint vdtree_populate_path_by_node(ViewDirTree *vdt, GtkCTreeNode *node, gint force);
static GtkCTreeNode *vdtree_populate_path(ViewDirTree *vdt, const gchar *path, gint expand);
static void vdtree_make_visible(ViewDirTree *vdt, GtkCTreeNode *node);


/*
 *----------------------------------------------------------------------------
 * utils
 *----------------------------------------------------------------------------
 */

static void set_cursor(GtkWidget *widget, GdkCursorType cursor_type)
{
	GdkCursor *cursor = NULL;

	if (!widget || !widget->window) return;

	if (cursor_type > -1) cursor = gdk_cursor_new (cursor_type);
	gdk_window_set_cursor (widget->window, cursor);
	if (cursor) gdk_cursor_destroy(cursor);
	gdk_flush();
}

static void vdtree_set_node_color(ViewDirTree *vdt, GtkCTreeNode *node, gint active, gint prelight)
{
	if (!node) return;

	if (GTK_CLIST(vdt->ctree)->selection &&
	    node == GTK_CTREE_NODE(GTK_CLIST(vdt->ctree)->selection->data))
		{
		active = FALSE;
		prelight = FALSE;
		}

	if (active)
		{
		gtk_ctree_node_set_background(GTK_CTREE(vdt->ctree), node,
			&GTK_WIDGET (vdt->ctree)->style->bg[GTK_STATE_ACTIVE]);
		gtk_ctree_node_set_foreground(GTK_CTREE(vdt->ctree), node,
			&GTK_WIDGET (vdt->ctree)->style->fg[GTK_STATE_ACTIVE]);
		}
	else if (prelight)
		{
		gtk_ctree_node_set_background(GTK_CTREE(vdt->ctree), node,
			&GTK_WIDGET (vdt->ctree)->style->bg[GTK_STATE_PRELIGHT]);
		gtk_ctree_node_set_foreground(GTK_CTREE(vdt->ctree), node,
			&GTK_WIDGET (vdt->ctree)->style->fg[GTK_STATE_PRELIGHT]);
		}
	else
		{
		gtk_ctree_node_set_background(GTK_CTREE(vdt->ctree), node, NULL);
		gtk_ctree_node_set_foreground(GTK_CTREE(vdt->ctree), node, NULL);
		}
}

static gint vdtree_rename_node_cb(ClistEditData *ced, const gchar *old, const gchar *new, gpointer data)
{
	ViewDirTree *vdt = data;
	GtkCTreeNode *node;
	NodeData *nd;
	gchar *new_path;
	gchar *base;

	node = gtk_ctree_node_nth(GTK_CTREE(vdt->ctree), (guint)ced->row);

	nd = gtk_ctree_node_get_row_data(GTK_CTREE(vdt->ctree), node);
	if (!nd) return FALSE;

	base = remove_level_from_path(nd->fd->path);
	new_path = concat_dir_and_file(base, new);
	g_free(base);

	if (rename(nd->fd->path, new_path) < 0)
		{
		gchar *buf;

		buf = g_strdup_printf(_("Failed to rename %s to %s."), old, new);
		warning_dialog("Rename failed", buf);
		g_free(buf);
		}
	else
		{
		gchar *old_path;

		old_path = g_strdup(nd->fd->path);
		vdtree_populate_path(vdt, new_path, TRUE);

		if (vdt->layout && strcmp(vdt->path, old_path) == 0)
			{
			layout_set_path(vdt->layout, new_path);
			}
		g_free(old_path);
		}

	g_free(new_path);
	return FALSE;
}

static void vdtree_rename_by_node(ViewDirTree *vdt, GtkCTreeNode *node)
{
	NodeData *nd;
	gint row;

	if (!node) return;

	nd = gtk_ctree_node_get_row_data(GTK_CTREE(vdt->ctree), node);
	if (!nd) return;

	vdtree_make_visible(vdt, node);

	row = g_list_position(GTK_CLIST(vdt->ctree)->row_list, (GList *)node);
	if (row < 0) return;

	clist_edit_by_row(GTK_CLIST(vdt->ctree), row, 0,
			  vdtree_rename_node_cb, vdt);
}

/*
 *-----------------------------------------------------------------------------
 * drop menu (from dnd)
 *-----------------------------------------------------------------------------
 */

static void vdtree_drop_menu_copy_cb(GtkWidget *widget, gpointer data)
{
	ViewDirTree *vdt = data;
	const gchar *path;
	GList *list;

	path = vdtree_node_get_path(vdt, vdt->drop_node);
	list = vdt->drop_list;

	vdt->drop_list = NULL;

	file_util_copy_simple(list, path);
}

static void vdtree_drop_menu_move_cb(GtkWidget *widget, gpointer data)
{
	ViewDirTree *vdt = data;
	const gchar *path;
	GList *list;

	path = vdtree_node_get_path(vdt, vdt->drop_node);
	list = vdt->drop_list;

	vdt->drop_list = NULL;

	file_util_move_simple(list, path);
}

static GtkWidget *vdtree_drop_menu(ViewDirTree *vdt, gint active)
{
	GtkWidget *menu;

	menu = popup_menu_short_lived();

	menu_item_add_sensitive(menu, _("Copy"), active, GTK_SIGNAL_FUNC(vdtree_drop_menu_copy_cb), vdt);
	menu_item_add_sensitive(menu, _("Move"), active, GTK_SIGNAL_FUNC(vdtree_drop_menu_move_cb), vdt);

	menu_item_add_divider(menu);
	menu_item_add(menu, _("Cancel"), NULL, vdt);

	return menu;
}

/*
 *-----------------------------------------------------------------------------
 * pop-up menu
 *-----------------------------------------------------------------------------
 */

static void vdtree_pop_menu_slide_cb(GtkWidget *widget, gpointer data)
{
	ViewDirTree *vdt = data;
	gchar *path;

	if (!vdt->layout) return;

	path = g_strdup(vdtree_node_get_path(vdt, vdt->click_node));
	if (!path) return;

	layout_set_path(vdt->layout, path);
	layout_select_none(vdt->layout);
	layout_image_slideshow_stop(vdt->layout);
	layout_image_slideshow_start(vdt->layout);

	g_free(path);
}

static void vdtree_pop_menu_slide_rec_cb(GtkWidget *widget, gpointer data)
{
	ViewDirTree *vdt = data;
	gchar *path;
	GList *list;

	if (!vdt->layout) return;

	path = g_strdup(vdtree_node_get_path(vdt, vdt->click_node));
	if (!path) return;

	list = path_list_recursive(path);

	layout_image_slideshow_stop(vdt->layout);
	layout_image_slideshow_start_from_list(vdt->layout, list);

	g_free(path);
}

static void vdtree_pop_menu_dupe(ViewDirTree *vdt, gint recursive)
{
	DupeWindow *dw;
	const gchar *path;
	GList *list = NULL;

	path = vdtree_node_get_path(vdt, vdt->click_node);
	if (!path) return;

	if (recursive)
		{
		list = g_list_append(list, g_strdup(path));
		}
	else
		{
		path_list(path, &list, NULL);
		list = path_list_filter(list, FALSE);
		}

	dw = dupe_window_new(DUPE_MATCH_NAME);
	dupe_window_add_files(dw, list, recursive);

	path_list_free(list);
}

static void vdtree_pop_menu_dupe_cb(GtkWidget *widget, gpointer data)
{
	ViewDirTree *vdt = data;
	vdtree_pop_menu_dupe(vdt, FALSE);
}

static void vdtree_pop_menu_dupe_rec_cb(GtkWidget *widget, gpointer data)
{
	ViewDirTree *vdt = data;
	vdtree_pop_menu_dupe(vdt, TRUE);
}

static void vdtree_pop_menu_new_cb(GtkWidget *widget, gpointer data)
{
	ViewDirTree *vdt = data;
	const gchar *path;
	gchar *new_path;
	gchar *buf;

	path = vdtree_node_get_path(vdt, vdt->click_node);
	if (!path) return;

	buf = concat_dir_and_file(path, _("new_folder"));
	new_path = unique_filename(buf, NULL, NULL, FALSE);
	g_free(buf);
	if (!new_path) return;

	if (mkdir(new_path, 0755) < 0)
		{
		gchar *text;

		text = g_strdup_printf(_("Unable to create directory:\n%s"), new_path);
		file_util_warning_dialog(_("Error creating directory"), text);
		g_free(text);
		}
	else
		{
		GtkCTreeNode *new_node;	

		new_node = vdtree_populate_path(vdt, new_path, TRUE);

		vdtree_rename_by_node(vdt, new_node);
		}

	g_free(new_path);
}

static void vdtree_pop_menu_rename_cb(GtkWidget *widget, gpointer data)
{
	ViewDirTree *vdt = data;

	vdtree_rename_by_node(vdt, vdt->click_node);
}

static void vdtree_pop_menu_tree_cb(GtkWidget *widget, gpointer data)
{
	ViewDirTree *vdt = data;

	if (vdt->layout) layout_views_set(vdt->layout, FALSE, vdt->layout->icon_view);
}

static void vdtree_pop_menu_refresh_cb(GtkWidget *widget, gpointer data)
{
	ViewDirTree *vdt = data;

	if (vdt->layout) layout_refresh(vdt->layout);
}

static GtkWidget *vdtree_pop_menu(ViewDirTree *vdt, GtkCTreeNode *node)
{
	GtkWidget *menu;
	gint active;

	active = (node != NULL);

	menu = popup_menu_short_lived();

	menu_item_add_sensitive(menu, _("Slideshow"), active,
				GTK_SIGNAL_FUNC(vdtree_pop_menu_slide_cb), vdt);
	menu_item_add_sensitive(menu, _("Slideshow recursive"), active,
				GTK_SIGNAL_FUNC(vdtree_pop_menu_slide_rec_cb), vdt);

	menu_item_add_divider(menu);
	menu_item_add_sensitive(menu, _("Find duplicates..."), active,
				GTK_SIGNAL_FUNC(vdtree_pop_menu_dupe_cb), vdt);
	menu_item_add_sensitive(menu, _("Find duplicates recursive..."), active,
				GTK_SIGNAL_FUNC(vdtree_pop_menu_dupe_rec_cb), vdt);

	menu_item_add_divider(menu);

	active = (node &&
		  access(vdtree_node_get_path(vdt, node), W_OK | X_OK) == 0);
	menu_item_add_sensitive(menu, _("New folder..."), active,
				GTK_SIGNAL_FUNC(vdtree_pop_menu_new_cb), vdt);

	active = (active && GTK_CTREE_ROW(node)->parent &&
		  access(vdtree_node_get_path(vdt, GTK_CTREE_ROW(node)->parent), W_OK | X_OK) == 0);
	menu_item_add_sensitive(menu, _("Rename..."), active,
				GTK_SIGNAL_FUNC(vdtree_pop_menu_rename_cb), vdt);

	menu_item_add_divider(menu);
	menu_item_add_check(menu, _("View as tree"), TRUE,
			    GTK_SIGNAL_FUNC(vdtree_pop_menu_tree_cb), vdt);
	menu_item_add(menu, _("Refresh"), GTK_SIGNAL_FUNC(vdtree_pop_menu_refresh_cb), vdt);

	return menu;
}

static void vdtree_popup_destroy_cb(GtkWidget *widget, gpointer data)
{
	ViewDirTree *vdt = data;

	vdtree_set_node_color(vdt, vdt->click_node, FALSE, FALSE);
	vdt->click_node = NULL;
	vdt->popup = NULL;

	path_list_free(vdt->drop_list);
	vdt->drop_list = NULL;
	vdt->drop_node = NULL;
}

/*
 *----------------------------------------------------------------------------
 * dnd
 *----------------------------------------------------------------------------
 */

static GtkTargetEntry vdtree_dnd_drop_types[] = {
	{ "text/uri-list", 0, TARGET_URI_LIST }
};
static gint vdtree_dnd_drop_types_count = 1;


static void vdtree_dnd_get(GtkWidget *widget, GdkDragContext *context,
			   GtkSelectionData *selection_data, guint info,
			   guint time, gpointer data)
{
	ViewDirTree *vdt = data;
	gchar *uri_text = NULL;
	NodeData *nd;
	gchar *path = NULL;

	nd = gtk_ctree_node_get_row_data(GTK_CTREE(vdt->ctree), vdt->click_node);
	if (!nd) return;

	path = nd->fd->path;

	switch (info)
		{
		case TARGET_URI_LIST:
			uri_text = g_strdup_printf("file:%s\r\n", path);
			break;
		case TARGET_TEXT_PLAIN:
			uri_text = g_strdup_printf("%s\r\n", path);
			break;
		default:
			break;
		}

	if (!uri_text) return;

	gtk_selection_data_set(selection_data, selection_data->target,
			       8, uri_text, strlen(uri_text));
	g_free(uri_text);
}

static void vdtree_dnd_begin(GtkWidget *widget, GdkDragContext *context, gpointer data)
{
	ViewDirTree *vdt = data;

	vdtree_set_node_color(vdt, vdt->click_node, TRUE, FALSE);
}

static void vdtree_dnd_end(GtkWidget *widget, GdkDragContext *context, gpointer data)
{
	ViewDirTree *vdt = data;

	vdtree_set_node_color(vdt, vdt->click_node, FALSE, FALSE);
	vdt->click_node = NULL;
}

static void vdtree_dnd_drop_receive(GtkWidget *widget,
				    GdkDragContext *context, gint x, gint y,
				    GtkSelectionData *selection_data, guint info,
				    guint time, gpointer data)
{
	ViewDirTree *vdt = data;
	gint row = -1;

	if (gtk_clist_get_selection_info(GTK_CLIST(vdt->ctree), x, y, &row, NULL))
		{
		vdt->drop_node = gtk_ctree_node_nth(GTK_CTREE(vdt->ctree), (guint)row);
		}

        if (info == TARGET_URI_LIST)
                {
		GList *list;
		gint active;

		list = get_uri_file_list(selection_data->data);
		if (!list) return;

		active = (access(vdtree_node_get_path(vdt, vdt->drop_node), W_OK | X_OK) == 0);

		vdt->popup = vdtree_drop_menu(vdt, active);
		gtk_menu_popup(GTK_MENU(vdt->popup), NULL, NULL, NULL, NULL, 0, time);
		gtk_signal_connect(GTK_OBJECT(vdt->popup), "destroy",
				   GTK_SIGNAL_FUNC(vdtree_popup_destroy_cb), vdt);

		vdt->drop_list = list;
		}
}

static gint vdtree_dnd_drop_expand_cb(gpointer data)
{
	ViewDirTree *vdt = data;

	if (vdt->drop_node)
		{
		NodeData *nd;
		nd = gtk_ctree_node_get_row_data(GTK_CTREE(vdt->ctree), vdt->drop_node);
		if (nd && !nd->expanded)
			{
			if (vdtree_populate_path_by_node(vdt, vdt->drop_node, FALSE))
				{
				gtk_ctree_expand(GTK_CTREE(vdt->ctree), vdt->drop_node);
				}
			else
				{
				vdt->drop_node = NULL;
				}
			}
		else if (nd && nd->expanded && !GTK_CTREE_ROW(vdt->drop_node)->expanded)
			{
			gtk_ctree_expand(GTK_CTREE(vdt->ctree), vdt->drop_node);
			}
		}

	vdt->drop_expand_id = -1;

	return FALSE;
}

static void vdtree_dnd_drop_expand_cancel(ViewDirTree *vdt)
{
	if (vdt->drop_expand_id != -1) gtk_timeout_remove(vdt->drop_expand_id);
	vdt->drop_expand_id = -1;
}

static void vdtree_dnd_drop_expand(ViewDirTree *vdt, GtkCTreeNode *node)
{
	if (vdt->drop_node == node) return;

	if (vdt->drop_node != vdt->click_node) vdtree_set_node_color(vdt, vdt->drop_node, FALSE, FALSE);
	if (node != vdt->click_node) vdtree_set_node_color(vdt, node, FALSE, TRUE);

	vdtree_dnd_drop_expand_cancel(vdt);
	vdt->drop_node = node;
	vdt->drop_expand_id = gtk_timeout_add(1000, vdtree_dnd_drop_expand_cb, vdt);
}

static void vdtree_dnd_drop_scroll_cancel(ViewDirTree *vdt)
{
        if (vdt->drop_scroll_id != -1) gtk_timeout_remove(vdt->drop_scroll_id);
        vdt->drop_scroll_id = -1;
}

static gint vdtree_dnd_drop_scroll_cb(gpointer data)
{
	ViewDirTree *vdt = data;

	if (vdt->drop_row == -1 || vdt->drop_list)
		{
		vdt->drop_scroll_id = -1;
		return FALSE;
		}

	if (vdt->drop_row < 1 || vdt->drop_row >= GTK_CLIST(vdt->ctree)->rows - 1) return TRUE;

	if (gtk_clist_row_is_visible(GTK_CLIST(vdt->ctree), vdt->drop_row - 1) != GTK_VISIBILITY_FULL)
		{
		gtk_clist_moveto(GTK_CLIST(vdt->ctree), vdt->drop_row - 1, 0, 0.0, 0.0);
		if (vdt->drop_row > 0)
			{
			vdt->drop_row--;
			if (vdt->drop_node) vdtree_dnd_drop_expand(vdt, GTK_CTREE_NODE_PREV(vdt->drop_node));
			}
		}
	else if (gtk_clist_row_is_visible(GTK_CLIST(vdt->ctree), vdt->drop_row + 1) != GTK_VISIBILITY_FULL)
		{
		gtk_clist_moveto(GTK_CLIST(vdt->ctree), vdt->drop_row + 1, 0, 1.0, 0.0);
		if (vdt->drop_row < GTK_CLIST(vdt->ctree)->rows - 1)
			{
			vdt->drop_row++;
			if (vdt->drop_node) vdtree_dnd_drop_expand(vdt, GTK_CTREE_NODE_NEXT(vdt->drop_node));
			}
		}

	return TRUE;
}

static gint vdtree_dnd_drop_motion(GtkWidget *widget, GdkDragContext *context,
				   gint x, gint y, guint time, gpointer data)
{
        ViewDirTree *vdt = data;
	NodeData *nd;
	gint row = -1;

	if (gtk_clist_get_selection_info(GTK_CLIST(vdt->ctree), x, y, &row, NULL))
		{
		GtkCTreeNode *node;
		node = gtk_ctree_node_nth(GTK_CTREE(vdt->ctree), (guint)row);
		nd = gtk_ctree_node_get_row_data(GTK_CTREE(vdt->ctree), node);
		if (vdt->drop_node != node)
			{
			vdtree_dnd_drop_expand(vdt, node);
			}
		}
	else
		{
		vdtree_dnd_drop_expand(vdt, NULL);
		}

	if (gtk_drag_get_source_widget(context) == vdt->ctree)
		{
		/* from same window */
#if 0
		if (TRUE)
			{
			gdk_drag_status(context, context->suggested_action, time);
			}
		else
			{
			gdk_drag_status (context, 0, time);
			}
#endif
			gdk_drag_status(context, 0, time);
		}
	else
		{
		gdk_drag_status(context, context->suggested_action, time);
		}

	/* delayed drop scrolling */
	if (vdt->drop_row == -1 && row != -1 && vdt->drop_scroll_id == -1)
		{
		vdt->drop_scroll_id = gtk_timeout_add(100, vdtree_dnd_drop_scroll_cb, vdt);
		}
	vdt->drop_row = row;

	return FALSE;
}

static void vdtree_dnd_drop_leave(GtkWidget *widget, GdkDragContext *context, guint time, gpointer data)
{
	ViewDirTree *vdt = data;

	if (vdt->drop_node != vdt->click_node) vdtree_set_node_color(vdt, vdt->drop_node, FALSE, FALSE);

	vdt->drop_node = NULL;
	vdt->drop_row = -1;

	vdtree_dnd_drop_expand_cancel(vdt);
}

static void vdtree_dnd_init(ViewDirTree *vdt)
{
	gtk_drag_source_set(vdt->ctree, GDK_BUTTON2_MASK,
			    dnd_file_drag_types, dnd_file_drag_types_count,
			    GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_ASK);
	gtk_signal_connect(GTK_OBJECT(vdt->ctree), "drag_data_get",
			   GTK_SIGNAL_FUNC(vdtree_dnd_get), vdt);
	gtk_signal_connect(GTK_OBJECT(vdt->ctree), "drag_begin",
			   GTK_SIGNAL_FUNC(vdtree_dnd_begin), vdt);
	gtk_signal_connect(GTK_OBJECT(vdt->ctree), "drag_end",
			   GTK_SIGNAL_FUNC(vdtree_dnd_end), vdt);
	gtk_drag_dest_set(vdt->ctree,
			GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP,
			vdtree_dnd_drop_types, vdtree_dnd_drop_types_count,
			GDK_ACTION_MOVE | GDK_ACTION_COPY);
	gtk_signal_connect(GTK_OBJECT(vdt->ctree), "drag_data_received",
			GTK_SIGNAL_FUNC(vdtree_dnd_drop_receive), vdt);
	gtk_signal_connect(GTK_OBJECT(vdt->ctree), "drag_motion",
			GTK_SIGNAL_FUNC(vdtree_dnd_drop_motion), vdt);
	gtk_signal_connect(GTK_OBJECT(vdt->ctree), "drag_leave",
			GTK_SIGNAL_FUNC(vdtree_dnd_drop_leave), vdt);
}

/*
 *----------------------------------------------------------------------------
 * parts lists
 *----------------------------------------------------------------------------
 */

static GList *parts_list(const gchar *path)
{
	GList *list = NULL;
	const gchar *strb, *strp;
	gint l;

	strp = path;

	if (*strp != '/') return NULL;

	strp++;
	strb = strp;
	l = 0;

	while (*strp != '\0')
		{
		if (*strp == '/')
			{
			if (l > 0) list = g_list_prepend(list, g_strndup(strb, l));
			strp++;
			strb = strp;
			l = 0;
			}
		else
			{
			strp++;
			l++;
			}
		}
	if (l > 0) list = g_list_prepend(list, g_strndup(strb, l));

	list = g_list_reverse(list);

	list = g_list_prepend(list, g_strdup("/"));

	return list;
}

static void parts_list_free(GList *list)
{
	GList *work = list;
	while (work)
		{
		PathData *pd = work->data;
		g_free(pd->name);
		g_free(pd);
		work = work->next;
		}

	g_list_free(list);
}

static GList *parts_list_add_node_points(ViewDirTree *vdt, GList *list)
{
	GList *work;
	GtkCTreeNode *sibling;

	sibling = (GtkCTreeNode *)(GTK_CLIST(vdt->ctree)->row_list);

	work = list;
	while (work)
		{
		PathData *pd;
		gchar *name;

		pd = g_new0(PathData, 1);
		name = work->data;

		while (sibling && name != NULL)
			{
			NodeData *nd = gtk_ctree_node_get_row_data(GTK_CTREE(vdt->ctree), sibling);
			if (nd && strcmp(nd->fd->name, name) == 0)
				{
				name = NULL;
				}
			else
				{
				sibling = GTK_CTREE_ROW(sibling)->sibling;
				}
			}
		pd->name = work->data;
		pd->node = sibling;
		work->data = pd;

		if (sibling) sibling = GTK_CTREE_ROW(sibling)->children;

		work = work->next;
		}

	return list;
}

/*
 *----------------------------------------------------------------------------
 * misc
 *----------------------------------------------------------------------------
 */

static void vdtree_make_visible(ViewDirTree *vdt, GtkCTreeNode *node)
{
	GtkCTreeNode *parent;

	parent = GTK_CTREE_ROW(node)->parent;

	gtk_clist_freeze(GTK_CLIST(vdt->ctree));
	while (parent)
		{
		if (!GTK_CTREE_ROW(parent)->expanded)
			{
			gtk_ctree_expand(GTK_CTREE(vdt->ctree), parent);
			}
		parent = GTK_CTREE_ROW(parent)->parent;
		}

	/* the realized test is a hack, otherwise the scrollbar is incorrect at start up */
	if (GTK_WIDGET_REALIZED(vdt->ctree) && gtk_ctree_node_is_visible(GTK_CTREE(vdt->ctree), node) != GTK_VISIBILITY_FULL)
		{
		gtk_ctree_node_moveto(GTK_CTREE(vdt->ctree), node, 0, 0.3, 0.0);
		}
	gtk_clist_thaw(GTK_CLIST(vdt->ctree));
}

static const gchar *vdtree_node_get_path(ViewDirTree *vdt, GtkCTreeNode *node)
{
	NodeData *nd;

	nd = gtk_ctree_node_get_row_data(GTK_CTREE(vdt->ctree), node);

	if (!nd) return NULL;
	return nd->fd->path;
}

static void vdtree_remove_node_cb(gpointer data)
{
	NodeData *nd = data;

	file_data_free(nd->fd);
	g_free(nd);
}

static void vdtree_setup_root(ViewDirTree *vdt)
{
	GtkCTreeNode *node;
	NodeData *nd;
	const gchar *path = "/";
	gchar *buf[2];

	nd = g_new(NodeData, 1);

	nd->fd = g_new0(FileData, 1);
	nd->fd->path = g_strdup(path);
	nd->fd->name = nd->fd->path;
	nd->fd->size = 0;
	nd->fd->date = filetime(path);

	nd->expanded = FALSE;
	nd->last_update = time(NULL);

	buf[0] = (gchar *)nd->fd->name;
	buf[1] = NULL;

	node = gtk_ctree_insert_node(GTK_CTREE(vdt->ctree), NULL, NULL, buf, VDTREE_PAD,
				     vdt->pf->close_pixmap, vdt->pf->close_mask,
				     vdt->pf->open_pixmap, vdt->pf->open_mask,
				     FALSE, TRUE);

	gtk_ctree_node_set_row_data_full(GTK_CTREE(vdt->ctree),
					 node, nd, vdtree_remove_node_cb);
}

/*
 *----------------------------------------------------------------------------
 * node traversal, management
 *----------------------------------------------------------------------------
 */

static GtkCTreeNode *vdtree_find_node_by_name(ViewDirTree *vdt, GtkCTreeNode *node, const gchar *name)
{
	GtkCTreeNode *sibling;

	sibling = GTK_CTREE_ROW(node)->children;

	while (sibling)
		{
		NodeData *nd;

		nd = gtk_ctree_node_get_row_data(GTK_CTREE(vdt->ctree), sibling);

		if (nd && strcmp(nd->fd->name, name) == 0) return sibling;
		sibling = GTK_CTREE_ROW(sibling)->sibling;
		}

	return NULL;
}

static void vdtree_add_by_data(ViewDirTree *vdt, FileData *fd, GtkCTreeNode *parent)
{
	GtkCTreeNode *node;
	gchar *buf[2];
	GList *list;
	NodeData *nd;
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	if (!fd) return;

	list = parts_list(fd->path);
	if (!list) return;

	buf[0] = (gchar *)fd->name;
	buf[1] = NULL;

	if (access(fd->path, X_OK) == 0)
		{
		pixmap = vdt->pf->close_pixmap;
		mask = vdt->pf->close_mask;
		}
	else
		{
		pixmap = vdt->pf->deny_pixmap;
		mask = vdt->pf->deny_mask;
		}

	node = gtk_ctree_insert_node(GTK_CTREE(vdt->ctree), parent, NULL, buf, VDTREE_PAD,
				     pixmap, mask,
				     vdt->pf->open_pixmap, vdt->pf->open_mask,
				     FALSE, FALSE);

	nd = g_new0(NodeData, 1);
	nd->fd = fd;
	nd->expanded = FALSE;
	nd->last_update = time(NULL);

	gtk_ctree_node_set_row_data_full(GTK_CTREE(vdt->ctree),
					 node, nd, vdtree_remove_node_cb);

	if (GTK_CTREE_ROW(node)->parent)
		{
		GtkCTreeNode *parent;
		NodeData *nd;

		parent = GTK_CTREE_ROW(node)->parent;
		nd = gtk_ctree_node_get_row_data(GTK_CTREE(vdt->ctree), parent);

		if (tree_descend_subdirs && GTK_CTREE_ROW(parent)->expanded && !nd->expanded)
			{
			vdtree_populate_path_by_node(vdt, node, FALSE);
			}
		}
}

static gint vdtree_populate_path_by_node(ViewDirTree *vdt, GtkCTreeNode *node, gint force)
{
	GList *list;
	GList *work;
	GList *old;
	GtkCTreeNode *sibling;
	time_t current_time;
	NodeData *nd;

	nd = gtk_ctree_node_get_row_data(GTK_CTREE(vdt->ctree), node);

	if (!nd) return FALSE;

	current_time = time(NULL);
	
	if (nd->expanded)
		{
		if (!force && current_time - nd->last_update < 10) return TRUE;
		if (!isdir(nd->fd->path))
			{
			gtk_ctree_remove_node(GTK_CTREE(vdt->ctree), node);
			if (vdt->click_node == node) vdt->click_node = NULL;
			if (vdt->drop_node == node) vdt->drop_node = NULL;
			return FALSE;
			}
		if (!force && filetime(nd->fd->path) == nd->fd->date) return TRUE;
		}

	if (!filelist_read(nd->fd->path, NULL, &list)) return FALSE;

	old = NULL;
	sibling = GTK_CTREE_ROW(node)->children;
	while (sibling)
		{
		old = g_list_prepend(old, sibling);
		sibling = GTK_CTREE_ROW(sibling)->sibling;
		}

	gtk_clist_freeze(GTK_CLIST(vdt->ctree));

	work = list;
	while (work)
		{
		FileData *fd;

		fd = work->data;
		work = work->next;

		if (strcmp(fd->name, ".") == 0 || strcmp(fd->name, "..") == 0)
			{
			file_data_free(fd);
			}
		else
			{
			GtkCTreeNode *n = vdtree_find_node_by_name(vdt, node, fd->name);
			if (n)
				{
				NodeData *nd;

				nd = gtk_ctree_node_get_row_data(GTK_CTREE(vdt->ctree), n);
				old = g_list_remove(old, n);

				if (nd->expanded && nd->fd->date != fd->date)
					{
					vdtree_populate_path_by_node(vdt, n, FALSE);
					}
				file_data_free(nd->fd);
				nd->fd = fd;
				}
			else
				{
				vdtree_add_by_data(vdt, fd, node);
				}
			}
		}

	work = old;
	while (work)
		{
		gtk_ctree_remove_node(GTK_CTREE(vdt->ctree), GTK_CTREE_NODE(work->data));
		if (vdt->click_node == node) vdt->click_node = NULL;
		if (vdt->drop_node == node) vdt->drop_node = NULL;
		work = work->next;
		}

	gtk_clist_thaw(GTK_CLIST(vdt->ctree));

	g_list_free(old);
	g_list_free(list);

	nd->expanded = TRUE;
	nd->last_update = current_time;

	return TRUE;
}

static GtkCTreeNode *vdtree_populate_path(ViewDirTree *vdt, const gchar *path, gint expand)
{
	GList *list;
	GList *work;
	GtkCTreeNode *node = NULL;

	if (!path) return NULL;

	list = parts_list(path);
	list = parts_list_add_node_points(vdt, list);

	gtk_clist_freeze(GTK_CLIST(vdt->ctree));

	work = list;
	while (work)
		{
		PathData *pd = work->data;
		if (pd->node == NULL)
			{
			if (work == list)
				{
				parts_list_free(list);
				gtk_clist_thaw(GTK_CLIST(vdt->ctree));
				return NULL;
				}
			else
				{
				PathData *ppd = (work->prev)->data;
				if (!vdtree_populate_path_by_node(vdt, ppd->node, TRUE))
					{
					printf("aborted at %s\n", ppd->name);
					parts_list_free(list);
					gtk_clist_thaw(GTK_CLIST(vdt->ctree));
					return NULL;
					}
				pd->node = vdtree_find_node_by_name(vdt, ppd->node, pd->name);
				if (expand) gtk_ctree_expand(GTK_CTREE(vdt->ctree), pd->node);
				}
			}
		if (!work->next && !vdtree_populate_path_by_node(vdt, pd->node, FALSE))
			{
			printf("warning: view_dir_tree.c: aborted at %s\n", pd->name);
			parts_list_free(list);
			gtk_clist_thaw(GTK_CLIST(vdt->ctree));
			return NULL;
			}
		work = work->next;
		}
	gtk_clist_thaw(GTK_CLIST(vdt->ctree));

	work = g_list_last(list);
	if (work)
		{
		PathData *pd = work->data;
		node = pd->node;
		}

	parts_list_free(list);
	return node;
}

/*
 *----------------------------------------------------------------------------
 * access
 *----------------------------------------------------------------------------
 */

gint vdtree_set_path(ViewDirTree *vdt, const gchar *path)
{
	GtkCTreeNode *node;

	if (!path) return FALSE;
	if (vdt->path && strcmp(path, vdt->path) == 0) return TRUE;

	g_free(vdt->path);
	vdt->path = g_strdup(path);

	node = vdtree_populate_path(vdt, vdt->path, TRUE);

	if (!node) return FALSE;

	gtk_ctree_select(GTK_CTREE(vdt->ctree), node);
	vdtree_make_visible(vdt, node);

	return TRUE;
}

#if 0
const gchar *vdtree_get_path(ViewDirTree *vdt)
{
	return vdt->path;
}
#endif

void vdtree_refresh(ViewDirTree *vdt)
{
	vdtree_populate_path(vdt, vdt->path, FALSE);
}

const gchar *vdtree_row_get_path(ViewDirTree *vdt, gint row)
{
	printf("FIXME: no get row path\n");
	return NULL;
}

/*
 *----------------------------------------------------------------------------
 * callbacks
 *----------------------------------------------------------------------------
 */

static void vdtree_expand_cb(GtkCTree *ctree, GtkCTreeNode *node, gpointer data)
{
	ViewDirTree *vdt = data;
	GtkCTreeNode *child;
	GtkWidget *top = gtk_widget_get_toplevel(vdt->widget);

	child = GTK_CTREE_ROW(node)->children;

	if (tree_descend_subdirs)
		{
		set_cursor(top, GDK_WATCH);

		gtk_clist_freeze(GTK_CLIST(vdt->ctree));
		while(child)
			{
			vdtree_populate_path_by_node(vdt, child, FALSE);
			child = GTK_CTREE_ROW(child)->sibling;
			}
		gtk_clist_thaw(GTK_CLIST(vdt->ctree));

		set_cursor(top, -1);
		}
}

static void vdtree_collapse_cb(GtkCTree *ctree, GtkCTreeNode *node, gpointer data)
{
	ViewDirTree *vdt = data;
	GtkCTreeNode *selection;

	if (!GTK_CLIST(vdt->ctree)->selection) return;

	selection = GTK_CTREE_NODE(GTK_CLIST(vdt->ctree)->selection->data);
	if (!selection)
		{
		return;
		}

	if (node && gtk_ctree_is_ancestor(GTK_CTREE(vdt->ctree), node, selection))
		{
		gtk_ctree_select(GTK_CTREE(vdt->ctree), node);
		}
}

static void vdtree_select_cb(GtkWidget *widget, GtkCTreeNode *node, gint column, gpointer data)
{
	ViewDirTree *vdt = data;
	NodeData *nd;

	nd = gtk_ctree_node_get_row_data(GTK_CTREE(vdt->ctree), node);

	if (!vdtree_populate_path_by_node(vdt, node, FALSE))
		{
		return;
		}

	if (nd && vdt->select_func)
		{
		vdt->select_func(vdt, nd->fd->path, vdt->select_data);
		}
}

static gint vdtree_press_cb(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	ViewDirTree *vdt = data;
	gint row = -1;

	gtk_clist_get_selection_info(GTK_CLIST(vdt->ctree), bevent->x, bevent->y, &row, NULL);
	if (row != -1)
		{
		vdt->click_node = gtk_ctree_node_nth(GTK_CTREE(vdt->ctree), (guint)row);
		}
	else
		{
		vdt->click_node = NULL;
		}

	if (bevent->button == 3)
		{
		vdtree_set_node_color(vdt, vdt->click_node, TRUE, FALSE);

		vdt->popup = vdtree_pop_menu(vdt, vdt->click_node);
		gtk_signal_connect(GTK_OBJECT(vdt->popup), "destroy",
				   GTK_SIGNAL_FUNC(vdtree_popup_destroy_cb), vdt);

		gtk_menu_popup(GTK_MENU(vdt->popup), NULL, NULL, NULL, NULL,
			       bevent->button, bevent->time);
		}

	return FALSE;
}

/*
 *----------------------------------------------------------------------------
 * core
 *----------------------------------------------------------------------------
 */

static void vdtree_destroy_cb(GtkWidget *widget, gpointer data)
{
	ViewDirTree *vdt = data;

	if (vdt->popup)
		{
		gtk_signal_disconnect_by_data(GTK_OBJECT(vdt->popup), vdt);
		}

	vdtree_dnd_drop_expand_cancel(vdt);
	vdtree_dnd_drop_scroll_cancel(vdt);
	path_list_free(vdt->drop_list);

	folder_icons_free(vdt->pf);

	g_free(vdt->path);
	g_free(vdt);
}

ViewDirTree *vdtree_new(const gchar *path, gint expand)
{
	ViewDirTree *vdt;

	vdt = g_new0(ViewDirTree, 1);

	vdt->path = NULL;
	vdt->click_node = NULL;

	vdt->drop_node = NULL;
	vdt->drop_list = NULL;
	vdt->drop_scroll_id = -1;
	vdt->drop_expand_id = -1;
	vdt->drop_row = -1;

	vdt->popup = NULL;

	vdt->widget = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(vdt->widget),
				       GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_signal_connect(GTK_OBJECT(vdt->widget), "destroy",
			   GTK_SIGNAL_FUNC(vdtree_destroy_cb), vdt);

	vdt->ctree = gtk_ctree_new(1, 0);
	gtk_ctree_set_indent(GTK_CTREE(vdt->ctree), VDTREE_INDENT); /* sorry, had to hard code it */
	gtk_ctree_set_line_style(GTK_CTREE(vdt->ctree), GTK_CTREE_LINES_DOTTED);
	gtk_clist_set_selection_mode(GTK_CLIST(vdt->ctree), GTK_SELECTION_BROWSE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(vdt->ctree), 0 , TRUE);

	gtk_signal_connect(GTK_OBJECT(vdt->ctree), "tree_expand",
			   GTK_SIGNAL_FUNC(vdtree_expand_cb), vdt);
	gtk_signal_connect(GTK_OBJECT(vdt->ctree), "tree_collapse",
			   GTK_SIGNAL_FUNC(vdtree_collapse_cb), vdt);

	gtk_signal_connect(GTK_OBJECT(vdt->ctree), "button_press_event",
			   GTK_SIGNAL_FUNC(vdtree_press_cb), vdt);
	gtk_signal_connect(GTK_OBJECT(vdt->ctree), "tree_select_row",
			   GTK_SIGNAL_FUNC(vdtree_select_cb), vdt);

	gtk_container_add(GTK_CONTAINER(vdt->widget), vdt->ctree);
	gtk_clist_set_auto_sort(GTK_CLIST(vdt->ctree), TRUE);
	gtk_widget_show(vdt->ctree);

	vdt->pf = folder_icons_new();

	vdtree_setup_root(vdt);

	vdtree_dnd_init(vdt);

	if (path) vdtree_set_path(vdt, path);

	return vdt;
}

void vdtree_set_select_func(ViewDirTree *vdt,
                            void (*func)(ViewDirTree *vdt, const gchar *path, gpointer data), gpointer data)
{
        vdt->select_func = func;
        vdt->select_data = data;
}

#if 0
void vdtree_set_click_func(ViewDirTree *vdt,
			   void (*func)(ViewDirTree *vdt, GdkEventButton *event, FileData *fd, gpointer), gpointer data)
{
	if (!td) return;
	vdt->click_func = func;
	vdt->click_data = data;
}
#endif

void vdtree_set_layout(ViewDirTree *vdt, LayoutWindow *layout)
{
	vdt->layout = layout;
}


