/*
 * (SLIK) SimpLIstic sKin functions
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif
#include "intl.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>

#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "ui_tabcomp.h"

#include "ui_fileops.h"

#include <gdk/gdkkeysyms.h> /* for key values */

#include "ui_tabcomp.xpm"

/* define this to enable a pop-up menu that shows possible matches
 * #define TAB_COMPLETION_ENABLE_POPUP_MENU
 */
#define TAB_COMPLETION_ENABLE_POPUP_MENU 1

#ifdef TAB_COMPLETION_ENABLE_POPUP_MENU
#include "ui_menu.h"
#endif


/* ----------------------------------------------------------------
   Tab completion routines, can be connected to any gtkentry widget
   using the tab_completion_add_to_entry() function.
   Use remove_trailing_slash() to strip the trailing '/'.
   ----------------------------------------------------------------*/

typedef struct _TabCompData TabCompData;
struct _TabCompData
{
	GtkWidget *entry;
	gchar *dir_path;
	GList *file_list;
	void (*enter_func)(const gchar *, gpointer);
	void (*tab_func)(const gchar *, gpointer);
	gpointer enter_data;
	gpointer tab_data;

	GtkWidget *combo;
	gint has_history;
	gchar *history_key;
	gint history_levels;
};

typedef struct _HistoryData HistoryData;
struct _HistoryData
{
	gchar *key;
	GList *list;
};

static GList *history_list = NULL;


static gchar *quoted_from_text(const gchar *text)
{
	const gchar *ptr;
	gint c = 0;
	gint l = strlen(text);

	if (l == 0) return NULL;

	while (c < l && text[c] !='"') c++;
	if (text[c] == '"')
		{
		gint e;
		c++;
		ptr = text + c;
		e = c;
		while (e < l && text[e] !='"') e++;
		if (text[e] == '"')
			{
			if (e - c > 0)
				{
				return g_strndup(ptr, e - c);
				}
			}
		}
	return NULL;
}

/*
 *-----------------------------------------------------------------------------
 * history lists
 *-----------------------------------------------------------------------------
 */

gint history_list_load(const gchar *path)
{
	FILE *f;
	gchar *key = NULL;
	gchar s_buf[1024];

	f = fopen(path, "r");
	if (!f) return FALSE;

	/* first line must start with History comment */
	if (!fgets(s_buf,1024,f) ||
	    strncmp(s_buf, "#History", 8) != 0)
		{
		fclose(f);
		return FALSE;
		}

	while (fgets(s_buf,1024,f))
		{
		if (s_buf[0]=='#') continue;
		if (s_buf[0]=='[')
			{
			gint c;
			gchar *ptr;

			ptr = s_buf + 1;
			c = 0;
			while(ptr[c] != ']' && ptr[c] != '\n' && ptr[c] != '\0') c++;

			g_free(key);
			key = g_strndup(ptr, c);
			}
		else
			{
			gchar *value;

			value = quoted_from_text(s_buf);
			if (value && key)
				{
				history_list_add_to_key(key, value, 0);
				}
			g_free(value);
			}
		}

	fclose(f);

	g_free(key);

	return TRUE;
}

gint history_list_save(const gchar *path)
{
	FILE *f;
	GList *list;

	f = fopen(path, "w");
	if (!f)
		{
		printf(_("Unable to write history lists to: %s\n"), path);
		return FALSE;
		}

	fprintf(f, "#History lists\n");
	fprintf(f, "\n");

	list = g_list_last(history_list);
	while(list)
		{
		HistoryData *hd;
		GList *work;

		hd = list->data;
		list = list->prev;

		fprintf(f, "[%s]\n", hd->key);

		/* save them inverted (oldest to newest)
		 * so that when reading they are added correctly
		 */
		work = g_list_last(hd->list);
		while(work)
			{
			fprintf(f, "\"%s\"\n", (gchar *)work->data);
			work = work->prev;
			}
		fprintf(f, "\n");
		}

	fprintf(f, "#end\n");

	fclose(f);

	return TRUE;
}

static void history_list_free(HistoryData *hd)
{
	GList *work;

	if (!hd) return;

	work = hd->list;
	while(work)
		{
		g_free(work->data);
		work = work->next;
		}

	g_free(hd->key);
	g_free(hd);
}

static HistoryData *history_list_find_by_key(const gchar* key)
{
	GList *work = history_list;
	while(work)
		{
		HistoryData *hd = work->data;
		if (strcmp(hd->key, key) == 0) return hd;
		work = work->next;
		}
	return NULL;
}

static const gchar *history_list_find_last_path_by_key(const gchar* key)
{
	HistoryData *hd;
	hd = history_list_find_by_key(key);
	if (!hd || !hd->list) return NULL;

	return hd->list->data;
}

void history_list_free_key(const gchar *key)
{
	HistoryData *hd;
	hd = history_list_find_by_key(key);
	if (!hd) return;

	history_list = g_list_remove(history_list, hd);
	history_list_free(hd);
}

void history_list_add_to_key(const gchar *key, const gchar *path, gint max)
{
	HistoryData *hd;
	GList *work;

	if (!key || !path) return;

	hd = history_list_find_by_key(key);
	if (!hd)
		{
		hd = g_new(HistoryData, 1);
		hd->key = g_strdup(key);
		hd->list = NULL;
		history_list = g_list_prepend(history_list, hd);
		}

	/* if already in the list, simply move it to the top */
	work = hd->list;
	while(work)
		{
		gchar *buf = work->data;
		work = work->next;
		if (strcmp(buf, path) == 0)
			{
			hd->list = g_list_remove(hd->list, buf);
			hd->list = g_list_prepend(hd->list, buf);
			return;
			}
		}

	hd->list = g_list_prepend(hd->list, g_strdup(path));

	if (max > 0)
		{
		while(hd->list && g_list_length(hd->list) > max)
			{
			GList *work = g_list_last(hd->list);
			gchar *buf = work->data;
			hd->list = g_list_remove(hd->list, buf);
			g_free(buf);
			}
		}
}

GList *history_list_get_by_key(const gchar *key)
{
	HistoryData *hd;

	hd = history_list_find_by_key(key);
	if (!hd) return NULL;

	return hd->list;
}

/*
 *-----------------------------------------------------------------------------
 * tab completion entry
 *-----------------------------------------------------------------------------
 */

static void tab_completion_free_list(TabCompData *td)
{
	GList *list;

	g_free(td->dir_path);
	td->dir_path = NULL;

	list = td->file_list;

	while(list)
		{
		g_free(list->data);
		list = list->next;
		}

	g_list_free(td->file_list);
	td->file_list = NULL;
}

static void tab_completion_read_dir(TabCompData *td, const gchar *path)
{
        DIR *dp;
        struct dirent *dir;
        GList *list = NULL;

	tab_completion_free_list(td);

        if((dp = opendir(path))==NULL)
                {
                /* dir not found */
                return;
                }
        while ((dir = readdir(dp)) != NULL)
                {
                /* skips removed files */
                if (dir->d_ino > 0)
                        {
			gchar *name = dir->d_name;
			if (strcmp(name, ".") != 0 && strcmp(name, "..") != 0)
				{
				list = g_list_prepend(list, g_strdup(name));
				}
                        }
		}
        closedir(dp);

	td->dir_path = g_strdup(path);
	td->file_list = list;
}

static void tab_completion_destroy(GtkWidget *widget, gpointer data)
{
	TabCompData *td = data;
	tab_completion_free_list(td);
	g_free(td->history_key);
	g_free(td);
}

static void tab_completion_emit_enter_signal(TabCompData *td)
{
	gchar *text;
	if (!td->enter_func) return;

	text = g_strdup(gtk_entry_get_text(GTK_ENTRY(td->entry)));

	if (text[0] == '~')
		{
		gchar *t = text;
		text = g_strconcat(homedir(), t + 1, NULL);
		g_free(t);
		}

	td->enter_func(text, td->enter_data);
	g_free(text);
}

static void tab_completion_emit_tab_signal(TabCompData *td)
{
	gchar *text;
	if (!td->tab_func) return;

	text = g_strdup(gtk_entry_get_text(GTK_ENTRY(td->entry)));

	if (text[0] == '~')
		{
		gchar *t = text;
		text = g_strconcat(homedir(), t + 1, NULL);
		g_free(t);
		}

	td->tab_func(text, td->tab_data);
	g_free(text);
}

#ifdef TAB_COMPLETION_ENABLE_POPUP_MENU

static gint tab_completion_popup_key_press(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	TabCompData *td = data;

	if (event->keyval == GDK_Tab ||
	    event->keyval == GDK_BackSpace ||
	    (event->keyval >= 0x20 && event->keyval <= 0xFF) )
		{
		if (event->keyval >= 0x20 && event->keyval <= 0xFF)
			{
			gchar buf[2];
			buf[0] = event->keyval;
			buf[1] = '\0';
			gtk_entry_append_text(GTK_ENTRY(td->entry), buf);
			}

		/*close the menu */
		gtk_menu_popdown(GTK_MENU(widget));
		/* doing this does not emit the "selection done" signal, unref it ourselves */
		gtk_widget_unref(widget);
			
		return TRUE;
		}

	return FALSE;
}

static void tab_completion_popup_cb(GtkWidget *widget, gpointer data)
{
	gchar *name = data;
	TabCompData *td;
	gchar *buf;
	gchar *ptr;

	td = gtk_object_get_data(GTK_OBJECT(widget), "tab_completion_data");
	if (!td) return;

	ptr = td->dir_path + strlen(td->dir_path) - 1;
	buf = g_strconcat(td->dir_path, (ptr[0] == '/') ? "" : "/", name, NULL);
	gtk_entry_set_text(GTK_ENTRY(td->entry), buf);
	gtk_entry_set_position(GTK_ENTRY(td->entry), strlen(buf));
	g_free(buf);

	tab_completion_emit_tab_signal(td);
}

/*
 * MAGIC_OFFSET: No simple way to determine frame size of menu.
 * MAX: Limit popup menu items to a sane number, will be fixed when a list is used instead.
 */

#define TAB_COMP_POPUP_MAGIC_OFFSET 3
#define TAB_COMP_POPUP_MAX 64

static void tab_completion_popup_pos_cb(GtkMenu *menu, gint *x, gint *y, gpointer data)
{
	TabCompData *td = data;
	gint height;

	gdk_window_get_origin(td->entry->window, x, y);

	height = MIN(td->entry->requisition.height, td->entry->allocation.height);
	*y += height;

	*x += GTK_ENTRY(td->entry)->char_offset[GTK_EDITABLE(td->entry)->current_pos];
	*x -= GTK_ENTRY(td->entry)->scroll_offset + TAB_COMP_POPUP_MAGIC_OFFSET;
}

static void tab_completion_popup_list(TabCompData *td, GList *list)
{
	GtkWidget *menu;
	GList *work;
	GdkEvent *event;
	guint32 etime;
	gint ebutton;
	gint count = 0;

	if (!list) return;

	/*
	 * well, the menu would be too long anyway...
	 * (listing /dev causes gtk+ window allocation errors, -> too big a window)
	 * this is why menu popups are disabled, this really should be a popup scrollable clist.
	 */
	if (g_list_length(list) > 200) return;

	menu = popup_menu_short_lived();

	work = list;
	while (work && count < TAB_COMP_POPUP_MAX)
		{
		gchar *name = work->data;
		GtkWidget *item;

		item = menu_item_add(menu, name, GTK_SIGNAL_FUNC(tab_completion_popup_cb), name);
		gtk_object_set_data(GTK_OBJECT(item), "tab_completion_data", td);

		work = work->next;
		count++;
		}

	gtk_signal_connect(GTK_OBJECT(menu), "key_press_event",
			   GTK_SIGNAL_FUNC(tab_completion_popup_key_press), td);

	/* peek at the current event to get the time, etc. */
	event = gtk_get_current_event();

	if (event && event->type == GDK_BUTTON_RELEASE)
		{
		ebutton = event->button.button;
		etime = event->button.time;
		}
	else if (event && event->type == GDK_KEY_PRESS)
		{
		ebutton = 0;
		etime = event->key.time;
		}
	else
		{
		ebutton = 0;
		etime = gdk_time_get();
		}

	if (event) gdk_event_free(event);

	gtk_menu_popup(GTK_MENU(menu), NULL, NULL,
		       tab_completion_popup_pos_cb, td, ebutton, etime);
}

static gint simple_sort(gconstpointer a, gconstpointer b)
{
        return strcmp((gchar *)a, (gchar *)b);
}

#endif

static gint tab_completion_do(TabCompData *td)
{
	const gchar *entry_text = gtk_entry_get_text(GTK_ENTRY(td->entry));
	const gchar *entry_file;
	gchar *entry_dir;
	gchar *ptr;
	gint home_exp = FALSE;

	/* home dir expansion */
	if (entry_text[0] == '~')
		{
		entry_dir = g_strconcat(homedir(), entry_text + 1, NULL);
		home_exp = TRUE;
		}
	else
		{
		entry_dir = g_strdup(entry_text);
		}

	entry_file = filename_from_path(entry_text);

	if (isfile(entry_dir))
		{
		if (home_exp)
			{
			gtk_entry_set_text(GTK_ENTRY(td->entry), entry_dir);
			gtk_entry_set_position(GTK_ENTRY(td->entry), strlen(entry_dir));
			}
		g_free(entry_dir);
		return home_exp;
		}
	if (isdir(entry_dir) && strcmp(entry_file, ".") != 0 && strcmp(entry_file, "..") != 0)
		{
		ptr = entry_dir + strlen(entry_dir) - 1;
		if (ptr[0] == '/')
			{
			if (home_exp)
				{
				gtk_entry_set_text(GTK_ENTRY(td->entry), entry_dir);
				gtk_entry_set_position(GTK_ENTRY(td->entry), strlen(entry_dir));
				}

#ifdef TAB_COMPLETION_ENABLE_POPUP_MENU

			tab_completion_read_dir(td, entry_dir);
			td->file_list = g_list_sort(td->file_list, simple_sort);
			tab_completion_popup_list(td, td->file_list);

#endif

			g_free(entry_dir);
			return home_exp;
			}
		else
			{
			gchar *buf = g_strconcat(entry_dir, "/", NULL);
			gtk_entry_set_text(GTK_ENTRY(td->entry), buf);
			gtk_entry_set_position(GTK_ENTRY(td->entry), strlen(buf));
			g_free(buf);
			g_free(entry_dir);
			return TRUE;
			}
		}

	ptr = (gchar *)filename_from_path(entry_dir);
	if (ptr > entry_dir) ptr--;
	ptr[0] = '\0';

	if (strlen(entry_dir) == 0)
		{
		g_free(entry_dir);
		entry_dir = g_strdup("/");
		}

	if (isdir(entry_dir))
		{
		GList *list;
		GList *poss = NULL;
		gint l = strlen(entry_file);

		if (!td->dir_path || !td->file_list || strcmp(td->dir_path, entry_dir) != 0)
			{
			tab_completion_read_dir(td, entry_dir);
			}

		if (strcmp(entry_dir, "/") == 0) entry_dir[0] = '\0';

		list = td->file_list;
		while(list)
			{
			gchar *file = list->data;
			if (strncmp(entry_file, file, l) == 0)
				{
				poss = g_list_prepend(poss, file);
				}
			list = list->next;
			}

		if (poss)
			{
			if (!poss->next)
				{
				gchar *file = poss->data;
				gchar *buf;

				buf = g_strconcat(entry_dir, "/", file, NULL);

				if (isdir(buf))
					{
					g_free(buf);
					buf = g_strconcat(entry_dir, "/", file, "/", NULL);
					}
				gtk_entry_set_text(GTK_ENTRY(td->entry), buf);
				gtk_entry_set_position(GTK_ENTRY(td->entry), strlen(buf));
				g_free(buf);
				g_list_free(poss);
				g_free(entry_dir);
				return TRUE;
				}
			else
				{
				gint c = strlen(entry_file);
				gint done = FALSE;
				gchar *test_file = poss->data;

				while (!done)
					{
					list = poss;
					if (!list) done = TRUE;
					while(list && !done)
						{
						gchar *file = list->data;
						if (strlen(file) < c || strncmp(test_file, file, c) != 0)
							{
							done = TRUE;
							}
						list = list->next;
						}
					c++;
					}
				c -= 2;
				if (c > 0)
					{
					gchar *file;
					gchar *buf;
					file = g_strdup(test_file);
					file[c] = '\0';
					buf = g_strconcat(entry_dir, "/", file, NULL);
					gtk_entry_set_text(GTK_ENTRY(td->entry), buf);
					gtk_entry_set_position(GTK_ENTRY(td->entry), strlen(buf));

#ifdef TAB_COMPLETION_ENABLE_POPUP_MENU

					poss = g_list_sort(poss, simple_sort);
					tab_completion_popup_list(td, poss);

#endif

					g_free(file);
					g_free(buf);
					g_list_free(poss);
					g_free(entry_dir);
					return TRUE;
					}
				}
			g_list_free(poss);
			}
		}

	g_free(entry_dir);

	return FALSE;
}

static gint tab_completion_key_pressed(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	TabCompData *td = data;
	gint stop_signal = FALSE;

	switch (event->keyval)
		{
                case GDK_Tab:
			if (tab_completion_do(td))
				{
				tab_completion_emit_tab_signal(td);
				}
			stop_signal = TRUE;
			break;
		case GDK_Return: case GDK_KP_Enter:
			tab_completion_emit_enter_signal(td);
			stop_signal = TRUE;
			break;
		default:
			break;
		}

	if (stop_signal)
		{
		if (stop_signal) gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		return TRUE;
		}

	return FALSE;
}

static void tab_completion_button_pressed(GtkWidget *widget, gpointer data)
{
	TabCompData *td;
	GtkWidget *entry = data;

	td = gtk_object_get_data(GTK_OBJECT(entry), "tab_completion_data");

	if (!td) return;

	if (!GTK_WIDGET_HAS_FOCUS(entry))
		{
		gtk_widget_grab_focus(entry);
		}

	if (tab_completion_do(td))
		{
		tab_completion_emit_tab_signal(td);
		}
}

static GtkWidget *tab_completion_create_complete_button(GtkWidget *entry)
{
	GtkWidget *button;
	GtkWidget *icon;
	GdkPixmap *pixmap = NULL;
	GdkBitmap *mask = NULL;
	GdkPixbuf *pixbuf;

	pixbuf = gdk_pixbuf_new_from_xpm_data((const char **)ui_tabcomp_xpm);
	gdk_pixbuf_render_pixmap_and_mask(pixbuf, &pixmap, &mask, 128);
	gdk_pixbuf_unref(pixbuf);

	button = gtk_button_new();
	GTK_WIDGET_UNSET_FLAGS(button, GTK_CAN_FOCUS);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(tab_completion_button_pressed), entry);
	
	icon = gtk_pixmap_new(pixmap, mask);
	gtk_container_add(GTK_CONTAINER(button), icon);
	gtk_widget_show(icon);

	gdk_pixmap_unref(pixmap);
	if (mask) gdk_bitmap_unref(mask);

	return button;
}

/*
 *----------------------------------------------------------------------------
 * public interface
 *----------------------------------------------------------------------------
 */

GtkWidget *tab_completion_new_with_history(GtkWidget **entry, const gchar *text,
					   const gchar *history_key, gint max_levels,
					   void (*enter_func)(const gchar *, gpointer), gpointer data)
{
	GtkWidget *combo;
	GtkWidget *button;
	HistoryData *hd;
	TabCompData *td;

	combo = gtk_combo_new();
	gtk_combo_set_use_arrows(GTK_COMBO(combo), FALSE);

	button = tab_completion_create_complete_button(GTK_COMBO(combo)->entry);
	gtk_box_pack_start(GTK_BOX(combo), button, FALSE, FALSE, 0);
	gtk_box_reorder_child(GTK_BOX(combo), button, 1);
	gtk_widget_show(button);
	
	tab_completion_add_to_entry(GTK_COMBO(combo)->entry, enter_func, data);

	td = gtk_object_get_data(GTK_OBJECT(GTK_COMBO(combo)->entry), "tab_completion_data");
	if (!td) return NULL; /* this should never happen! */

	td->combo = combo;
	td->has_history = TRUE;
	td->history_key = g_strdup(history_key);
	td->history_levels = max_levels;

	hd = history_list_find_by_key(td->history_key);
	if (hd && hd->list)
		{
		gtk_combo_set_popdown_strings(GTK_COMBO(combo), hd->list);
		}

	if (text) gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), text);

	if (entry) *entry = GTK_COMBO(combo)->entry;
	return combo;
}

const gchar *tab_completion_set_to_last_history(GtkWidget *entry)
{
	TabCompData *td = gtk_object_get_data(GTK_OBJECT(entry), "tab_completion_data");
	const gchar *buf;

	if (!td || !td->has_history) return NULL;

	buf = history_list_find_last_path_by_key(td->history_key);
	if (buf)
		{
		gtk_entry_set_text(GTK_ENTRY(td->entry), buf);
		}

	return buf;
}

void tab_completion_append_to_history(GtkWidget *entry, const gchar *path)
{
	TabCompData *td = gtk_object_get_data(GTK_OBJECT(entry), "tab_completion_data");
	HistoryData *hd;

	if (!path) return;

	if (!td || !td->has_history) return;

	history_list_add_to_key(td->history_key, path, td->history_levels);
	hd = history_list_find_by_key(td->history_key);
	if (hd && hd->list)
		{
		gchar *buf;

		buf = g_strdup(gtk_entry_get_text(GTK_ENTRY(td->entry)));
		gtk_combo_set_popdown_strings(GTK_COMBO(td->combo), hd->list);
		if (strcmp(buf, gtk_entry_get_text(GTK_ENTRY(td->entry))) != 0)
			{
			gtk_entry_set_text(GTK_ENTRY(td->entry), buf);
			}
		g_free(buf);
		}
}

GtkWidget *tab_completion_new(GtkWidget **entry, const gchar *text,
			      void (*enter_func)(const gchar *, gpointer), gpointer data)
{
	GtkWidget *hbox;
	GtkWidget *button;
	GtkWidget *newentry;

	hbox = gtk_hbox_new(FALSE, 0);

	newentry = gtk_entry_new();
	if (text) gtk_entry_set_text(GTK_ENTRY(newentry), text);
	gtk_box_pack_start(GTK_BOX(hbox), newentry, TRUE, TRUE, 0);
	gtk_widget_show(newentry);

	button = tab_completion_create_complete_button(newentry);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	tab_completion_add_to_entry(newentry, enter_func, data);

	if (entry) *entry = newentry;
	return hbox;
}

void tab_completion_add_to_entry(GtkWidget *entry, void (*enter_func)(const gchar *, gpointer), gpointer data)
{
	TabCompData *td;
	if (!entry)
		{
		printf("Tab completion error: entry != NULL\n");
		return;
		}

	td = g_new0(TabCompData, 1);
	td->entry = entry;
	td->dir_path = NULL;
	td->file_list = NULL;
	td->enter_func = enter_func;
	td->enter_data = data;
	td->tab_func = NULL;
	td->tab_data = NULL;

	td->has_history = FALSE;
	td->history_key = NULL;
	td->history_levels = 0;

	gtk_object_set_data(GTK_OBJECT(td->entry), "tab_completion_data", td);

	gtk_signal_connect(GTK_OBJECT(entry), "key_press_event",
			   GTK_SIGNAL_FUNC(tab_completion_key_pressed), td);
	gtk_signal_connect(GTK_OBJECT(entry), "destroy",
			   GTK_SIGNAL_FUNC(tab_completion_destroy), td);
}

void tab_completion_add_tab_func(GtkWidget *entry, void (*tab_func)(const gchar *, gpointer), gpointer data)
{
	TabCompData *td = gtk_object_get_data(GTK_OBJECT(entry), "tab_completion_data");

	if (!td) return;

	td->tab_func = tab_func;
	td->tab_data = data;
}

gchar *remove_trailing_slash(const gchar *path)
{
	gchar *ret;
	gint l;
	if (!path) return NULL;

	ret = g_strdup(path);
	l = strlen(ret);
	if (l > 1 && ret[l - 1] == '/') ret[l - 1] = '\0';

	return ret;
}

