/*
 * GQview
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "gqview.h"
#include "layout_config.h"


#include "layout.h"


typedef struct _LayoutStyle LayoutStyle;
struct _LayoutStyle
{
	LayoutLocation a, b, c;
};

typedef struct _LayoutConfig LayoutConfig;
struct _LayoutConfig
{
	GtkWidget *box;

	GList *style_widgets;

	GtkWidget *clist;

	gint style;
	gint a, b, c;
};


static LayoutStyle layout_config_styles[] = {
	/* 1, 2, 3 */
	{ LAYOUT_LEFT | LAYOUT_TOP, LAYOUT_LEFT | LAYOUT_BOTTOM, LAYOUT_RIGHT },
	{ LAYOUT_LEFT | LAYOUT_TOP, LAYOUT_RIGHT | LAYOUT_TOP, LAYOUT_BOTTOM },
	{ LAYOUT_LEFT, LAYOUT_RIGHT | LAYOUT_TOP, LAYOUT_RIGHT | LAYOUT_BOTTOM },
	{ LAYOUT_TOP, LAYOUT_LEFT | LAYOUT_BOTTOM, LAYOUT_RIGHT | LAYOUT_BOTTOM }
};

static gint layout_config_style_count = sizeof(layout_config_styles) / sizeof (LayoutStyle);


static void layout_config_destroy(GtkWidget *widget, gpointer data)
{
	LayoutConfig * lc = data;

	g_list_free(lc->style_widgets);
	g_free(lc);
}

static void layout_config_set_order(LayoutLocation l, gint n,
				    LayoutLocation *a, LayoutLocation *b, LayoutLocation *c)
{
	switch (n)
		{
		case 0:
			*a = l;
			break;
		case 1:
			*b = l;
			break;
		case 2: default:
			*c = l;
			break;
		}
}

static void layout_config_from_data(gint style, gint oa, gint ob, gint oc,
				    LayoutLocation *la, LayoutLocation *lb, LayoutLocation *lc)
{
	LayoutStyle ls;

	style = CLAMP(style, 0, layout_config_style_count);

	ls = layout_config_styles[style];

	layout_config_set_order(ls.a, oa, la, lb, lc);
	layout_config_set_order(ls.b, ob, la, lb, lc);
	layout_config_set_order(ls.c, oc, la, lb, lc);
}

void layout_config_parse(gint style, const gchar *order,
			 LayoutLocation *a, LayoutLocation *b, LayoutLocation *c)
{
	gint na, nb, nc;

	layout_config_order_from_text(order, &na, &nb, &nc);
	layout_config_from_data(style, na, nb, nc, a, b, c);
}

#if 0
static gint layout_style_match(LayoutStyle ls, LayoutLocation l)
{
	return (ls.a == l || ls.b == l || ls.c == l);
}

static gint layout_config_get_style(LayoutLocation a, LayoutLocation b, LayoutLocation c)
{
	gint i;

	for (i = 0; i < layout_config_style_count; i++)
		{
		LayoutStyle ls;

		ls = layout_config_styles[i];
		if (layout_style_match(ls, a) &&
		    layout_style_match(ls, b) &&
		    layout_style_match(ls, c)) return i;
		}
	return 0;
}

static void layout_config_get_order(LayoutStyle ls, LayoutLocation l, gint *n)
{
	if (ls.a == l)
		{
		*n = 0;
		}
	else if (ls.b == l)
		{
		*n = 1;
		}
	else
		{
		*n = 2;
		}
}

static void layout_config_to_styledata(LayoutLocation la, LayoutLocation lb, LayoutLocation lc,
				       gint *style, gint *oa, gint *ob, gint *oc)
{
	LayoutStyle ls;

	*style = layout_config_get_style(la, lb, lc);

	ls = layout_config_styles[*style];
	layout_config_get_order(ls, la, oa);
	layout_config_get_order(ls, lb, ob);
	layout_config_get_order(ls, lc, oc);
}
#endif

static void layout_config_list_order(GtkWidget *clist, gint src, gint dest)
{
	gint row;

	row = gtk_clist_find_row_from_data(GTK_CLIST(clist), GINT_TO_POINTER(src));
	if (row != dest)
		{
		gtk_clist_row_move(GTK_CLIST(clist), row, dest);
		}
}

void layout_config_set(GtkWidget *widget, gint style, const gchar *order)
{
	LayoutConfig *lc;
	GtkWidget *button;
	gint a, b, c;

	lc = gtk_object_get_data(GTK_OBJECT(widget), "layout_config");

	if (!lc) return;

	style = CLAMP(style, 0, layout_config_style_count);
	button = g_list_nth_data(lc->style_widgets, style);
	if (!button) return;

	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), TRUE);

	layout_config_order_from_text(order, &a, &b, &c);

	layout_config_list_order(lc->clist, a, 0);
	layout_config_list_order(lc->clist, b, 1);
	layout_config_list_order(lc->clist, c, 2);
}

gchar *layout_config_get(GtkWidget *widget, gint *style)
{
	LayoutConfig *lc;

	lc = gtk_object_get_data(GTK_OBJECT(widget), "layout_config");

	/* this should not happen */
	if (!lc) return NULL;

	*style = lc->style;

	lc->a = GPOINTER_TO_INT(gtk_clist_get_row_data(GTK_CLIST(lc->clist), 0));
	lc->b = GPOINTER_TO_INT(gtk_clist_get_row_data(GTK_CLIST(lc->clist), 1));
	lc->c = GPOINTER_TO_INT(gtk_clist_get_row_data(GTK_CLIST(lc->clist), 2));

	return layout_config_order_to_text(lc->a, lc->b, lc->c);
}

static void layout_config_widget_click_cb(GtkWidget *widget, gpointer data)
{
	LayoutConfig *lc;

	lc = gtk_object_get_data(GTK_OBJECT(widget), "layout_config");
	
	if (lc) lc->style = GPOINTER_TO_INT(data);
}

static void layout_config_table_button(GtkWidget *table, LayoutLocation l, const gchar *text)
{
	GtkWidget *button;

	gint x1, y1;
	gint x2, y2;

	x1 = 0;
	y1 = 0;
	x2 = 2;
	y2 = 2;

	if (l & LAYOUT_LEFT) x2 = 1;
	if (l & LAYOUT_RIGHT) x1 = 1;
	if (l & LAYOUT_TOP) y2 = 1;
	if (l & LAYOUT_BOTTOM) y1 = 1;

	button = gtk_button_new_with_label(text);
	gtk_widget_set_sensitive(button, FALSE);
	GTK_WIDGET_UNSET_FLAGS(button, GTK_CAN_FOCUS);
	gtk_table_attach_defaults(GTK_TABLE(table), button, x1, x2, y1, y2);
	gtk_widget_show(button);
}

#define LAYOUT_STYLE_SIZE 48

static GtkWidget *layout_config_widget(GtkWidget *group, GtkWidget *box, gint style, LayoutConfig *lc)
{
	GtkWidget *table;
	LayoutStyle ls;

	ls = layout_config_styles[style];

	if (group)
		{
		group = gtk_radio_button_new(gtk_radio_button_group(GTK_RADIO_BUTTON(group)));
		}
	else
		{
		group = gtk_radio_button_new(NULL);
		}
	gtk_object_set_data(GTK_OBJECT(group), "layout_config", lc);
	gtk_signal_connect(GTK_OBJECT(group), "clicked",
			   GTK_SIGNAL_FUNC(layout_config_widget_click_cb), GINT_TO_POINTER(style));
	gtk_box_pack_start(GTK_BOX(box), group, FALSE, FALSE, 0);

	table = gtk_table_new(2, 2, TRUE);

	layout_config_table_button(table, ls.a, "1");
	layout_config_table_button(table, ls.b, "2");
	layout_config_table_button(table, ls.c, "3");

	gtk_widget_set_usize(table, LAYOUT_STYLE_SIZE, LAYOUT_STYLE_SIZE);
	gtk_container_add(GTK_CONTAINER(group), table);
	gtk_widget_show(table);

	gtk_widget_show(group);
	
	return group;
}

static void layout_config_row_move(GtkWidget *clist, gint source_row, gint dest_row, gpointer data)
{
	gchar *text;
	gint i;

	for (i = 0; i < 3; i++)
		{
		text = g_strdup_printf("%d", i + 1);
		gtk_clist_set_text(GTK_CLIST(clist), i, 0, text);
		g_free(text);
		}
}

GtkWidget *layout_config_new(void)
{
	LayoutConfig *lc;
	GtkWidget *vbox;
	GtkWidget *group = NULL;
	GtkWidget *scrolled;
	GtkWidget *label;
	gchar *layout_titles[] = { _("Tools"), _("Files"), _("Image") };
	gint i;

	lc = g_new0(LayoutConfig, 1);

	lc->box = gtk_hbox_new(TRUE, 5);
	gtk_object_set_data(GTK_OBJECT(lc->box), "layout_config", lc);

	gtk_signal_connect(GTK_OBJECT(lc->box), "destroy",
			   GTK_SIGNAL_FUNC(layout_config_destroy), lc);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(lc->box), vbox, FALSE, FALSE, 0);
	for (i = 0; i < layout_config_style_count; i++)
		{
		group = layout_config_widget(group, vbox, i, lc);
		lc->style_widgets = g_list_append(lc->style_widgets, group);
		}
	gtk_widget_show(vbox);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(lc->box), vbox, FALSE, FALSE, 0);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				       GTK_POLICY_NEVER, GTK_POLICY_NEVER);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	gtk_widget_show(scrolled);

	lc->clist = gtk_clist_new(2);
	gtk_clist_set_column_auto_resize(GTK_CLIST(lc->clist), 0, TRUE);
	gtk_clist_set_reorderable(GTK_CLIST(lc->clist), TRUE);
	for (i = 0; i < 3; i++)
		{
		gchar *buf[3];
		gint row;

		buf[0] = g_strdup_printf("%d", i + 1);
		buf[1] = layout_titles[i];
		buf[2] = NULL;

		row = gtk_clist_append(GTK_CLIST(lc->clist), buf);
		gtk_clist_set_row_data(GTK_CLIST(lc->clist), row, GINT_TO_POINTER(i));

		g_free(buf[0]);
		}

	gtk_signal_connect_after(GTK_OBJECT(lc->clist), "row_move",
				 GTK_SIGNAL_FUNC(layout_config_row_move), NULL);
	
	gtk_container_add(GTK_CONTAINER(scrolled), lc->clist);
	gtk_widget_show(lc->clist);

	label = gtk_label_new(_("(drag to change order)"));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);
	
	gtk_widget_show(vbox);

	return lc->box;
}

static char num_to_text_char(gint n)
{
	switch (n)
		{
		case 1:
			return '2';
			break;
		case 2:
			return '3';
			break;
		}
	return '1';
}

gchar *layout_config_order_to_text(gint a, gint b, gint c)
{
	gchar *text;

	text = g_strdup("   ");

	text[0] = num_to_text_char(a);
	text[1] = num_to_text_char(b);
	text[2] = num_to_text_char(c);

	return text;
}

static gint text_char_to_num(const gchar *text, gint n)
{
	if (text[n] == '3') return 2;
	if (text[n] == '2') return 1;
	return 0;
}

void layout_config_order_from_text(const gchar *text, gint *a, gint *b, gint *c)
{
	if (!text || strlen(text) < 3)
		{
		*a = 0;
		*b = 1;
		*c = 2;
		}
	else
		{
		*a = text_char_to_num(text, 0);
		*b = text_char_to_num(text, 1);
		*c = text_char_to_num(text, 2);
		}
}


