/*
 * GQview
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "fullscreen.h"

#include "image.h"

/*
 *----------------------------------------------------------------------------
 * full screen functions
 *----------------------------------------------------------------------------
 */

static void clear_mouse_cursor(GtkWidget *widget, gint clear)
{
	if (clear)
		{
		GdkCursor *cursor;
		GdkPixmap *p;

		p = gdk_bitmap_create_from_data(widget->window, "\0\0\0", 1, 1);

		cursor = gdk_cursor_new_from_pixmap(p, p,
						    &widget->style->fg[GTK_STATE_ACTIVE],
						    &widget->style->bg[GTK_STATE_ACTIVE],
						    0, 0);

		gdk_window_set_cursor (widget->window, cursor);

		if (cursor) gdk_cursor_destroy (cursor);
		gdk_pixmap_unref(p);
		}
	else
		{
		gdk_window_set_cursor (widget->window, NULL);
		}
}

static gint fullscreen_hide_mouse_cb(gpointer data)
{
	FullScreenData *fs = data;

	if (fs->hide_mouse_id == -1) return FALSE;

	clear_mouse_cursor(fs->window, TRUE);

	fs->hide_mouse_id = -1;
	return FALSE;
}

static void fullscreen_hide_mouse_disable(FullScreenData *fs)
{
	if (fs->hide_mouse_id != -1)
		{
		gtk_timeout_remove(fs->hide_mouse_id);
		fs->hide_mouse_id = -1;
		}
}

static void fullscreen_hide_mouse_reset(FullScreenData *fs)
{
	fullscreen_hide_mouse_disable(fs);
	fs->hide_mouse_id = gtk_timeout_add(FULL_SCREEN_HIDE_MOUSE_DELAY, fullscreen_hide_mouse_cb, fs);
}

static gint fullscreen_mouse_moved(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	FullScreenData *fs = data;

	clear_mouse_cursor(fs->window, FALSE);
	fullscreen_hide_mouse_reset(fs);

	return FALSE;
}

static gint fullscreen_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	FullScreenData *fs = data;

	fullscreen_stop(fs);
	return TRUE;
}

FullScreenData *fullscreen_start(GtkWidget *window, ImageWindow *imd,
				 void (*stop_func)(FullScreenData *, gpointer), gpointer stop_data)
{
	FullScreenData *fs;
	gint w;
	gint h;

	if (!window || !imd) return NULL;

	fs = g_new0(FullScreenData, 1);

	fs->hide_mouse_id = -1;
	fs->normal_window = window;
	fs->normal_imd = imd;

	fs->stop_func = stop_func;
	fs->stop_data = stop_data;

	w = gdk_screen_width();
	h = gdk_screen_height();

	fs->window = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_container_border_width(GTK_CONTAINER(fs->window), 0);
	gtk_signal_connect(GTK_OBJECT(fs->window), "delete_event",
			   GTK_SIGNAL_FUNC(fullscreen_delete_cb), fs);

	gtk_window_set_title(GTK_WINDOW(fs->window), _("GQview full screen"));
	gtk_widget_set_usize(fs->window, w, h);
	gtk_widget_set_uposition(fs->window, 0, 0);

	fs->imd = image_new(FALSE);

	gtk_container_add(GTK_CONTAINER(fs->window), fs->imd->widget);

	/* set background to black */
	if (BLACK_BACKGROUND)
		{
		image_background_set_black(fs->imd, TRUE);
		}

	gtk_widget_show(fs->imd->widget);

	image_change_from_image(fs->imd, fs->normal_imd);

	gtk_window_set_modal(GTK_WINDOW(fs->window), TRUE);
	gtk_widget_show(fs->window);

	gdk_keyboard_grab(fs->window->window, TRUE, GDK_CURRENT_TIME);
	gtk_grab_add(fs->window);
	gtk_widget_grab_focus(fs->window);

	/* for hiding the mouse */
	gtk_signal_connect(GTK_OBJECT(fs->imd->image), "motion_notify_event",
			   GTK_SIGNAL_FUNC(fullscreen_mouse_moved), fs);
	clear_mouse_cursor(fs->window, TRUE);
#if 0
	fullscreen_hide_mouse_reset(fs);
#endif

	/* hide normal window
	 * FIXME: properly restore this window on show
	 */
#ifdef HIDE_WINDOW_IN_FULLSCREEN
	gtk_widget_hide(fs->normal_window);
#endif
	image_change_path(fs->normal_imd, NULL, image_zoom_get(fs->normal_imd));

	return fs;
}

void fullscreen_stop(FullScreenData *fs)
{
	if (!fs) return;

	fullscreen_hide_mouse_disable(fs);
	gdk_keyboard_ungrab(GDK_CURRENT_TIME);

	image_change_from_image(fs->normal_imd, fs->imd);
#ifdef HIDE_WINDOW_IN_FULLSCREEN
	gtk_widget_show(fs->normal_window);
#endif
	if (fs->stop_func) fs->stop_func(fs, fs->stop_data);

	gtk_grab_remove(fs->window);

	gtk_widget_destroy(fs->window);

	g_free(fs);
}

