/*
 * GQview
 * (C) 2002 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "dupe.h"

#include "cache.h"
#include "collect.h"
#include "collect-table.h"
#include "dnd.h"
#include "filelist.h"
#include "image-load.h"
#include "img-view.h"
#include "info.h"
#include "layout.h"
#include "layout_image.h"
#include "menu.h"
#include "thumb.h"
#include "utilops.h"
#include "ui_clist_edit.h"
#include "ui_fileops.h"
#include "ui_menu.h"

#include <gdk/gdkkeysyms.h> /* for keyboard values */


#include <utime.h>


#define DUPE_DEF_WIDTH 600
#define DUPE_DEF_HEIGHT 400

/* column assignment order (simply change them here) */
#define DUPE_COLUMN_RANK	0
#define DUPE_COLUMN_THUMB	1
#define DUPE_COLUMN_NAME	2
#define DUPE_COLUMN_SIZE	3
#define DUPE_COLUMN_DATE	4
#define DUPE_COLUMN_DIMENSIONS	5
#define DUPE_COLUMN_PATH	6

#define DUPE_COLUMN_COUNT	7	/* total columns */


static GList *dupe_window_list = NULL;	/* list of open DupeWindow *s */

/*
 * Well, after adding the 'compare two sets' option things got a little sloppy in here
 * because we have to account for two 'modes' everywhere. (be careful).
 */


static DupeData *dupe_data_find(DupeWindow *dw, DupeItem *di);
static void dupe_thumb_step(DupeWindow *dw);
static gint dupe_check_cb(gpointer data);

static void dupe_second_add(DupeWindow *dw, DupeItem *di);
static void dupe_second_remove(DupeWindow *dw, DupeItem *di);

static void dupe_dnd_init(DupeWindow *dw);

static void dupe_window_update_count(DupeWindow *dw, gint count_only)
{
	gchar *text;

	if (!dw->list)
		{
		gtk_label_set(GTK_LABEL(dw->status_label), _("Drop files to compare them."));
		return;
		}

	if (count_only)
		{
		text = g_strdup_printf(_("%d files"), g_list_length(dw->list));
		}
	else
		{
		text = g_strdup_printf(_("%d matches found in %d files"), g_list_length(dw->dupes), g_list_length(dw->list));
		}
	gtk_label_set(GTK_LABEL(dw->status_label), text);

	g_free(text);
}

static gint dupe_iterations(gint n)
{
	return (n * ((n + 1) / 2));
}

static void dupe_window_update_progress(DupeWindow *dw, const gchar *status, gfloat value)
{
	const gchar *status_text;

	if (status)
		{
		gtk_progress_set_value(GTK_PROGRESS(dw->extra_label), value);

		if (dw->setup_count > 0)
			{
			if (dw->setup_n % 10 == 0)
				{
				dw->setup_time_count = (gdk_time_get() - dw->setup_time) / 1000;
				}

			if (dw->setup_time_count > 3)
				{
				gchar *buf;
				gint t;
				gint d;
				guint32 rem;

				if (dw->setup_done)
					{
					if (dw->second_set)
						{
						t = dw->setup_count;
						d = dw->setup_count - dw->setup_n;
						}
					else
						{
						t = dupe_iterations(dw->setup_count);
						d = dupe_iterations(dw->setup_count - dw->setup_n);
						}
					}
				else
					{
					t = dw->setup_count;
					d = dw->setup_count - dw->setup_n;
					}

				rem = (t - d) ? ((float)dw->setup_time_count / (t - d)) * d : 0;

				buf = g_strdup_printf("%s %d:%02d ", status, rem / 60, rem % 60);

				gtk_progress_set_format_string(GTK_PROGRESS(dw->extra_label), buf);
				g_free(buf);
				return;
				}
			}
		status_text = status;
		}	
	else
		{
		gtk_progress_set_value(GTK_PROGRESS(dw->extra_label), 0.0);
		status_text = "";
		}

	gtk_progress_set_format_string(GTK_PROGRESS(dw->extra_label), status_text);
}

static GtkStyle *dupe_clist_new_match_style(DupeWindow *dw, gint row)
{
	GtkStyle *row_style;
	GtkStyle *style = NULL;

	if (row < 0 || GTK_CLIST(dw->clist)->rows < 1) return NULL;

	row_style = gtk_clist_get_row_style(GTK_CLIST(dw->clist), row);

	if (!row_style)
		{
		style = gtk_style_copy(gtk_widget_get_style(dw->clist));
		clist_edit_shift_color(style);
		}

	return style;	
}

static void dupe_clist_realign_styles(DupeWindow *dw)
{
	gint length;
	GtkStyle *style = NULL;
	DupeData *dd = NULL;
	gint i;

	length = GTK_CLIST(dw->clist)->rows;

	gtk_clist_freeze(GTK_CLIST(dw->clist));

	for (i = 0; i < length; i++)
		{
		DupeItem *di;

		di = gtk_clist_get_row_data(GTK_CLIST(dw->clist), i);

		if (!dd || g_list_find(dd->list, di) == NULL)
			{
			dd = dupe_data_find(dw, di);
			if (style)
				{
				gtk_style_unref(style);
				style = NULL;
				}
			else
				{
				if (i == 0)
					{
					style = gtk_clist_get_row_style(GTK_CLIST(dw->clist), 0);
					if (style) gtk_style_ref(style);
					}
				else
					{
					style = dupe_clist_new_match_style(dw, i - 1);
					}
				}
			}

		gtk_clist_set_row_style(GTK_CLIST(dw->clist), i, style);
		}
	if (style) gtk_style_unref(style);

	gtk_clist_thaw(GTK_CLIST(dw->clist));
}

static DupeItem *dupe_item_new(const gchar *path, gint size, time_t date)
{
	DupeItem *di;

	di = g_new0(DupeItem, 1);

	di->path = g_strdup(path);
	di->name = filename_from_path(di->path);
	di->size = size;
	di->date = date;

	di->dupe = FALSE;

	di->simd = NULL;
	di->checksum = 0;
	di->width = 0;
	di->height = 0;

	return di;
}

static void dupe_item_free(DupeItem *di)
{
	g_free(di->path);
	image_sim_free(di->simd);
	if (di->pixmap) gdk_pixmap_unref(di->pixmap);
	if (di->mask) gdk_bitmap_unref(di->mask);

	g_free(di);
}

static DupeItem *dupe_item_find_path_by_list(const gchar *path, GList *work)
{
	while (work)
		{
		DupeItem *di = work->data;

		if (strcmp(di->path, path) == 0) return di;

		work = work->next;
		}

	return NULL;
}

static DupeItem *dupe_item_find_path(DupeWindow *dw, const gchar *path)
{
	DupeItem *di;

	di = dupe_item_find_path_by_list(path, dw->list);
	if (!di && dw->second_set) di = dupe_item_find_path_by_list(path, dw->second_list);

	return di;
}

static void dupe_item_read_cache(DupeItem *di)
{
	gchar *path;
	CacheData *cd;

	if (!di) return;

	path = cache_find_location(di->path, GQVIEW_CACHE_SIMIL_EXT);
	if (!path) return;

	if (filetime(di->path) != filetime(path))
		{
		g_free(path);
		return;
		}

	cd = cache_sim_data_load(path);
	g_free(path);

	if (cd)
		{
		if (!di->simd && cd->sim)
			{
			di->simd = cd->sim;
			cd->sim = NULL;
			}
		if (di->width == 0 && di->height == 0 && cd->dimensions)
			{
			di->width = cd->width;
			di->height = cd->height;
			}
		if (di->checksum == 0 && cd->have_checksum)
			{
			di->checksum = cd->checksum;
			}
		cache_sim_data_free(cd);
		}
}

static void dupe_item_write_cache(DupeItem *di)
{
	gchar *base;
	mode_t mode = 0755;

	if (!di) return;

	base = cache_get_location(di->path, FALSE, NULL, &mode);
	if (cache_ensure_dir_exists(base, mode))
		{
		CacheData *cd;

		cd = cache_sim_data_new();
		cd->path = cache_get_location(di->path, TRUE, GQVIEW_CACHE_SIMIL_EXT, NULL);

		if (di->width != 0) cache_sim_data_set_dimensions(cd, di->width, di->height);
		if (di->checksum != 0) cache_sim_data_set_checksum(cd, di->checksum);
		if (di->simd) cache_sim_data_set_similarity(cd, di->simd);

		if (cache_sim_data_save(cd))
			{
			struct utimbuf ut;
			/* set time to that of source file */

			ut.actime = ut.modtime = filetime(di->path);
 			if (ut.modtime > 0)
 				{
 				utime(cd->path, &ut);
 				}
			}
		cache_sim_data_free(cd);
		}
	g_free(base);
}

static DupeData *dupe_data_new(void)
{
	DupeData *dd;

	dd = g_new0(DupeData, 1);
	dd->list = NULL;
	return dd;
}

static void dupe_data_free(DupeData *dd)
{
	g_list_free(dd->list);
	g_free(dd);
}

static void dupe_data_add(DupeWindow *dw, DupeData *dd, DupeItem *di)
{
	gint row;
	gchar *text[9];
	GtkStyle *style = NULL;

	if (di->dupe) return;
	di->dupe = TRUE;

	dd->list = g_list_prepend(dd->list, di);

	if (dd->list->next)
		{
		row = gtk_clist_find_row_from_data(GTK_CLIST(dw->clist), (GList *)(dd->list->next)->data);
		style = gtk_clist_get_row_style(GTK_CLIST(dw->clist), row);
		if (style) gtk_style_ref(style);
		row++;
		}
	else
		{
		row = 0;
		style = dupe_clist_new_match_style(dw, row);
		}

	if (di->rank == 0 || !dd->list->next)
		{
		text[DUPE_COLUMN_RANK] = g_strdup("");
		}
	else
		{
		text[DUPE_COLUMN_RANK] = g_strdup_printf("%d", di->rank);
		}

	text[DUPE_COLUMN_THUMB] = "";
	text[DUPE_COLUMN_NAME] = (gchar *)di->name;
	text[DUPE_COLUMN_SIZE] = text_from_size(di->size);
	text[DUPE_COLUMN_DATE] = (gchar *)text_from_time(di->date);
	if (di->width > 0 && di->height > 0)
		{
		text[DUPE_COLUMN_DIMENSIONS] = g_strdup_printf("%d x %d", di->width, di->height);
		}
	else
		{
		text[DUPE_COLUMN_DIMENSIONS] = g_strdup("");
		}
	text[DUPE_COLUMN_PATH] = di->path;
	text[DUPE_COLUMN_COUNT] = NULL;
	row = gtk_clist_insert(GTK_CLIST(dw->clist), row, text);
	g_free(text[DUPE_COLUMN_RANK]);
	g_free(text[DUPE_COLUMN_SIZE]);
	g_free(text[DUPE_COLUMN_DIMENSIONS]);

	if (style)
		{
		gtk_clist_set_row_style(GTK_CLIST(dw->clist), row, style);
		gtk_style_unref(style);
		}
	gtk_clist_set_row_data(GTK_CLIST(dw->clist), row, di);
}

static void dupe_clist_remove_item(DupeWindow *dw, DupeItem *di)
{
	gint row;

	row = gtk_clist_find_row_from_data(GTK_CLIST(dw->clist), di);
	if (row < 0) return;
	gtk_clist_remove(GTK_CLIST(dw->clist), row);
}

static void dupe_data_remove(DupeWindow *dw, DupeData *dd, DupeItem *di)
{
	if (!dd || !di || !g_list_find(dd->list, di)) return;

	dd->list = g_list_remove(dd->list, di);
	dupe_clist_remove_item(dw, di);
	di->dupe = FALSE;

	if (dd->list && !dd->list->next)
		{
		DupeItem *last = dd->list->data;
		dd->list = g_list_remove(dd->list, last);
		last->dupe = FALSE;
		dupe_clist_remove_item(dw, last);
		}

	if (!dd->list)
		{
		dw->dupes = g_list_remove(dw->dupes, dd);
		g_free(dd);
		if (GTK_CLIST(dw->clist)->freeze_count == 0) dupe_clist_realign_styles(dw);
		}
}

static DupeData *dupe_data_find(DupeWindow *dw, DupeItem *di)
{
	GList *work;
	work = dw->dupes;
	while (work)
		{
		GList *w;
		DupeData *dd = work->data;
		work = work->next;

		w = dd->list;
		while(w)
			{
			if (w->data == di) return dd;
			w = w->next;
			}
		}

	return NULL;
}

static void dupe_list_free(GList *list)
{
	GList *work = list;
	while(work)
		{
		DupeData *dd = work->data;
		work = work->next;
		dupe_data_free(dd);
		}
	g_list_free(list);
}

static void dupe_files_free(GList *list)
{
	GList *work = list;
	while(work)
		{
		DupeItem *di = work->data;
		work = work->next;
		dupe_item_free(di);
		}
	g_list_free(list);
}

static gint dupe_match(DupeItem *a, DupeItem *b, DupeMatchType mask)
{
	gint match = 0;

	if (a == b) return FALSE;

	if (mask & DUPE_MATCH_PATH)
		{
		if (strcmp(a->path, b->path) == 0)
			{
			match = 1;
			a->rank = 0;
			b->rank = 0;
			}
		else
			{
			return FALSE;
			}
		}
	if (mask & DUPE_MATCH_NAME)
		{
		if (strcmp(a->name, b->name) == 0)
			{
			match = 1;
			a->rank = 0;
			b->rank = 0;
			}
		else
			{
			return FALSE;
			}
		}
	if (mask & DUPE_MATCH_SIZE)
		{
		if (a->size == b->size)
			{
			match = 1;
			a->rank = 0;
			b->rank = 0;
			}
		else
			{
			return FALSE;
			}
		}
	if (mask & DUPE_MATCH_DATE)
		{
		if (a->date == b->date)
			{
			match = 1;
			a->rank = 0;
			b->rank = 0;
			}
		else
			{
			return FALSE;
			}
		}
	if (mask & DUPE_MATCH_SUM)
		{
		if (a->checksum == 0) a->checksum = checksum_simple(a->path);
		if (b->checksum == 0) b->checksum = checksum_simple(b->path);
		if (a->checksum == b->checksum)
			{
			match = 1;
			a->rank = 0;
			b->rank = 0;
			}
		else
			{
			return FALSE;
			}
		}
	if (mask & DUPE_MATCH_DIM)
		{
		if (a->width == 0) image_load_dimensions(a->path, &a->width, &a->height);
		if (b->width == 0) image_load_dimensions(b->path, &b->width, &b->height);
		if (a->width == b->width && a->height == b->height)
			{
			match = 1;
			a->rank = 0;
			b->rank = 0;
			}
		else
			{
			return FALSE;
			}
		}
	if (mask & DUPE_MATCH_SIM_HIGH ||
	    mask & DUPE_MATCH_SIM_MED ||
	    mask & DUPE_MATCH_SIM_LOW ||
	    mask & DUPE_MATCH_SIM_CUSTOM)
		{
		gfloat f;
		gfloat m;

		if (mask & DUPE_MATCH_SIM_HIGH) m = 0.95;
		else if (mask & DUPE_MATCH_SIM_MED) m = 0.90;
		else if (mask & DUPE_MATCH_SIM_CUSTOM) m = (float)dupe_custom_threshold / 100.0;
		else m = 0.85;

		f = image_sim_compare_fast(a->simd, b->simd, m);

		if (f > m)
			{
			if (debug) printf("similar: %32s %32s = %f\n", a->name, b->name, f);
			match = 1;
			a->rank = b->rank = (gint)(f * 100.0);
			}
		else
			{
			return FALSE;
			}
		}

	return match;
}

static void dupe_list_check_match(DupeWindow *dw, DupeItem *needle, GList *start)
{
	GList *work;

	if (dw->second_set)
		{
		work = dw->second_list;
		}
	else if (start)
		{
		work = start;
		}
	else
		{
		work = g_list_last(dw->list);
		}

	while (work)
		{
		DupeItem *di = work->data;

		/* speed opt: forward for second set, back for simple compare */
		if (dw->second_set)
			work = work->next;
		else
			work = work->prev;

		if ((!di->dupe || !needle->dupe) && dupe_match(di, needle, dw->match_mask))
			{
			DupeData *dd = NULL;

			if (di->dupe) dd = dupe_data_find(dw, di);
			if (needle->dupe) dd = dupe_data_find(dw, needle);
			if (!dd)
				{
				dd = dupe_data_new();
				dw->dupes = g_list_prepend(dw->dupes, dd);
				}
			dupe_data_add(dw, dd, needle);
			dupe_data_add(dw, dd, di);
			}
		}
}

static void dupe_clist_set_thumb(DupeWindow *dw, DupeItem *di)
{
	gint row;

	row = gtk_clist_find_row_from_data(GTK_CLIST(dw->clist), di);
	if (row < 0) return;

	gtk_clist_set_pixmap(GTK_CLIST(dw->clist), row, DUPE_COLUMN_THUMB, di->pixmap, di->mask);
}

static void dupe_thumb_do(DupeWindow *dw)
{
	DupeItem *di;

	if (!dw->thumb_loader || !dw->thumb_item) return;
	di = dw->thumb_item;

	if (di->pixmap) gdk_pixmap_unref(di->pixmap);
	if (di->mask) gdk_bitmap_unref(di->mask);
	if (thumb_loader_to_pixmap(dw->thumb_loader, &di->pixmap, &di->mask) < 0)
		{
		thumb_from_xpm_d(img_unknown, thumb_max_width, thumb_max_height, &di->pixmap, &di->mask);
		}

	dupe_clist_set_thumb(dw, di);
}

static void dupe_thumb_error_cb(ThumbLoader *tl, gpointer data)
{
	DupeWindow *dw = data;

	dupe_thumb_do(dw);
	dupe_thumb_step(dw);
}

static void dupe_thumb_done_cb(ThumbLoader *tl, gpointer data)
{
	DupeWindow *dw = data;

	dupe_thumb_do(dw);
	dupe_thumb_step(dw);
}

static void dupe_thumb_step(DupeWindow *dw)
{
	DupeItem *di = NULL;
	gint i;
	gint length;

	i = 0;
	length = GTK_CLIST(dw->clist)->rows;

	while (!di && i < length)
		{
		if (!gtk_clist_get_pixmap(GTK_CLIST(dw->clist), i, DUPE_COLUMN_THUMB, NULL, NULL))
			{
			di = gtk_clist_get_row_data(GTK_CLIST(dw->clist), i);
			
			/* cached? then use it */
			if (di && di->pixmap)
				{
				dupe_clist_set_thumb(dw, di);
				di = NULL;
				}
			}
		i++;
		}

	if (!di)
		{
		dw->thumb_item = NULL;
		thumb_loader_free(dw->thumb_loader);
		dw->thumb_loader = NULL;

		dupe_window_update_progress(dw, NULL, 0.0);
		return;
		}

	dupe_window_update_progress(dw, _("Loading thumbs..."),
				    length == 0 ? 0.0 : (float)(i - 1) / length);

	dw->thumb_item = di;
	thumb_loader_free(dw->thumb_loader);
	dw->thumb_loader = thumb_loader_new(di->path, thumb_max_width, thumb_max_height);

	/* setup error */
	thumb_loader_set_error_func(dw->thumb_loader, dupe_thumb_error_cb, dw);

	/* start it */
	if (!thumb_loader_start(dw->thumb_loader, dupe_thumb_done_cb, dw))
		{
		/* error, handle it, do next */
		if (debug) printf("error loading thumb for %s\n", di->path);
		dupe_thumb_do(dw);
		dupe_thumb_step(dw);
		}
}

static void dupe_check_stop(DupeWindow *dw)
{
	if (dw->idle_id != -1 || dw->img_loader || dw->thumb_loader)
		{
		gtk_idle_remove(dw->idle_id);
		dw->idle_id = -1;
		dupe_window_update_progress(dw, NULL, 0.0);
		gtk_clist_thaw(GTK_CLIST(dw->clist));
		}

	thumb_loader_free(dw->thumb_loader);
	dw->thumb_loader = NULL;

	image_loader_free(dw->img_loader);
	dw->img_loader = NULL;
}

static void dupe_loader_done_cb(ImageLoader *il, gpointer data)
{
	DupeWindow *dw = data;
	GdkPixbuf *pixbuf;

	pixbuf = image_loader_get_pixbuf(il);

	if (dw->setup_point)
		{
		DupeItem *di = dw->setup_point->data;

		if (!di->simd) di->simd = image_sim_new_from_pixbuf(pixbuf);
		if (di->width == 0 && di->height == 0)
			{
			di->width = gdk_pixbuf_get_width(pixbuf);
			di->height = gdk_pixbuf_get_height(pixbuf);
			}
		if (enable_thumb_caching)
			{
			dupe_item_write_cache(di);
			}
		}

	image_loader_free(dw->img_loader);
	dw->img_loader = NULL;

	dw->idle_id = gtk_idle_add(dupe_check_cb, dw);
}

static void dupe_setup_reset(DupeWindow *dw)
{
	dw->setup_point = NULL;
	dw->setup_n = 0;
	dw->setup_time = gdk_time_get();
	dw->setup_time_count = 0;
}

static GList *dupe_setup_point_step(DupeWindow *dw, GList *p)
{
	if (!p) return NULL;

	if (p->next) return p->next;

	if (dw->second_set && g_list_first(p) == dw->list) return dw->second_list;

	return NULL;
}

static gint dupe_check_cb(gpointer data)
{
	DupeWindow *dw = data;

	if (dw->idle_id == -1) return FALSE;

	if (!dw->setup_done)
		{
		if ((dw->match_mask & DUPE_MATCH_SUM) &&
		    !(dw->setup_mask & DUPE_MATCH_SUM) )
			{
			if (!dw->setup_point) dw->setup_point = dw->list;

			while(dw->setup_point)
				{
				DupeItem *di = dw->setup_point->data;

				dw->setup_point = dupe_setup_point_step(dw, dw->setup_point);
				dw->setup_n++;

				if (di->checksum == 0)
					{
					dupe_window_update_progress(dw, _("Reading checksums..."),
						dw->setup_count == 0 ? 0.0 : (float)dw->setup_n / dw->setup_count);

					if (enable_thumb_caching)
						{
						dupe_item_read_cache(di);
						if (di->checksum != 0) return TRUE;
						}

					di->checksum = checksum_simple(di->path);
					if (enable_thumb_caching)
						{
						dupe_item_write_cache(di);
						}
					return TRUE;
					}
				}
			dw->setup_mask |= DUPE_MATCH_SUM;
			dupe_setup_reset(dw);
			}
		if ((dw->match_mask & DUPE_MATCH_DIM) &&
		    !(dw->setup_mask & DUPE_MATCH_DIM) )
			{
			if (!dw->setup_point) dw->setup_point = dw->list;

			while(dw->setup_point)
				{
				DupeItem *di = dw->setup_point->data;

				dw->setup_point = dupe_setup_point_step(dw, dw->setup_point);
				if (di->width == 0 && di->height == 0)
					{
					dupe_window_update_progress(dw, _("Reading dimensions..."),
						dw->setup_count == 0 ? 0.0 : (float)dw->setup_n / dw->setup_count);

					if (enable_thumb_caching)
						{
						dupe_item_read_cache(di);
						if (di->width != 0 || di->height != 0) return TRUE;
						}

					image_load_dimensions(di->path, &di->width, &di->height);
					if (enable_thumb_caching)
						{
						dupe_item_write_cache(di);
						}
					return TRUE;
					}
				dw->setup_n++;
				}
			dw->setup_mask |= DUPE_MATCH_DIM;
			dupe_setup_reset(dw);
			}
		if ((dw->match_mask & DUPE_MATCH_SIM_HIGH ||
		     dw->match_mask & DUPE_MATCH_SIM_MED ||
		     dw->match_mask & DUPE_MATCH_SIM_LOW ||
		     dw->match_mask & DUPE_MATCH_SIM_CUSTOM) &&
		    !(dw->setup_mask & DUPE_MATCH_SIM_MED) )
			{
			if (!dw->setup_point) dw->setup_point = dw->list;

			while(dw->setup_point)
				{
				DupeItem *di = dw->setup_point->data;

				if (!di->simd)
					{
					dupe_window_update_progress(dw, _("Reading similarity data..."),
						dw->setup_count == 0 ? 0.0 : (float)dw->setup_n / dw->setup_count);

					if (enable_thumb_caching)
						{
						dupe_item_read_cache(di);
						if (di->simd) return TRUE;
						}

					dw->img_loader = image_loader_new(di->path);
					image_loader_set_buffer_size(dw->img_loader, 8);
					image_loader_set_error_func(dw->img_loader, dupe_loader_done_cb, dw);

					if (!image_loader_start(dw->img_loader, dupe_loader_done_cb, dw))
						{
						di->simd = image_sim_new();
						image_loader_free(dw->img_loader);
						dw->img_loader = NULL;
						return TRUE;
						}
					dw->idle_id = -1;
					return FALSE;
					}

				dw->setup_point = dupe_setup_point_step(dw, dw->setup_point);
				dw->setup_n++;
				}
			dw->setup_mask |= DUPE_MATCH_SIM_MED;
			dupe_setup_reset(dw);
			}
		dupe_window_update_progress(dw, _("Comparing..."), 0.0);
		dw->setup_done = TRUE;
		dupe_setup_reset(dw);
		dw->setup_count = g_list_length(dw->list);
		}

	if (!dw->working)
		{
		dw->idle_id = -1;
		dw->setup_count = 0;
		dupe_window_update_count(dw, FALSE);
		dupe_window_update_progress(dw, NULL, 0.0);
		gtk_clist_thaw(GTK_CLIST(dw->clist));

		/* check thumbs */
		if (dw->show_thumbs) dupe_thumb_step(dw);

		return FALSE;
		}

	dupe_list_check_match(dw, (DupeItem *)dw->working->data, dw->working);
	dupe_window_update_progress(dw, _("Comparing..."), dw->setup_count == 0 ? 0.0 : (float) dw->setup_n / dw->setup_count);
	dw->setup_n++;

	dw->working = dw->working->prev;

	return TRUE;
}

static void dupe_check_start(DupeWindow *dw)
{
	dw->setup_done = FALSE;

	dw->setup_count = g_list_length(dw->list);
	if (dw->second_set) dw->setup_count += g_list_length(dw->second_list);

	dw->setup_mask = 0;
	dupe_setup_reset(dw);

	dw->working = g_list_last(dw->list);

	dupe_window_update_count(dw, TRUE);

	if (dw->idle_id != -1) return;

	dw->idle_id = gtk_idle_add(dupe_check_cb, dw);
	gtk_clist_freeze(GTK_CLIST(dw->clist));
}

static void dupe_item_remove(DupeWindow *dw, DupeItem *di)
{
	DupeData *dd;

	if (!di) return;

	dd = dupe_data_find(dw, di);
	if (dd)	dupe_data_remove(dw, dd, di);

	/* handle things that may be in progress... */
	if (dw->working && dw->working->data == di)
		{
		dw->working = dw->working->prev;
		}
	if (dw->thumb_loader && dw->thumb_item == di)
		{
		dupe_thumb_step(dw);
		}
	if (dw->setup_point && dw->setup_point->data == di)
		{
		dw->setup_point = dupe_setup_point_step(dw, dw->setup_point);
		if (dw->img_loader)
			{
			image_loader_free(dw->img_loader);
			dw->img_loader = NULL;
			dw->idle_id = gtk_idle_add(dupe_check_cb, dw);
			}
		}

	if (dw->second_list && g_list_find(dw->second_list, di))
		{
		dupe_second_remove(dw, di);
		}
	else
		{
		dw->list = g_list_remove(dw->list, di);
		}
	dupe_item_free(di);

	dupe_window_update_count(dw, FALSE);
}

static gint dupe_item_remove_by_path(DupeWindow *dw, const gchar *path)
{
	DupeItem *di;

	di = dupe_item_find_path(dw, path);
	if (!di) return FALSE;

	dupe_item_remove(dw, di);

	return TRUE;
}

static gint dupe_item_set_path(DupeWindow *dw, const gchar *source, const gchar *dest)
{
	DupeItem *di;

	di = dupe_item_find_path(dw, source);
	if (!di) return FALSE;

	g_free(di->path);
	di->path = g_strdup(dest);
	di->name = filename_from_path(di->path);

	if ( (dw->match_mask & DUPE_MATCH_NAME) || (dw->match_mask & DUPE_MATCH_PATH) )
		{
		DupeData *dd;

		/* only effects matches on name or path */

		dd = dupe_data_find(dw, di);
		if (dd)	dupe_data_remove(dw, dd, di);

		/* re-compare (only this one) */
		dupe_list_check_match(dw, di, NULL);
		dupe_window_update_count(dw, FALSE);

		if ((dw->thumb_item == di && dw->thumb_loader) ||
		    (!dw->working && !dw->thumb_loader && !dw->img_loader && dw->show_thumbs)) dupe_thumb_step(dw);
		}
	else
		{
		gint row;

		/* only update the clist */

		row = gtk_clist_find_row_from_data(GTK_CLIST(dw->clist), di);
		if (row >= 0)
			{
			gtk_clist_set_text(GTK_CLIST(dw->clist), row, DUPE_COLUMN_NAME, di->name);
			gtk_clist_set_text(GTK_CLIST(dw->clist), row, DUPE_COLUMN_PATH, di->path);
			}
		}

	return TRUE;
}

static void dupe_files_add(DupeWindow *dw, CollectionData *collection, CollectInfo *info,
			   const gchar *path, gint recurse)
{
	DupeItem *di = NULL;

	if (info)
		{
		di = dupe_item_new(info->path, info->size, info->date);
		}
	else if (path)
		{
		if (isfile(path))
			{
			di = dupe_item_new(path, filesize(path), filetime(path));
			}
		else if (isdir(path) && recurse)
			{
			GList *f, *d;
			if (path_list(path, &f, &d))
				{
				GList *work;

				f = path_list_filter(f, FALSE);
				d = path_list_filter(d, TRUE);

				work = f;
				while(work)
					{
					dupe_files_add(dw, NULL, NULL, (gchar *)work->data, TRUE);
					work = work->next;
					}
				path_list_free(f);
				work = d;
				while(work)
					{
					dupe_files_add(dw, NULL, NULL, (gchar *)work->data, TRUE);
					work = work->next;
					}
				path_list_free(d);
				}
			}
		}

	if (!di) return;

	if (dw->second_drop)
		{
		dupe_second_add(dw, di);
		}
	else
		{
		dw->list = g_list_prepend(dw->list, di);
		}
}

void dupe_window_add_collection(DupeWindow *dw, CollectionData *collection)
{
	CollectInfo *info;

	if (dw->second_drop) gtk_clist_freeze(GTK_CLIST(dw->second_clist));

	info = collection_get_first(collection);
	while(info)
		{
		dupe_files_add(dw, collection, info, NULL, FALSE);
		info = collection_next_by_info(collection, info);
		}

	if (dw->second_drop) gtk_clist_thaw(GTK_CLIST(dw->second_clist));

	dupe_check_start(dw);
}

void dupe_window_add_files(DupeWindow *dw, GList *list, gint recurse)
{
	GList *work;

	if (dw->second_drop) gtk_clist_freeze(GTK_CLIST(dw->second_clist));

	work = list;
	while(work)
		{
		gchar *path = work->data;
		work = work->next;

		dupe_files_add(dw, NULL, NULL, path, recurse);
		}

	if (dw->second_drop) gtk_clist_thaw(GTK_CLIST(dw->second_clist));

	dupe_check_start(dw);
}

static void dupe_list_reset_dupes(GList *work)
{
	while(work)
		{
		DupeItem *di = work->data;
		di->dupe = FALSE;
		work = work->next;
		}
}

static void dupe_window_recompare(DupeWindow *dw)
{
	dupe_check_stop(dw);

	gtk_clist_clear(GTK_CLIST(dw->clist));

	dupe_list_free(dw->dupes);
	dw->dupes = NULL;

	dupe_list_reset_dupes(dw->list);
	dupe_list_reset_dupes(dw->second_list);

	dupe_check_start(dw);
}

#if 0
static const gchar *dupe_clist_get_path(DupeWindow *dw, gint row)
{
	DupeItem *di;

	di = gtk_clist_get_row_data(GTK_CLIST(dw->clist), row);

	if (di) return di->path;
	return NULL;
}
#endif

static GList *dupe_clist_get_selected(DupeWindow *dw, GtkWidget *clist)
{
	GList *work;
	GList *list = NULL;

	work = GTK_CLIST(clist)->selection;
	while(work)
		{
		DupeItem *di;

		di = gtk_clist_get_row_data(GTK_CLIST(clist), GPOINTER_TO_INT(work->data));
		if (di)
			{
			list = g_list_append(list, g_strdup(di->path));
			}
		work = work->next;
		}

	return list;
}

static gint dupe_clist_row_is_selected(DupeWindow *dw, gint row, GtkWidget *clist)
{
	GList *work;

	work = GTK_CLIST(clist)->selection;
	while(work)
		{
		gint n = GPOINTER_TO_INT(work->data);
		if (n == row) return TRUE;
		work = work->next;
		}

	return FALSE;
}

static void dupe_menu_view(DupeWindow *dw, DupeItem *di, gint new_window)
{
	if (!di) return;

	if (di->collection && collection_info_valid(di->collection, di->info))
		{
		if (new_window)
			{
			view_window_new_from_collection(di->collection, di->info);
			}
		else
			{
			layout_image_set_collection(NULL, di->collection, di->info);
			}
		}
	else
		{
		if (new_window)
			{
			view_window_new(di->path);
			}
		else
			{
			layout_image_set_path(NULL, di->path);
			}
		}
}

static void dupe_window_remove_selected(DupeWindow *dw, GtkWidget *clist)
{
	GList *work;
	GList *list = NULL;

	work = GTK_CLIST(clist)->selection;
	if (!work) return;

	gtk_clist_freeze(GTK_CLIST(clist));

	while (work)
		{
		DupeItem *di;
		gint row;

		row = GPOINTER_TO_INT(work->data);
		work = work->next;

		di = gtk_clist_get_row_data(GTK_CLIST(clist), row);
		list = g_list_prepend(list, di);
		}

	work = list;
	while (work)
		{
		DupeItem *di;

		di = work->data;
		work = work->next;
		dupe_item_remove(dw, di);
		}

	g_list_free(list);

	/* force a style realignment */
	dupe_clist_realign_styles(dw);

	gtk_clist_thaw(GTK_CLIST(clist));
}

static void dupe_window_edit_selected(DupeWindow *dw, gint n)
{
	GList *list;

	list = dupe_clist_get_selected(dw, dw->clist);

	start_editor_from_path_list(n, list);

	path_list_free(list);
}

static void dupe_window_collection_from_selected(DupeWindow *dw)
{
	CollectWindow *w;
	GList *list;

	list = dupe_clist_get_selected(dw, dw->clist);
	w = collection_window_new(NULL);
	collection_table_add_path_list(w->table, list);
	path_list_free(list);
}

static void dupe_window_append_file_list(DupeWindow *dw, gint on_second)
{
	GList *list;

	dw->second_drop = (dw->second_set && on_second);

	list = layout_list(NULL);
	dupe_window_add_files(dw, list, FALSE);
	path_list_free(list);
}

/*
 *-------------------------------------------------------------------
 * main pop-up menu callbacks
 *-------------------------------------------------------------------
 */

static void dupe_menu_view_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	if (dw->click_row < 0) return;
	dupe_menu_view(dw, gtk_clist_get_row_data(GTK_CLIST(dw->clist), dw->click_row), FALSE);
}

static void dupe_menu_viewnew_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	if (dw->click_row < 0) return;
	dupe_menu_view(dw, gtk_clist_get_row_data(GTK_CLIST(dw->clist), dw->click_row), TRUE);
}

static void dupe_menu_select_all_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	gtk_clist_select_all(GTK_CLIST(dw->clist));
}

static void dupe_menu_select_none_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	gtk_clist_unselect_all(GTK_CLIST(dw->clist));
}

static void dupe_menu_edit_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw;
	gint n;

	dw = submenu_item_get_data(widget);
	n = GPOINTER_TO_INT(data);
	if (!dw) return;

	dupe_window_edit_selected(dw, n);
}

static void dupe_menu_info_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	info_window_new(NULL, dupe_clist_get_selected(dw, dw->clist));
}

static void dupe_menu_collection_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	dupe_window_collection_from_selected(dw);
}

static void dupe_menu_copy_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	file_util_copy(NULL, dupe_clist_get_selected(dw, dw->clist), NULL);
}

static void dupe_menu_move_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	file_util_move(NULL, dupe_clist_get_selected(dw, dw->clist), NULL);
}

static void dupe_menu_rename_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	file_util_rename(NULL, dupe_clist_get_selected(dw, dw->clist));
}

static void dupe_menu_delete_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	file_util_delete(NULL, dupe_clist_get_selected(dw, dw->clist));
}

static void dupe_menu_remove_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	dupe_window_remove_selected(dw, dw->clist);
}

static void dupe_menu_clear_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	dupe_window_clear(dw);
}

static void dupe_menu_close_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	dupe_window_close(dw);
}

static gint dupe_clist_press_cb(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	DupeWindow *dw = data;

	gint row = -1;
	gint col = -1;

	gtk_clist_get_selection_info(GTK_CLIST(dw->clist), bevent->x, bevent->y, &row, &col);

	if (bevent->button == 3)
		{
		/* right click menu */
		GtkWidget *menu;
		GtkWidget *item;

		menu = popup_menu_short_lived();
		item = menu_item_add(menu, _("View"), GTK_SIGNAL_FUNC(dupe_menu_view_cb), dw);
		if (row < 0) gtk_widget_set_sensitive(item, FALSE);
		item = menu_item_add(menu, _("View in new window"), GTK_SIGNAL_FUNC(dupe_menu_viewnew_cb), dw);
		if (row < 0) gtk_widget_set_sensitive(item, FALSE);
		menu_item_add_divider(menu);
		item = menu_item_add(menu, _("Select all"), GTK_SIGNAL_FUNC(dupe_menu_select_all_cb), dw);
		if (dw->dupes == NULL) gtk_widget_set_sensitive(item, FALSE);
		item = menu_item_add(menu, _("Select none"), GTK_SIGNAL_FUNC(dupe_menu_select_none_cb), dw);
		if (dw->dupes == NULL) gtk_widget_set_sensitive(item, FALSE);
		menu_item_add_divider(menu);
		submenu_add_edit(menu, &item, GTK_SIGNAL_FUNC(dupe_menu_edit_cb), dw);
		if (row < 0) gtk_widget_set_sensitive(item, FALSE);
		item = menu_item_add(menu, _("Properties"), GTK_SIGNAL_FUNC(dupe_menu_info_cb), dw);
		if (row < 0) gtk_widget_set_sensitive(item, FALSE);
		item = menu_item_add(menu, _("Add to new collection"), GTK_SIGNAL_FUNC(dupe_menu_collection_cb), dw);
		if (row < 0) gtk_widget_set_sensitive(item, FALSE);
		menu_item_add_divider(menu);
		item = menu_item_add(menu, _("Copy..."), GTK_SIGNAL_FUNC(dupe_menu_copy_cb), dw);
		if (row < 0) gtk_widget_set_sensitive(item, FALSE);
		item = menu_item_add(menu, _("Move..."), GTK_SIGNAL_FUNC(dupe_menu_move_cb), dw);
		if (row < 0) gtk_widget_set_sensitive(item, FALSE);
		item = menu_item_add(menu, _("Rename..."), GTK_SIGNAL_FUNC(dupe_menu_rename_cb), dw);
		if (row < 0) gtk_widget_set_sensitive(item, FALSE);
		item = menu_item_add(menu, _("Delete..."), GTK_SIGNAL_FUNC(dupe_menu_delete_cb), dw);
		if (row < 0) gtk_widget_set_sensitive(item, FALSE);
		menu_item_add_divider(menu);
		item = menu_item_add(menu, _("Remove"), GTK_SIGNAL_FUNC(dupe_menu_remove_cb), dw);
		if (row < 0) gtk_widget_set_sensitive(item, FALSE);
		item = menu_item_add(menu, _("Clear"), GTK_SIGNAL_FUNC(dupe_menu_clear_cb), dw);
		if (dw->list == NULL) gtk_widget_set_sensitive(item, FALSE);
		menu_item_add_divider(menu);
		menu_item_add(menu, _("Close window"), GTK_SIGNAL_FUNC(dupe_menu_close_cb), dw);

		gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, bevent->button, bevent->time);
		}

	if (row < 0 || col < 0) return FALSE;

	dw->click_row = row;

	if (bevent->button == 1 && bevent->type == GDK_2BUTTON_PRESS)
		{
		dupe_menu_view(dw, gtk_clist_get_row_data(GTK_CLIST(dw->clist), row), FALSE);
		}

	if (bevent->button == 2 || bevent->button == 3)
		{
		if (!dupe_clist_row_is_selected(dw, row, dw->clist))
			{
			gtk_clist_unselect_all(GTK_CLIST(dw->clist));
			gtk_clist_select_row(GTK_CLIST(dw->clist), row, col);
			}
		}

	return FALSE;
}

/*
 *-------------------------------------------------------------------
 * second set stuff
 *-------------------------------------------------------------------
 */

static void dupe_second_update_status(DupeWindow *dw)
{
	gchar *buf;

	buf = g_strdup_printf(_("%d files"), g_list_length(dw->second_list));
	gtk_label_set(GTK_LABEL(dw->second_status_label), buf);
	g_free(buf);
}

static void dupe_second_add(DupeWindow *dw, DupeItem *di)
{
	gint row;
	gchar *buf[2];

	if (!di) return;

	dw->second_list = g_list_prepend(dw->second_list, di);

	buf[0] = di->path;
	buf[1] = NULL;

	row = gtk_clist_append(GTK_CLIST(dw->second_clist), buf);
	gtk_clist_set_row_data(GTK_CLIST(dw->second_clist), row, di);

	dupe_second_update_status(dw);
}

static void dupe_second_remove(DupeWindow *dw, DupeItem *di)
{
	gint row;

	row = gtk_clist_find_row_from_data(GTK_CLIST(dw->second_clist), di);
	if (row >= 0) gtk_clist_remove(GTK_CLIST(dw->second_clist), row);

	dw->second_list = g_list_remove(dw->second_list, di);

	dupe_second_update_status(dw);
}

static void dupe_second_clear(DupeWindow *dw)
{
	dupe_list_free(dw->dupes);
	dw->dupes = NULL;

	gtk_clist_clear(GTK_CLIST(dw->second_clist));

	dupe_files_free(dw->second_list);
	dw->second_list = NULL;

	dupe_second_update_status(dw);
}

static void dupe_second_menu_view_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	if (dw->click_row < 0) return;
	dupe_menu_view(dw, gtk_clist_get_row_data(GTK_CLIST(dw->second_clist), dw->click_row), FALSE);
}

static void dupe_second_menu_viewnew_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	if (dw->click_row < 0) return;
	dupe_menu_view(dw, gtk_clist_get_row_data(GTK_CLIST(dw->second_clist), dw->click_row), TRUE);
}

static void dupe_second_menu_select_all_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	gtk_clist_select_all(GTK_CLIST(dw->second_clist));
}

static void dupe_second_menu_select_none_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	gtk_clist_unselect_all(GTK_CLIST(dw->second_clist));
}

static void dupe_second_menu_remove_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	dupe_window_remove_selected(dw, dw->second_clist);
}

static void dupe_second_menu_clear_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	dupe_second_clear(dw);
	dupe_window_recompare(dw);
}

static gint dupe_second_clist_press_cb(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	DupeWindow *dw = data;

	gint row = -1;
	gint col = -1;

	gtk_clist_get_selection_info(GTK_CLIST(dw->second_clist), bevent->x, bevent->y, &row, &col);

	if (bevent->button == 3)
		{
		/* right click menu */
		GtkWidget *menu;
		GtkWidget *item;

		menu = popup_menu_short_lived();
		item = menu_item_add(menu, _("View"), GTK_SIGNAL_FUNC(dupe_second_menu_view_cb), dw);
		if (row < 0) gtk_widget_set_sensitive(item, FALSE);
		item = menu_item_add(menu, _("View in new window"), GTK_SIGNAL_FUNC(dupe_second_menu_viewnew_cb), dw);
		if (row < 0) gtk_widget_set_sensitive(item, FALSE);
		menu_item_add_divider(menu);
		item = menu_item_add(menu, _("Select all"), GTK_SIGNAL_FUNC(dupe_second_menu_select_all_cb), dw);
		if (dw->second_list == NULL) gtk_widget_set_sensitive(item, FALSE);
		item = menu_item_add(menu, _("Select none"), GTK_SIGNAL_FUNC(dupe_second_menu_select_none_cb), dw);
		if (dw->second_list == NULL) gtk_widget_set_sensitive(item, FALSE);
		menu_item_add_divider(menu);
		item = menu_item_add(menu, _("Remove"), GTK_SIGNAL_FUNC(dupe_second_menu_remove_cb), dw);
		if (row < 0) gtk_widget_set_sensitive(item, FALSE);
		item = menu_item_add(menu, _("Clear"), GTK_SIGNAL_FUNC(dupe_second_menu_clear_cb), dw);
		if (dw->second_list == NULL) gtk_widget_set_sensitive(item, FALSE);
		menu_item_add_divider(menu);
		menu_item_add(menu, _("Close window"), GTK_SIGNAL_FUNC(dupe_menu_close_cb), dw);

		gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, bevent->button, bevent->time);
		}

	if (row < 0 || col < 0) return FALSE;

	dw->click_row = row;

	if (bevent->button == 1 && bevent->type == GDK_2BUTTON_PRESS)
		{
		dupe_menu_view(dw, gtk_clist_get_row_data(GTK_CLIST(dw->second_clist), row), FALSE);
		}

	if (bevent->button == 2 || bevent->button == 3)
		{
		if (!dupe_clist_row_is_selected(dw, row, dw->second_clist))
			{
			gtk_clist_unselect_all(GTK_CLIST(dw->second_clist));
			gtk_clist_select_row(GTK_CLIST(dw->second_clist), row, col);
			}
		}

	return FALSE;
}

static void dupe_second_set_toggle_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	dw->second_set = GTK_TOGGLE_BUTTON(widget)->active;

	if (dw->second_set)
		{
		dupe_second_update_status(dw);
		gtk_table_set_col_spacings(GTK_TABLE(dw->table), 5);
		gtk_widget_show(dw->second_vbox);
		}
	else
		{
		gtk_table_set_col_spacings(GTK_TABLE(dw->table), 0);
		gtk_widget_hide(dw->second_vbox);
		dupe_second_clear(dw);
		}

	dupe_window_recompare(dw);
}

/*
 *-------------------------------------------------------------------
 * match type menu
 *-------------------------------------------------------------------
 */

static void dupe_menu_type_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	dw->match_mask = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(widget), "dupe_type"));

	dupe_window_recompare(dw);
}

static void dupe_menu_add_item(GtkWidget *menu, const gchar *text, DupeMatchType type, DupeWindow *dw)
{
	GtkWidget *item;

	item = gtk_menu_item_new_with_label(text);
	gtk_object_set_data(GTK_OBJECT(item), "dupe_type", GINT_TO_POINTER(type));
	gtk_signal_connect(GTK_OBJECT(item), "activate",
			   GTK_SIGNAL_FUNC(dupe_menu_type_cb), dw);
	gtk_menu_append(GTK_MENU(menu), item);
	gtk_widget_show(item);
}

static void dupe_menu_setup(DupeWindow *dw)
{
	GtkWidget *menu;
	gint n;

	menu = gtk_menu_new();

	dupe_menu_add_item(menu, _("Name"), DUPE_MATCH_NAME, dw);
	dupe_menu_add_item(menu, _("Size"), DUPE_MATCH_SIZE, dw);
	dupe_menu_add_item(menu, _("Date"), DUPE_MATCH_DATE, dw);
	dupe_menu_add_item(menu, _("Dimensions"), DUPE_MATCH_DIM, dw);
	dupe_menu_add_item(menu, _("Checksum"), DUPE_MATCH_SUM, dw);
	dupe_menu_add_item(menu, _("Path"), DUPE_MATCH_PATH, dw);
	dupe_menu_add_item(menu, _("Similarity (high)"), DUPE_MATCH_SIM_HIGH, dw);
	dupe_menu_add_item(menu, _("Similarity"), DUPE_MATCH_SIM_MED, dw);
	dupe_menu_add_item(menu, _("Similarity (low)"), DUPE_MATCH_SIM_LOW, dw);
	dupe_menu_add_item(menu, _("Similarity (custom)"), DUPE_MATCH_SIM_CUSTOM, dw);

	gtk_option_menu_set_menu(GTK_OPTION_MENU(dw->dupe_omenu), menu);

	switch (dw->match_mask)
		{
		case DUPE_MATCH_NAME:
			n = 0;
			break;
		case DUPE_MATCH_SIZE:
			n = 1;
			break;
		case DUPE_MATCH_DATE:
			n = 2;
			break;
		case DUPE_MATCH_DIM:
			n = 3;
			break;
		case DUPE_MATCH_SUM:
			n = 4;
			break;
		case DUPE_MATCH_PATH:
			n = 5;
			break;
		case DUPE_MATCH_SIM_HIGH:
			n = 6;
			break;
		case DUPE_MATCH_SIM_MED:
			n = 7;
			break;
		case DUPE_MATCH_SIM_LOW:
			n = 8;
			break;
		case DUPE_MATCH_SIM_CUSTOM:
			n = 9;
			break;
		default:
			n = 0;
			break;
		}

	gtk_option_menu_set_history(GTK_OPTION_MENU(dw->dupe_omenu), n);
}

/*
 *-------------------------------------------------------------------
 * misc cb
 *-------------------------------------------------------------------
 */

static void dupe_window_show_thumb_cb(GtkWidget *widget, gpointer data)
{
	DupeWindow *dw = data;

	dw->show_thumbs = GTK_TOGGLE_BUTTON(widget)->active;

	if (dw->show_thumbs)
		{
		if (!dw->working) dupe_thumb_step(dw);
		}
	else
		{
		gint i;
		gint length;

		thumb_loader_free(dw->thumb_loader);
		dw->thumb_loader = NULL;

		gtk_clist_freeze(GTK_CLIST(dw->clist));
		length = GTK_CLIST(dw->clist)->rows;
		for (i = 0; i < length; i++)
			{
			gtk_clist_set_text(GTK_CLIST(dw->clist), i, DUPE_COLUMN_THUMB, "");
			}
		gtk_clist_thaw(GTK_CLIST(dw->clist));
		dupe_window_update_progress(dw, NULL, 0.0);
		}

	gtk_clist_set_row_height(GTK_CLIST(dw->clist), dw->show_thumbs ? thumb_max_height : 0);
}

static gint dupe_window_keypress_cb(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	DupeWindow *dw = data;
	gint stop_signal = FALSE;
	gint row = -1;
	gint on_second;
	GtkWidget *clist;
	DupeItem *di = NULL;

	on_second = GTK_WIDGET_HAS_FOCUS(dw->second_clist);

	if (on_second)
		{
		clist = dw->second_clist;
		}
	else
		{
		clist = dw->clist;
		}

	if (GTK_CLIST(dw->clist)->selection)
		{
		GList *last = g_list_last(GTK_CLIST(clist)->selection);

		/* last is newest selected file */
		if (last) row = GPOINTER_TO_INT(last->data);
		di = gtk_clist_get_row_data(GTK_CLIST(clist), row);
		}

	if (event->state & GDK_CONTROL_MASK)
		{
		gint edit_val = -1;

		if (!on_second)
		    switch (event->keyval)
			{
			case '1':
				edit_val = 0;
				break;
			case '2':
				edit_val = 1;
				break;
			case '3':
				edit_val = 2;
				break;
			case '4':
				edit_val = 3;
				break;
			case '5':
				edit_val = 4;
				break;
			case '6':
				edit_val = 5;
				break;
			case '7':
				edit_val = 6;
				break;
			case '8':
				edit_val = 7;
				break;
			case 'C': case 'c':
				stop_signal = TRUE;
				file_util_copy(NULL, dupe_clist_get_selected(dw, dw->clist), NULL);
				break;
			case 'M': case 'm':
				file_util_move(NULL, dupe_clist_get_selected(dw, dw->clist), NULL);
				stop_signal = TRUE;
				break;
			case 'R': case 'r':
				file_util_rename(NULL, dupe_clist_get_selected(dw, dw->clist));
				stop_signal = TRUE;
				break;
			case 'D': case 'd':
				file_util_delete(NULL, dupe_clist_get_selected(dw, dw->clist));
				stop_signal = TRUE;
				break;
			default:
				break;
			}

		switch (event->keyval)
			{
			case 'A': case 'a':
				if (event->state & GDK_SHIFT_MASK)
					{
					gtk_clist_unselect_all(GTK_CLIST(clist));
					}
				else
					{
					gtk_clist_select_all(GTK_CLIST(clist));
					}
				stop_signal = TRUE;
				break;
			case GDK_Delete: case GDK_KP_Delete:
				if (on_second)
					{
					dupe_second_clear(dw);
					dupe_window_recompare(dw);
					}
				else
					{
					dupe_window_clear(dw);
					}
				stop_signal = TRUE;
				break;
			case 'L': case 'l':
				dupe_window_append_file_list(dw, FALSE);
				stop_signal = TRUE;
				break;
			case 'W': case 'w':
				dupe_window_close(dw);
				stop_signal = TRUE;
				break;
			default:
				break;
			}

		if (edit_val >= 0)
			{
			dupe_window_edit_selected(dw, edit_val);
			stop_signal = TRUE;
			}
		}
	else
		{
		switch (event->keyval)
			{
			case GDK_Return: case GDK_KP_Enter:
				dupe_menu_view(dw, di, FALSE);
				stop_signal = TRUE;
				break;
			case 'V': case 'v':
				stop_signal = TRUE;
				dupe_menu_view(dw, di, TRUE);
				break;
			case GDK_Delete: case GDK_KP_Delete:
				dupe_window_remove_selected(dw, clist);
				stop_signal = TRUE;
				break;
			case 'C': case 'c':
				if (!on_second)
					{
					dupe_window_collection_from_selected(dw);
					stop_signal = TRUE;
					}
				break;
			default:
				break;
			}
		}

	return stop_signal;
}


void dupe_window_clear(DupeWindow *dw)
{
	dupe_check_stop(dw);
	gtk_clist_clear(GTK_CLIST(dw->clist));

	dupe_list_free(dw->dupes);
	dw->dupes = NULL;
	dupe_files_free(dw->list);
	dw->list = NULL;

	dupe_window_update_count(dw, FALSE);
	dupe_window_update_progress(dw, NULL, 0.0);
}

void dupe_window_close(DupeWindow *dw)
{
	dupe_check_stop(dw);

	dupe_window_list = g_list_remove(dupe_window_list, dw);
	gtk_widget_destroy(dw->window);

	dupe_list_free(dw->dupes);
	dupe_files_free(dw->list);

	dupe_files_free(dw->second_list);

	g_free(dw);
}

static gint dupe_window_delete(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	DupeWindow *dw = data;
	dupe_window_close(dw);

	return TRUE;
}

/* collection and files can be NULL */
DupeWindow *dupe_window_new(DupeMatchType match_mask)
{
	DupeWindow *dw;
	GtkWidget *vbox;
	GtkWidget *scrolled;
	GtkWidget *frame;
	GtkWidget *status_box;
	GtkWidget *label;
	GtkWidget *button;
	GtkWidget *sep;
	gchar *title[] = {
		"",
		"",
		_("Name"),
		_("Size"),
		_("Date"),
		_("Dimensions"),
		_("Path"),
		NULL
		};
	gchar *second_title[] = { _("Compare to:"), NULL };
	gint i;

	dw = g_new0(DupeWindow, 1);

	dw->list = NULL;
	dw->dupes = NULL;
	dw->match_mask = match_mask;
	dw->show_thumbs = FALSE;

	dw->idle_id = -1;

	dw->second_set = FALSE;

	dw->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
        window_set_icon(dw->window, NULL, NULL);

	gtk_window_set_default_size(GTK_WINDOW(dw->window), DUPE_DEF_WIDTH, DUPE_DEF_HEIGHT);

	gtk_window_set_policy(GTK_WINDOW(dw->window), TRUE, TRUE, FALSE);
	gtk_window_set_title(GTK_WINDOW(dw->window), _("Find duplicates - GQview"));
        gtk_window_set_wmclass(GTK_WINDOW(dw->window), "dupe", "GQview");
        gtk_container_border_width (GTK_CONTAINER (dw->window), 0);

        gtk_signal_connect(GTK_OBJECT(dw->window), "delete_event",
			   GTK_SIGNAL_FUNC(dupe_window_delete), dw);
	gtk_signal_connect(GTK_OBJECT(dw->window), "key_press_event",
			   GTK_SIGNAL_FUNC(dupe_window_keypress_cb), dw);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(dw->window), vbox);
	gtk_widget_show(vbox);

	dw->table = gtk_table_new(1, 3, FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), dw->table, TRUE, TRUE, 0);
	gtk_widget_show(dw->table);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_table_attach_defaults(GTK_TABLE(dw->table), scrolled, 0, 2, 0, 1);
        gtk_widget_show(scrolled);

	dw->clist = gtk_clist_new_with_titles(DUPE_COLUMN_COUNT, title);
	gtk_clist_set_selection_mode(GTK_CLIST(dw->clist), GTK_SELECTION_EXTENDED);
	
	for (i = 0; i < DUPE_COLUMN_COUNT; i++)
		{
		gtk_clist_set_column_auto_resize(GTK_CLIST(dw->clist),  i, TRUE);
		}
	gtk_clist_set_column_justification(GTK_CLIST(dw->clist), DUPE_COLUMN_SIZE, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification(GTK_CLIST(dw->clist), DUPE_COLUMN_DATE, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification(GTK_CLIST(dw->clist), DUPE_COLUMN_DIMENSIONS, GTK_JUSTIFY_CENTER);
	gtk_signal_connect(GTK_OBJECT(dw->clist), "button_press_event",
			   GTK_SIGNAL_FUNC(dupe_clist_press_cb), dw);
	gtk_container_add(GTK_CONTAINER(scrolled), dw->clist);
	if (dw->show_thumbs) gtk_clist_set_row_height(GTK_CLIST(dw->clist), thumb_max_height);
        gtk_widget_show(dw->clist);

	dw->second_vbox = gtk_vbox_new(FALSE, 0);
	gtk_table_attach_defaults(GTK_TABLE(dw->table), dw->second_vbox, 2, 3, 0, 1);
	if (dw->second_set)
		{
		gtk_table_set_col_spacings(GTK_TABLE(dw->table), 5);
		gtk_widget_show(dw->second_vbox);
		}
	else
		{
		gtk_table_set_col_spacings(GTK_TABLE(dw->table), 0);
		}

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(dw->second_vbox), scrolled, TRUE, TRUE, 0);
        gtk_widget_show(scrolled);

	dw->second_clist = gtk_clist_new_with_titles(1, second_title);
	gtk_clist_set_selection_mode(GTK_CLIST(dw->second_clist), GTK_SELECTION_EXTENDED);
	
	for (i = 0; i < 1; i++)
		{
		gtk_clist_set_column_auto_resize(GTK_CLIST(dw->second_clist),  i, TRUE);
		}
	gtk_signal_connect(GTK_OBJECT(dw->second_clist), "button_press_event",
			   GTK_SIGNAL_FUNC(dupe_second_clist_press_cb), dw);
	gtk_container_add(GTK_CONTAINER(scrolled), dw->second_clist);
        gtk_widget_show(dw->second_clist);

	dw->second_status_label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(dw->second_vbox), dw->second_status_label, FALSE, FALSE, 0);
	gtk_widget_show(dw->second_status_label);

	sep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(dw->second_vbox), sep, FALSE, FALSE, 0);
	gtk_widget_show(sep);

	status_box = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(vbox), status_box, FALSE, FALSE, 0);
        gtk_widget_show(status_box);

	label = gtk_label_new(_("Compare by:"));
	gtk_box_pack_start(GTK_BOX(status_box), label, FALSE, FALSE, 5);
	gtk_widget_show(label);

	dw->dupe_omenu = gtk_option_menu_new();
	dupe_menu_setup(dw);
	gtk_box_pack_start(GTK_BOX(status_box), dw->dupe_omenu, FALSE, FALSE, 0);
	gtk_widget_show(dw->dupe_omenu);

	button = gtk_check_button_new_with_label(_("Thumbnails"));
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), dw->show_thumbs);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(dupe_window_show_thumb_cb), dw);
	gtk_box_pack_start(GTK_BOX(status_box), button, FALSE, FALSE, 5);
	gtk_widget_show(button);

	button = gtk_check_button_new_with_label(_("Compare two file sets"));
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), dw->second_set);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			   GTK_SIGNAL_FUNC(dupe_second_set_toggle_cb), dw);
	gtk_box_pack_end(GTK_BOX(status_box), button, FALSE, FALSE, 0);
	gtk_widget_show(button);

	status_box = gtk_hbox_new(FALSE, 0);
        gtk_box_pack_start(GTK_BOX(vbox), status_box, FALSE, FALSE, 0);
        gtk_widget_show(status_box);

	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(status_box), frame, TRUE, TRUE, 0);
	gtk_widget_show(frame);

	dw->status_label = gtk_label_new("");
	gtk_container_add(GTK_CONTAINER(frame), dw->status_label);
	gtk_widget_show(dw->status_label);

	dw->extra_label = gtk_progress_bar_new();
	gtk_progress_set_show_text(GTK_PROGRESS(dw->extra_label), TRUE);
	gtk_progress_configure(GTK_PROGRESS(dw->extra_label), 0.0, 0.0, 1.0);
	gtk_box_pack_end(GTK_BOX(status_box), dw->extra_label, FALSE, FALSE, 0);
	gtk_widget_show(dw->extra_label);

	dupe_dnd_init(dw);
	gtk_widget_show(dw->window);

	dupe_window_update_count(dw, TRUE);
	dupe_window_update_progress(dw, NULL, 0.0);

	dupe_window_list = g_list_append(dupe_window_list, dw);

	return dw;
}

/*
 *-------------------------------------------------------------------
 * dnd confirm dir
 *-------------------------------------------------------------------
 */

typedef struct {
	DupeWindow *dw;
	GList *list;
} CDupeConfirmD;

static void confirm_dir_list_cancel(GtkWidget *widget, gpointer data)
{
	/* do nothing */
}

static void confirm_dir_list_add(GtkWidget *widget, gpointer data)
{
	CDupeConfirmD *d = data;
	GList *work;

	dupe_window_add_files(d->dw, d->list, FALSE);

	work = d->list;
	while (work)
		{
		gchar *path = work->data;
		work = work->next;
		if (isdir(path))
			{
			GList *list = NULL;

			path_list(path, &list, NULL);
			list = path_list_filter(list, FALSE);
			if (list)
				{
				dupe_window_add_files(d->dw, list, FALSE);
				path_list_free(list);
				}
			}
		}
}

static void confirm_dir_list_recurse(GtkWidget *widget, gpointer data)
{
	CDupeConfirmD *d = data;
	dupe_window_add_files(d->dw, d->list, TRUE);
}

static void confirm_dir_list_skip(GtkWidget *widget, gpointer data)
{
	CDupeConfirmD *d = data;
	dupe_window_add_files(d->dw, d->list, FALSE);
}

static void confirm_dir_list_destroy(GtkWidget *widget, gpointer data)
{
	CDupeConfirmD *d = data;
	path_list_free(d->list);
	g_free(d);
}

static GtkWidget *dupe_confirm_dir_list(DupeWindow *dw, GList *list)
{
	GtkWidget *menu;
	CDupeConfirmD *d;

	d = g_new(CDupeConfirmD, 1);
	d->dw = dw;
	d->list = list;

	menu = popup_menu_short_lived();
	gtk_signal_connect(GTK_OBJECT(menu), "destroy",
			   GTK_SIGNAL_FUNC(confirm_dir_list_destroy), d);

	menu_item_add(menu, _("Dropped list includes directories."), NULL, NULL);
	menu_item_add_divider(menu);
	menu_item_add(menu, _("Add contents"), GTK_SIGNAL_FUNC(confirm_dir_list_add), d);
	menu_item_add(menu, _("Add contents recursive"), GTK_SIGNAL_FUNC(confirm_dir_list_recurse), d);
	menu_item_add(menu, _("Skip directories"), GTK_SIGNAL_FUNC(confirm_dir_list_skip), d);
	menu_item_add_divider(menu);
	menu_item_add(menu, _("Cancel"), GTK_SIGNAL_FUNC(confirm_dir_list_cancel), d);

	return menu;
}

/*
 *-------------------------------------------------------------------
 * dnd
 *-------------------------------------------------------------------
 */

static GtkTargetEntry dupe_drag_types[] = {
        { "text/uri-list", 0, TARGET_URI_LIST },
        { "text/plain", 0, TARGET_TEXT_PLAIN }
};
static gint n_dupe_drag_types = 2;

static GtkTargetEntry dupe_drop_types[] = {
        { "application/x-gqview-collection-member", 0, TARGET_APP_COLLECTION_MEMBER },
        { "text/uri-list", 0, TARGET_URI_LIST }
};
static gint n_dupe_drop_types = 2;

static void dupe_dnd_data_set(GtkWidget *widget, GdkDragContext *context,
			      GtkSelectionData *selection_data, guint info,
			      guint time, gpointer data)
{
	DupeWindow *dw = data;
	gchar *uri_text;
	gint length;
	GList *list;

	switch (info)
		{
		case TARGET_URI_LIST:
		case TARGET_TEXT_PLAIN:
			list = dupe_clist_get_selected(dw, dw->clist);
			if (!list) return;
			uri_text = make_uri_file_list(list, info == TARGET_TEXT_PLAIN, &length);
			path_list_free(list);
			break;
		default:
			uri_text = NULL;
			break;
		}

	if (uri_text) gtk_selection_data_set(selection_data, selection_data->target,
					     8, uri_text, length);
	g_free(uri_text);
}

static void dupe_dnd_data_get(GtkWidget *widget, GdkDragContext *context,
			      gint x, gint y,
			      GtkSelectionData *selection_data, guint info,
			      guint time, gpointer data)
{
	DupeWindow *dw = data;
	GList *list = NULL;
	GList *work;

	if (gtk_drag_get_source_widget(context) == dw->clist) return;

	dw->second_drop = (dw->second_set && widget == dw->second_clist);

	switch (info)
		{
		case TARGET_APP_COLLECTION_MEMBER:
			collection_from_dnd_data((gchar *)selection_data->data, &list, NULL);
			break;
		case TARGET_URI_LIST:
			list = get_uri_file_list(selection_data->data);
			work = list;
			while(work)
				{
				if (isdir((gchar *)work->data))
					{
					GtkWidget *menu;
					menu = dupe_confirm_dir_list(dw, list);
					gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, time);
					return;
					}
				work = work->next;
				}
			break;
		default:
			list = NULL;
			break;
		}

	if (list)
		{
		dupe_window_add_files(dw, list, FALSE);
		path_list_free(list);
		}
}

static void dupe_dnd_init(DupeWindow *dw)
{
	gtk_drag_source_set(dw->clist, GDK_BUTTON2_MASK,
			    dupe_drag_types, n_dupe_drag_types,
			    GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK);

	gtk_signal_connect(GTK_OBJECT(dw->clist), "drag_data_get",
			   GTK_SIGNAL_FUNC(dupe_dnd_data_set), dw);

	gtk_drag_dest_set(dw->clist,
			  GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP,
			  dupe_drop_types, n_dupe_drop_types,
			  GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_ASK);
	gtk_signal_connect(GTK_OBJECT(dw->clist), "drag_data_received",
			   GTK_SIGNAL_FUNC(dupe_dnd_data_get), dw);

	gtk_drag_dest_set(dw->second_clist,
			  GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP,
			  dupe_drop_types, n_dupe_drop_types,
			  GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_ASK);
	gtk_signal_connect(GTK_OBJECT(dw->second_clist), "drag_data_received",
			   GTK_SIGNAL_FUNC(dupe_dnd_data_get), dw);
}

/*
 *-------------------------------------------------------------------
 * maintenance (move, delete, etc.)
 *-------------------------------------------------------------------
 */

void dupe_maint_removed(const gchar *path)
{
	GList *work;

	work = dupe_window_list;
	while (work)
		{
		DupeWindow *dw = work->data;
		work = work->next;

		while (dupe_item_remove_by_path(dw, path));
		}
}

void dupe_maint_renamed(const gchar *source, const gchar *dest)
{
	GList *work;

	work = dupe_window_list;
	while (work)
		{
		DupeWindow *dw = work->data;
		work = work->next;

		while (dupe_item_set_path(dw, source, dest));
		}

}

