"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDirectQuery = void 0;
var _react = require("react");
var _constants = require("../constants");
var _types = require("../types");
var _query_session_utils = require("../utils/query_session_utils");
var _shared = require("../utils/shared");
var _use_polling = require("../utils/use_polling");
var _sql = require("../requests/sql");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const useDirectQuery = (http, notifications, dataSourceMDSId) => {
  const sqlService = new _sql.SQLService(http);
  const [loadStatus, setLoadStatus] = (0, _react.useState)(_types.DirectQueryLoadingStatus.SCHEDULED);
  const {
    data: pollingResult,
    loading: _pollingLoading,
    error: pollingError,
    startPolling,
    stopPolling: stopLoading
  } = (0, _use_polling.usePolling)(params => {
    return sqlService.fetchWithJobId(params, dataSourceMDSId || '');
  }, _constants.ASYNC_POLLING_INTERVAL);
  const startLoading = requestPayload => {
    setLoadStatus(_types.DirectQueryLoadingStatus.SCHEDULED);
    const sessionId = (0, _query_session_utils.getAsyncSessionId)(requestPayload.datasource);
    if (sessionId) {
      requestPayload = {
        ...requestPayload,
        sessionId
      };
    }
    sqlService.fetch(requestPayload, dataSourceMDSId).then(result => {
      (0, _query_session_utils.setAsyncSessionId)(requestPayload.datasource, (0, _shared.get)(result, 'sessionId', null));
      if (result.queryId) {
        startPolling({
          queryId: result.queryId
        });
      } else {
        // eslint-disable-next-line no-console
        console.error('No query id found in response');
        setLoadStatus(_types.DirectQueryLoadingStatus.FAILED);
      }
    }).catch(e => {
      var _e$body;
      setLoadStatus(_types.DirectQueryLoadingStatus.FAILED);
      const formattedError = (0, _shared.formatError)('', 'The query failed to execute and the operation could not be complete.', (_e$body = e.body) === null || _e$body === void 0 ? void 0 : _e$body.message);
      notifications.toasts.addError(formattedError, {
        title: 'Query Failed'
      });
      // eslint-disable-next-line no-console
      console.error(e);
    });
  };
  (0, _react.useEffect)(() => {
    // cancel direct query
    if (!pollingResult) return;
    const {
      status: anyCaseStatus,
      datarows,
      error
    } = pollingResult;
    const status = anyCaseStatus === null || anyCaseStatus === void 0 ? void 0 : anyCaseStatus.toLowerCase();
    if (status === _types.DirectQueryLoadingStatus.SUCCESS || datarows) {
      setLoadStatus(status);
      stopLoading();
    } else if (status === _types.DirectQueryLoadingStatus.FAILED) {
      setLoadStatus(status);
      stopLoading();
      const formattedError = (0, _shared.formatError)('', 'The query failed to execute and the operation could not be complete.', error);
      notifications.toasts.addError(formattedError, {
        title: 'Query Failed'
      });
    } else {
      setLoadStatus(status);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [pollingResult, pollingError, stopLoading]);
  return {
    loadStatus,
    startLoading,
    stopLoading,
    pollingResult
  };
};
exports.useDirectQuery = useDirectQuery;