/* -*-c-*- */
/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

%option prefix="netfilter"
%option outfile="netfilter.cc"
%option noyywrap

%{
using namespace std;

#define YY_NO_UNPUT

#include <unistd.h>
#include <string.h>
#include <ctype.h>

#include "netfilter.h"
#include "realtime.h"
#include "common.h"
#include "defs.h"


static void netfilter_parse_prefix(char *input);
static void netfilter_parse_noprefix(char *input);
static void netfilter_parse_proto(char *input);

static wf_logentry* logentry;
%}

MONTH	("Jan"|"Feb"|"Mar"|"Apr"|"May"|"Jun"|"Jul"|"Aug"|"Sep"|"Oct"|"Nov"|"Dec")
STRING	[a-zA-Z][a-zA-Z0-9.-]*
PREFIX	([ -OQ-~-]+|([ -OQ-~-]*P[ -GI-~-]*)|([ -OQ-~-]*PH[ -XZ-~-]*)|([ -OQ-~-]*PHY[ -RT-~-]*))
LOGHOST	[0-9.a-zA-Z_-]*
DIGIT	[0-9]
NUMBER	{DIGIT}+
BYTE	{DIGIT}{1,3}
IPADDR	{BYTE}\.{BYTE}\.{BYTE}\.{BYTE}
PORT	{DIGIT}{1,5}
HEXDIGIT	[0-9a-fA-F]
HEXNUM	"0x"{HEXDIGIT}+
PROTO	("TCP"|"UDP"|"ICMP"|"ESP"|"AH"|"ICMPv6"|{NUMBER})

%%

{MONTH}[ ]{1,2}{DIGIT}{1,2}[ ]{DIGIT}{2}:{DIGIT}{2}:{DIGIT}{2}[ ]{LOGHOST}	parse_date(netfiltertext, logentry);
" kernel: IN="{STRING}?			netfilter_parse_noprefix(netfiltertext+12);
" kernel: "{PREFIX}"IN="{STRING}?	netfilter_parse_prefix(netfiltertext+9);
" klogd: IN="{STRING}?			netfilter_parse_noprefix(netfiltertext+11);
" klogd: "{PREFIX}"IN="{STRING}?	netfilter_parse_prefix(netfiltertext+8);
" "{PREFIX}"IN="{STRING}?		netfilter_parse_prefix(netfiltertext+1);
"OUT="{STRING}?				logentry->output_iface = netfiltertext+4;
"PHYSIN="{STRING}?	/* ignore */
"PHYSOUT="{STRING}?	/* ignore */
 /* MAC is dst_mac:src_mac:08:00 */
"MAC="(({HEXDIGIT}{HEXDIGIT}:){13}{HEXDIGIT}{HEXDIGIT})?	{
  if (netfiltertext[4] != '\0') {
    netfiltertext[21] = '\0';
    netfiltertext[39] = '\0'; /* ignore last 08:00 */
    logentry->smacaddr.set(netfiltertext + 22);
    logentry->dmacaddr.set(netfiltertext + 4);
  }
}
"SRC="{IPADDR}	{
  logentry->sipaddr.set(netfiltertext + 4);
//  convert_ip(netfiltertext + 4, &logentry->shost);
}
"DST="{IPADDR}	{
  logentry->dipaddr.set(netfiltertext + 4);
  //  convert_ip(netfiltertext + 4, &logentry->dhost);
}
"LEN="{NUMBER}		logentry->datalen = atoi(netfiltertext+4);
"TOS="{HEXNUM}		/* ignore */
"PREC="{HEXNUM}		/* ignore */
"TTL="{NUMBER}		/* ignore */
"ID="{NUMBER}		/* ignore */
"CE"			/* ignore */
"DF"			/* ignore */
"MF"			/* ignore */
"FRAG:"{NUMBER}		/* ignore */
"PROTO="{PROTO}		netfilter_parse_proto(netfiltertext+6);
"INCOMPLETE ["{NUMBER}" bytes]"	/* ignore */
"TYPE="{NUMBER}		{ logentry->sport = atoi(netfiltertext+5); }
"CODE="{NUMBER}		{ logentry->dport = atoi(netfiltertext+5); }
"SEQ="{NUMBER}		/* ignore */
"ACK="{NUMBER}		/* ignore */
"SPT="{PORT}		{ logentry->sport = atoi(netfiltertext+4); }
"DPT="{PORT}		{ logentry->dport = atoi(netfiltertext+4); }
"WINDOW="{NUMBER}	/* ignore */
"RES="{HEXNUM}		/* ignore */
"FIN"			logentry->tcpflags |= TCP_FIN;
"SYN"			logentry->tcpflags |= TCP_SYN;
"RST"			logentry->tcpflags |= TCP_RST;
"PSH"			logentry->tcpflags |= TCP_PSH;
"ACK"			logentry->tcpflags |= TCP_ACK;
"URG"			logentry->tcpflags |= TCP_URG;
"ECE"			logentry->tcpflags |= TCP_ECE;
"CWR"			logentry->tcpflags |= TCP_CWR;
"URGP="{NUMBER}		/* ignore */
"OPT ("[0-9A-F]*")"	/* ignore */
"SPI="{HEXNUM}		/* ignore */
"GATEWAY="{IPADDR}	/* ignore */
"TC="{NUMBER}		/* ignore */
"HOPLIMIT="{NUMBER}	/* ignore */
"FLOWLBL="{NUMBER}	/* ignore */
"[".+"]"		/* ignore */
[ ]+			/* ignore whitespace */
[\n]			return 0;
{STRING}		if (verbose) fprintf(stderr, _("Unrecognized token: %s\n"), netfiltertext);
.			if (verbose) fprintf(stderr, _("Unrecognized character: %s\n"), netfiltertext);

%%

static void
netfilter_parse_prefix(char* input) {
  char* pos = strstr(input, "IN=");
  if (pos != NULL) {
    *pos = '\0';
    logentry->chainlabel = input;
    logentry->input_iface = pos + 3;
  }
}

static void
netfilter_parse_noprefix(char *input) {
  logentry->chainlabel = "-";
  logentry->input_iface = input;
}

static void
netfilter_parse_proto(char* input) {
  if (isdigit(input[0]))
    logentry->protocol = atoi(input);
  else {
    if (!strncmp(input, "TCP", 3))
      logentry->protocol = IPPROTO_TCP;
    else if (!strncmp(input, "UDP", 3))
      logentry->protocol = IPPROTO_UDP;
    else if (!strncmp(input, "ICMP", 4))
      logentry->protocol = IPPROTO_ICMP;
    else if (!strncmp(input, "ESP", 3))
      logentry->protocol = IPPROTO_ESP;
    else if (!strncmp(input, "AH", 2))
      logentry->protocol = IPPROTO_AH;
    else if (!strncmp(input, "ICMPv6", 6))
      logentry->protocol = IPPROTO_ICMPV6;
  }
}


wf_inmodule_netfilter::wf_inmodule_netfilter() {
}

bool
wf_inmodule_netfilter::match(const string& line) const {
  //  return (line.find(" OUT=") != string::npos);

  /* First we search "kernel:", then "OUT=". */
  const char str[] = " kernel: ";
  string::size_type i;
  return ((i = line.find(str)) != string::npos &&
	  line.find(" OUT=", i + sizeof(str) - 1) != string::npos);
}

enum wf_logentry_parsing_result
wf_inmodule_netfilter::parse(wf_logentry** retlogentry,
			     const string& line, int linenum) {
  logentry = new wf_logentry();
  if (logentry == NULL)
    return WF_LOGENTRY_PARSING_ERROR;

  netfilter_scan_string(line.c_str());
  netfilterlex();

  logentry->count = 1; /* only one log line per packet */
  logentry->branchname = "-"; /* not defined */
  logentry->format = "netfilter";

  /*
  if ((logentry->protocol == 6 || logentry->protocol == 17) &&
      parser == (NF_DATE | NF_PROTO | NF_IN | NF_SRC | NF_DST | NF_SPT | NF_DPT))
    *retlogentry = logentry;
    return WF_LOGENTRY_PARSING_OK;

  if (logentry->protocol == 1 &&
      parser == (NF_DATE | NF_PROTO | NF_IN | NF_SRC | NF_DST | NF_TYPE))
  */
    *retlogentry = logentry;
    return WF_LOGENTRY_PARSING_OK;

  if (verbose)
    fprintf(stderr, _("netfilter parse error in line %d, ignoring.\n"),
	    linenum);

  delete logentry;
  return WF_LOGENTRY_PARSING_ERROR;
}

extern "C" wf_inmodule*
wf_inmodule_netfilter_init() {
#ifdef ENABLE_NFULOG
  wf_inmodule_netfilter_realtime_init();
#endif
  return new wf_inmodule_netfilter();
}
