/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developpers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
#ifndef TRACE_HPP
#define TRACE_HPP

/*!
 *
 * \file Trace.hpp
 * \brief This file contains the datas definition for the storage of the 
 * \date 30 janvier 09
 *
 */

#include <list>
#include <vector>

using std::list;
using std::vector;

#include "values/Value.hpp"
#include "values/Date.hpp"
#include "values/String.hpp"
#include "values/Name.hpp"
#include "values/Integer.hpp"
#include "values/Hex.hpp"
#include "values/Double.hpp"
#include "values/Color.hpp"

#include "ContainerType.hpp"
#include "Container.hpp"
#include "StateType.hpp"
#include "EventType.hpp"
#include "LinkType.hpp"
#include "VariableType.hpp"
#include "EntityValue.hpp"
#include "State.hpp"
#include "Event.hpp"
#include "Link.hpp"

/*!
 *
 * \class Trace
 * \brief This class contains the trace
 *
 */

class Trace {
private:
    Date                  _max_date;
    list<ContainerType *> _root_container_types;
    list<Container *>     _root_containers;
    list<StateType *>     _state_types;
    list<EventType *>     _event_types;
    list<LinkType *>      _link_types;
    list<VariableType *>  _variable_types;

public :
    Trace();
    ~Trace();

    /*!
     * \brief Define a container type
     * \param alias Name of the container type
     * \param parent_container_type Type of the parent container
     * \param opt Extra fields
     */
    void define_container_type(Name &alias, ContainerType *parent_container_type, map<std::string, Value *> &opt);

    /*!
     * \brief Create a container
     * \param time Date of the event
     * \param alias Name of the container
     * \param type Type of the container
     * \param parent Parent of the container (NULL if the container is root)
     * \param opt Extra fields
     */
    void create_container(Date &time, Name &alias, ContainerType *type, Container *parent, map<std::string, Value *> &opt);

    /*!
     * \brief Destroy a container
     * \param time Date of the event
     * \param cont Container to destroy
     * \param type Type of the container
     * \param opt Extra fields
     */
    void destroy_container(Date &time, Container *cont, ContainerType *type, map<std::string, Value *> &opt);


    /*!
     * \brief Define a type of event
     * \param alias Name of the type
     * \param container_type Type of the container for these events
     * \param opt Extra fields
     */
    void define_event_type(Name &alias, ContainerType *container_type, map<std::string, Value *> &opt);


    /*!
     * \brief Define a type of state
     * \param alias Name of the type
     * \param container_type Type of the container for these states
     * \param opt Extra fields
     */
    void define_state_type(Name &alias, ContainerType *container_type, map<std::string, Value *> &opt);

    /*!
     * \brief Define a type of variable
     * \param alias Name of the type
     * \param container_type Type of the container for these variables
     * \param opt Extra fields
     */
    void define_variable_type(Name &alias, ContainerType *container_type, map<std::string, Value *> &opt);


    /*!
     * \brief Define a type of link
     * \param alias Name of the type
     * \param ancestor Type of the ancestor container
     * \param source Type of the source container
     * \param destination Type of the destination container
     * \param opt Extra fields
     */
    void define_link_type(Name &alias, ContainerType *ancestor, ContainerType *source, ContainerType *destination, map<std::string, Value *> &opt);


    /*!
     * \brief Define an entity value
     * \param alias Name of the value
     * \param entity_type Type of the entity
     * \param opt Extra fields
     */
    void define_entity_value(Name &alias, EntityType *entity_type, map<std::string, Value *> &opt);


    /*!
     * \brief Set the state of a container
     * \param time Date of the event
     * \param type Type of the state
     * \param container Container whose state is changed
     * \param value Value of the state
     * \param opt Extra fields
     */
    void set_state(Date &time, StateType *type, Container *container, EntityValue *value, map<std::string, Value *> &opt);


    /*!
     * \brief Set the stae of a container and save the previous one
     * \param time Date of the event
     * \param type Type of the state
     * \param container Container whose state is changed
     * \param value Value of the state
     * \param opt Extra fields
     */
    void push_state(Date &time, StateType *type, Container *container, EntityValue *value, map<std::string, Value *> &opt);


    /*!
     * \brief Restore a previously saved state of a container
     * \param time Date of the event
     * \param type Type of the state
     * \param container Container whose state is changed
     * \param opt Extra fields
     */
    void pop_state(Date &time, StateType *type, Container *container, map<std::string, Value *> &opt);


    /*!
     * \brief Add a new event to a container
     * \param time Date of the event
     * \param type Type of the event
     * \param container Container of the event
     * \param value Value of the event
     * \param opt Extra fields
     */
    void new_event(Date &time, EventType *type, Container *container, EntityValue *value, map<std::string, Value *> &opt);


    /*!
     * \brief Set the value of a variable
     * \param time Date of the new value
     * \param type Type of the variable whose value is changed
     * \param container Container of the variable
     * \param value New value of the variable
     * \param opt Extra fields
     */
    void set_variable(Date &time, VariableType *type, Container *container, Double value, map<std::string, Value *> &opt);


    /*!
     * \brief Add a value to a variable
     * \param time Date of the new value
     * \param type Type of the variable whose value is changed
     * \param container Container of the variable
     * \param value Value to add
     * \param opt Extra fields
     */
    void add_variable(Date &time, VariableType *type, Container *container, Double value, map<std::string, Value *> &opt);


    /*!
     * \brief Substract a value to a variable
     * \param time Date of the new value
     * \param type Type of the variable whose value is changed
     * \param container Container of the variable
     * \param value Value to substract
     * \param opt Extra fields
     */
    void sub_variable(Date &time, VariableType *type, Container *container, Double value, map<std::string, Value *> &opt);


    /*!
     * \brief Start a new link identified by key
     * \param time Date of the event
     * \param type Type of the link
     * \param ancestor Ancestor container
     * \param source Source container
     * \param value Value of the link
     * \param key Key to match the end of the link
     * \param opt Extra fields
     */
    void start_link(Date &time, LinkType *type, Container *ancestor, Container *source, EntityValue *value, String key, map<std::string, Value *> &opt);


    /*!
     * \brief End a link identified by key
     * \param time Date of the event
     * \param type Type of the link
     * \param ancestor Ancestor container
     * \param destination Destination container
     * \param value Value of the link
     * \param key Key to match the Start of the link
     * \param opt Extra fields
     */
    void end_link(Date &time, LinkType *type, Container *ancestor, Container *destination, EntityValue *value, String key, map<std::string, Value *> &opt);

    /*!
     * \fn finish()
     * \brief Finish to initialize the trace
     */
    void finish();

    /*!
     * \fn get_root_containers() const
     * \brief Get the list of the root containers
     */
    const list <Container *> *get_root_containers() const;
    
    /*!
     * \brief Define a type of state
     * \param alias Name of the type
     * \param container_type Type of the container for these states
     * \param opt Extra fields
     */
    const list <StateType*> *get_state_types() const;

    /*!
     * \fn search_container_type(String name) const
     * \brief Search a container type by his name or alias
     */
    ContainerType *search_container_type(String name) const;
    
    /*!
     * \fn search_container(String name) const
     * \brief Search a container by his name or alias
     */
    Container *search_container(String name) const;
    
    /*!
     * \fn search_event_type(String name) const
     * \brief Search a event type by his name or alias
     */
    EventType *search_event_type(String name) const;


    /*!
     * \fn search_state_type(String name) const
     * \brief Search a container by his name or alias
     */
    StateType *search_state_type(String name) const;


    /*!
     * \fn search_variable_type(String name) const
     * \brief Search a variable type by his name or alias
     */
    VariableType *search_variable_type(String name) const;


    /*!
     * \fn search_link_type(String name) const
     * \brief Search a container by his name or alias
     */
    LinkType *search_link_type(String name) const;
    
    /*!
     * \fn search_entity_type(String name) const
     * \brief Search an entity type by his name or alias
     */
    EntityType *search_entity_type(String name) const;                    
    
    /*!
     * \fn search_entity_value(String name,  EntityType *entity_type) const
     * \brief Search a container by his name or alias
     */
    EntityValue *search_entity_value(String name,  EntityType *entity_type) const;

    /*!
     * \fn get_max_date()
     * \brief Function that return the max of the date
     */
    Date get_max_date(){
        return _max_date;
    }

};//end class

#endif
