/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developpers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
/*!
 *\file Render_area.hpp
 */


#ifndef RENDER_AREA_HPP
#define RENDER_AREA_HPP




class Render_area;
class Interface_console;
class Render_opengl;

/* Global informations */
#include "../info/info.hpp"
#include "resource.hpp"



/*!
 * \brief Structure used to store event information.
 */
struct Event_{
    /*!
     * \brief time of an event
     */
    Element_pos time;
    /*!
     * \brief height of an event
     */
    Element_pos height;
    /*!
     * \brief height of the container
     */
    Element_pos container_height;
};

/*!
 * \brief Structure used to store arrow information.
 */
struct Arrow_{
    /*!
     * \brief time when the arrow starts.
     */
    Element_pos start_time;
    /*!
     * \brief time when the arrow ends.
     */
    Element_pos end_time;
    /*!
     * \brief The start height of the arrow.
     */
    Element_pos start_height;
    /*!
     * \brief The end height of the arrow.
     */
    Element_pos end_height;
};


/*!
 * \brief Structure used to store selection coordinate information.
 */
struct Selection_{
    /*!
     * \brief x scale.
     */
    Element_pos x_scale;

    /*!
     * \brief y scale.
     */
    Element_pos y_scale;

    /*!
     * \brief x translation.
     */
    Element_pos x_translate;

    /*!
     * \brief y translation.
     */
    Element_pos y_translate;
};










/*!
 * \brief This class redefined the OpenGL widget - QGLWidget - to display the trace.
 */
class Render_area : public QGLWidget/*, public Render*/
{
    Q_OBJECT



protected:


    /*!
     * \brief Contains container text coordinates.
     */
    std::list<Element_pos> _text_pos;

    /*!
     * \brief Contains container strings.
     */
    std::list<std::string> _text_value;

    /*!
     * \brief Contains arrow informations.
     */
    std::vector<Arrow_> _arrows;

    /*!
     * \brief Contains event informations.
     */
    std::vector<Event_> _events;

    /*!
     * \brief Contains the previous selection coordinates.
     */
    std::stack<Selection_> _previous_selection;

    /*!
     * \brief Contains the OpenGL render instance.
     */
    Render_opengl* _render_instance;


    /*!
     * \brief Contains the parent instance.
     */
    Interface_console* _core;


    /***********************************
     *
     * Render area state attributes.
     *
     **********************************/

    /*!
     * \brief State when there is no file opened.
     */
    static const int DRAWING_STATE_WAITING;

    /*!
     * \brief State when the application is drawing traces.
     */
    static const int DRAWING_STATE_DRAWING;


    /*!
     * \brief Contains the kind of state for the render area (drawing, waiting, etc.).
     */
    int _state;

    /*!
     * \brief Indicated if mouse is pressed or not.
     */
    bool _mouse_pressed;

    /*!
     * \brief Used to store the mouse last x position.
     */
    int _mouse_x;

    /*!
     * \brief Used to store the mouse last y position.
     */
    int _mouse_y;

    /*!
     * \brief Used to store the mouse current x position.
     */
    int _new_mouse_x;

    /*!
     * \brief Used to store the mouse current y position.
     */
    int _new_mouse_y;

    /*!
     * \brief Indicate if the scroll was asked by key capture in openGL render area.
     */
    bool _key_scrolling;

    /*!
     * \brief Alpha color of the selection rectangle.
     */
    double _selection_rectangle_alpha;

    /*!
     * \brief Define the minimum width and height to draw the selection rectangle (avoid bas manipulations).
     */
    Element_pos _minimum_distance_for_selection;

    /*!
     * \brief Define the scrolling factor when CTRL key is pressed.
     */
    static const int _ctrl_scroll_factor;

    /*!
     * \brief Define the scrolling factor when CTRL key is pressed.
     */
    static const int _ctrl_zoom_factor;


    /***********************************
     *
     * The wait screen drawing.
     *
     **********************************/

    /***********************************
     * The wait list Attributes.
     **********************************/


    /*!
     * \brief Rotation angle for the wait.
     */
    float _wait_angle;

    /*!
     * \brief Y rotation angle for the wait.
     */
    float _wait_angle_y;

    /*!
     * \brief Time in ms between two frames for the waiting screen.
     */
    static const int DRAWING_TIMER_DEFAULT;

    /*!
     * \brief Wait animation seconds per frame.
     */
    int _wait_spf;

    /*!
     * \brief Timer to animate the wait.
     */
    QTimer* _wait_timer;




    /***********************************
     *
     * The trace drawing.
     *
     **********************************/


    /***********************************
     * The drawing list Attributes.
     **********************************/



    /*!
     * \brief The waiting screen display list.
     */
     GLuint _wait_list;

    /*!
     * \brief The ruler display list.
     */
    GLuint _list_ruler;

    /*!
     * \brief Used for draw counter.
     */
    bool _line_already_begun;

    /*!
     * \brief The last x position of the point (for counter).
     */
    Element_pos _counter_last_x;

    /*!
     * \brief The last y position of the point (for counter).
     */
    Element_pos _counter_last_y;

    /*!
     * \brief The opengl render area width in pixels.
     */
    Element_pos _screen_width;

    /*!
     * \brief The opengl render area height in pixels.
     */
    Element_pos _screen_height;

    /*!
     * \brief The opengl visibled scene width in the OpenGL units.
     */
    Element_pos _render_width;

    /*!
     * \brief The opengl visibled scene height in the OpenGL units.
     */
    Element_pos _render_height;

    /*!
     * \brief The width of container area draw.
     */
    Element_pos _container_x_max;

    /*!
     * \brief The height of container area draw.
     */
    Element_pos _container_y_max;

    /*!
     * \brief The x base of container area draw.
     */
    Element_pos _container_x_min;

    /*!
     * \brief The y base of container area draw.
     */
    Element_pos _container_y_min;

    /*!
     * \brief The width of state area draw.
     */
    Element_pos _state_x_max;

    /*!
     * \brief The height of state area draw.
     */
    Element_pos _state_y_max;

    /*!
     * \brief The x base of state area draw.
     */
    Element_pos _state_x_min;

    /*!
     * \brief The y base of state area draw.
     */
    Element_pos _state_y_min;

    /*!
     * \brief z position for the ruler.
     */
    Element_pos _z_ruler;

    /*!
     * \brief z position for objects over the ruler.
     */
    Element_pos _z_ruler_over;

    /*!
     * \brief z position for objects under the ruler.
     */
    Element_pos _z_ruler_under;

    /*!
     * \brief z position for containers.
     */
    Element_pos _z_container;

    /*!
     * \brief z position for objects under containers.
     */
    Element_pos _z_container_under;

    /*!
     * z position for states.
     */
    Element_pos _z_state;

    /*!
     * z position for events.
     */
    Element_pos _z_event;

    /*!
     * z position for arrows.
     */
    Element_pos _z_arrow;

    /*!
     * z position for counters.
     */
    Element_pos _z_counter;

    /*!
     * Default offset of entities drawing.
     */
    Element_pos _default_entity_x_translate;

    /*!
     * Distance between two ruler measures.
     */
    Element_pos _ruler_distance;

    /*!
     * Height of the ruler.
     */
    Element_pos _ruler_height;

    /*!
     *  Highness of the ruler.
     */
    Element_pos _ruler_y;

    /*!
     *  Width of the ruler.
     */
    Element_pos _ruler_width;

    /*!
     * \brief The percentage taken by container display in the render area.
     */
    Element_pos _x_scale_container_state;

    /*!
     * \brief the x scale of state drawing.
     */
    Element_pos _x_state_scale;

    /*!
     * \brief the y scale of state drawing.
     */
    Element_pos _y_state_scale;

    /*!
     * \brief The x position of camera view for state drawing area.
     */
    Element_pos _x_state_translate;

    /*!
     * \brief The y position of camera view for state drawing area.
     */
    Element_pos _y_state_translate;

    /*!
     * \brief The x position of the horizontal scroll bar.
     */
    Element_pos _x_scroll_pos;

    /*!
     * \brief The y position of the vertical scroll bar.
     */
    Element_pos _y_scroll_pos;



    /***********************************
     *
     * Default QWidget functions.
     *
     **********************************/

    /*!
     * \brief This functions receives all mouse press events.
     * \param event The event triggered by mouse.
     */
    void mousePressEvent(QMouseEvent * event);

    /*!
     * \brief If user press, this functions receives all mouse move events until user release mouse.
     * \param event The event triggered by mouse.
     */
    void mouseMoveEvent(QMouseEvent * event);

    /*!
     * \brief This functions receives all mouse release events.
     * \param event The event triggered by mouse.
     */
    void mouseReleaseEvent(QMouseEvent * event);

    /*!
     * \brief This functions receives all mouse wheel events.
     * \param event The event triggered by the mouse wheel.
     */
    void wheelEvent(QWheelEvent * event);

    /*!
     * \brief This functions receives all keyboard events.
     * \param event The event triggered by the keyboard event.
     */
    void keyPressEvent(QKeyEvent * event);

    /*!
     * \brief This functions receives all keyboard release events.
     * \param event The event triggered by a keyboard release.
     */
    void keyReleaseEvent(QKeyEvent * event);




    /***********************************
     *
     * Render OpenGL drawing functions.
     *
     **********************************/

    /*!
     * \brief Display a wait on the screen if there is no file opened.
     * \return Asset value of the wait.
     */
    GLuint draw_wait();

    /*!
     * \brief Create the ruler display list.
     * \return Asset value of the ruler.
     */
    GLuint draw_ruler();

    /*!
     * \brief Draw the ruler display list.
     */
    void call_ruler();


    /***********************************
     *
     * Constructor and destructor.
     *
     **********************************/

    /*!
     * \brief The default constructor.
     *
     * Default constructor has a private scope to avoid any direct instantiation of Render_area.
     */
    Render_area();


public:


    /*!
     * \brief The constructor.
     * \param render_instance The instance of a drawing class.
     */
    Render_area(Render_opengl* render_instance, Interface_console* core, QWidget *parent);

    /*!
     * \brief The destructor
     */
    virtual ~Render_area();



    /***********************************
     *
     * Scaling and scrolling functions.
     *
     **********************************/


    /*!
     * \brief Change the scale of state drawing.
     * \param scale_coeff The new coefficient scale value to add to the current scale.
     */
    void change_scale(Element_pos scale_coeff);

    /*!
     * \brief Change the scale of the y state drawing.
     * \param scale_coeff The new coefficient scale value to add to the current scale.
     */
    void change_scale_y(Element_pos scale_coeff);

    /*!
     * \brief Replace the current scale by a new scale.
     * \param new_scale The new scale value to replace the current scale.
     */
    void replace_scale(Element_pos new_scale);

    /*!
     * \brief Change the x position of camera view for state drawing area.
     * \param translate The new position.
     */
    void change_translate(int translate);/* temporary -> to change the translate to view states */

    /*!
     * \brief Change the y position of camera view for state drawing area.
     * \param translate The new position.
     */
    void change_translate_y(int translate);/* temporary -> to change the translate to view states */

    /*!
     * \brief Replace the current x translate by a new translate.
     * \param new_translate The new translate value to replace the current translate.
     */
    void replace_translate(Element_pos new_translate);

    /*!
     * \brief Replace the current y translate by a new translate.
     * \param new_translate The new translate value to replace the current translate.
     */
    void replace_translate_y(Element_pos new_translate);

    /*!
     * \brief Pre registered translation values (for x or y translate).
     * \param id The pre registered translation id.
     */
    void registered_translate(int id);

    /*!
     * \brief Refresh scroll bar positions when shortcuts execute movements
     * \param LENGTH_CHANGED If true the total length of the scroll bar will be updated.
     */
    void refresh_scroll_bars(bool LENGTH_CHANGED = false);

    /*!
     * \brief Change the percentage taken by container display in the render area.
     * \param view_size The new percentage (between 0 to 100).
     */
    void change_scale_container_state(int view_size);/* temporary -> to change the size of container view */

    /*!
     * \brief This function updates the visible interval value (for export).
     */
    void update_visible_interval_value();

    /*!
     * \brief Scale the current view to the zoom box shape.
     * \param x_min the x minimum position.
     * \param x_max the x maximum position.
     * \param y_min the y minimum position.
     * \param y_max the y maximum position.
     */
    void apply_zoom_box(Element_pos x_min, Element_pos x_max, Element_pos y_min, Element_pos y_max);



    /***********************************
     *
     * Coordinate convert functions.
     *
     **********************************/

    /*!
     * \brief This function convert a X screen coordinate into render coordinate.
     * \param e A X screen coordinate.
     * \return A X render coordinate corresponding to e but in render area coordinate.
     */
    Element_pos screen_to_render_x(Element_pos e);

    /*!
     * \brief This function convert a Y screen coordinate into render coordinate.
     * \param e A Y screen coordinate.
     * \return A Y render coordinate corresponding to e but in render area coordinate.
     */
    Element_pos screen_to_render_y(Element_pos e);

    /*!
     * \brief This function convert a X render coordinate into trace coordinate.
     * \param e A X render coordinate.
     * \return A X trace coordinate corresponding to e but in trace coordinate.
     */
    Element_pos render_to_trace_x(Element_pos e);

    /*!
     * \brief This function convert a Y render coordinate into trace coordinate.
     * \param e A Y render coordinate.
     * \return A Y trace coordinate corresponding to e but in trace coordinate.
     */
    Element_pos render_to_trace_y(Element_pos e);

    /*!
     * \brief Dual of screen_to_render_x.
     * \param e A X render coordinate.
     * \return A X screen coordinate corresponding to e but in screen coordinate.
     */
    Element_pos render_to_screen_x(Element_pos e);

    /*!
     * \brief Dual of screen_to_render_y.
     * \param e A Y render coordinate.
     * \return A Y screen coordinate corresponding to e but in screen coordinate.
     */
    Element_pos render_to_screen_y(Element_pos e);

    /*!
     * \brief Dual of render_to_trace_x.
     * \param e A X trace coordinate.
     * \return A X render coordinate corresponding to e but in render area coordinate.
     */
    Element_pos trace_to_render_x(Element_pos e);

    /*!
     * \brief  Dual of render_to_trace_y.
     * \param e A Y trace coordinate.
     * \return A Y render coordinate corresponding to e but in render area coordinate.
     */
    Element_pos trace_to_render_y(Element_pos e);


    /***********************************
     *
     * Building functions.
     *
     **********************************/

    /*!
     * \brief This function draws the trace.
     */
    bool build();

    /*!
     * \brief This function releases the trace.
     */
    bool unbuild();
};



#endif
