/**
 * Copyright (C) 2007-2012 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev: 427 $
 * $Date: 2012-03-04 10:50:17 +0800 (Sun, 04 Mar 2012) $
 */
#include "Document.hpp"

#include "../../gnome/gnome.hpp"
#include "../meta.hpp"
#include "../image/ImageManipulation.hpp"

#include "wx/button.h"
#include "wx/sizer.h"
#include "wx/cshelp.h"
#include "wx/config.h"
#include "wx/stdpaths.h"
#include "wx/filename.h"

using namespace indii;

Document::Document(wxWindow* parent, ImageResource* res,
    Model* model, Controller* control) : wxPanel(parent), res(res),
    model(model), control(control), isSaved(false), havePath(false) {
  init();
}

Document::Document(wxWindow* parent) : wxPanel(parent), res(NULL),
    model(NULL), control(NULL), isSaved(false), havePath(false) {
  //
}

void Document::set(ImageResource* res, Model* model, Controller* control) {
  this->res = res;
  this->model = model;
  this->control = control;

  init();
}

void Document::init() {
  wxBoxSizer* sizer = new wxBoxSizer(wxVERTICAL);
  previewPane = new PreviewPane(this, res, model, control);
  sizer->Add(previewPane, 1, wxEXPAND);
  this->SetSizer(sizer);
}

void Document::save() {
  /* pre-condition */
  assert (havePath);

  unsigned width = res->getWidth();
  unsigned height = res->getHeight();

  wxImage img(width, height);
  model->calcFg(width, height, img);
  img.SetOption(wxIMAGE_OPTION_QUALITY, 95);
  wxLogNull logNo;
  isSaved = img.SaveFile(path);
  if (!isSaved) {
    isSaved = img.SaveFile(path, _("image/png"));
    if (!isSaved) {
      wxMessageBox(_("Error attempting to save file. File not saved."),
          _("Error"), wxOK|wxICON_HAND, this);
    } else {
      wxMessageBox(_("Unrecognised file extension. File saved as PNG."),
                   _("Warning"), wxOK|wxICON_INFORMATION, this);
    }
  }
}

void Document::saveAs() {
  wxString savePath;
  wxConfig config(_(PACKAGE_TARNAME), _("indii.org"));
  if (!config.Read(_("SavePath"), &savePath)) {
    savePath = wxStandardPaths::Get().GetDocumentsDir();
  }

  #ifdef __WXGTK__
  wxString defaultName(_("image.png"));
  if (havePath) {
    defaultName = wxFileName(path).GetName() + _(".png");
  }
  #else
  wxString defaultName(_("image"));
  if (havePath) {
    defaultName = wxFileName(path).GetName();
  }
  #endif

  wxString path = wxFileSelector(wxFileSelectorPromptStr, savePath,
      defaultName, _("png"), wildcards(true),
      wxFD_SAVE|wxFD_OVERWRITE_PROMPT, this);
  if (!path.empty()) {
    wxFileName file(path);
    savePath = file.GetPath();
    config.Write(_("SavePath"), savePath);

    setPath(path);
    save();
  }
}

const wxString& Document::getPath() const {
  return this->path;
}

void Document::setPath(const wxString& path) {
  this->path = path;
  havePath = true;
}

BEGIN_EVENT_TABLE(Document, wxPanel)
END_EVENT_TABLE()
