/////////////////////////////////////////////////////////////////////////////
// File:        module.h
// Author:      Cesar Mauri Loba (cesar at crea-si dot com)
// Copyright:   (C) 2010 Cesar Mauri Loba - CREA Software Systems
// 
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
/////////////////////////////////////////////////////////////////////////////
#ifndef SPCORE_MODULE_H
#define SPCORE_MODULE_H

#include "spcore/coreversion.h"
#include "spcore/basetype.h"
#include "spcore/iterator.h"
#include "spcore/component.h"
#include "include/spcore/baseobj.h"
#include <vector>
using namespace std;

namespace spcore { template<class T> class IIterator; } 
namespace spcore { class ITypeFactory; } 
namespace spcore { class IComponentFactory; } 

namespace spcore { class IActivity; } 
namespace spcore {

class IModule : public IBaseObject {
  protected:
    //Protected destructor to prevent the use of the delete operator
    inline virtual ~IModule();


  public:
    // Returns the version of spcore this module was build against
    virtual int GetCoreVersion() const = 0;

    virtual int GetModuleVersion() const = 0;

    virtual const char* GetName() const = 0;

    // Return a string encoded in UTF-8 containing a description of the module.
    virtual const char* GetDescription() const = 0;

    //Type names that this modules provides. This names are either for
    //pins and type instances.
    virtual SmartPtr<IIterator<ITypeFactory*> > GetTypeFactories() = 0;

    virtual SmartPtr<IIterator<IComponentFactory*> > GetComponentFactories() = 0;

};
//Protected destructor to prevent the use of the delete operator
inline IModule::~IModule() {
}

class CModuleAdapter : public IModule {
  public:
    inline virtual ~CModuleAdapter();

    inline virtual int GetCoreVersion() const;

    inline virtual int GetModuleVersion() const;

    // Return a string encoded in UTF-8 containing a description of the module.
    inline virtual const char* GetDescription() const;

    ///**
    //Type names that this modules provides. This names are either for
    //pins and type instances.
    //*/
    inline virtual SmartPtr<IIterator<ITypeFactory*> > GetTypeFactories();

    inline virtual SmartPtr<IIterator<IComponentFactory*> > GetComponentFactories();


  private:
    vector<IComponentFactory *> m_componentFactories;

    vector<ITypeFactory *> m_typeFactories;


  protected:
    inline int RegisterTypeFactory(SmartPtr<ITypeFactory> f);

    inline int RegisterComponentFactory(SmartPtr<IComponentFactory> c);

};
inline CModuleAdapter::~CModuleAdapter() {
  	vector<IComponentFactory *>::iterator itcf;
  	for (itcf= m_componentFactories.begin(); itcf!= m_componentFactories.end(); ++itcf)
  		(*itcf)->Release();
  	m_componentFactories.clear();
  
  	vector<ITypeFactory *>::iterator ittf;
  	for (ittf= m_typeFactories.begin(); ittf!= m_typeFactories.end(); ++ittf)
  		(*ittf)->Release();
  	m_typeFactories.clear();
}

inline int CModuleAdapter::GetCoreVersion() const {
  	return SPCORE_VERSION;
}

inline int CModuleAdapter::GetModuleVersion() const {
  	return 1;
}

// Return a string encoded in UTF-8 containing a description of the module.
inline const char* CModuleAdapter::GetDescription() const {
  	return "";
}

///**
//Type names that this modules provides. This names are either for
//pins and type instances.
//*/
inline SmartPtr<IIterator<ITypeFactory*> > CModuleAdapter::GetTypeFactories() {
  	return SmartPtr<IIterator<ITypeFactory*> >(new CIteratorVector<ITypeFactory*>(m_typeFactories), false);
}

inline SmartPtr<IIterator<IComponentFactory*> > CModuleAdapter::GetComponentFactories() {
  	return SmartPtr<IIterator<IComponentFactory*> >(new CIteratorVector<IComponentFactory*>(m_componentFactories), false);
}

inline int CModuleAdapter::RegisterTypeFactory(SmartPtr<ITypeFactory> f) {
  	m_typeFactories.push_back(f.get());
  	f->AddRef();
  	return 0;
}

inline int CModuleAdapter::RegisterComponentFactory(SmartPtr<IComponentFactory> c) {
  	m_componentFactories.push_back(c.get());
  	c->AddRef();
  	return 0;
}


} // namespace spcore
#endif
