/*
 * Scilab (http://www.scilab.org/) - This file is part of Scilab
 * Copyright (C) 2009 - INRIA - Allan SIMON
 * 
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at    
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 *
 */


package org.scilab.modules.xpad.utils;

import java.awt.Color;
import java.awt.Font;
import java.awt.Toolkit;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Hashtable;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.scilab.modules.console.GuiManagement;
import org.scilab.modules.gui.utils.Position;
import org.scilab.modules.gui.utils.Size;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;


public final class ConfigXpadManager {
	private static final int BUFSIZE = 1024;

	private static final int MARGIN = 20;

	private static final String ERROR_READ = "Could not load file: ";
	private static final String ERROR_WRITE = "Could not save file: ";
	private static final String VALUE = "value";
	private static final String STYLE = "style";
	private static final String FONT_SIZE = "FontSize";
	private static final String FONT_STYLE = "FontStyle";
	private static final String FONT_NAME = "FontName";
	private static final String DEFAULT = "default";
	private static final String WIDTH = "width";
	private static final String HEIGHT = "height";
	private static final String XCOORD = "x";
	private static final String YCOORD = "y";
	private static final String MAINWINPOSITION = "MainWindowPosition";
	private static final String MAINWINSIZE = "MainWindowSize";
	private static final String AUTOINDENT = "AutoIndent";
	private static final String AUTOCOLORIZE = "AutoColorize";
	private static final String DEFAULTENCONDING = "DefaultEncoding";

	private static final String FOREGROUNDCOLOR = "ForegroundColor";
	private static final String BACKGROUNDCOLOR = "BackgroundColor";
	private static final String COLORPREFIX = "#";

	private static final String NAME = "name";

	private static final String PROFILE = "Profile";

	private static String RECENT_SEARCH = "recentSearch";
	private static String SEARCH = "search";
	private static String RECENT_REPLACE = "recentReplace";
	private static String REPLACE = "replace";
	private static String EXPRESSION = "exp"; 
	private static String REGULAR_EXPRESION = "regularExp"; 
	private static String WORD_WARP = "wordWarp"; 
	private static String WHOLE_WORD = "wholeWord"; 
	private static String CASE_SENSITIVE = "caseSensitive"; 
	private static String STATE_FLAG = "state"; 
	
	private static String SETTING = "Setting";
	private static String XPAD = "xpad";

	//private static final String XPAD_CONFIG_FILE = "/home/allan/scilab5-2/scilab/modules/xpad/etc/xpadConfiguration.xml";
	private static final String XPAD_CONFIG_FILE = System.getenv("SCI") + "/modules/xpad/etc/xpadConfiguration.xml";

	//private static final String USER_XPAD_CONFIG_FILE = "/home/allan/Bureau/xpadConfiguration.xml";
	private static final String USER_XPAD_CONFIG_FILE = GuiManagement.getSCIHOME() + "/xpadConfiguration.xml";
	private static final int PLAIN = 0;
	private static final int BOLD =  1;
	private static final int ITALIC = 2;
	private static final int BOLDITALIC = 3;

	private static final int DEFAULT_WIDTH = 650;
	private static final int DEFAULT_HEIGHT = 550;

	private static final int MAX_RECENT_FILES = 10;
	private static final int MAX_RECENT_SEARCH = 20;
	private static final int MAX_RECENT_REPLACE = 20;


	private static Document document;



	/**
	 * Constructor
	 */
	private ConfigXpadManager() {
		throw new UnsupportedOperationException();
	}

	/**
	 * Create a copy of Scilab configuration file in the user directory
	 */
	public static void createUserCopy() {
		/*TODO*/
		File fileConfig = new File(USER_XPAD_CONFIG_FILE);
		if (!fileConfig.exists() || (fileConfig.length() == 0)) {
			/* Create a local copy of the configuration file */
			copyFile(new File(XPAD_CONFIG_FILE), new File(USER_XPAD_CONFIG_FILE));

		}
	}
	/**
	 * Get the name of the user configuration file
	 * @return the name of the configuration file
	 */
	public static String getUserConfigFile() {
		/*TODO*/
		return USER_XPAD_CONFIG_FILE;
	}

	/**
	 * Get all Style name 
	 * @return a array list of all style name
	 */

	public static ArrayList<String> getAllStyleName() {
		ArrayList<String> stylesName = new ArrayList<String>();

		readDocument();


		Element root = document.getDocumentElement();
		NodeList styles = root.getElementsByTagName(STYLE);

		for (int i = 0; i < styles.getLength(); ++i) {
			Element style = (Element) styles.item(i);

			stylesName.add(style.getAttribute(NAME));


		}		
		return stylesName;


	}

	/**
	 * Get the font name
	 * @return the name of the font
	 */
	public static String getFontName() {

		/*load file*/
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList fontNameElement = xpadProfile.getElementsByTagName(FONT_NAME);
		Element fontName = (Element) fontNameElement.item(0);
		return fontName.getAttribute(VALUE);

	}

	/**
	 * Get the font size
	 * @return the font size
	 */
	public static int getFontSize() {
		/*load file*/
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList fontSizeElement = xpadProfile.getElementsByTagName(FONT_SIZE);
		Element fontSize = (Element) fontSizeElement.item(0);
		return Integer.parseInt(fontSize.getAttribute(VALUE));

	}

	/**
	 * Get all font style 
	 * @return true if the font style is bold , false otherwise
	 */
	public static Hashtable<String, Boolean> getAllisBold() {
		/*load file*/
		readDocument();
		Hashtable<String, Boolean > stylesIsBoldTable = new Hashtable<String, Boolean>();

		Element root = document.getDocumentElement();
		NodeList styles = root.getElementsByTagName(STYLE);

		for (int i = 0; i < styles.getLength(); ++i) {
			Element style = (Element) styles.item(i);


			NodeList fontStyleElement = style.getElementsByTagName(FONT_STYLE);
			Element fontStyle = (Element) fontStyleElement.item(0);
			int value = Integer.parseInt(fontStyle.getAttribute(VALUE));

			if (value  == BOLD || value == BOLDITALIC) {
				stylesIsBoldTable.put(style.getAttribute(NAME), true);
			} else {
				stylesIsBoldTable.put(style.getAttribute(NAME), false);
			}
		}		
		return stylesIsBoldTable;
	}


	/**
	 * Get the font setting
	 * @return the font
	 */
	public static Font getFont() {

		/*load file*/
		readDocument();
		Font font;
		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList fontSizeElement = xpadProfile.getElementsByTagName(FONT_SIZE);
		Element fontSize = (Element) fontSizeElement.item(0);
		int size = Integer.parseInt(fontSize.getAttribute(VALUE));

		NodeList fontNameElement = xpadProfile.getElementsByTagName(FONT_NAME);
		Element fontName = (Element) fontNameElement.item(0);
		String name = fontName.getAttribute(VALUE);

		NodeList fontStyleElement = xpadProfile.getElementsByTagName(FONT_STYLE);
		Element fontStyle = (Element) fontStyleElement.item(0);
		int style = Integer.parseInt(fontStyle.getAttribute(VALUE));

		if (style == PLAIN) {
			font = new Font(name, Font.PLAIN, size); 

		} else if (style == BOLD) {
			font = new Font(name, Font.BOLD, size); 		

		} else if (style == ITALIC) {
			font = new Font(name, Font.ITALIC, size); 

		} else if (style == BOLDITALIC) {
			font = new Font(name, Font.BOLD | Font.ITALIC , size); 

		} else {
			font = new Font(name, Font.PLAIN, size); 
		}

		return font;
	}

	/**
	 * Get Default Font Settings
	 * @return the default font
	 */

	public static Font getDefaultFont() {
		/*load file*/
		readDocument();
		Font font;
		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList fontSizeElement = xpadProfile.getElementsByTagName(FONT_SIZE);
		Element fontSize = (Element) fontSizeElement.item(0);
		int size = Integer.parseInt(fontSize.getAttribute(DEFAULT));

		NodeList fontNameElement = xpadProfile.getElementsByTagName(FONT_NAME);
		Element fontName = (Element) fontNameElement.item(0);
		String name = fontName.getAttribute(DEFAULT);

		NodeList fontStyleElement = xpadProfile.getElementsByTagName(FONT_STYLE);
		Element fontStyle = (Element) fontStyleElement.item(0);
		int style = Integer.parseInt(fontStyle.getAttribute(DEFAULT));

		if (style == PLAIN) {
			font = new Font(name, Font.PLAIN, size); 

		} else if (style == BOLD) {
			font = new Font(name, Font.BOLD, size); 		

		} else if (style == ITALIC) {
			font = new Font(name, Font.ITALIC, size); 

		} else if (style == BOLDITALIC) {
			font = new Font(name, Font.BOLD | Font.ITALIC , size); 

		} else {
			font = new Font(name, Font.PLAIN, size); 
		}
		return font;

	}

	/**
	 * Save a new font setting
	 * @param font the new font
	 */
	public static void saveFont(Font font) {
		/*TODO*/
		/*load file */
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList fontSizeElement = xpadProfile.getElementsByTagName(FONT_SIZE);
		Element fontSize = (Element) fontSizeElement.item(0);
		fontSize.setAttribute(VALUE, Integer.toString(font.getSize()));		

		NodeList fontNameElement = xpadProfile.getElementsByTagName(FONT_NAME);
		Element fontName = (Element) fontNameElement.item(0);
		fontName.setAttribute(VALUE, font.getFontName());

		NodeList fontStyleElement = xpadProfile.getElementsByTagName(FONT_STYLE);
		Element fontStyle = (Element) fontStyleElement.item(0);

		if (!font.isBold() && !font.isItalic()) {
			fontStyle.setAttribute(VALUE, "0");
		} else if (font.isBold() && font.isItalic()) {
			fontStyle.setAttribute(VALUE, "3");
		} else if (font.isBold()) {
			fontStyle.setAttribute(VALUE, "1");
		} else {
			fontStyle.setAttribute(VALUE, "2");
		}

		/* Save changes */
		writeDocument();
	}

	/**
	 * Copy a file
	 * @param in src file
	 * @param out dest file
	 */
	private static void copyFile(File in, File out) {
		FileInputStream fis = null;
		try {
			fis = new FileInputStream(in);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		FileOutputStream fos = null;
		try {
			fos = new FileOutputStream(out);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		byte[] buf = new byte[BUFSIZE];
		int i = 0;
		try {
			while ((i = fis.read(buf)) != -1) {
				fos.write(buf, 0, i);
			}
			fis.close();
			fos.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
		/*TODO*/
	}

	/**
	 * Get the background Color 
	 * @return the background Color
	 */
	public static Color getXpadBackgroundColor() {
		/* Load file */
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList allSizeElements = xpadProfile.getElementsByTagName(BACKGROUNDCOLOR);
		Element xpadBackground = (Element) allSizeElements.item(0);

		/*direct create a Color with "#FF00FF" string from the xml */
		return Color.decode(xpadBackground.getAttribute(VALUE));
	}

	/**
	 * Save Xpad BackgroundColor
	 * @param color the new Color
	 */
	public static void saveXpadBackground(Color color) {

		/* Load file */
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList allSizeElements = xpadProfile.getElementsByTagName(FOREGROUNDCOLOR);
		Element xpadForeground = (Element) allSizeElements.item(0);

		String rgb = Integer.toHexString(color.getRGB());
		xpadForeground.setAttribute(VALUE, COLORPREFIX + rgb.substring(2, rgb.length()));

		/* Save changes */
		writeDocument();	

	}

	/**
	 * Save Xpad autoIndent or not
	 * @param boolean if autoIndent should be used or not
	 */
	public static void saveAutoIndent(boolean activated) {

		/* Load file */
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList allSizeElements = xpadProfile.getElementsByTagName(AUTOINDENT);
		Element xpadAutoIndent = (Element) allSizeElements.item(0);
		if (xpadAutoIndent == null){
			Element autoIndent = document.createElement(AUTOINDENT);

			autoIndent.setAttribute(VALUE, new Boolean(activated).toString());

			xpadProfile.appendChild((Node) autoIndent);
		} else {
			xpadAutoIndent.setAttribute(VALUE, new Boolean(activated).toString()  );
		}
		/* Save changes */
		writeDocument();	

	}


	/**
	 * Save Xpad autoIndent or not
	 * @param boolean if autoIndent should be used or not
	 */
	public static boolean getAutoIndent() {

		/* Load file */
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList allSizeElements = xpadProfile.getElementsByTagName(AUTOINDENT);
		Element autoIndent = (Element) allSizeElements.item(0);

		if(autoIndent == null){
			return true;
		} else {
			return new Boolean(autoIndent.getAttribute(VALUE));
		}
	}


	/**
	 * Save Xpad autoColorize or not
	 * @param boolean if autoIndent should be used or not
	 */
	public static void saveAutoColorize(boolean activated) {

		/* Load file */
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList allSizeElements = xpadProfile.getElementsByTagName(AUTOCOLORIZE);
		Element xpadAutoIndent = (Element) allSizeElements.item(0);
		if (xpadAutoIndent == null){
			Element autoColorize = document.createElement(AUTOCOLORIZE);

			autoColorize.setAttribute(VALUE, new Boolean(activated).toString());

			xpadProfile.appendChild((Node) autoColorize);
		} else {
			xpadAutoIndent.setAttribute(VALUE, new Boolean(activated).toString()  );
		}
		/* Save changes */
		writeDocument();	

	}


	/**
	 * Save Xpad autoIndent or not
	 * @param boolean if autoIndent should be used or not
	 */
	public static boolean getAutoColorize() {

		/* Load file */
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList allSizeElements = xpadProfile.getElementsByTagName(AUTOCOLORIZE);
		Element autoColorize = (Element) allSizeElements.item(0);

		if(autoColorize == null){
			return true;
		} else {
			return new Boolean(autoColorize.getAttribute(VALUE));
		}
	}


	/**
	 * Save Xpad autoIndent or not
	 * @param boolean if autoIndent should be used or not
	 */
	public static void saveDefaultEncoding(String encoding) {

		/* Load file */
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList allSizeElements = xpadProfile.getElementsByTagName(DEFAULTENCONDING);
		Element xpadAutoIndent = (Element) allSizeElements.item(0);
		if (xpadAutoIndent == null){
			Element defaultEncoding = document.createElement(DEFAULTENCONDING);

			defaultEncoding.setAttribute(VALUE, encoding);

			xpadProfile.appendChild((Node) defaultEncoding);
		} else {
			xpadAutoIndent.setAttribute(VALUE, encoding  );
		}
		/* Save changes */
		writeDocument();	

	}


	/**
	 * Save Xpad autoIndent or not
	 * @param boolean if autoIndent should be used or not
	 */
	public static String getDefaultEncoding() {

		/* Load file */
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList allSizeElements = xpadProfile.getElementsByTagName(DEFAULTENCONDING);
		Element defaultEncoding = (Element) allSizeElements.item(0);

		if (defaultEncoding == null || defaultEncoding.getAttribute(VALUE).equals("")) {
			// If no default encoding read then used system default
			saveDefaultEncoding(Charset.defaultCharset().name());
			return Charset.defaultCharset().name();
		} else {
			return defaultEncoding.getAttribute(VALUE);
		}
	}

	/**
	 * Get all the foreground Colors 
	 * @return a Hashtable with the styles and the associated colors.
	 */

	public static Hashtable<String, Color> getAllForegroundColors() {
		/* Load file */
		readDocument();

		Hashtable<String, Color> stylesColorsTable = new Hashtable<String, Color>();

		Element root = document.getDocumentElement();
		NodeList styles = root.getElementsByTagName(STYLE);

		for (int i = 0; i < styles.getLength(); ++i) {
			Element style = (Element) styles.item(i);

			NodeList allForegroundElements = style.getElementsByTagName(FOREGROUNDCOLOR);
			Element styleForeground = (Element) allForegroundElements.item(0);
			Color styleColor = Color.decode(styleForeground.getAttribute(VALUE));

			stylesColorsTable.put(style.getAttribute(NAME), styleColor);
		}

		return stylesColorsTable;
	}

	/**
	 * get all default foreground colors of xpad
	 * @return a Hashtable with the styles and the associated default colors.
	 */
	public static Hashtable<String, Color> getAllDefaultForegroundColors() {
		/* Load file */
		readDocument();

		Hashtable<String, Color> stylesDefaultColorsTable = new Hashtable<String, Color>();

		Element root = document.getDocumentElement();
		NodeList styles = root.getElementsByTagName(STYLE);

		for (int i = 0; i < styles.getLength(); ++i) {
			Element style = (Element) styles.item(i);

			NodeList allForegroundElements = style.getElementsByTagName(FOREGROUNDCOLOR);
			Element styleForeground = (Element) allForegroundElements.item(0);
			Color styleColor = Color.decode(styleForeground.getAttribute(DEFAULT));

			stylesDefaultColorsTable.put(style.getAttribute(NAME), styleColor);
		}

		return stylesDefaultColorsTable;
	}	

	/**
	 * save all foreground colors
	 *@param stylesColorsTable a hashtable containing styles and the associated colors
	 */
	public static void saveAllForegroundColors(Hashtable<String, Color> stylesColorsTable) {
		/* Load file */
		readDocument();

		Element root = document.getDocumentElement();
		NodeList styles = root.getElementsByTagName(STYLE);

		for (int i = 0; i < styles.getLength(); ++i) {
			Element style = (Element) styles.item(i);

			String styleName = style.getAttribute(NAME);
			NodeList allForegroundElements = style.getElementsByTagName(FOREGROUNDCOLOR);
			Element styleForeground = (Element) allForegroundElements.item(0);

			Color color = stylesColorsTable.get(styleName);

			String rgb = Integer.toHexString(color.getRGB());
			styleForeground.setAttribute(VALUE, COLORPREFIX + rgb.substring(2, rgb.length()));


		}
		/* Save changes */
		writeDocument();
	}

	/**
	 * Get the position of Xpad Main Window
	 * @return the position
	 */
	public static Position getMainWindowPosition() {
		/* Load file */
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList allPositionElements = xpadProfile.getElementsByTagName(MAINWINPOSITION);
		Element mainWindowPosition = (Element) allPositionElements.item(0);
		if (mainWindowPosition != null) {
			int x = Integer.parseInt(mainWindowPosition.getAttribute(XCOORD));
			int y = Integer.parseInt(mainWindowPosition.getAttribute(YCOORD));
			/* Avoid Xpad Main Window to be out of the screen */
			if (x <= (Toolkit.getDefaultToolkit().getScreenSize().width - MARGIN)
					&& y <= (Toolkit.getDefaultToolkit().getScreenSize().height - MARGIN)) {
				return new Position(x, y);
			} else {
				return new Position(0, 0);
			}
		} else {
			return new Position(0, 0);
		}

		/*TODO*/
	}


	/**
	 * Save the position of Xpad Main Window
	 * @param position the position of Xpad main Window
	 */
	public static void saveMainWindowPosition(Position position) {
		/*TODO*/
		/* Load file */
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList allPositionElements = xpadProfile.getElementsByTagName(MAINWINPOSITION);
		Element mainWindowPosition = (Element) allPositionElements.item(0);


		mainWindowPosition.setAttribute(XCOORD, Integer.toString(position.getX()));
		mainWindowPosition.setAttribute(YCOORD, Integer.toString(position.getY()));

		/* Save changes */
		writeDocument();	
	}


	/**
	 * Save the size of Xpad Main Window
	 * @param size the size of Xpad main Window
	 */
	public static void saveMainWindowSize(Size size) {
		/*TODO*/
		/* Load file */
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList allPositionElements = xpadProfile.getElementsByTagName(MAINWINSIZE);
		Element mainWindowSize = (Element) allPositionElements.item(0);


		mainWindowSize.setAttribute(WIDTH, Integer.toString(size.getWidth()));
		mainWindowSize.setAttribute(HEIGHT, Integer.toString(size.getHeight()));

		/* Save changes */
		writeDocument();

	}

	/**
	 * Get the size of Xpad Main Window
	 * @return the size
	 */
	public static Size getMainWindowSize() {

		/* Load file */
		readDocument();

		Element root = document.getDocumentElement();

		NodeList profiles = root.getElementsByTagName(PROFILE);
		Element xpadProfile = (Element) profiles.item(0);

		NodeList allSizeElements = xpadProfile.getElementsByTagName(MAINWINSIZE);
		Element mainWindowSize = (Element) allSizeElements.item(0);
		if (mainWindowSize != null) {
			return new Size(Integer.parseInt(mainWindowSize.getAttribute(WIDTH)), Integer.parseInt(mainWindowSize.getAttribute(HEIGHT)));
		} else {
			return new Size(DEFAULT_WIDTH, DEFAULT_HEIGHT);
		}
	}

	/**
	 * Get all the  recent opened files
	 * @return a array of uri
	 */

	public static ArrayList<File> getAllRecentOpenedFiles() {
		ArrayList<File> files = new ArrayList<File>();

		readDocument();


		Element root = (Element)document.getDocumentElement().getElementsByTagName("recentFiles").item(0);
		if (root != null) {
			NodeList recentFiles = root.getElementsByTagName("document");

			for (int i = 0; i < recentFiles.getLength(); ++i) {
				Element style = (Element) recentFiles.item(i);

				File temp = new File(style.getAttribute("path"));

				if (temp.exists()) {
					files.add(temp);
				} else {
					root.removeChild((Node) style);
				}

				/* Save changes */
				writeDocument();
			}		
		}
		return files;
	}

	/**
	 * Add a file to recent Opened Files
	 * @param filePath the path of the files to add 
	 */
	public static void saveToRecentOpenedFiles(String filePath) {

		readDocument();

		Element root = (Element) document.getDocumentElement().getElementsByTagName("recentFiles").item(0);
		NodeList recentFiles = root.getElementsByTagName("document");
		int numberOfFiles = recentFiles.getLength();

		// we remove all the duplicate
		for (int i = 0; i < recentFiles.getLength();  ++i) {
			Element style = (Element) recentFiles.item(i);


			if (filePath.equals(style.getAttribute("path"))) {
				root.removeChild((Node) style);
				numberOfFiles--;
			}

		}


		// if we have reached the maximun , we remove the oldest files
		while (recentFiles.getLength() >= MAX_RECENT_FILES) {
			root.removeChild(root.getFirstChild());
		}

		Element newFile =  document.createElement("document");

		newFile.setAttribute("path", filePath);

		root.appendChild((Node) newFile);

		/* Save changes */
		writeDocument();



	}



	/**
	 * Read the file to modify
	 */
	private static void readDocument() {

		File xml = null;
		DocumentBuilder docBuilder = null;

		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
			docBuilder = factory.newDocumentBuilder();

			// read content of a XML file with DOM
			xml = new File(USER_XPAD_CONFIG_FILE);
			document = docBuilder.parse(xml);

		} catch (ParserConfigurationException pce) {
			System.out.println(ERROR_READ + USER_XPAD_CONFIG_FILE);
		} catch (SAXException se) {
			System.out.println(ERROR_READ + USER_XPAD_CONFIG_FILE);
		} catch (IOException ioe) {
			System.out.println(ERROR_READ + USER_XPAD_CONFIG_FILE);
		}

	}
	/**
	 * Save the modifications
	 */
	private static void writeDocument() {

		Transformer transformer = null; 
		try {
			transformer = TransformerFactory.newInstance().newTransformer();
		} catch (TransformerConfigurationException e1) {
			System.out.println(ERROR_WRITE + USER_XPAD_CONFIG_FILE);
		} catch (TransformerFactoryConfigurationError e1) {
			System.out.println(ERROR_WRITE + USER_XPAD_CONFIG_FILE);
		}
		transformer.setOutputProperty(OutputKeys.INDENT, "yes");

		StreamResult result = new StreamResult(new File(USER_XPAD_CONFIG_FILE));
		DOMSource source = new DOMSource(document);
		try {
			transformer.transform(source, result);
		} catch (TransformerException e) {
			System.out.println(ERROR_WRITE + USER_XPAD_CONFIG_FILE);
		}


	}
	/**
	 * Add a file to recent Opened Files
	 * 
	 * @param exp
	 *            the path of the files to add
	 */
	public static void saveRecentSearch(String exp) {

		Node root = getXcosRoot();
		if(root == null || exp == null || exp.compareTo("") == 0) {
			return;
		}

		Node recents = getNodeChild(root, RECENT_SEARCH);
		if(recents == null) {
			recents = document.createElement(RECENT_SEARCH);
			root.appendChild(recents);
		}

		ArrayList<Node> search = getNodeChildren(recents, SEARCH);

		while(search.size() >= MAX_RECENT_SEARCH) {
			removeRecentSearch(((Element)search.get(0)).getAttribute(EXPRESSION));
			search = getNodeChildren(recents, SEARCH);
		}
		//if path already in file no need to add it
		for(Node item : search) {
			if (exp.compareTo(((Element)item).getAttribute(EXPRESSION)) == 0) {
				return;
			}
		}

		Element newSearch = document.createElement(SEARCH);

		newSearch.setAttribute(EXPRESSION, exp);

		recents.appendChild((Node) newSearch);

		/* Save changes */
		writeDocument();
	}

	public static void removeRecentSearch(String exp) {

		Node root = getXcosRoot();
		if(root == null) {
			return;
		}

		Node recent = getNodeChild(root, RECENT_SEARCH);
		ArrayList<Node> search = getNodeChildren(recent, SEARCH);

		// remove node if exists
		for(Node file : search) {
			if (exp.compareTo(((Element)file).getAttribute(EXPRESSION)) == 0) {
				recent.removeChild(file);
				break;
			}

		}
		/* Save changes */
		writeDocument();

	}

	public static ArrayList<String> getRecentSearch() {
		ArrayList<String> files = new ArrayList<String>();

		Node root = getXcosRoot();
		if(root == null) {
			return files;
		}

		Node recent = getNodeChild(root, RECENT_SEARCH);
		ArrayList<Node> searches = getNodeChildren(recent, SEARCH);
		for(Node search : searches) {
			String exp = ((Element)search).getAttribute(EXPRESSION);
			if(exp != null && exp.compareTo("") != 0) {
				files.add(exp);
			}
		}

		return files;
	}

	public static void saveRecentReplace(String exp) {

		Node root = getXcosRoot();
		if(root == null || exp == null || exp.compareTo("") == 0) {
			return;
		}

		Node recent = getNodeChild(root, RECENT_REPLACE);
		if(recent == null) {
			recent = document.createElement(RECENT_REPLACE);
			root.appendChild(recent);
		}

		ArrayList<Node> replace = getNodeChildren(recent, REPLACE);

		while(replace.size() >= MAX_RECENT_REPLACE) {
			removeRecentReplace(((Element)replace.get(0)).getAttribute(EXPRESSION));
			replace = getNodeChildren(recent, REPLACE);
		}
		//if path already in file no need to add it
		for(Node item : replace) {
			if (exp.compareTo(((Element)item).getAttribute(EXPRESSION)) == 0) {
				return;
			}
		}

		Element newReplace = document.createElement(REPLACE);

		newReplace.setAttribute(EXPRESSION, exp);

		recent.appendChild((Node) newReplace);

		/* Save changes */
		writeDocument();
	}

	public static void removeRecentReplace(String filePath) {

		Node root = getXcosRoot();
		if(root == null) {
			return;
		}

		Node recent = getNodeChild(root, RECENT_REPLACE);
		ArrayList<Node> replace = getNodeChildren(recent, REPLACE );

		// remove node if exists
		for(Node exp : replace) {
			if (filePath.compareTo(((Element)exp).getAttribute(EXPRESSION)) == 0) {
				recent.removeChild(exp);
				break;
			}

		}
		/* Save changes */
		writeDocument();

	}

	public static ArrayList<String> getRecentReplace() {
		ArrayList<String> exps = new ArrayList<String>();

		Node root = getXcosRoot();
		if(root == null) {
			return exps;
		}

		Node recent = getNodeChild(root, RECENT_REPLACE);
		ArrayList<Node> replace = getNodeChildren(recent, REPLACE);
		for(Node file : replace) {
			String exp = ((Element)file).getAttribute(EXPRESSION);
			if(exp != null && exp.compareTo("") != 0) {
				exps.add(exp);
			}
		}

		return exps;
	}

	public static boolean getRegularExpression() {
		return getBooleanAttribute(REGULAR_EXPRESION, STATE_FLAG, false);
	}
	
	public static void saveRegularExpression(boolean regualExp) {
		saveBooleanAttribute(REGULAR_EXPRESION, STATE_FLAG, regualExp);
	}
	
	public static boolean getWholeWord() {
		return getBooleanAttribute(WHOLE_WORD, STATE_FLAG, false);
	}
	
	public static void saveWholeWord(boolean wholeWord) {
		saveBooleanAttribute(WHOLE_WORD, STATE_FLAG, wholeWord);
	}
	
	public static boolean getWordWarp() {
		return getBooleanAttribute(WORD_WARP, STATE_FLAG, true);
	}
	
	public static void saveWordWarp(boolean wordWarp) {
		saveBooleanAttribute(WORD_WARP, STATE_FLAG, wordWarp);
	}
	
	public static boolean getCaseSensitive() {
		return getBooleanAttribute(CASE_SENSITIVE, STATE_FLAG, false);
	}
	
	public static void saveCaseSensitive(boolean caseSensitive) {
		saveBooleanAttribute(CASE_SENSITIVE, STATE_FLAG, caseSensitive);
	}
	
	private static boolean getBooleanAttribute(String node, String attrib, boolean defaultValue) {
		boolean flag = false;
		Node root = getXcosRoot();
		if(root == null) {
			return flag;
		}
		Node recent = getNodeChild(root, node);
		if(recent != null) {
			String exp = ((Element)recent).getAttribute(attrib);
			if(exp.compareTo("true") == 0) {
				flag = true;
			}
		} else {
			return defaultValue;
		}
		return flag;
	}

	private static void saveBooleanAttribute(String node, String attrib, boolean state) {
		Node root = getXcosRoot();
		if(root == null) {
			return;
		}

		Node recent = getNodeChild(root, node);
		if(recent == null) {
			recent = document.createElement(node);
			root.appendChild(recent);
		}


		((Element)recent).setAttribute(attrib, new Boolean(state).toString());
		
		root.appendChild((Node) recent);

		/* Save changes */
		writeDocument();
	}
	
	private static Node getNodeChild(Node parent, String nodeName) {

		if(parent == null) {
			if(document == null) {
				readDocument();
				if(document == null) {
					return null;
				}
			}
			parent = document;
		}

		Node currentNode = parent.getFirstChild();
		while(currentNode != null) {
			if(currentNode.getNodeName().compareTo(nodeName) == 0){
				return currentNode;
			}
			currentNode = currentNode.getNextSibling();
		}
		return currentNode;
	}

	private static ArrayList<Node> getNodeChildren(Node parent, String childName) {
		ArrayList<Node> nodes = new ArrayList<Node>();

		if(parent == null) {
			if(document == null) {
				readDocument();
				if(document == null) {
					return nodes;
				}
			}
			parent = document;
		}

		Node currentNode = parent.getFirstChild();
		while(currentNode != null) {
			if(currentNode.getNodeName().compareTo(childName) == 0){
				nodes.add(currentNode);
			}
			currentNode = currentNode.getNextSibling();
		}
		return nodes;

	}

	private static Node getXcosRoot() {

		if(document == null) {
			readDocument();
			if(document == null) {
				return null;
			}
		}

		Node setting = getNodeChild(null, SETTING);

		if(setting != null) {
			ArrayList<Node> nodes = getNodeChildren(setting, PROFILE);
			for(Node node : nodes) {
				if(((Element)node).getAttribute(NAME).compareTo(XPAD) == 0) {
					return node;
				}
			}
		}
		return null;
	}	
}
