# -*- coding: utf-8 -*-

# Copyright (C) 2010-2011 by Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# Python X2go is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Python X2go is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

"""\
X2goProxy classes - proxying your connection through NX3 and others.

"""
__NAME__ = 'x2goproxynx3-pylib'

# modules
import gevent
import os
import sys
import types
import time
import copy
import threading

# Python X2go modules
import x2go.forward as forward
import x2go.utils as utils
import x2go.log as log
import base

from x2go.defaults import X2GOCLIENT_OS as _X2GOCLIENT_OS

class X2goProxyNX3(base.X2goProxyBASE):
    """\
    X2goNX3Proxy is a NX version 3 based X2go proxy connection class.

    It basically fills L{X2goProxyBASE} variables with sensible content. Its 
    methods mostly wrap around the corresponding methods of the parent class.

    """
    def __init__(self, *args, **kwargs):
        """\
        For available parameters refer to L{X2goProxyBASE} class documentation.

        """
        base.X2goProxyBASE.__init__(self, *args, **kwargs)

        # setting some default environment variables, nxproxy paths etc.
        if _X2GOCLIENT_OS == "Windows":
            _nxproxy_paths = [
                os.path.join(os.environ["ProgramFiles"], os.path.normpath("PyHoca-GUI/nxproxy/nxproxy.exe")),
                os.path.join(os.environ["ProgramFiles"], os.path.normpath("x2goclient/nxproxy.exe")),
                os.path.join(os.environ["ProgramFiles"], os.path.normpath("NX Client for Windows/bin/nxproxy.exe")),
            ]
            if os.environ.has_key('NXPROXY_BINARY'):
                _nxproxy_paths.insert(0, os.environ['NXPROXY_BINARY'])
            for _nxproxy_cmd in _nxproxy_paths:
                if os.path.exists(_nxproxy_cmd):
                    break
            self.PROXY_CMD = _nxproxy_cmd
        else:
            self.PROXY_CMD = "/usr/bin/nxproxy"
        self.PROXY_ENV.update({
            "NX_CLIENT": "/bin/true",
            "NX_ROOT": self.sessions_rootdir
        })
        self.PROXY_MODE = '-S'
        if _X2GOCLIENT_OS == "Windows":
            self.PROXY_OPTIONS = [
                "nx/nx" ,
                "retry=5",
                "composite=1",
                "connect=127.0.0.1",
                "clipboard=1",
                "cookie=%s" % self.session_info.cookie,
                "port=%d" % self.session_info.graphics_port,
                "errors=%s" % os.path.join(".", "..", "S-%s" % self.session_info.name, self.session_log, ),
            ]
        else:
            self.PROXY_OPTIONS = [
                "nx/nx" ,
                "retry=5",
                "composite=1",
                "connect=127.0.0.1",
                "clipboard=1",
                "cookie=%s" % self.session_info.cookie,
                "port=%d" % self.session_info.graphics_port,
                "errors=%s" % os.path.join(self.session_info.local_container, self.session_log, ),
            ]

        self.PROXY_DISPLAY = self.session_info.display

    def _update_local_proxy_socket(self, port):
        for idx, a in enumerate(self.PROXY_OPTIONS):
            if a.startswith('port='):
                self.PROXY_OPTIONS[idx] = 'port=%s' % port

    def _generate_cmdline(self):

        if _X2GOCLIENT_OS == "Windows":
            _options_filename = os.path.join(self.session_info.local_container, 'options')
            options = open(_options_filename, 'w')
            options.write('%s:%s' % (','.join(self.PROXY_OPTIONS), self.PROXY_DISPLAY))
            options.close()
            self.PROXY_OPTIONS= [ 'nx/nx', 'options=%s' % os.path.join("\\", "..", "S-%s" % self.session_info.name, 'options'), ]

        cmd_line = [ self.PROXY_CMD, ]
        cmd_line.append(self.PROXY_MODE)
        _proxy_options = "%s:%s" % (",".join(self.PROXY_OPTIONS), self.PROXY_DISPLAY)
        cmd_line.append(_proxy_options)
        return cmd_line

    def process_proxy_options(self):

        if self.proxy_options.has_key('defkeymap') and self.proxy_options['defkeymap']:

            # first: draw in xkb_rules_names from xprop output
            xkb_rules_names = utils.xkb_rules_names()
            _proxy_options = {
                'xkbrules': xkb_rules_names['rules'],
                'xkbmodel': xkb_rules_names['model'],
                'xkblayout': xkb_rules_names['layout'],
                'xkbvariant': xkb_rules_names['variant'],
                'xkboptions': xkb_rules_names['options'],
            }

            # merge self.proxy_options into the proxy_options we obtained from xprop
            _proxy_options.update(self.proxy_options)
            self.proxy_options = _proxy_options

            del self.proxy_options['defkeymap']

            # create keyboard file
            _keyboard = """\
rules=%s
model=%s
layout=%s
variant=%s
options=%s""" % ( self.proxy_options['xkbrules'],
                  self.proxy_options['xkbmodel'],
                  self.proxy_options['xkblayout'],
                  self.proxy_options['xkbvariant'],
                  self.proxy_options['xkboptions'], )

            # remove processed options from self.process_options
            del self.proxy_options['xkbrules']
            del self.proxy_options['xkbmodel']
            del self.proxy_options['xkblayout']
            del self.proxy_options['xkbvariant']
            del self.proxy_options['xkboptions']

            # write the keyboard file into the remote session directory
            self.session_instance.control_session._x2go_sftp_write('%s/keyboard' % self.session_info.remote_container, _keyboard)

        # run the base variant of this method (basically for logging of ignored process_options)
        base.X2goProxyBASE.process_proxy_options(self)

    def start_proxy(self):
        self.logger('starting local NX3 proxy...', loglevel=log.loglevel_INFO)
        self.logger('NX3 Proxy mode is server, cookie=%s, host=127.0.0.1, port=%s.' % (self.session_info.cookie, self.session_info.graphics_port,), loglevel=log.loglevel_DEBUG)
        self.logger('NX3 proxy writes session log to %s.' % os.path.join(self.session_info.local_container, 'session.log'), loglevel=log.loglevel_DEBUG)

        p = base.X2goProxyBASE.start_proxy(self)

        if self.ok():
            self.logger('NX3 proxy is up and running.', loglevel=log.loglevel_INFO)
        else:
            self.logger('Bringing up NX3 proxy failed.', loglevel=log.loglevel_ERROR)

        return p
