/*
  Plee the Bear

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file status_layer.hpp
 * \brief The status layer contains the area where are printed the score,
 *        energy, remaining lifes of the players.
 * \author Julien Jorge
 */
#ifndef __PTB_STATUS_LAYER_HPP__
#define __PTB_STATUS_LAYER_HPP__

#include "ptb/gui/horizontal_gauge.hpp"
#include "ptb/gui/vertical_gauge.hpp"
#include "ptb/throwable_item/throwable_item.hpp"

#include "generic_items/timer.hpp"

#include "engine/layer/gui_layer.hpp"
#include "visual/animation.hpp"
#include "visual/writing.hpp"
#include "universe/derived_item_handle.hpp"

namespace ptb
{
  class player;

  /**
   * \brief The status layer contains the area where are printed the score,
   *        energy, remaining lifes of the players.
   */
  class status_layer:
    public bear::engine::gui_layer
  {
  public:
    /** \brief The type of a list of scene elements retrieved from the layer. */
    typedef bear::engine::gui_layer::scene_element_list scene_element_list;

  private:
    /** \brief An handle on the timer of the level. */
    typedef
    bear::universe::const_derived_item_handle_maker<bear::timer>::handle_type
    timer_handle;

  private:
    /**
     * \brief All the informations we want to show about a player.
     */
    class player_status
    {
    public:
      player_status
      ( bear::engine::level_globals& glob,
        const bear::visual::font& f,
        bear::universe::const_derived_item_handle_maker
        <player>::handle_type& p);

      void progress( bear::engine::level_globals& glob,
                     bear::universe::time_type elapsed_time );
    private:
      bool update_throwable_item(unsigned int index);
      void update_tries( bear::engine::level_globals& glob,
                          bear::universe::time_type elapsed_time,
                          unsigned int index);
      void update_corrupting_bonus( bear::engine::level_globals& glob,
                                    bear::universe::time_type elapsed_time);
      void update_throwable_item_animation();
      void progress_gauge
      ( vertical_gauge& gauge, bool is_active, double value,
        double max_value, bool& new_active,
        bear::visual::coordinate_type& offset_x);

    public:
      /** \brief Level of energy. */
      horizontal_gauge energy;

      /** \brief Level of oxygen. */
      vertical_gauge oxygen;

      /** \brief The fire gauge. */
      vertical_gauge fire_gauge;

      /** \brief The ice gauge. */
      vertical_gauge ice_gauge;

      /** \brief The score. */
      bear::visual::writing score;

      /** \brief Number of throwable items. */
      bear::visual::writing throwable_items;

      /** \brief Number of remaining tries. */
      bear::visual::writing tries;

      /** \brief Last number of remaining tries. */
      unsigned int m_last_tries;

      /** \brief The time since the last trie. */
      bear::universe::time_type m_tries_time;

      /** \brief Pointer to the player. */
      bear::universe::const_derived_item_handle_maker<player>::handle_type
      m_player;

      /** \brief Indicates if the oxygen gauge is activated. */
      bool oxygen_active;

      /** \brief The offset of the oxygen gauge. */
      bear::visual::coordinate_type oxygen_offset_x;

      /** \brief Indicates if the fire gauge is activated. */
      bool fire_gauge_active;

      /** \brief The offset of the fire gauge. */
      bear::visual::coordinate_type fire_gauge_offset_x;

      /** \brief Indicates if the ice gauge is activated. */
      bool ice_gauge_active;

      /** \brief The offset of the ice gauge. */
      bear::visual::coordinate_type ice_gauge_offset_x;

      /** \brief The current animation of throwable_item. */
      bear::visual::animation throwable_item_animation;

    private:
      /** \brief The air power status in the last iteration. */
      bool m_air_power;

      /** \brief The fire power status in the last iteration. */
      bool m_fire_power;

      /** \brief The water power status in the last iteration. */
      bool m_water_power;

      /** \brief Indicates if the current throwable_item could be dropped in the
          last iteration. */
      bool m_last_throw_status;

      /** \brief The current throwable_item in the last iteration. */
      const throwable_item* m_last_throwable_item;

      /** \brief The length of the bars. */
      const static double s_bar_length;

    }; // class player_status

  public:
    status_layer();
    virtual ~status_layer();

    void build();
    void progress( bear::universe::time_type elapsed_time );
    void render( scene_element_list& e ) const;

    void set_timer( const timer_handle& t );

  private:
    void progress_time( bear::universe::time_type elapsed_time );
    void progress_corrupting_bonus( bear::universe::time_type elapsed_time );

    void render_player1( scene_element_list& e ) const;
    void render_player2( scene_element_list& e ) const;
    void render_corrupting_bonus( scene_element_list& e ) const;
    void render_hazelnut( scene_element_list& e ) const;
    void render_honeypots( scene_element_list& e ) const;
    void search_players();

  private:
    /** \brief Data of the first player. */
    player_status* m_data_1;

    /** \brief Data of the second player. */
    player_status* m_data_2;

    /** \brief The first player. */
    bear::universe::const_derived_item_handle_maker<player>::handle_type
    m_first_player;

    /** \brief The second player. */
    bear::universe::const_derived_item_handle_maker<player>::handle_type
    m_second_player;

    /** \brief The component in which we show the time. */
    bear::visual::writing m_text_time;

    /** \brief Indicate if the time must be displayed. */
    bool m_time_on;

    /** \brief The timer of the level. */
    timer_handle m_timer;

    /** \brief A small Plee next to the remaining tries. */
    bear::visual::sprite m_small_player;

    /** \brief Animation for corrupting bonus. */
    bear::visual::animation* m_corrupting_bonus;

    /** \brief The component in which we show the corrupting bonus count. */
    bear::visual::writing m_text_corrupting_bonus;

    /** \brief Last number of corrupting_bonus. */
    unsigned int m_last_corrupting_bonus;

    /** \brief The time since the last corrupting_bonus decreasing. */
    bear::universe::time_type m_corrupting_bonus_time;

    /** \brief Animation for hazelnut. */
    bear::visual::animation* m_hazelnut;

    /** \brief Animation for honeypots. */
    bear::visual::sprite m_honeypot;

    /** \brief Distance between the elements of the layer. */
    const static unsigned int s_margin;

  }; // class status_layer
} // namespace ptb

#endif // __PTB_STATUS_LAYER_HPP__
