/*
  Plee the Bear

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file code/players_present.cpp
 * \brief Implementation of the ptb::players_present class.
 * \author Sbastien Angibaud
 */
#include "ptb/item/players_present.hpp"

BASE_ITEM_EXPORT( players_present, ptb )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
ptb::players_present::players_present()
  : m_check_all_players(true)
{
  set_weak_collisions(true);
} // players_present::players_present()

/*----------------------------------------------------------------------------*/
/**
 * \brief Copy constructor.
 * \param that The item copied.
 */
ptb::players_present::players_present( const players_present& that)
  : super(that), m_check_all_players(that.m_check_all_players)
{

} // players_present::players_present()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c boolean.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::players_present::set_bool_field
( const std::string& name, bool value )
{
   bool ok = true;

  if (name == "players_present.check_all_players")
    m_check_all_players = value;
  else
    ok = super::set_bool_field(name,value);

  return ok;
} // players_present::set_bool_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c list of item.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool ptb::players_present::set_item_list_field
( const std::string& name, const std::vector<bear::engine::base_item*>& value)
{
  bool ok = true;

  if (name == "players_present.toggles")
    for( std::size_t i=0; i!=value.size(); ++i )
      m_items.push_back
        ( handle_type((bear::universe::physical_item*)value[i]) );
  else
    ok = super::set_item_list_field(name, value);

  return ok;
} // players_present::set_item_list_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Activation by one player among the two players.
 * \param p The player.
 */
void ptb::players_present::on_one_player( player* p )
{
  super::on_one_player(p);

  if ( !m_check_all_players )
    activate_toggles();
} // players_present::on_one_player()

/*----------------------------------------------------------------------------*/
/**
 * \brief Activation by all players.
 * \param p1 The first player.
 * \param p2 The second player. NULL in a single player game.
 */
void ptb::players_present::on_all_players( player* p1, player* p2 )
{
  super::on_all_players(p1,p2);

  if ( m_check_all_players )
    activate_toggles();
} // players_present::on_all_players()

/*----------------------------------------------------------------------------*/
/**
 * \brief Activate all toggles.
 */
void ptb::players_present::activate_toggles()
{
  handle_list::iterator it;
  std::list<handle_list::iterator> dead;

  for (it=m_items.begin(); it!=m_items.end(); ++it)
    if ( *it == (bear::engine::with_toggle*)NULL )
      dead.push_back(it);
    else
      (*it)->toggle(true,this);

  for ( ; !dead.empty(); dead.pop_front() )
    m_items.erase(dead.front());
} // players_present::activate_toggles()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the items concerned by a progress/move of this one.
 * \param d (out) A list to which are added such items.
 */
void ptb::players_present::get_dependent_items
( std::list<bear::universe::physical_item*>& d ) const
{
  super::get_dependent_items(d);

  handle_list::const_iterator it;

  for (it=m_items.begin(); it!=m_items.end(); ++it)
    if ( *it != (physical_item*)NULL )
      d.push_back(it->get_item());
} // players_present::get_dependent_items()
