/*
  Bear Engine

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file ambient_sound.cpp
 * \brief Implementation of the bear::ambient_sound class.
 * \author Julien Jorge
 */
#include "generic_items/ambient_sound.hpp"

BASE_ITEM_EXPORT( ambient_sound, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::ambient_sound::ambient_sound()
  : m_fadeout(0), m_sample(NULL)
{
  set_can_move_items(false);
  set_phantom(true);
} // ambient_sound::ambient_sound()

/*----------------------------------------------------------------------------*/
/**
 * \brief Destructor.
 */
bear::ambient_sound::~ambient_sound()
{
  delete m_sample;
} // ambient_sound::~ambient_sound()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c <sample>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::ambient_sound::set_sample_field
( const std::string& name, audio::sample* value )
{
  bool ok = true;

  if (name == "ambient_sound.sample")
    m_sample = value;
  else
    ok = super::set_sample_field(name, value);

  return ok;
} // ambient_sound::set_sample_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c <real>.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool bear::ambient_sound::set_real_field
( const std::string& name, double value )
{
  bool ok = true;

  if (name == "ambient_sound.fadeout")
    m_fadeout = value;
  else
    ok = super::set_real_field(name,value);

  return ok;
} // ambient_sound::set_real_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the item is well initialized.
 */
bool bear::ambient_sound::is_valid() const
{
  return (m_sample != NULL) && super::is_valid();
} // ambient_sound::is_valid()

/*----------------------------------------------------------------------------*/
/**
 * \brief Play the sample.
 * \param activator (ignored) The item that activates the toggle, if any.
 */
void bear::ambient_sound::on_toggle_on( engine::base_item* activator )
{
  audio::sound_effect effect(m_sample->get_effect());

  if (!is_global())
    effect.set_position( get_center_of_mass() );

  m_sample->play(effect);
} // ambient_sound::on_toggle_on()

/*----------------------------------------------------------------------------*/
/**
 * \brief Stop the sample.
 * \param activator (ignored) The item that activates the toggle, if any.
 */
void bear::ambient_sound::on_toggle_off( engine::base_item* activator )
{
  m_sample->stop(m_fadeout);
} // ambient_sound::on_toggle_off()
