/*
  Bear Engine

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file gui/checkbox.cpp
 * \brief Implementation of the gui::checkbox class.
 * \author Julien Jorge
 */
#include "gui/checkbox.hpp"

#include "gui/static_text.hpp"
#include "input/keyboard.hpp"
#include "visual/scene_sprite.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owner The control owning this one.
 * \param off The sprite displayed when the box is not checked.
 * \param on The sprite displayed when the box is checked.
 */
bear::gui::checkbox::checkbox
( visual_component* owner, const visual::sprite& off, const visual::sprite& on )
  : visual_component(owner), m_text(NULL), m_checked(false), m_off(off),
    m_on(on)
{
  create();
} // checkbox::checkbox()

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owner The control owning this one.
 * \param off The sprite displayed when the box is not checked.
 * \param on The sprite displayed when the box is checked.
 * \param f The font used to display the text.
 */
bear::gui::checkbox::checkbox
( visual_component* owner, const visual::sprite& off, const visual::sprite& on,
  font_type f )
  : visual_component(owner), m_text(NULL), m_checked(false), m_off(off),
    m_on(on)
{
  create();
  m_text->set_font(f);
} // checkbox::checkbox()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the font of the text.
 * \param f The new font.
 */
void bear::gui::checkbox::set_font( font_type f )
{
  set_size_maximum();
  m_text->set_font(f);
  fit();
} // checkbox::set_font()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the text of the control.
 * \param text The new text.
 */
void bear::gui::checkbox::set_text( const std::string& text )
{
  set_size_maximum();
  m_text->set_text(text);
  fit();
} // checkbox::set_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check/uncheck the box.
 * \param b Tell if the box is checked or not.
 */
void bear::gui::checkbox::check( bool b )
{
  if ( b == m_checked )
    return;

  m_checked = b;

  if ( m_checked )
    m_checked_callback.execute();
  else
    m_unchecked_callback.execute();
} // checkbox::check()

/*----------------------------------------------------------------------------*/
/**
 * \brief Toggle the value of the box.
 */
void bear::gui::checkbox::toggle_value()
{
  check( !checked() );
} // checkbox::toggle_value()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the text of the control.
 */
const std::string& bear::gui::checkbox::get_text() const
{
  return m_text->get_text();
} // checkbox::get_text()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the box is checked or not.
 */
bool bear::gui::checkbox::checked() const
{
  return m_checked;
} // checkbox::checked()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the callback called when the box becomes checked.
 * \param c The callback.
 */
void bear::gui::checkbox::set_checked_callback( const callback& c )
{
  m_checked_callback = c;
} // checkbox::set_checked_callback()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the callback called when the box becomes unchecked.
 * \param c The callback.
 */
void bear::gui::checkbox::set_unchecked_callback( const callback& c )
{
  m_unchecked_callback = c;
} // checkbox::set_unchecked_callback()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the scene elements of the component.
 * \param e (out) The scene elements.
 */
void bear::gui::checkbox::display( std::list<visual::scene_element>& e ) const
{
  claw::math::coordinate_2d<unsigned int> p(bottom_left());

  if (m_checked)
    {
      p.y += (height() - m_on.height()) / 2;
      e.push_back( visual::scene_sprite(p.x, p.y, m_on) );
    }
  else
    {
      p.y += (height() - m_off.height()) / 2;
      e.push_back( visual::scene_sprite(p.x, p.y, m_off) );
    }
} // checkbox::display()

/*----------------------------------------------------------------------------*/
/**
 * \brief Initialize the control.
 */
void bear::gui::checkbox::create()
{
  set_size_maximum();

  m_text = new static_text(this);
  m_text->set_auto_size(true);
  m_text->set_position( std::max(m_off.width(), m_on.width() ) + 5, 0 );

  fit();
} // checkbox::create()

/*----------------------------------------------------------------------------*/
/**
 * \brief Adjust the size of the control to the size of its sub controls.
 */
void bear::gui::checkbox::fit()
{
  set_size( std::max(m_on.width(), m_off.width()) + m_text->width() + 5,
            std::max( std::max(m_on.height(), m_off.height()),
                      m_text->height() ) );
} // checkbox::fit()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a key has been pressed.
 * \param key The code of the key.
 */
bool bear::gui::checkbox::on_key_press( const bear::input::key_info& key )
{
  bool result = true;

  if ( key.is_enter() || (key.get_code() == input::keyboard::kc_space) )
    toggle_value();
  else
    result = false;

  return result;
} // checkbox::on_key_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a joystick button has been pressed.
 * \param button The code of the button.
 * \param joy_index The index of the joytick.
 */
bool bear::gui::checkbox::on_button_press
( bear::input::joystick::joy_code button, unsigned int joy_index )
{
  bool result = true;

  switch( button )
    {
    case bear::input::joystick::jc_button_1:
    case bear::input::joystick::jc_button_2:
    case bear::input::joystick::jc_button_3:
    case bear::input::joystick::jc_button_4:
    case bear::input::joystick::jc_button_5:
    case bear::input::joystick::jc_button_6:
    case bear::input::joystick::jc_button_7:
    case bear::input::joystick::jc_button_8:
    case bear::input::joystick::jc_button_9:
    case bear::input::joystick::jc_button_10:
    case bear::input::joystick::jc_button_11:
    case bear::input::joystick::jc_button_12:
    case bear::input::joystick::jc_button_13:
    case bear::input::joystick::jc_button_14:
    case bear::input::joystick::jc_button_15:
    case bear::input::joystick::jc_button_16:
      toggle_value();
      break;
    default:
      result = false;
    }

  return result;
} // checkbox::on_button_press()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell that a mouse button has been pressed.
 * \param key The code of the button.
 * \param pos The position of the mouse.
 */
bool bear::gui::checkbox::on_mouse_press
( input::mouse::mouse_code key,
  const claw::math::coordinate_2d<unsigned int>& pos )
{
  toggle_value();
  return true;
} // checkbox::on_mouse_press()
