/*
    Bear Engine - Level editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/gui_level.hpp
 * \brief A level in the graphical user interface.
 * \author Julien Jorge
 */
#ifndef __BF_GUI_LEVEL_HPP__
#define __BF_GUI_LEVEL_HPP__

#include "bf/item_selection.hpp"
#include "bf/level.hpp"

#include <vector>

#include <claw/non_copyable.hpp>

namespace bf
{
  /**
   * \brief A level in the graphical user interface.
   * \author Julien Jorge
   */
  class gui_level:
    public level,
    claw::concept::non_copyable
  {
  public:
    /**
     * \brief Constructor.
     * \param width The width of the level.
     * \param height The height of the level.
     * \param mus The default music in the level.
     */
    gui_level
    ( unsigned int width, unsigned int height, const std::string& mus );

    /**
     * \brief Tell if the selection of a layer is not empty.
     * \param layer_index The layer to check.
     */
    bool has_selection( unsigned int layer_index ) const;

    /** \brief Tell if the selection of the layer on which we are working is not
        empty. */
    bool has_selection() const;

    /**
     * \brief Get the selection in a layer.
     * \param layer_index The index of the layer.
     */
    const item_selection& get_selection( unsigned int layer_index ) const;

    /** \brief Get the selection in the layer on which we are working. */
    const item_selection& get_selection() const;

    /**
     * \brief Get the main selection in a layer.
     * \param layer_index The index of the layer.
     */
    item_instance* get_main_selection( unsigned int layer_index ) const;

    /** \brief Get the main selection in the layer on which we are working. */
    item_instance* get_main_selection() const;

    /**
     * \brief Set the selection of a layer.
     * \param layer_index The index of the layer.
     * \param s The selection in the layer.
     * \pre All items in \a s are in the layer number \a layer_index.
     */
    void set_selection( unsigned int layer_index, const item_selection& s );

    /**
     * \brief Tell if an item is selected.
     * \param layer_index The layer to search the item in.
     * \param item The item to search.
     */
    bool item_is_selected
    ( unsigned int layer_index, item_instance const* item ) const;

    /**
     * \brief Tell if an item is selected in the layer in which we are working.
     * \param item The item to search.
     */
    bool item_is_selected( item_instance const* item ) const;

    /**
     * \brief Tell if an item is the main selection.
     * \param layer_index The layer to search the item in.
     * \param item The item to search.
     */
    bool item_is_main_selection
    ( unsigned int layer_index, item_instance const* item ) const;

    /**
     * \brief Tell if an item is the main selection.
     * \param layer_index The layer to search the item in.
     * \param item The item to search.
     */
    bool item_is_main_selection( item_instance const* item ) const;

    /**
     * \brief Add a selection in the selection of a layer.
     * \param layer_index The layer in which we select the items.
     * \param s The items to add.
     */
    void add_to_selection( unsigned int layer_index, const item_selection& s );

    /**
     * \brief Add an item in the selection of a layer.
     * \param layer_index The layer in which we select the item.
     * \param item The item to select.
     * \param main_selection Set the item as the main selection.
     */
    void add_to_selection
    ( unsigned int layer_index, item_instance* item,
      bool main_selection = false );

    /**
     * \brief Add an item in the selection of the layer on which we are working.
     * \param item The item to select.
     * \param main_selection Set the item as the main selection.
     */
    void add_to_selection( item_instance* item, bool main_selection = false );

    /**
     * \brief Remove an item from the selection of a layer.
     * \param layer_index The layer in which we deselect the item.
     * \param item The item to deselect.
     */
    void remove_from_selection( unsigned int layer_index, item_instance* item );

    /**
     * \brief Remove a selection from the selection of a layer.
     * \param layer_index The layer in which we deselect the items.
     * \param s The items to remove.
     */
    void
    remove_from_selection( unsigned int layer_index, const item_selection& s );

    /**
     * \brief Remove an item from the selection of the layer on which we are
     *        working.
     * \param item The item to deselect.
     */
    void remove_from_selection( item_instance* item );

    /**
     * \brief Clear the selection of a layer.
     * \param layer_index The layer in which we clear the selection.
     */
    void clear_selection( unsigned int layer_index );

    /** \brief Clear the selection in the layer in which we are working. */
    void clear_selection();

    /**
     * \brief Remove an item from the level.
     * \param layer_index The layer in which we remove the item.
     * \param item The item to remove.
     */
    void remove_item( unsigned int layer_index, item_instance* item );

    /**
     * \brief Tell if a layer is visible.
     * \param layer_index The index of the layer.
     */
    bool layer_is_visible( unsigned int layer_index ) const;

    /**
     * \brief Set the visibility of a layer.
     * \param layer_index The index of the layer.
     * \param b Visible or not?
     */
    void set_layer_visibility( unsigned int layer_index, bool b );

    /**
     * \brief Set the index of the layer to work on.
     * \param layer_index The index of the layer.
     */
    void set_active_layer( unsigned int layer_index );

    /** \brief Get the layer on which we are working. */
    layer& get_active_layer();

    /** \brief Get the layer on which we are working. */
    const layer& get_active_layer() const;

    /** \brief Get the index of the layer on which we are working. */
    unsigned int get_active_layer_index() const;

    layer& add_layer
    ( unsigned int width, unsigned int height, const std::string& class_name );
    void add_layer( layer* lay, unsigned int layer_index );
    layer* remove_layer( unsigned int layer_index );

    void move_backward( unsigned int layer_index );
    void move_forward( unsigned int layer_index );

  private:
    /** \brief The visibility of the layers. */
    std::vector<bool> m_layer_visibility;

    /** \brief The selection in each layer. */
    std::vector<item_selection> m_selection_by_layer;

    /** \brief Index of the layer on which we are working. */
    unsigned int m_active_layer;

  }; // class gui_level
} // namespace bf

#endif // __BF_GUI_LEVEL_HPP__
