/*
  Bear Engine

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file visual_component.hpp
 * \brief Base class for all gui components.
 * \author Julien Jorge
 */
#ifndef __GUI_VISUAL_COMPONENT_HPP__
#define __GUI_VISUAL_COMPONENT_HPP__

#include "visual/scene_element.hpp"
#include "input/input_listener.hpp"

#include <vector>

#include <claw/non_copyable.hpp>

#include "gui/class_export.hpp"

namespace bear
{
  namespace gui
  {
    /**
     * \brief Base class for all gui components.
     * \author Julien Jorge
     */
    class GUI_EXPORT visual_component :
      public claw::concept::non_copyable,
      public input::input_listener
    {
    public:
      typedef std::list<visual::scene_element> scene_element_list;

    private:
      typedef std::vector<visual_component*> component_list;

    public:
      visual_component( visual_component* owner );
      virtual ~visual_component();

      void render( scene_element_list& e ) const;

      bool key_pressed( const input::key_info& key );
      bool char_pressed( const input::key_info& key );
      bool button_pressed
      ( input::joystick::joy_code button, unsigned int joy_index );
      bool mouse_pressed( input::mouse::mouse_code button,
                          const claw::math::coordinate_2d<unsigned int>& pos );

      bool mouse_move( const claw::math::coordinate_2d<unsigned int>& pos );

      void set_size_maximum();
      void set_size( const claw::math::coordinate_2d<unsigned int>& size );
      void set_size( unsigned int w, unsigned int h );
      void set_width( unsigned int w );
      void set_height( unsigned int h );
      void set_position( const claw::math::coordinate_2d<unsigned int>& pos );
      void set_position( unsigned int x, unsigned int y );
      void set_visible( bool b );
      void set_tab_order( unsigned int v );
      void set_input_priority( bool this_first );

      unsigned int width() const;
      unsigned int height() const;
      unsigned int left() const;
      unsigned int right() const;
      unsigned int bottom() const;
      unsigned int top() const;
      claw::math::coordinate_2d<unsigned int> top_left() const;
      claw::math::coordinate_2d<unsigned int> top_right() const;
      claw::math::coordinate_2d<unsigned int> bottom_left() const;
      claw::math::coordinate_2d<unsigned int> bottom_right() const;

      claw::math::coordinate_2d<unsigned int> get_size() const;
      const claw::math::coordinate_2d<unsigned int>& get_position() const;
      const claw::math::rectangle<unsigned int>& get_rectangle() const;
      bool get_visible() const;

      void set_border_color( const claw::graphic::rgba_pixel& clr );

    protected:
      void add_component( visual_component* that );
      void
      change_tab_position( const visual_component* that, unsigned int pos );

      virtual void on_resized();

      virtual void display( scene_element_list& e ) const;

      virtual bool on_key_press( const input::key_info& key );
      virtual bool on_char_pressed( const input::key_info& key );
      virtual bool on_button_press( input::joystick::joy_code button,
                                    unsigned int joy_index );
      virtual bool on_mouse_press
      ( input::mouse::mouse_code key,
        const claw::math::coordinate_2d<unsigned int>& pos );
      virtual
      bool on_mouse_move( const claw::math::coordinate_2d<unsigned int>& pos );

    private:
      bool broadcast_mouse_move
      ( const claw::math::coordinate_2d<unsigned int>& pos );
      bool broadcast_mouse_press
      ( input::mouse::mouse_code button,
        const claw::math::coordinate_2d<unsigned int>& pos );

      void stay_in_owner();

    private:
      /**
       * \brief The box in which the component will be. The position is relative
       *        to the top left corner of the owner.
       */
      claw::math::rectangle<unsigned int> m_box;

      /** \brief The component containing this component. */
      visual_component* m_owner;

      /** \brief The components in this component. */
      component_list m_components;

      /** \brief The component having the focus. */
      int m_focused_component;

      /** \brief Tell if the component is visible or not. */
      bool m_visible;

      /** \brief Tell if the component catches the inputs before its
          children. */
      bool m_input_priority;

      /** \brief The color of the borders of the control. */
      claw::graphic::rgba_pixel m_border_color;

    }; // class visual_component
  } // namespace gui
} // namespace bear

#endif // __GUI_VISUAL_COMPONENT_HPP__
