/*
 *  Copyright 2007 Fabrice Colin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <unistd.h>
#include <fileref.h>
#include <tfile.h>
#include <tag.h>

#include "TagLibMusicFilter.h"

using std::string;
using namespace Dijon;

#ifdef _DYNAMIC_DIJON_FILTERS
bool get_filter_types(std::set<std::string> &mime_types)
{
	mime_types.clear();
	mime_types.insert("audio/mpeg");
	mime_types.insert("audio/x-mp3");
	mime_types.insert("application/ogg");
	mime_types.insert("audio/x-flac+ogg");
	mime_types.insert("audio/x-flac");

	return true;
}

bool check_filter_data_input(int data_input)
{
	Filter::DataInput input = (Filter::DataInput)data_input;

	if ((input == Filter::DOCUMENT_DATA) ||
		(input == Filter::DOCUMENT_STRING) ||
		(input == Filter::DOCUMENT_FILE_NAME))
	{
		return true;
	}

	return false;
}

Filter *get_filter(const std::string &mime_type)
{
	return new TagLibMusicFilter(mime_type);
}
#endif

TagLibMusicFilter::TagLibMusicFilter(const string &mime_type) :
	Filter(mime_type),
	m_parseDocument(false)
{
}

TagLibMusicFilter::~TagLibMusicFilter()
{
	rewind();
}

bool TagLibMusicFilter::is_data_input_ok(DataInput input) const
{
	if ((input == DOCUMENT_DATA) ||
		(input == DOCUMENT_STRING) ||
		(input == DOCUMENT_FILE_NAME))
	{
		return true;
	}

	return false;
}

bool TagLibMusicFilter::set_property(Properties prop_name, const string &prop_value)
{
	return false;
}

bool TagLibMusicFilter::set_document_data(const char *data_ptr, unsigned int data_length)
{
	return false;
}

bool TagLibMusicFilter::set_document_string(const string &data_str)
{
	return false;
}

bool TagLibMusicFilter::set_document_file(const string &file_path, bool unlink_when_done)
{
	if (Filter::set_document_file(file_path, unlink_when_done) == true)
	{
		m_parseDocument = true;

		return true;
	}

	return false;
}

bool TagLibMusicFilter::set_document_uri(const string &uri)
{
	return false;
}

bool TagLibMusicFilter::has_documents(void) const
{
	return m_parseDocument;
}

bool TagLibMusicFilter::next_document(void)
{
	if (m_parseDocument == true)
	{
		m_parseDocument = false;

		TagLib::FileRef fileRef(m_filePath.c_str(), false);
		if (fileRef.isNull() == false)
		{
			TagLib::Tag *pTag = fileRef.tag();

			if ((pTag != NULL) &&
				(pTag->isEmpty() == false))
			{
				char yearStr[64];

				string trackTitle(pTag->title().toCString(true));
				trackTitle += " ";
				trackTitle += pTag->artist().toCString(true);

				string pseudoContent(trackTitle);
				pseudoContent += " ";
				pseudoContent += pTag->album().toCString(true);
				pseudoContent += " ";
				pseudoContent += pTag->comment().toCString(true);
				pseudoContent += " ";
				pseudoContent += pTag->genre().toCString(true);
				snprintf(yearStr, 64, " %u", pTag->year());
				pseudoContent += yearStr;

				m_metaData["content"] = pseudoContent;
				m_metaData["title"] = trackTitle;
				m_metaData["ipath"] = "";
				m_metaData["mimetype"] = "text/plain";
				m_metaData["charset"] = "utf-8";
				m_metaData["author"] = pTag->artist().toCString(true);
			}
			else
			{
				// This file doesn't have any tag
				m_metaData["content"] = "";
				string::size_type filePos = m_filePath.find_last_of("/");
				if ((filePos != string::npos) &&
					(m_filePath.length() - filePos > 1))
				{
					m_metaData["title"] = m_filePath.substr(filePos + 1);
				}
				else
				{
					m_metaData["title"] = m_filePath;
				}
				m_metaData["ipath"] = "";
				m_metaData["mimetype"] = "text/plain";
				m_metaData["charset"] = "utf-8";
			}

			return true;
		}
	}

	return false;
}

bool TagLibMusicFilter::skip_to_document(const string &ipath)
{
	if (ipath.empty() == true)
	{
		return next_document();
	}

	return false;
}

string TagLibMusicFilter::get_error(void) const
{
	return "";
}

void TagLibMusicFilter::rewind(void)
{
	Filter::rewind();

	m_parseDocument = false;
}
