/*  Version 0.67  June, 2002
 *
 * pencam2; A program to download pictures, automatically take pictures at 
 * selected intervals, and change various camera parameters. Based on
 * pencam-0.50 and pensnap-0.21. 
 *
 *  by Kevin Sisson  <kjsisson@bellsouth.net>
 *
 * Originally based on " pencam- a quick and dirty picture dumper for my 
 * pencam.       (C) Copyright 2001 by Bart Hartgers "
 *
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * First version, Jan 2002. Combined pencam and pensnap, and added the 
 * ability to read/write a .pencam2rc file to store parameters
 * 
 * Ver 0.65: Jan - Feb, 2002. Fixed a few bugs, added new auto light filter 
 * routine from Kurt Garloff, added ability to read simple scripts.
 *
 * Ver 0.67: Oct, 2002. Added erase pictures function, and the 
 * beginnings of support for multiple cameras. Also added STV driver connect/
 * disconnect for MODIFIED usbcore.o code
 */


/* uncomment this to compile code to disconnect STV680 driver.
*  NOTE: this ONLY works if you have also modified the usbcore.c code
*  This will hopefully be in future kernels, beginning (maybe) with 2.4.21.
*  It should already be in the 2.5 series.
*/
//#define SHARE

/* Define this to be 1 to get more information; normally it is 0   */
#define DEBUG 0


#include <stdio.h>
#include <usb.h>
#include <stdlib.h>
#include <term.h>
#include <string.h>
#include <malloc.h>
#include <ctype.h>
#include <math.h>

#include "pencam2.h"
#include "unsharp.h"

extern struct termios kborig, kbnew;
extern int peek;


/*****  camera init and get picture routines *****/

int count_cameras(camera_count *camera_addr, int cnt)
{
	struct      usb_bus *bus;
	struct      usb_device *dev;
	int         i, j, l, num_cams;
	char        tmp_addr[20];
	
	    
	for (i=0; i<=MAX_CAMERAS; i++) {
		strcpy (camera_addr->bus_dirname[i], "");
		strcpy (camera_addr->dev_filename[i], "");
		if (camera_addr->in_use[i] != -1)
			camera_addr->in_use[i] = 0;
	}

	usb_init ();

	if ((i = usb_find_busses ()) < 0)
		fprintf (stderr, " Error returned from usb_find_busses (%i)\n", i);
	if ((i = usb_find_devices ()) < 0)
		fprintf (stderr, " Error returned from usb_find_devices (%i)\n", i);

	i = 0;
	for (bus=usb_busses; bus; bus=bus->next) {
		for (dev=bus->devices; dev; dev=dev->next) {
			if ((dev->descriptor.idVendor == 0x0553) &&
			    (dev->descriptor.idProduct == 0x0202)) {
				l = 0;
				for (j=0; j<(i+1); j++) {
					if ((strcmp (bus->dirname, camera_addr->bus_dirname[j]) == 0) &&
					    (strcmp (dev->filename, camera_addr->dev_filename[j]) == 0))
						l++;
				}
				if (l == 0) {   /* count from 1 not 0 */
					strcpy (camera_addr->bus_dirname[i+1], bus->dirname);
					strcpy (camera_addr->dev_filename[i+1], dev->filename);
					camera_addr->in_use[i+1] = 1;
					i++;
					if (i >= MAX_CAMERAS)
					    goto cc1;
				}
			}
		}
	}

cc1:
	num_cams = i;

        /* Do some sorting; bus_dirname first */
	if (num_cams > 1) {
		for (l=1; l<=(num_cams-1); l++) {
    			for (i=1; i<=num_cams-1; i++) {
				if (strcmp (camera_addr->bus_dirname[i], camera_addr->bus_dirname[i+1]) > 0) {
			    		strcpy (tmp_addr, camera_addr->dev_filename[i]);
					strcpy (camera_addr->dev_filename[i], camera_addr->dev_filename[i+1]);
					strcpy (camera_addr->dev_filename[i+1], tmp_addr);
			 		strcpy (tmp_addr, camera_addr->bus_dirname[i]);
					strcpy (camera_addr->bus_dirname[i], camera_addr->bus_dirname[i+1]);
					strcpy (camera_addr->bus_dirname[i+1], tmp_addr);
					j = camera_addr->in_use[i];
					camera_addr->in_use[i] = camera_addr->in_use[i+1];
					camera_addr->in_use[i+1] = j;
				}
			}
		}
	}
	/* now dev_filename */
	if (num_cams > 1) {
		for (l=1; l<=(num_cams-1); l++) {
    			for (i=1; i<=num_cams-1; i++) {
				if ((strcmp (camera_addr->dev_filename[i], camera_addr->dev_filename[i+1]) > 0)
				    && (strcmp (camera_addr->bus_dirname[i], camera_addr->bus_dirname[i+1]) == 0)) {
			    		strcpy (tmp_addr, camera_addr->dev_filename[i]);
					strcpy (camera_addr->dev_filename[i], camera_addr->dev_filename[i+1]);
					strcpy (camera_addr->dev_filename[i+1], tmp_addr);
			 		strcpy (tmp_addr, camera_addr->bus_dirname[i]);
					strcpy (camera_addr->bus_dirname[i], camera_addr->bus_dirname[i+1]);
					strcpy (camera_addr->bus_dirname[i+1], tmp_addr);
					j = camera_addr->in_use[i];
					camera_addr->in_use[i] = camera_addr->in_use[i+1];
					camera_addr->in_use[i+1] = j;
				}
			}
		}
	}

	return num_cams; 
}  /*  count_cameras  */


usb_dev_handle *pencam_open (usb_stv *stv680)
{
	struct      usb_bus *bus;
	struct      usb_device *dev = NULL;
	static int  first = 1;

	if (first) {
		usb_init ();
		usb_find_busses ();
		usb_find_devices ();
		first = 0;
	}
	for (bus=usb_busses; bus; bus=bus->next) {
		for (dev=bus->devices; dev; dev=dev->next) {
			if ((dev->descriptor.idVendor == 0x0553) &&
			    (dev->descriptor.idProduct == 0x0202)) {
				strcpy (stv680->bus_dirname, bus->dirname);
				strcpy (stv680->dev_filename, dev->filename);
				return usb_open (dev);
			}
		}
	}
	return NULL;
}

usb_dev_handle *pencam_open_all (usb_stv *stv680, char *cam_bus_dirname, char *cam_dev_filename)
{
	struct      usb_bus *bus;
	struct      usb_device *dev = NULL;
	static int  first = 1;

	if (first) {
		usb_init ();
		usb_find_busses ();
		usb_find_devices ();
		first = 0;
	}
	for (bus=usb_busses; bus; bus=bus->next) {
		for (dev=bus->devices; dev; dev=dev->next) {
			if ((dev->descriptor.idVendor == 0x0553) &&
			    (dev->descriptor.idProduct == 0x0202) &&
			    (strcmp(bus->dirname, cam_bus_dirname) == 0) &&
			    (strcmp(dev->filename, cam_dev_filename) == 0)) {
				strcpy (stv680->bus_dirname, bus->dirname);
				strcpy (stv680->dev_filename, dev->filename);
				return usb_open (dev);
			}
		}
	}
	return NULL;
}

int clear_all_images (usb_dev_handle *pc, usb_stv *stv680)
{
	unsigned char  buffer[16];

	pencam_vendor (1, pc, 0x06, 0, 0, buffer);
	return 0;

}


int get_pictures (usb_dev_handle *pc, usb_stv *stv680, int first_pic, 
		  int last_pic)
{
	int    i, pic_count;


	/**** get the pictures and process them  *****/

	if ((first_pic < 0) || (last_pic < 0) || (last_pic > stv680->num_pics) 
	    || (first_pic > last_pic))
		return -1;

	pic_count = 0;
	fprintf (stdout, " Please wait. Preparing to download pictures...\n");

	for (i=first_pic; i<last_pic; ++i) 
	{
		stv680->pic_no = i+1;
		
		if (pencam_get_picture (pc, stv680, i) < 0) {
			sleep(1);
			if (pencam_get_picture (pc, stv680, i) < 0)
				fprintf (stderr, "\a\a Made 2 tries to get picture, moving to next picture\n");
		}			
		else {
			pic_count++;
		}
	}  /* for */
	    
	return pic_count;

}  /*  get_pictures  */


int pencam_get_picture (usb_dev_handle *pc, usb_stv *stv680, int pic_no)
{
	int       s;
	unsigned  width, height;
	unsigned long int sofar;
	unsigned char  buffer[16];
	unsigned char  *raw, *pic;
	
	
	/* get image header */
	if (pencam_vendor (0, pc, 0x8f, 16, pic_no, buffer) < 16)
		return -1;

	stv680->rawbufsize = (buffer[0]<<24) | (buffer[1]<<16)
			    | (buffer[2]<<8) | (buffer[3]);
	width = (buffer[4]<<8)|(buffer[5]); 	stv680->cwidth = width;
	height = (buffer[6]<<8)|(buffer[7]); 	stv680->cheight = height;

	if ((stv680->QVGA==1) && (width >= 318) && (width <= 324))  { /* QVGA */
	    stv680->vwidth = 320;	stv680->vheight = 240;
	}	
	else if ((stv680->VGA==1) && (width >= 638) && (width <= 646)) { /* VGA */
	    stv680->vwidth = 640;	stv680->vheight = 480;
	}	
	else if ((stv680->CIF==1) && (width >= 350) && (width <= 356)) { /* CIF */
	    stv680->vwidth = 352;	stv680->vheight = 288;
	}	
	else if ((stv680->QCIF==1) && (width >=172) && (width <= 180)) { /* QCIF */
	    stv680->vwidth = 176;	stv680->vheight = 144;
	}	
	else {
	    fprintf(stderr, "\a\a Picture dimensions not valid!\n");
	    return -1;
	}
	stv680->FineExp = (buffer[8]<<8) | (buffer[9]);
	stv680->CoarseExp = (buffer[10]<<8) | (buffer[11]);
	stv680->origGain = buffer[12];
	stv680->CLKDIV = buffer[13];
	stv680->AvgPixVal = buffer[14];
		
	raw = malloc (stv680->rawbufsize);
	if (raw == NULL) return -1;

	pic = malloc (3*stv680->rawbufsize);
	if (pic == NULL) {
		free (raw);
		return -1;
	}

	/*  set up bulk channel */
	if (pencam_vendor (0, pc, 0x83, 16, pic_no, buffer) < 16) {
		free (raw);
		free (pic);
		return -1;
	}
	
	sofar = 0;  /*  get picture */

	s = usb_bulk_read (pc, 0x82 , (raw +sofar), (int)(stv680->rawbufsize - sofar), 2*PENCAM_TIMEOUT);  
	if (s <= 0) {
		fprintf (stderr, "\a\a\n Usb_bulk_read error in pencam_get_picture\n");
		/* clear comms error */
		pencam_vendor (0, pc, 0x0, 0, 0, buffer);
		sleep (1);
		/* clear buffer */
		free (raw);
		free (pic);
		return -1;
	}

	/* do image processing and save to disk */
	common_image_process (pc, stv680, raw, pic, "PENCAM");
	if (common_save_image (stv680, stv680->pic_no, pic, "PENCAM") < 0)
		fprintf (stderr, "\a\a\n Error saving picture %i in pencam_get_picture!\n", pic_no);

	free (raw);
	free (pic);
	return 0;
}  /*  pencam_get_picture  */

int pencam_vendor(int set, usb_dev_handle *pc, int type, int size,
	      	  int value,  unsigned char *buffer)
{
	int ret = -1;  /* VENDOR=0x40, DEVICE=0x0, ENDPOINT=0x01  */

	switch (set) {   
		case 0:     /*  0xc1  */
   			ret = usb_control_msg (pc, (0x80 | USB_TYPE_VENDOR | USB_RECIP_DEVICE),
            				type, value, 0, buffer, size, PENCAM_TIMEOUT);
			if (ret != size)
				ret = -1;
			break;
	
		case 1:     /*  0x41  */
   			ret = usb_control_msg (pc, (USB_TYPE_VENDOR | USB_RECIP_ENDPOINT),
            				type, value, 0, buffer, size, PENCAM_TIMEOUT);
			break;
	
		case 2:     /*  0x80  */
   			ret = usb_control_msg (pc, (0x80 | USB_RECIP_DEVICE),
            				type, value, 0, buffer, size, PENCAM_TIMEOUT);
			break;
	
		case 3:     /*  0x40  */
   			ret = usb_control_msg (pc, (USB_TYPE_VENDOR | USB_RECIP_DEVICE),
            				type, value, 0, buffer, size, PENCAM_TIMEOUT);
			break;
	}

        if ((ret < 0)) {
    		usb_control_msg (pc, (0x80 | USB_TYPE_VENDOR | USB_RECIP_DEVICE),
		                 0x80, 0, 0, buffer, 0x02, PENCAM_TIMEOUT);
	        fprintf (stderr, "\a\a usb_control_msg error: %i,  command = 0x%x\n", 
				  buffer[0], buffer[1]);
		return -1;
	}
	return ret;
}

int pencam_configuration( usb_dev_handle *pc, int conf, int intrface, int altiface )
{
	if (usb_set_configuration (pc, conf) < 0) {
		fprintf (stderr, "\a\a pencam_set_configuration error\n");
		return -1;
	}
	if (usb_claim_interface (pc, intrface) < 0) {
		fprintf (stderr,"\a\a usb_claim_interface error; is the stv680 driver loaded?\n");
		return -1;
	}
	if (usb_set_altinterface (pc, altiface) < 0) {
		fprintf (stderr,"\a\a usb_set_altinterface error\n");
		return -1;
	}
	return 0;
}

int pencam_init(usb_dev_handle *pc, usb_stv *stv680)
{
        int i, check=0;
	unsigned char buffer[40];

pi1:
        memset (buffer,0xff,40);
  
	/* mimick windows driver setup */
        if (pencam_configuration (pc,1,0,0) < 0)
		return -1;

	/* ping camera to be sure STV0680 is present */
        pencam_vendor (0, pc, 0x88, 0x02, 0x7856, buffer); 
	if ((buffer[0] != 0x78) || (buffer[1] != 0x56))  {
		check++;
		if (check >= 3) {
			fprintf(stderr, "\a\a Camera ping failed!! Check connections and try again!\n");
			return -1;
		} else {
			sleep (1);
			goto pi1;
		}
	}

        /* this returns camera info */
	if ((i = pencam_vendor (0, pc, 0x85, 0x10, 0, buffer)) != 0x10) {
	    fprintf(stderr, "\a\a\n Error getting camera info in pencam_init\n"); 
	    return -1;
	}
	stv680->SupportedModes = buffer[7];   i = stv680->SupportedModes;
        stv680->CIF = 0; stv680->VGA = 0; stv680->QVGA = 0; stv680->QCIF = 0;
	if (i & 1)
	    stv680->CIF = 1;
        if (i & 2)
    	    stv680->VGA = 1;
	if (i & 4)
	    stv680->QCIF = 1;
	if (i & 8)
	    stv680->QVGA = 1;
	if (stv680->SupportedModes == 0) {
	    fprintf(stdout, " There are NO supported STV680 modes!!");
	    return -1; 
	}
        /* FW rev, ASIC rev, sensor ID  */
	stv680->FW = ((float)(((int)buffer[0])*10 + (int)buffer[1]))/10;
        stv680->ASICrev =  ((float)(((int)buffer[2])*10 + (int)buffer[3]))/10;
	stv680->SensorID =  ((int)buffer[4])*16 + (int)(buffer[5]>>4);
        stv680->SensorIDrev =  buffer[5] & 0x0f;

        memset(buffer,0x0,40);
	/* get current mode */
	if ((i = pencam_vendor (0, pc, 0x87, 0x08, 0, buffer)) != 0x08) 
	    fprintf (stderr, "\a\a Error getting camera mode in pencam_init\n"); 
        else  {
	    stv680->currentMode = buffer[0];
	}

	/* set up for still image download */    
        if (pencam_configuration (pc,1,0,1) < 0)
		return -1;

	if ((i = pencam_vendor (0, pc, 0x8d, 0x08, 0, buffer)) != 0x08) return -1; /* get user info */
	stv680->num_pics = buffer[3];

        return 0;
	
}  /*  pencam_init  */


/*****   menu routines   *****/

void pencam_do_function (usb_dev_handle *pc, usb_stv *stv680, char choice)
{
        int     i, first_pic, last_pic, num_pics;
        double  r;


	common_do_function (pc, stv680, "PENCAM", choice);

	num_pics = stv680->num_pics;

	switch (choice) {

	    case 'i':
		fprintf (stdout, "\n Press <enter> to download all, or number of 1st picture: ");
		r = get_val (1, (float) num_pics);
		if (r == -11)
			break;
		else if (r == -1) {
			i = get_pictures (pc, stv680, 0, stv680->num_pics);
			goto inxt;		
		} else {
	 		first_pic = (int) rint(r); 
			first_pic--;
		        if (first_pic < 0)
		    		first_pic = 0;
			fprintf (stdout, " Press <enter> for just this picture, or number of last picture: ");
			r = get_val (1, (float) num_pics);    
			if (r == -11) 
		   	        break;
			else if (r == -1) 
				last_pic = first_pic + 1;		   
			else 
				last_pic = (int) rint(r);
			i = get_pictures (pc, stv680, first_pic, last_pic);
		}

inxt:		
		fflush (stdin);
		fprintf (stdout, " Press any key to continue ");
		get_key();
		fprintf (stdout, "\n");
		break;		

	    case 'r':
		clear_all_images (pc, stv680);
		break;

	    case 'z':
		fprintf (stdout, " exiting\n");
		break;
	    		
	    default:
		fprintf (stdout, "\n");
		break;
	}  /* switch */

	fprintf (stdout, "\n");
	return;

} /* pencam_do_function */


void load_defaults (usb_stv *stv680)
{
        char pathname[100], basename[50];

        /* pencam */
	stv680->light_type = 3;  /* auto select, based on Fine and Coarse Exp */
        stv680->saturate_toggle = 1;
	stv680->saturation_data = 40;
        stv680->lightness_data = 50;
	stv680->hue_data = 50;
        stv680->brightness = 1.0;
	stv680->sharp_toggle = 1;
        stv680->sharpen_percent = 55;
	stv680->unsharpen_radius = 7.0; //unsharp_params.radius;
        stv680->unsharpen_amount = 0.5; //unsharp_params.amount;
	stv680->unsharpen_threshold = 4; //unsharp_params.threshold;
    
        /* pensnap */
	stv680->beep_yes = 1;
        stv680->time_delay = 60;
	stv680->overwrite = 1;
        stv680->light_type_ps = 3;  /* auto select, based on Fine and Coarse Exp */
	stv680->saturate_toggle_ps = 1;
        stv680->saturation_data_ps = 40;
	stv680->lightness_data_ps = 50;
        stv680->hue_data_ps = 50;
	stv680->brightness_ps = 1.0;
        stv680->sharp_toggle_ps = 1;
        stv680->sharpen_percent_ps = 55;
	stv680->unsharpen_radius_ps = 7.0; //unsharp_params.radius;
        stv680->unsharpen_amount_ps = 0.5; //unsharp_params.amount;
	stv680->unsharpen_threshold_ps = 4; //unsharp_params.threshold;
	stv680->annotate_ps = 1;
	strcpy (stv680->picmsg_ps, "STVcam ");

	stv680->convert = -1;
	strcpy (stv680->convert_path, " ");	
	
        /* default path: user's home directory ; default pic name: pencam */
	strcpy (pathname, getenv("HOME"));
        strcat (pathname, "/");
	strcpy (stv680->pathname, pathname);     /* pencam */
        strcpy (basename, "pencam");
	strcpy (stv680->basename, basename);

        strcpy (stv680->pathname_ps, pathname);  /* pensnap */
	strcpy (stv680->basename_ps, "snapshot");

}  /*  load_defaults  */


int execute_script (int argc, char *argv[], usb_dev_handle *pc, usb_stv *stv680)
{
	char   outpath[75], script[2810] = " ";
	char   str_cmd[20][140], num[10];
	FILE   *of;
	int    i, j, k, l, m, sl, first_pic, last_pic, cmd_num, slmax = 138;
	
	
	if (argc == 1)
		return 0;
		
	if (argc > 1) {
		strcpy (outpath, getenv("HOME"));
		strcat (outpath, "/");
		strcat (outpath, argv[1]);
		if ((of = fopen (outpath,"rb")) == NULL) {
			fprintf (stderr, "\n\a ** no script file **\n");
			return 0;
		} else
			fprintf (stdout, "Executing script file %s\n", outpath);
		fread (script, 2805, 1, of);
		fclose (of);
	}

	stv680->num_pics = pencam_get_num_pics (pc, stv680);
	
	/* break script file into commands */
	j = 0;		cmd_num = 0;		l = 1;
	for (i=0; i<2805; i++) {
		if (script[i] == '\0') 
			break;
		if (script[i] != '\n') {
			str_cmd[cmd_num][j] = script[i]; 
			j++;
			if (j > (slmax-1))
				script[i] = '\n';
		}
		if (script[i] == '\n')  {
			if (str_cmd[cmd_num][0] == '#') {
				fprintf (stdout, "Found a comment on line %i, proceeding..\n", l);
				j = 0;
				l++;
				goto ec1;
			}  else {
				str_cmd[cmd_num][j] = '\0';
				cmd_num++;
				j = 0;
				l++;
			}
		}
ec1:		k = 0;
	}

	/*  cmd_num = # of commands  */
	fprintf (stdout, "\nThere are %i commands:\n", cmd_num);
	for (i=0; i<cmd_num; i++) 
		fprintf (stdout, "%s\n", str_cmd[i]);
	fprintf (stdout, "\n");
		
	/* figure out what to do */
	for (i=0; i<cmd_num; i++) {
		
	    	if (str_cmd[i][0] ==  '#') 
			continue;
		if ((strpbrk(str_cmd[i], "i") != NULL) && (strpbrk(str_cmd[i], "CONVERT") == NULL)) {

			if (stv680->num_pics == 0) {
				fprintf (stdout, "The camera contains no pictures, procededing..\n");
				goto ec2;
			}
			fprintf (stdout, "Downloading images\n");
			first_pic = 0;
			last_pic = stv680->num_pics;
			k = 0;		m = 0;		l = 0;
			for (j=0; j<10; j++)
					num[j] = '\0';
			sl = strlen(str_cmd[i]);
			if (sl > slmax)
				sl = slmax;
			for (j=0; j<sl; j++) {
				if (str_cmd[i][j] == 'i') {
				        l = j;
					l++;
					break;
				}
			}
			for (j=l; j<sl; j++) {
			
				if (str_cmd[i][j] == '\n')
					break;
				while ((str_cmd[i][j] != ' ') && (str_cmd[i][j] != '\n')) {
					num[m] = str_cmd[i][j];
					m++;
					j++;
					if (k == 0)
					    k = 1;
					else if (k == 2)
					    k = 3;
				}
				if ((k == 1) && (m > 0)) {
					first_pic = atoi (num);
					first_pic --;
					if (first_pic < 0)
						first_pic = 0;
					k = 2;
					m = 0;
				} else if ((k == 3) && (m > 0)) {
					last_pic = atoi (num);
				}
			} 

			if (k == 2)
				last_pic = first_pic + 1;
			if (last_pic > stv680->num_pics)
				last_pic = stv680->num_pics;
			get_pictures (pc, stv680, first_pic, last_pic);
		}  /*  'i'  */
ec2:
		if (strpbrk(str_cmd[i], "CONVERT") != NULL) {
			l = 0;
			sl = strlen(str_cmd[i]);
			if (sl > slmax)
				sl = slmax;
			for (j=1; j<sl; j++) {
				if (str_cmd[i][j] == '"')
					l++;
			}
			if (l != 2) {
				fprintf (stderr, "\a\aDid you enclose the command/path in double quotes?\n");
				goto ec3;
			}
			j = 1;
			while (str_cmd[i][j] != '"') {
				j++;
			}
			j++;
			m = 0;
			for (k=j; k<sl; k++) {
				if (str_cmd[i][k] != '"') {
					stv680->convert_path[m] = str_cmd[i][k];
					m++;
				} else {
					stv680->convert_path[m] = '\0';
					stv680->convert = 1;					
					goto ec3;
				}
			}  /* for k  */
		}  /*  CONVERT  */
ec3:	
		if ((strpbrk(str_cmd[i], "e") != NULL)  && (strpbrk(str_cmd[i], "CONVERT") == NULL)) {
			fprintf (stdout, "Reading config file\n");
		    	pencam_read_config (pc, stv680);
		}
		
		if ((strpbrk(str_cmd[i], "r") != NULL)  && (strpbrk(str_cmd[i], "CONVERT") == NULL)) {
			fprintf (stdout, "Erasing all pictures in camera\n");
		    	clear_all_images (pc, stv680);
		}

		if ((strpbrk(str_cmd[i], "x") != NULL)  && (strpbrk(str_cmd[i], "CONVERT") == NULL)) {
		    	fprintf (stdout, "Starting snaphot process\n");
		    	get_snapshot (pc, stv680);
		}

		if ((strpbrk(str_cmd[i], "q") != NULL) && (strpbrk(str_cmd[i], "CONVERT") == NULL)) {
			return -1;
		}
	}

	return 0;
}

#ifdef SHARE

/*
*  Program to permit sharing of device between kernel driver and user space
*  Originally by Dan Streetman <ddstreet@us.ibm.com>
*
*  example usage (stv cameras use interface 0):
*  <prog> /proc/bus/usb/001/005 -0
*  to disconnect interface number (not position) 0 from device 5 on HC 1
*
*  to connect,
*  <prog> /proc/bus/usb/001/005 0
*/

#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/usbdevice_fs.h>


#define CONNECT 1
#define DISCONNECT 2

#define USBDEVFS_DISCONNECT        _IO('U', 22)
#define USBDEVFS_CONNECT           _IO('U', 23)
#define USBDEVFS_IOCTL             _IOWR('U', 18, struct usbdevfs_ioctl)


int share_device (char *camera_addr_busname, char *camera_addr_devname , int pcmode)
{
	struct   usbdevfs_ioctl *connect_ioctl;
	int      iface_num=0, fd, mode, errno=0, i=0;
	char     devnode[40];
	char     *modestr, *disconnect = "disconnect", *connect = "connect";


	if (!(connect_ioctl = malloc(sizeof(*connect_ioctl)))) {
    		fprintf (stderr, "\a\a Couldn't malloc connect_ioctl struct.\n");
		return -1;
	}

	strcpy (devnode, "/proc/bus/usb/");
	strcat (devnode, camera_addr_busname);
	strcat (devnode, "/");
	strcat (devnode, camera_addr_devname);

	if (pcmode >= 0) 
		mode = CONNECT;
	else if (pcmode < 0)
		mode = DISCONNECT;

	modestr = mode == DISCONNECT ? disconnect : connect;

	connect_ioctl->ifno = abs( iface_num );
	connect_ioctl->ioctl_code = mode == DISCONNECT ? USBDEVFS_DISCONNECT : USBDEVFS_CONNECT;
	connect_ioctl->data = NULL;

	if (0 > (fd = open( devnode, O_RDWR ))) {
    		fprintf (stderr, "\a\a Couldn't open devnode %i.\n", i);
		return -1;
	}

	if ((errno = ioctl( fd, USBDEVFS_IOCTL, connect_ioctl )) != 0) {
		fprintf( stderr, "\a\a Could not %s interface %d : %s\n", modestr, connect_ioctl->ifno, strerror(errno) );
		free(connect_ioctl);
		return -errno;
	}

	if (DEBUG != 0)
		fprintf (stdout, " Stv driver %sed for camera at %s:%s.\n", modestr, camera_addr_busname, camera_addr_devname);

	free(connect_ioctl);
	return 0;
}
#endif


/****************************   MAIN   ****************************/

int main( int argc, char *argv[ ])
{
        int    i, j=0, num_cams = 1, current_cam = 1, orig_num_cams = 1;
	char   choice = ' ', outpath[75];
	FILE   *of;
        usb_stv        *stv680;
	usb_dev_handle *pc=NULL;    
	camera_count   *camera_addr=NULL, *orig_camera_addr=NULL;
	usb_stv        *stv[MAX_CAMERAS+1];
	usb_dev_handle *pca[MAX_CAMERAS+1];    


	/* The current (active) camera is always {stv680, pc}. stv[] and pca[] 
	   are the arays that hold the other cameras. When the active camera 
	   is switched to another, stv680 and pc get copied over with the new 
	   camera values. When a camer is removed (unplugged), the corresponding 
	   stv and pca are NULLed. The pca subscripts start with 1, to 
	   correspond to camera 1, etc.                                          */

	/* Assume there is only 1 camera */
        if ((stv680 = malloc (sizeof(*stv680))) == NULL) {
	        fprintf (stderr, "\a\a Couldn't malloc stv680 struct.\n");
    	        goto exit;
	}

	memset (stv680, 0, sizeof(*stv680));
        load_defaults (stv680);
	pca[0] = NULL;

	/* Set up for MAX_CAMERAS */
	for (i=1; i<=MAX_CAMERAS; i++) {
    		if ((stv[i] = malloc (sizeof(*stv680))) == NULL) {
    		fprintf (stderr, "\a\a Couldn't malloc stv[%i] struct\n", i);
    		goto exit;
		}
	    	memcpy (stv[i], stv680, sizeof(*stv680)); /* copy defaults */
		pca[i] = NULL;
	}

        if ((camera_addr = malloc (sizeof(*camera_addr))) == NULL) {
	        fprintf (stderr, "\a\a Couldn't malloc camera_addr struct.\n");
    	        goto exit;
	}
        if ((orig_camera_addr = malloc (sizeof(*orig_camera_addr))) == NULL) {
	        fprintf (stderr, "\a\a Couldn't malloc orig_camera_addr struct.\n");
    	        goto exit;
	}


	/* Get actual number of cameras */
	num_cams = count_cameras (camera_addr, 0);
	memcpy (orig_camera_addr, camera_addr, sizeof(*camera_addr));
	orig_num_cams = num_cams;

	if (DEBUG != 0) {
		fprintf (stdout,"\n current:  ");
		for (i=1; i<=MAX_CAMERAS; i++) {
			if (camera_addr->in_use[i] == 1)
				fprintf (stdout,"%s : %s  **  ", camera_addr->bus_dirname[i], camera_addr->dev_filename[i]);
		}
		fprintf (stdout,"\n original: ");
		for (i=1; i<=MAX_CAMERAS; i++) {
			if (orig_camera_addr->in_use[i] == 1)
				fprintf (stdout,"%s : %s  **  ", orig_camera_addr->bus_dirname[i], orig_camera_addr->dev_filename[i]);
		}
		fprintf (stdout,"\n");
	}
	
	/* This part of the code (#ifdef SHARE) only works if usbcore.o has 
	   been modified to allow the device to be 'surrendered' to a user 
	   space program. The modified usbcore should be in kernel 2.4.21 or 
	   higher. */

#ifdef SHARE
	/* Disconnect cameras */
	fprintf (stdout, "\n");
	for (i=1; i<=num_cams; i++) {
		if (share_device (camera_addr->bus_dirname[i], camera_addr->dev_filename[i], -1) < 0) {
			fprintf (stderr,"\a\a pencam_open error: Could not disconnect from STV driver for camera %i.\n", i);
			goto exit;
		}
	}
#endif

	/* we always have 1 camera */
	if ((pc = pencam_open_all(stv680, camera_addr->bus_dirname[1], 
				  camera_addr->dev_filename[1])) == NULL) {
		fprintf (stderr,"\a\a pencam_open error: Cannot open camera. Are you sure it is connected to the computer?\n");
		goto exit;
	}
	if (pencam_init (pc, stv680) != 0) {
		fprintf (stderr, "\a\a Error initializing camera in main!\n");
		goto exit;
	}
	
	/* Update camera 1 info */
	pca[1] = pc;
	memcpy (stv[1], stv680, sizeof(*stv680));

	/* Initialize structures for other cameras */
	if (num_cams >= 2) {
		for (i=2; i<=num_cams; i++) {
			if ((pca[i] = pencam_open_all(stv[i], camera_addr->bus_dirname[i], 
						    camera_addr->dev_filename[i])) == NULL) {
	    			fprintf (stderr,"\a\a pencam_open_all error: Cannot open camera %i\n", i);
				goto exit;
			}
    			if (pencam_init (pca[i], stv[i]) != 0) {
				fprintf (stderr, "\a\a Error initializing camera %i in main!\n", i);
				goto exit;
			}
		} 
	}  /*  num_cams  */

        /* kbhit stuff */
	tcgetattr(0, &kborig);
        kbnew = kborig;
	kbnew.c_lflag &= ~ICANON;
        kbnew.c_lflag &= ~ECHO;
	kbnew.c_lflag &= ~ISIG;
        kbnew.c_cc[VMIN] = 1;
	kbnew.c_cc[VTIME] = 0;
	/**                       **/

        stv680->origMode = pencam_get_mode (pc, stv680);    
	stv680->ssMode = stv680->currentMode;

	/* check for scripts; ret=-1 means 'q' command was found */
	if ((execute_script (argc, argv, pc, stv680)) == -1)
		goto exit;
	

        /*  MAIN MENU LOOP  */

	while ( choice != 'q') {

        mm:
		if (DEBUG == 0)
		    clr_lines (4);
		else
		    clr_lines (1);

		/* recheck cameras */
		orig_num_cams = num_cams;
		num_cams = count_cameras (camera_addr, 1);

		/* This section checks for a change in the number of cameras
		   and tries to adjust the stv and pc structures accordingly.
		   It would be better to put it in a subroutine, but there are
		   too many arguments to pass.   */
		
		j = 0;
		for (i=1; i<=MAX_CAMERAS; i++) {
			if (strcmp (camera_addr->bus_dirname[i], orig_camera_addr->bus_dirname[i]) != 0)
				j++;
			if (strcmp (camera_addr->dev_filename[i], orig_camera_addr->dev_filename[i]) != 0)
				j++;
		}

		if ((j != 0) && (DEBUG != 0)) {
			fprintf (stderr,"\a\a\n Number of cameras or an address has changed!\n");
			fprintf (stdout,"\n current:  ");
			for (i=1; i<=MAX_CAMERAS; i++) {
				if (camera_addr->in_use[i] == 1)
					fprintf (stdout,"%s : %s  **  ", camera_addr->bus_dirname[i], camera_addr->dev_filename[i]);
			}
			fprintf (stdout,"\n original: ");
			for (i=1; i<=MAX_CAMERAS; i++) {
				if (orig_camera_addr->in_use[i] == 1)
					fprintf (stdout,"%s : %s  **  ", orig_camera_addr->bus_dirname[i], orig_camera_addr->dev_filename[i]);
			}
			fprintf (stdout,"\n");
		}

		if (num_cams > orig_num_cams) {
#ifdef SHARE
			for (i=2; i<=num_cams; i++) {
				if (share_device (camera_addr->bus_dirname[i], camera_addr->dev_filename[i], -1) < 0) {
					fprintf (stderr,"\a\a pencam_open error: Could not disconnect from STV driver for camera %i.\n", i);
					goto exit;
				}
			}
#endif
			for (i=2; i<=num_cams; i++) {
				if ((pca[i] = pencam_open_all(stv[i], camera_addr->bus_dirname[i], 
							    camera_addr->dev_filename[i])) == NULL) {
	    				fprintf (stderr,"\a\a pencam_open_all error: Cannot open camera %i\n", i);
					goto exit;
				}
    				if (pencam_init (pca[i], stv[i]) != 0) {
					fprintf (stderr, "\a\a Error initializing camera %i in main!\n", i);
					goto exit;
				}
			}
		} 

		if (num_cams < orig_num_cams) {
			for (i=1; i<=MAX_CAMERAS; i++) {
				if ((camera_addr->in_use[i] == 0) && (orig_camera_addr->in_use[i] == 1))  {
					camera_addr->in_use[i] = -1;
					if (pca[i] != NULL)
						usb_close (pca[i]);
				}
			}
		} 
		
		if (j != 0) {
			memcpy (orig_camera_addr, camera_addr, sizeof(*camera_addr));
			orig_num_cams = num_cams;
		}			
		
		/*  End of change in num_cams routines  */
		
		menu_header (pc, stv680, num_cams, camera_addr);
    
    		fprintf(stdout, " YOUR CHOICES ARE:  (NOTE: choices are case sensitive)\n\n"); 

	        /*  check for .pencam2rc file */
		i = 0;
	        strcpy (outpath, getenv("HOME"));
		strcat (outpath, "/.pencam2rc");
		of = fopen (outpath,"rb");
		if (of == NULL) 
			i = -1;
		else
			fclose (of);
		if ( i == -1) {
	    		fprintf (stdout, " \a\aNOTE: %s file is NOT present.\n", 
				outpath); 
			fprintf (stdout, " Press 'w' to create file now.\n\n"); 
		}
		/****                      ****/

		fprintf (stdout, "   C = PenCam functions and settings    ");
		if (num_cams >= 2) 
			fprintf (stdout, "S = PenSnap functions and settings\n\n"); 
		else
			fprintf (stdout, "\n\n   S = PenSnap functions and settings\n\n");
		fprintf (stdout, "   D = Load default values         ");
		if (num_cams >= 2) {
			fprintf (stdout, "     A = Toggle active camera (%i)\n\n", current_cam);
			fprintf (stdout, "   R = Release a camera                 G = Grab a camera\n\n");
		}
		else
			fprintf (stdout, "\n\n");		
		fprintf (stdout, "   q = Quit.\n\n");
    		fprintf (stdout, " Enter your choice: ");

		choice = get_key();

		switch (choice)  {
    
			case 'C':
				while (1) {
					choice = common_menu (pc, stv680, "PENCAM");
		    			pencam_do_function (pc, stv680, choice);
					if (choice == 'z')
						break;
				}
				break;

			case 'S':
				while (1) {
					choice = common_menu (pc, stv680, "PENSNAP");
		    			pensnap_do_function (pc, stv680, choice);
					if (choice == 'z')
						break;
				}
				break;


			case 'w':
		        	if (i == -1) {
					pencam_write_config (pc, stv680);
					i = 1;
				}
				break;

			case 'D':
				load_defaults (stv680);
				fprintf (stdout, " Default values loaded. Press any key ");
				get_key();
				break;

			case 'A':
				if (num_cams == 1)
					break;
				memcpy (stv[current_cam], stv680, sizeof(*stv680));
				pca[current_cam] = pc;
		    		current_cam++;
				if (current_cam > num_cams)
					current_cam = 1;
				if (pca[current_cam] != NULL) {
					memcpy (stv680, stv[current_cam], sizeof(*stv680));
					pc = pca[current_cam];
					fprintf (stdout, "\n");
				} else {
					current_cam--;
					if (current_cam < 1)
					current_cam = num_cams;
				}
				break;
				
			case 'R':
				if (num_cams == 1)
					break;
				fprintf (stdout, " (You must) enter camera number: (");
				for (i=1; i<=num_cams; i++) {
					if (i != current_cam)	
						fprintf (stdout, " %i ", i);
				}
				fprintf (stdout, " ) ");
				i = get_val(1, num_cams);
				if ((i == -11) || ( i == -1))
					break;
				if (i == current_cam) {
					fprintf (stdout, "\a\a Select a non-active camera. Press any key ");
					get_key ();
					break;
				}	
				camera_addr->in_use[i] = -1;
				if (pca[i] != NULL)
					usb_close (pca[i]);
				pca[i] = NULL;
				fprintf (stdout, "\n");
#ifdef SHARE
				/* reconnect STV driver */
	    			if (share_device (camera_addr->bus_dirname[i], camera_addr->dev_filename[i], 1) < 0) 
					fprintf (stderr,"\a\a pencam_close error: Could not connect to STV driver.\n");
#endif
				break;

			case 'G':
				if (num_cams == 1)
					break;
				fprintf (stdout, " (You must) enter camera number: (");
				for (i=1; i<=num_cams; i++) {
					if (i != current_cam)	
						fprintf (stdout, " %i ", i);
				}
				fprintf (stdout, " ) ");
				i = get_val(1, num_cams);
				if ((i == -11) || ( i == -1))
					break;
				if (i == current_cam) {
					fprintf (stdout, "\a\a Select a non-active camera. Press any key ");
					get_key ();
					break;
				}	
#ifdef SHARE
				/* disconnect STV driver */
	    			if (share_device (camera_addr->bus_dirname[i], camera_addr->dev_filename[i], -1) < 0) 
					fprintf (stderr,"\a\a pencam_close error: Could not disconnect STV driver.\n");
#endif
				camera_addr->in_use[i] = 1;
				if ((pca[i] = pencam_open_all(stv[i], camera_addr->bus_dirname[i], 
						    camera_addr->dev_filename[i])) == NULL) {
	    				fprintf (stderr,"\a\a pencam_open_all error: Cannot open camera %i\n", i);
					break;
				}
    			        load_defaults (stv[i]);
				if (pencam_init (pca[i], stv[i]) != 0) {
					fprintf (stderr, "\a\a Error initializing camera %i in main!\n", i);
					break;
				}
				fprintf (stdout, "\n");
				break;
			
			case 'q':
				fprintf (stdout, "\a\a Press 'q' again to confirm: ");
				choice = get_key();
				if (choice == 'q')  
					goto exit;
				break;

			default:
				break;
	    
		}  /* switch */

		choice = ' ';
		goto mm;
	
	}  /* while */
    
exit:
	fprintf (stdout, "\n\n");
        fflush (stdout);	fflush (stdin);
	if (pc != NULL)
		usb_close (pc);
	free (stv680);
	for (i=1; i<=MAX_CAMERAS; i++) {
		if (pca[i] != NULL)
			usb_close (pca[i]);
		free (stv[i]);
	}

#ifdef SHARE
	/* re-connect STV camera(s) */
	for (i=1; i<=num_cams; i++) {
		if (share_device (camera_addr->bus_dirname[i], camera_addr->dev_filename[i], 1) < 0) 
			fprintf (stderr,"\a\a pencam_close error: Could not connect to STV driver.\n");
	}
	fprintf (stdout, "\n");
#endif

	free (camera_addr);
	free (orig_camera_addr);

	return 0;
}
