/* $Id: nws_proxy.c,v 1.92 2005/07/08 17:41:17 graziano Exp $ */

#include "config_nws.h"

#include <stdio.h>
#include <string.h>

#include "diagnostic.h"
#include "protocol.h"
#include "host_protocol.h"
#include "osutil.h"
#include "dnsutil.h"
#include "messages.h"
#include "register.h"
#include "skills.h"
#include "strutil.h"
#include "nws_proxy.h"
#include "protocol.h"
#include "nws_api.h"
#include "nws_daemon.h"

/* we have here the list of series (and respective memories) that we are
 * keeping track of. We don't use the timeout (in registrations) so we
 * use it to keep track of the index where we can find the right
 * forecaster. #getOld# contains the object-list of series we need to get
 * old datas for. #forecasts#
 * contains a list of forecasts (NULL for empty slot) and
 * #highestForecastIndex# is the length of the array. 
 * #lost# is the list of series we need to restart autofetching for.
 */
static registrations *series;
static ObjectSet toStart;
static Object upTo;
static char *getOld, *lost;
static int getOldLen, lostLen;
static NWSAPI_ForecastState **forecasts;
static int highestForecastIndex = 0;
static int local = 1;

/* performance counter */
static long updates = 0;

static void *lock = NULL;	 	/* lock for the module */


#if 0
/* remove the data and stop the autofetching for a single series */
static int
DeleteSingleForecast(	const char *seriesName) {
	int i;

	/* sanity check */
	if (seriesName == NULL) {
		WARN("DeleteSingleForecast: NULL parameter\n");
		return 0;
	}

	GetNWSLock(&lock);
	/* look for the registration */
	if (!SearchForName(series, seriesName, 1, &i)) {
		ReleaseNWSLock(&lock);
		LOG1("DeleteSingleForecast: no such series (%s)\n", seriesName);
		return 0;
	}
	/* delete forecaster and registration */
	NWSAPI_FreeForecastState(&forecasts[series->expirations[i]]);
	DeleteRegistration(series, i);
	ReleaseNWSLock(&lock);

	return 1;
}
#endif

/* trap SIGTERM */
static int 
MyExit(void) {
	return 1;
}

/* retireve data from a nameserver. If we are not local we can call our
 * nameserver (the one we register with) but if we are local, we need to
 * call the nws api. */
static int
GetFromNS(	const char *filter,
		ObjectSet *whereTo) {
	if (!local) {
		return RetrieveFromMyNameserver(filter, whereTo);
	}

	return NWSAPI_GetObjects(filter, whereTo);
}

/* add a forecaster */
static int
AddSingleForecast(	const Object seriesName) {
	int i;
	long j;
	char *name;

	/* sanity check */
	if (seriesName == NULL) {
		ERROR("AddSingleForecaster: NULL parameter.\n");
		return 0;
	}
	/* let's be sure it's a series and does have name attribute */
	name = NwsAttributeValue_r(FindNwsAttribute(seriesName, "objectClass"));
	if (name == NULL || strcmp(name, "nwsSeries")) {
		FREE(name);
		WARN("AddSingleForecaster: object is not a series!\n");
		return 0;
	}
	FREE(name);
	name = NwsAttributeValue_r(FindNwsAttribute(seriesName, "name"));
	if (name == NULL) {
		WARN("AddSingleForecaster: object with no name??\n");
		return 0;
	}
	FREE(name);

	/* since we are using index to global structure we should lock
	 * all the function */
	GetNWSLock(&lock);

	/* search for empty forecaster */
	for (j = 0; j < highestForecastIndex; j++) {
		if (forecasts[j] == NULL) {
			break;
		}
	}
	if (j >= highestForecastIndex) {
		/* make room for the forecaster */
		forecasts = REALLOC(forecasts, sizeof(NWSAPI_ForecastState **)*(highestForecastIndex + 1), 0);
		if (forecasts == NULL) {
			ReleaseNWSLock(&lock);
			ERROR("AddSingleForecaster: out of memory\n");
			return 0;
		}
		j = highestForecastIndex;
		forecasts[j] = NULL;
		highestForecastIndex++;
	}

	/* let's see if we already have it */
	if (!SearchForObject(series, seriesName, &i)) {
		if (!InsertRegistration(series, seriesName, j, i)) {
			ReleaseNWSLock(&lock);
			ERROR1("AddSingleForecaster: failed to insert %s\n", seriesName);
			return 0;
		}
		forecasts[j] = NWSAPI_NewForecastState();
		if (forecasts[j] == NULL) {
			DeleteRegistration(series, i);
			ReleaseNWSLock(&lock);
			ERROR("AddSingleForecaster: out of memory\n");
			return 0;
		}
		series->expirations[i] = j;
	}
	ReleaseNWSLock(&lock);

	return 1;
}

#define HOW_MANY_FORECAST	(50)
static int
RestartForecast(	const char *name) {
	int i;
	unsigned int howMany;
	NWSAPI_Measurement meas[HOW_MANY_FORECAST];

	/* sanity check */
	if (name == NULL) {
		ERROR("RestartForecast: NULL parameter.\n");
		return 0;
	}

	/* since we are using index to global structure we should lock
	 * all the function */
	GetNWSLock(&lock);

	/* let's see if we already have it */
	if (!SearchForName(series, name, 1, &i)) {
		ReleaseNWSLock(&lock);
		ERROR1("RestartForecast: unknow series %s\n", name);
		return 0;
	}

	/* let's fetch the measurements */
	howMany = 0;
	if (!NWSAPI_GetMeasurements(name, NWSAPI_BEGINNING_OF_TIME, meas, HOW_MANY_FORECAST, &howMany)) {
		ReleaseNWSLock(&lock);
		ERROR1("RestartForecast: couldn't get measurements for %s\n", name);
		return 0;
	}

	/* let's restart the Forecaster */
	NWSAPI_FreeForecastState(&forecasts[series->expirations[i]]);
	forecasts[series->expirations[i]] = NWSAPI_NewForecastState();
	if (forecasts[series->expirations[i]] == NULL) {
		ReleaseNWSLock(&lock);
		ERROR("RestartForecast: out of memory\n");
		return 0;
	}

	/* let's update the forecaster state */
	NWSAPI_UpdateForecast(forecasts[series->expirations[i]], meas, howMany);
	ReleaseNWSLock(&lock);

	/* feed back for the new series */
	INFO1("RestartForecast: got old data for %s\n", name);

	return 1;
}

int
FindMeasurement(	char *host,
			char *dst,
			char *resource,
			char *opts,
			NWSAPI_ForecastCollection *col) {
	int i, done, ret;
	char *name, *ptr, src[MAX_HOST_NAME + 8];

	/* we might change the source, so we copy it */
	if (strlen(host) > MAX_HOST_NAME + 4) {
		ERROR1("FindMeasurement: host name too long (%s)\n", host);
		return 0;
	}
	strcpy(src, host);

	/* let's try a simple search now: we do a non-exact match beacuse
	 * we could specify the host without the port. */
	done = 0;
	if (!SearchForName(series, src, 0, &i)) {
		/* we didn't find it: let's try to be more aggressive
		 * with the name in case there is a port but not a domain
		 * name*/
		ptr = strchr(src, ':');
		if (!(ptr && strchr(src, '.') == NULL)) {
			return 0;
		}

		INFO1("FindMeasurement: %s has port and not domain name: expanding the search\n", src);
		name = MALLOC(strlen(src) + 3, 1);

		/* now add .* before the : */
		ret = ptr - src;
		memcpy(name, src, ret);
		name[ret++] = '.';
		name[ret++] = '*';
		memcpy(name + ret, ptr, strlen(ptr) + 1);
		strcpy(src, name);
		FREE(name);

		/* now let's try it again with a sloppy match */
		if (!SearchForName(series, src, 0, &i)) {
			/* there is no such host */
			return 0;
		}
	}

	for (name = NULL; i >= 0; i--) {
		/* we need to be sure that the registration is the right
		 * one, so we have to look for the host, the target and
		 * the resource. */

		/* now let's check that the host is the right one .. */
		name = NwsAttributeValue_r(FindNwsAttribute(series->vals[i], "host"));
		if (name == NULL) {
			continue;
		}
		ret = NWSAPI_HostCompare(name, host);
		FREE(name);
		if (!ret) {
			/* wrong host */
			break;
		}

		/* now let's check if the resource is right */
		name = NwsAttributeValue_r(FindNwsAttribute(series->vals[i], "resource"));
		if (name == NULL) {
			continue;
		}
		ret = strcmp(name, resource);
		FREE(name);
		if (ret != 0) {
			/* wrong resource */
			continue;
		}

		/* now let's check the target */
		if (NWSAPI_IntermachineResource(resource)) {
			name = NwsAttributeValue_r(FindNwsAttribute(series->vals[i], "target"));
			if (name == NULL) {
				continue;
			}
			ret =  NWSAPI_HostCompare(name, dst);
			FREE(name);
			if (!ret) {
				/* nope, wrong target */
				continue;
			}
		}

		/* got it! */
		if (!NWSAPI_ComputeForecast(forecasts[series->expirations[i]], col)) {
			ERROR("FindMeasurement: failed to compute forecasts!\n");
		} else {
			/* done for this round */
			done = 1;
			break;
		}
	}

	return done;
}

/* used by ProcessRequest to service GET_FORECASTS. We returns the
 * forecasts in #col# for #resource# (possibly specifying which options
 * we'd like to have) for the #howMany#-long list of #hosts#. */
static int
GetAllForecasts(	char *resource,
			char *opts,
			char **hosts,
			int howMany,
			NWSAPI_ForecastCollection *col) {
	int i, interMachine, j;
	
	/* sanity check */
	if (resource==NULL || hosts == NULL || col == NULL) {
		ERROR("GetAllForecasts: NULL parameter(s)\n");
		return 0;
	}
	if (NWSAPI_IsResourceValid(resource) == -1) {
		WARN1("GetAllForecasts: unknown resource (%s)\n", resource);
		return 0;
	}

	/* let's see if the resource is a 2 hosts resources (ie bw and
	 * lat) */
	interMachine = NWSAPI_IntermachineResource(resource);

	/* we don't parse options right now */
	if (opts != NULL || strlen(opts) > 0) {
		WARN("GetAllForecasts: we don't parse options right now!\n");
	}

	GetNWSLock(&lock);
	/* let's go through all the hosts */
	for (j = 0; j < howMany; j++) {
		/* if we are asking for a single resource series,
		 * we don't need to go further then j = 0 */
		if (!interMachine && j > 0) {
			break;
		}
		for (i = 0; i < howMany; i++) {
			char *tmp;

			/* set to not-good by default */
			col[i + howMany*j].measurement.timeStamp = 0;

			/* if we got interMachine and j == i we won't
			 * find that measurement */
			if (interMachine) {
				if (i == j) {
					continue;
				}
				tmp = hosts[j];
			} else {
				tmp = hosts[i];
			}

			FindMeasurement(tmp, hosts[i], resource, opts, &(col[i + howMany*j]));
		}
	}
	ReleaseNWSLock(&lock);

	return 1;
}

/* we go through the list of forecasts to find our which hosts we don't
 * have. We could keep a global list, but we should do so with resources
 * and options, so we keep it simple and build a list on the fly */
static int
CheckEmpty(	int interMachine,
		char **hosts,
		int howMany,
		NWSAPI_ForecastCollection *col,
		int **list) {
	int i, j, soFar;

	*list = (int *)MALLOC(sizeof(int) * howMany, 1);
	(*list)[0] = -1;
	soFar = 0;

	for (i = 0; i < howMany; i++) {
		for (j = 0; j < howMany; j++) {
			/* only the interMachine experiments have a j > 0 */
			if (!interMachine && j > 0) {
				j = howMany;
				break;
			}

			/* i = src and j = dst: do we have a measurement? */
			if (col[i + howMany*j].measurement.timeStamp != 0) {
				break;
			}
			/* we need to check the other way around only on
			 * 2 hosts resources */
			if (interMachine && col[j + i*howMany].measurement.timeStamp != 0) {
				break;
			}
		}
		if (j < howMany) {
			/* we do have measurement of this host */
			(*list)[soFar++] = i;
		}
	}

	return (soFar);
}

static int
AskForAllSeries(	char *resource,
			char *opts,
			char **hosts) {
	int interMachine, i, j;
	char *filter, tmp[MAX_HOST_NAME], *ptr;
	NWSAPI_ObjectSet whereTo;
	NWSAPI_Object obj;

	/* sanity check: opts can be NULL */
	if (resource == NULL || hosts == NULL || hosts[0] == NULL) {
		WARN("AskForAllSeries: NULL parameter\n");
		return 0;
	}

	interMachine = NWSAPI_IntermachineResource(resource);

	/* let's start to allocate some memory for the filter: this
	 * should be enough to get going. We'll realloc later on */
	filter = MALLOC(40 + strlen(resource), 0);
	if (filter == NULL) {
		ERROR("AskForAllSeries: out of memory!!\n");
		return 0;
	}
	filter[0] = '\0';
	sprintf(filter, "&(objectclass=nwsSeries)(resource=%s)(|", resource);

	/* now we have to add all the hosts: if we have an
	 * interMachineResource we have to couple host=* and resource=*,
	 * otherwise host=* is sufficient */
	for (i = 0; hosts[i] != NULL; i++) {
		for (j = 0; hosts[j] != NULL; j++) {
			/* if we are on a single host resource we are
			 * done after after one iteration */
			if (!interMachine && j > 0) {
				break;
			}

			/* if we do interMachine, then skip the i = j */
			if (i == j && interMachine) {
				continue;
			}

			/* get same memory for the filter */
			filter = REALLOC(filter, strlen(filter) + strlen(hosts[i]) + strlen(hosts[j]) + 25, 0);
			if (filter == NULL) {
				ERROR("AskForAllSeries: out of memory\n");
				return 0;
			}
			/* add the host= part */
			strcat(filter, "(&(host=");
			
			/* thanks to our damn department, the DNS is
			 * broken (machines reports no domain name) so we
			 * need to add .* */
			strcpy(tmp, hosts[i]);
			ptr = strchr(tmp, ':');
			if (!ptr && !strchr(tmp, '.')) {
				/* easy: there is no port (:port) nor
				 * domain name: we add a .* to the name */
				strcat(tmp, ".*");
			} else if (ptr && !strchr(tmp, '.')) {
				/* we got a port but not a domain name:
				 * we need to insert * in between: I'd
				 * like to use .* but bart:8888 won't be
				 * matched ...*/
				memmove(ptr + 1, ptr, strlen(ptr) + 1);
				*ptr = '*';
			} else if (!ptr && strchr(tmp, '.')) {
				/* we got domain but not the port: let's
				 * add a * to match everything */
				strcat(tmp, "*");
			}
			strcat(filter, tmp);

			/* do we need the target= ?*/
			if (interMachine) {
				strcat(filter, ")(target=");
				/* we need to fix the target name too! */
				strcpy(tmp, hosts[j]);
				ptr = strchr(tmp, ':');
				if (!ptr && !strchr(tmp, '.')) {
					strcat(tmp, ".*");
				} else if (ptr && !strchr(tmp, '.')) {
					memmove(ptr + 1, ptr, strlen(ptr) + 1);
					*ptr = '*';
				} else if (!ptr && strchr(tmp, '.')) {
					strcat(tmp, "*");
				}
				strcat(filter, tmp);
			}

			/* close this subclause */
			strcat(filter, "))");
		}
	}
	/* close the whole filter */
	strcat(filter, ")");

	/* let's query the nameserver: if the query is big the namserver
	 * will take sometime to answer. */
	i = GetFromNS(filter, &whereTo);
	FREE(filter);

	if (i == 0) {
		ERROR("AskForAllSeries: failed to talk to the nameserver\n");
	} else {
		/* let's add it to the todo list */
		NWSAPI_ForEachObject(whereTo, obj) {
			NWSAPI_AddObject(&toStart, obj);
		}
		NWSAPI_FreeObjectSet(&whereTo);
	}

	return (i);
}

/* fills the resource, options and hosts coming from the net. Returns the
 * # of hosts, 0 on error */
static int
FillHosts(	Socket *sd,
		size_t dataSize,
		char **resource,
		char **opts,
		char ***h) {
	char *tmp, *ptr, **hosts;
	DataDescriptor des = SIMPLE_DATA(CHAR_TYPE, 0);
	int i;

	/* allocate memory */
	tmp = MALLOC(dataSize * sizeof(char), 0);
	if (tmp == NULL) {
		return 0;
	}
	des.repetitions = dataSize;
	/* get all the data */
	if (!RecvData(*sd, tmp, &des, 1, -1)) {
		FREE(tmp);
		return 0;
	}
	tmp[dataSize - 1] = '\0';
	ptr = tmp;

	/* first is the resource name */
	*resource = strdup(ptr);
	ptr += strlen(ptr) + 1;

	/* then the options */
	*opts = strdup(ptr);
	ptr += strlen(ptr) + 1;

	/* check out of memory conditions */
	if (*resource == NULL || *opts == NULL) {
		ABORT("FillHost: out of memory!");
	}

	/* then all the hostnames: a lot of realloc down here */
	hosts = NULL;
	for(i = 0; (ptr - tmp + 1) <= dataSize; i++) {
		/* allocate space for the pointer first */
		hosts = (char **)REALLOC(hosts, sizeof(char *)*(i + 2), 1);
		hosts[i] = strdup(ptr);
		if (hosts[i] == NULL) {
			ABORT("FillHosts: out of memory!");
		}
		hosts[i + 1] = NULL;
		ptr += strlen(ptr) + 1;
	}
	FREE(tmp);

	/* check if we do indeed have hosts */
	if (hosts != NULL) {
		*h = hosts;
	}

	return i;
}

static void
ProcessRequest(	Socket *sd,
		MessageHeader header) {
	NWSAPI_Measurement meas;
	char name[255 + 1], *resource, *opts, **hosts;
	int i, j, howMany;
	NWSAPI_ForecastCollection *col;
	DataDescriptor howManyDes = SIMPLE_DATA(INT_TYPE, 1);
	DataDescriptor strDes = SIMPLE_DATA(CHAR_TYPE, 0);
	static const int sizeofForecast = sizeof(NWSAPI_ForecastCollection);
	NWSAPI_ObjectSet objs;
	NWSAPI_Object obj;

	resource = opts = NULL;

	switch(header.message) {
	case STATE_FETCHED:
		/* we just received and update */
		if (!NWSAPI_AutoFetchMessage(*sd, name, 256, &meas)) {
			ERROR("ProcessRequest: failed to autofetch!\n");
			break;
		}

		/* let's update */
		GetNWSLock(&lock);
		if (!SearchForName(series, name, 1, &i)) {
			WARN1("ProcessRequest: %s unknown series\n", name);
		} else {
			updates++;
			NWSAPI_UpdateForecast(forecasts[series->expirations[i]], &meas, 1);
			DDEBUG1("ProcessRequest: updates %s\n", name);
		}
		ReleaseNWSLock(&lock);
		
		break;

	case GET_FORECASTS:
		i = FillHosts(sd, header.dataSize, &resource, &opts, &hosts);
		if (i == 0) {
			SendMessage(*sd, PROXY_FAILED, -1);
			ERROR("ProcessRequest: error in receiving request!\n");
			DROP_SOCKET(sd);
			FREE(resource);
			FREE(opts);
			break;
		}

		/* let's allocate room for the forecasts */
		col = (NWSAPI_ForecastCollection *)MALLOC(sizeofForecast*i*i, 0);
		if (col == NULL) {
			ERROR("ProcessRequest: out of memory!\n");
			SendMessage(*sd, PROXY_FAILED, -1);
			DROP_SOCKET(sd);
		} else if (!GetAllForecasts(resource, opts, hosts, i, col)) {
			SendMessage(*sd, PROXY_FAILED, -1);
			DROP_SOCKET(sd);
		} else {
			int *list, z, inter;

			inter= NWSAPI_IntermachineResource(resource);

			/* let's see if we have non-known hosts, and
			 * calculate how many forecasts we are sending */
			howMany = CheckEmpty(inter, hosts, i, col, &list);
			
			/* send how many data we are shipping */
			howManyDes.repetitions = 1;
			if (!SendMessageAndData(*sd, GOT_FORECASTS, &howMany, &howManyDes, 1, -1)) {
				ERROR("ProcessRequest: failed to send GOT_FORECASTS\n");
				DROP_SOCKET(sd);
			} else if (howMany > 0) {
				/* we send the list only if we have
				 * something to send */
				howManyDes.repetitions = howMany;
				if (!SendData(*sd, list, &howManyDes, 1, -1 )) {
					ERROR("ProcessRequest: failed to send GOT_FORECASTS list\n");
					DROP_SOCKET(sd);
				} else for (j = 0; j < howMany; j++) {
					/* j is destination and on single
					 * host resource cannot be > 0 */
					if (!inter && j > 0) {
						break;
					}
					for (z = 0; z < howMany; z++) {
						int ciccio;

						/* let's compute the
						 * index we are sending */
						if (inter && j == z) {
							/* we don't send
							 * the diagonal
							 * in a 2 host
							 * resource */
							continue;
						} else if (!inter) {
							/* it's a simple
							 * index if we
							 * have just 1
							 * host resource*/
							ciccio = list[z];
						} else {
							ciccio = list[z] + list[j]*i;
						}

						/* we need to send one
						 * forecast at a time */
						if (!SendData(*sd, &col[ciccio], forecastCollectionDescriptor, forecastCollectionDescriptorLength, -1 )) {
							ERROR("ProcessRequest: failed to send GOT_FORECASTS\n");
							DROP_SOCKET(sd);
							break;
						}
					}
					if (z < howMany) {
						/* got an error */
						break;
					}
				}
			}
			FREE(list);
		}

		/* free the memory */
		for (i--; i >= 0; i--) {
			FREE(hosts[i]);
		}
		FREE(hosts);
		FREE(resource);
		FREE(opts);
		FREE(col);
		
		break;

	case START_FETCHING:
		i = FillHosts(sd, header.dataSize, &resource, &opts, &hosts);
		if (i == 0) {
			SendMessage(*sd, PROXY_FAILED, -1);
			ERROR("ProcessRequest: error in reading hosts!\n");
			DROP_SOCKET(sd);
			FREE(resource);
			FREE(opts);
			break;
		}

		/* no error message to the client from here */
		SendMessage(*sd, PROXY_ACK, -1);

		/* get all the series names */
		AskForAllSeries(resource, opts, hosts);

		/* done: free memory */
		for (i-- ;i >= 0; i--) {
			FREE(hosts[i]);
		}
		FREE(hosts);
		FREE(resource);
		FREE(opts);

		break;

	case START_ACTIVITY_FETCHING:
		strDes.repetitions = header.dataSize;
		resource = MALLOC(strDes.repetitions + 1, 1);
		if (strDes.repetitions > 220 || !RecvData(*sd, resource, &strDes, 1, -1)) {
			SendMessage(*sd, PROXY_FAILED, -1);
			ERROR("ProcessRequest: error in reading clique!\n");
			DROP_SOCKET(sd);
			FREE(resource);
			break;
		}
		resource[strDes.repetitions] = '\0';

		/* let's query the nameserver for all the series relative
		 * to this activity */
		sprintf(name, "(activity=%s)", resource);
		FREE(resource);

		objs = NWSAPI_NewObjectSet();
		if (GetFromNS(name, &objs)) {
			/* let's add it to the todo list */
			NWSAPI_ForEachObject(objs, obj) {
				NWSAPI_AddObject(&toStart, obj);
			}

			/* no error message to the client from here */
			SendMessage(*sd, PROXY_ACK, -1);
		} else {
			ERROR("ProcessRequest: failed to talk to nameserver\n");
			SendMessage(*sd, PROXY_FAILED, -1);
		}
		NWSAPI_FreeObjectSet(&objs);


		break;
	
	default:
		ERROR1("ProcessRequest: unknown message %d\n", header.message);
		DROP_SOCKET(sd);
		break;
	}

	/* done with this socket */
	SocketIsAvailable(*sd);
}

static int
AddToTabList(	char **list,
		int len,
		const char *src) {
	int srcLen;

	srcLen = strlen(src);
	*list = REALLOC(*list, len + srcLen + 2, 0);
	if (*list == NULL) {
		ERROR("out of memory\n");
		return 0;
	}
	if (len > 0) {
		(*list)[len] = '\t';
		len++;
	}
	memcpy(*list + len, src, srcLen + 1);
	len += srcLen;

	return len;
}

void
usage() {
	DaemonUsage("nws_proxy", "Proxy");
	printf("\t-s                  system wide proxy (register with nameserver)\n");
	printf("\t-A                  autofetch all the registered series\n");
}


#define NO_MORE_THEN 1000

char * GenerateList(const registrations *r);

int
main (int argc, char **argv) {
	int opt, startFetching, ind;
	extern char *optarg;
	extern int optind;
	double now, wakeup, nextErrorCheck, nextBeatTime, nextGetOldData;
	char *tmp, name[255];
	ObjectSet mySet;

	/* default values  */
	startFetching = 0;
	
	/* no error message from getopt() */
	opterr = 0;

	ind = InitializeDaemonStructure(PROXY_HOST, "sA", MyExit);

	while((int)(opt = getopt(argc, argv, DaemonAllSwitches(ind))) != EOF) {
		if (DaemonSwitch(opt, optarg, ind)) {
			continue;
		}

		switch(opt) {
		case 's':
			local = 0;
			break;

		case 'A':
			startFetching = 1;
			break;

		default:
			usage();
			exit(1);
			break;
		}
	}

	/* if we are local we only open a listening port, otherwise we do
	 * the full registration with our nameserver */
	if (!InitializeDaemon(local, 1, 0, ind)) {
		exit(-1);
	}

	/* we want to do relax matching */
	NWSAPI_RelaxHostNameCheck(1);

	/* initialize the series structure */
	GetNWSLock(&lock);
	if (!InitRegistrations(&series)) {
		ABORT("failed to init internal structure\n");
	}
	toStart = NewObjectSet();
	upTo = NULL;
	lost = getOld = NULL;
	lostLen = getOldLen = 0;
	ReleaseNWSLock(&lock);

	RegisterListener(STATE_FETCHED, "STATE_FETCHED,", &ProcessRequest);
	RegisterListener(GET_FORECASTS, "GET_FORECASTS,", &ProcessRequest);
	RegisterListener(START_FETCHING, "START_FETCHING,", &ProcessRequest);
	RegisterListener(START_ACTIVITY_FETCHING, "START_ACTIVITY_FETCHING,", &ProcessRequest);

	/* main service loop */
	wakeup = nextErrorCheck = nextBeatTime = nextGetOldData = 0;
	while (1) {
		now = CurrentTime();

		/* we might want to register with the nameserver here */
		if (nextBeatTime < now) {
			if (!local) {
				RegisterHost(DEFAULT_HOST_BEAT * 2);
			}
			LOG1("main: we have %d series\n", series->howMany);

			/* some feedback */
			LOG1("got %d updates\n", updates);
			updates = 0;

			nextBeatTime = now + DEFAULT_HOST_BEAT/5;
		}

		if (nextErrorCheck < now) {
			/* let's try to get failed series */
			tmp = NWSAPI_AutoFetchError();
			if (tmp != NULL) {
				lostLen = AddToTabList(&lost, lostLen, tmp);
				if (lostLen == 0) {
					ERROR("failed to rescue lost series\n");
				}
				FREE(tmp);
			}
			nextErrorCheck = now + 20;
		}

		/* now, if we've been asked to fetch all the series from
		 * a nameserver (-A) we do so periodically to be sure we
		 * keep getting new series */
		if (startFetching > 0 && startFetching < now) {
			startFetching = now + 60*60;

			/* let's get all the series names */
			if (!GetFromNS("(objectclass=nwsSeries)", &toStart)) {
				WARN("main: cannot get series!\n");
			}
		}


		if (nextBeatTime > nextErrorCheck) {
			wakeup = nextErrorCheck;
		} else {
			wakeup = nextBeatTime;
		}
		wakeup -= now;


		/* we do add new series for up to 1 seconds to avoid
		 * becoming unresponsive to the memory autofetching
		 * messages */
		upTo = NextObject(toStart, upTo);
		mySet = NewObjectSet();
		while (upTo != NULL && (CurrentTime() - now) < 1) {
			/* let's create the forecaster */
			if (!AddSingleForecast(upTo)) {
				WARN("failed to add forecaster\n");
			} else {
				NWSAPI_AddObject(&mySet, upTo);

				/* add the series to the restart list */
				tmp = NwsAttributeValue_r(FindNwsAttribute(upTo, "name"));
				getOldLen = AddToTabList(&getOld, getOldLen, tmp);
				if (getOldLen == 0) {
					ABORT("out of memory!\n");
				}
				FREE(tmp);
			}
			upTo = NextObject(toStart, upTo);
			if (upTo == NULL) {
				/* well, we are done! */
				FreeObjectSet(&toStart);
				toStart = NewObjectSet();
				if (toStart == NULL) {
					ABORT("out of memory\n");
				}
				LOG("done adding new series\n");
			}
		}

		/* now let's deal with lost series: we need to add to the
		 * mySet the list of objects to restart. Notice that we
		 * are using the same now, hence we do either the new
		 * series or we rescue the lost one. */
		while (lostLen > 0 && (CurrentTime() - now) < 1) {
			/* get the next series and remove it from the
			 * todo list */
			GETWORD(name, lost, (const char **)&tmp);
			memmove(lost, tmp, lostLen - (tmp - lost) + 1);
			lostLen -= (tmp - lost);
			if (lostLen == 0) {
				FREE(lost);
				LOG("done recovering lost series\n");
			} else {
				/* let's come back quickly */
				wakeup = -1;
			}

			/* now let's add it to the working list */
			if (!SearchForName(series, name, 1, &opt)) {
				WARN1("non existent series (%s)??\n", name);
			} else {
				NWSAPI_AddObject(&mySet, series->vals[opt]);
			}

		}
		/* let's start autofetching */
		if (strlen(mySet) > 5) {
			/* in case we already have these series, we stop
			 * and restart the autofetching */
			NWSAPI_AutoFetchEnd(mySet);
			if (!NWSAPI_AutoFetchBeginObject(mySet)) {
				LOG("failed to start AutoFetch\n");
			}

			/* since we started autofetch we better give some
			 * time to serve all the memory message */
			wakeup = 0;
		}
		FreeObjectSet(&mySet);

		/* now let's try to fill the forecasts with old data:
		 * this will overload the memory if we have a lot of
		 * series, so we do it no more then once a second. */
		if (getOldLen > 0 && nextGetOldData < now) {
			GETWORD(name, getOld, (const char **)&tmp);
			if (!RestartForecast(name)) {
				/* we got problem with restart: let's
				 * wait a bit longer before asking it
				 * again */
				nextGetOldData = CurrentTime() + 30;
			} else {
				nextGetOldData = CurrentTime() + 1;
			}
			memmove(getOld, tmp, getOldLen - (tmp - getOld) + 1);
			getOldLen -= (tmp - getOld);
			if (getOldLen == 0) {
				FREE(getOld);
				LOG("done getting old series values\n");
			}
			wakeup = 1;
		}

		/* let's serve the messages: we want to be responsive
		 * with the memory, so we try to serve all messages
		 * first, while we are ramping up our effort. */
		while(getOldLen > 0 && ListenForMessages(-1)) {
			;	/* nothing to do */
		}

		/* now we can do the normal stint */
		ListenForMessages((wakeup > 0) ? wakeup : -1);
	}

	return 1;
}

