#include "portable.h"

#include <stdio.h>

#include <ac/ctype.h>
#include <ac/socket.h>
#include <ac/string.h>
#include <ac/time.h>
#include <ac/regex.h>

#include <pwd.h>

#include "slap.h"
#include "external.h"
#include <ldap_pvt.h>

#include "nws_back.h"

#include "nws_back_cache.h"


nwsSeriesCache *
nws_get_series(nwsBackConfig *nbc, NWSobj *nobj) {

  struct nws_series_cache *series;
  char *tmpc;
  int mem_len;
  
  /* Debug(LDAP_DEBUG_TRACE,"==>nws_get_series()\n",0,0,0);  */

  /* try to find this series in the cache */
  series = nbc->nser;
  while(series->next != NULL) {
	if (!strcmp(series->name, nobj->name)) {
	  /* Debug(LDAP_DEBUG_TRACE,"<==nws_get_series()\n",0,0,0); */
	  return(series);
	}
	else {
	  series = series->next;
	}
  }

  /* this (dealing with "->next") is to make the append below easier */
  if (!strcmp(series->name, nobj->name)) {
	/* Debug(LDAP_DEBUG_TRACE,"<==nws_get_series()\n",0,0,0); */
	  return(series);
  }


  /* can't find it, so create one */
  series->next = malloc(sizeof(struct nws_series_cache));
  series->next->prev = series;
  series = series->next;
  series->next = NULL;
  series->memory = strdup(nobj->at.ns.memory);
  series->name = strdup(nobj->name);

  tmpc = strchr(series->memory,':');
  mem_len = tmpc - series->memory;
  strncpy((char *)&series->mem_cookie.name, series->memory, mem_len);
  series->mem_cookie.name[mem_len] = '\0';
  tmpc++;
  series->mem_cookie.port = atoi(tmpc);
  series->mem_cookie.sd = NO_SOCKET;
  series->exper = NULL;
  series->forc = NULL;
  series->forecast_state = InitForcl(nbc->max_forc_entries,
									 nbc->max_data_entries); 

  memset(&series->state, 0, sizeof(struct nws_memory_state));
  series->state.rec_count = -1; /* set to -1 for first nws_update_series */
  series->count = 0; /* how many are there? */
  series->lastupdate.tv_sec = series->lastupdate.tv_usec = 0;
  series->state.seq_no = series->seq_no = 0.0;
  pthread_mutex_init(&series->nser_mutex, NULL);

  strcpy(series->state.id, series->name);

  series->period = 120;

  /* Debug(LDAP_DEBUG_TRACE,"<==nws_get_series()\n",0,0,0); */
  return(series);

  /* nothing is ever free()d (or free) */
}

int
nws_update_series(nwsBackConfig *nbc, struct nws_series_cache *series) {

  char *exp_array;
  size_t recv_size;
  int timeout = -1;
  int i=0;

  DataDescriptor contentsDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);

  Debug(LDAP_DEBUG_TRACE,"==>nws_update_series()\n",0,0,0);

  series->state.seq_no = series->seq_no = 0.0;
  series->state.rec_size = 0;
  series->state.time_out = 0.0;

  if (series->state.rec_count < 0) {
	series->state.rec_count = nbc->initial_fetch_size;
  }
  else {
	series->state.rec_count = 1;
  }

  i=0;
  ldap_pvt_thread_mutex_lock(&nbc->nws_connect_mutex);
  while(!ConnectToHost(&series->mem_cookie, &series->mem_cookie.sd)) {
  	Debug(LDAP_DEBUG_TRACE,
		  "nws_cache_connect(): ConnectToHost(%s:%d) failed\n",
		  series->mem_cookie.name, series->mem_cookie.port,0);
	i++;
	if (i==3) {
	  ldap_pvt_thread_mutex_unlock(&nbc->nws_connect_mutex);
	  DisconnectHost(&series->mem_cookie);
	  return(-1);
	}
	else {
	  ldap_pvt_thread_sleep(1);
	}
  }
  ldap_pvt_thread_mutex_unlock(&nbc->nws_connect_mutex);

  /*
  Debug(LDAP_DEBUG_TRACE,"==>nws_update_series() connected.\n",0,0,0);
  */

  if(!SendMessageAndData(series->mem_cookie.sd, FETCH_STATE, &series->state,
                     stateDescriptor, stateDescriptorLength,
						 timeout)) {
	DisconnectHost(&series->mem_cookie);
	Debug(LDAP_DEBUG_TRACE,"nws_update_series(): SendMessageAndData failed\n"
		  ,0,0,0);
  }

  Debug(LDAP_DEBUG_TRACE,"==>nws_update_series() message sent.\n",0,0,0);

  if(!RecvMessage(series->mem_cookie.sd, STATE_FETCHED, &recv_size, timeout)) {
	DisconnectHost(&series->mem_cookie);
	Debug(LDAP_DEBUG_TRACE,"nws_update_series(): RecvMessage failed\n",0,0,0);
	return(-1);
  }

  Debug(LDAP_DEBUG_TRACE,"==>nws_update_series() message received.\n",0,0,0);
	
  if(!RecvData(series->mem_cookie.sd,
			   &series->state,
			   stateDescriptor,
			   stateDescriptorLength,
			   timeout)) {
	DisconnectHost(&series->mem_cookie);
	Debug(LDAP_DEBUG_TRACE,"nws_update_series(): RecvData() failed for state data.\n",0,0,0);
	return(-1);
  }

  /*
  Debug(LDAP_DEBUG_TRACE,"==>nws_update_series() descriptor received.\n",0,0,0);
  */

  contentsDescriptor.repetitions =
	series->state.rec_size * series->state.rec_count;
 
  exp_array = (char *)malloc(contentsDescriptor.repetitions + 1);
  if(exp_array == NULL) {
	DisconnectHost(&series->mem_cookie);
	Debug(LDAP_DEBUG_TRACE,"nws_update_series(): malloc() failed.\n",0,0,0);
	return(-1);
  }

  if(!RecvData(series->mem_cookie.sd,
			   exp_array,
			   &contentsDescriptor,
			   1,
			   timeout)) {
	free(exp_array);
	DisconnectHost(&series->mem_cookie);
	Debug(LDAP_DEBUG_TRACE,"nws_update_series(): RecvData() failed for measurement data.\n",0,0,0);
	return(-1);
  }

  Debug(LDAP_DEBUG_TRACE,"==>nws_update_series() data received.\n",0,0,0);

  exp_array[contentsDescriptor.repetitions] = '\0';

  if (series->exper != NULL) {
	free(series->exper);
  }

  series->exper = malloc(sizeof(NWSAPI_Measurement) * contentsDescriptor.repetitions);
  if (series->exper == NULL) {
	DisconnectHost(&series->mem_cookie);
	Debug(LDAP_DEBUG_TRACE,"nws_update_series(): malloc() failed.\n",
		   0,0,0);
  }
  if(!UnpackExperiments(&series->state, exp_array,
						series->state.rec_count, series->exper,
						(unsigned int *)&series->exp_count)) {
	free(exp_array);
	free(series->exper);
	Debug(LDAP_DEBUG_TRACE,"nws_update_series(): UnpackExperiments failed\n",
		   0,0,0);
	return(-1);
  }

  /*
  Debug(LDAP_DEBUG_TRACE,"==>nws_update_series() data unpacked.\n",0,0,0);
  */

  for(i=0; i < series->state.rec_count; i++) {
	/* printf("series->exper[i].timeStamp: %f, series->exper[i].value: %f\n", series->exper[i].timeStamp, series->exper[i].value); */
	UpdateForecasts(series->forecast_state, (double)series->exper[i].timeStamp,
				  (double)series->exper[i].measurement);
  }

  /*
  Debug(LDAP_DEBUG_TRACE,"==>nws_update_series() forecasts updated.\n",0,0,0);
  */

  if(gettimeofday(&series->lastupdate, NULL) < 0) {
	Debug(LDAP_DEBUG_TRACE,"nws_update_series(): gettimeofday() failed.\n"
		  ,0,0,0);
	series->lastupdate.tv_sec = 0;
	series->lastupdate.tv_usec = 0;
  }
  free(exp_array);

  ldap_pvt_thread_mutex_lock(&nbc->nws_connect_mutex);
  DisconnectHost(&series->mem_cookie);
  ldap_pvt_thread_mutex_unlock(&nbc->nws_connect_mutex);

  Debug(LDAP_DEBUG_TRACE,"<==nws_update_series()\n",0,0,0);
  return(0);
}


