/* $Id: nws_ctrl.c,v 1.66 2005/05/24 19:38:25 graziano Exp $ */

#include "config_nws.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#if TIME_WITH_SYS_TIME
#	include <sys/time.h>
#	include <time.h>
#else
#	if HAVE_SYS_TIME_H
#		include <sys/time.h>
#	else
#		include <time.h>
#	endif
#endif
#include "math.h"

#include "diagnostic.h"
#include "messages.h"
#include "protocol.h"
#include "strutil.h"
#include "osutil.h"
#include "skills.h"
#include "clique_protocol.h"
#include "register.h"

#define NWSAPI_SHORTNAMES 1
#include "nws_api.h" 

#include "host_protocol.h"

typedef enum {DEAD, HEALTHY, SICK, UNRESPONSIVE} HostStatii;

/* local function that prints the status of a host (in cookie) */
static void
PrintStatus(	struct host_cookie *cookie,
		HostStatii stat,
		HostInfo *info) {
	const char *TYPES[] = {"forecaster", "memory", "name_server", "sensor"};
	const char *STATII[] = {"dead", "healthy", "sick", "unresponsive"};

	if (cookie == NULL) {
		return;
	}

	printf("%s:", HostCImage(cookie));
	/* if we have the info, we don't need stat */
	if (info != NULL) {
		printf(" %s ", TYPES[info->hostType]);
		printf("(believes to be %s) ", info->registrationName);
		printf("registered with %s ", info->nameServer);
	}
	printf(" %s\n", STATII[stat]);
}

static int
FillCookie(	struct host_cookie *cookie,
		const char *host,
		HostInfo *info,
		double timeout) {
	/* sanity check */
	if (cookie == NULL || host == NULL || info == NULL) {
		ABORT("FillCookie: NULL parameter\n");
	}

	Host2Cookie(host, GetDefaultPort(SENSOR_HOST), cookie);
	if (!ConnectToHost(cookie, NULL)) {
		PrintStatus(cookie, DEAD, NULL);
		return 0;
	}
	if (!GetHostInfo(cookie, info, timeout)) {
		PrintStatus(cookie, UNRESPONSIVE, NULL);
		return 0;
	}

	return 1;
}

#define SWITCHES "s:t:Vv:f:XFN:p:"

static void
usage() {
	printf("\nUsage: nws_ctrl [OPTIONS] command host [command-options]\n");
	printf("Control the hosts for the Network Weather Service\n");
	printf("\nOPTIONS can be:\n");
	printf("\t-t timeout          specify a timeout\n");
	printf("\t-X                  print result from proxy cmd in XML\n");
	printf("\t-p password         the password of the remote nwsHost\n");
	printf("\t-N nameserver       the nameserver to use\n");
	printf("\t-F                  force an operation\n");
	printf("\t-f file             read host and command options from #file#\n");
	printf("\t-s size,buffer,msg  instruct ping to use these values\n");
	printf("\t-v level            verbose level (up to 5)\n");
	printf("\t-V                  print version\n");
	printf("\nand commands can be:\n");
	printf("\tregister            #host# will change nameserver to #command-options#\n");
	printf("\tmemory              #host# will change memory to #command-options#\n");
	printf("\thalt                halt #host#\n");
	printf("\tlog                 toggle logs on #host#: accept optional error and log\n");
	printf("\tadd                 ask #host# to add a sensor to a clique\n");
	printf("\tremove              ask #host# to remove a sensor from a clique\n");
	printf("\tstart               start an activity on #host#: you need to specify name:<activity> controlName:<control> skillName:<skill>. Others attributes are optional.\n");
	printf("\tstop                stop an <activityName> on #host#\n");
	printf("\tfetch               #host# is the proxy and the first options is the resource, the reminder are the hosts\n");
	printf("\tfetchClique         #host# is the proxy and the first options is the clique name\n");
	printf("\tclean               #host# is the memory, the first option is the idle time and reminders are series (perform a MEMORY_CLEAN)\n");
	printf("Report bugs to <nws@nws.cs.ucsb.edu>.\n\n");
}

int
main(		int argc,
		char *argv[]) {
	const char *command; 
	char tmp[255], *commandLine, *ptr;
	const char *password;
	struct host_cookie cookie;
	HostSpec NS;
	int c, sizes[3], z, inXML, force;
	double tout;
	unsigned short verbose;
	extern char *optarg;
	extern int optind;
	HostInfo info;
	FILE *optionsFile;

	/* deafult values */
	password = NULL;
	commandLine = strdup("");;
	optionsFile = NULL;
	tout = -1;
	verbose = 2;
	force = 0;
	sizes[0] = 256;		/* experiment size */
	sizes[1] = 32;		/* buffer size */
	sizes[2] = 16;		/* message size */
	inXML = 0;
	NS.machineName[0] = '\0';

	while((c = getopt(argc, argv, SWITCHES)) != EOF) {
		switch(c) {
		case 'p':
			password = optarg;
			break;

		case 'F':
			force = 1;
			break;

		case 'X':
			inXML = 1;
			break;

		case 'N':
			Host2Cookie(optarg, DefaultHostPort(NAME_SERVER_HOST), &cookie);
			NS = *MakeHostSpec(cookie.name, cookie.port);
			if (!UseNameServer(&NS)) {
				WARN("Failed to set nameserver\n");
			}
			break;

		case 't':
			tout = strtod(optarg, NULL);
			break;

		case 'f':
			optionsFile = fopen(optarg, "r");
			if (optionsFile == NULL) {
				printf("Cannot open file %s\n", optarg);
				exit(1);
			}
			break;

		case 's':
			/* Parse the comma-delimited size list, allowing
			 * each element to default if unspecified (e.g.
			 * "32", "10,,5" and ",,8" are all legal). */
			command = optarg;
			for (z = 0; z < 3 && *command != '\0'; z++) {
				if (*command != ',') {
					if (GETTOK(tmp, command, ",", &command)) {
						sizes[z] = (int)strtol(tmp, NULL, 10);
					}
				}
				if (*command != '\0') {
					command++;
				}
			}
			break;

		case 'v':
			verbose = (unsigned short)atol(optarg);
			break;

		case 'V':
			printf("nws_ctrl for NWS version %s", VERSION);
#ifdef HAVE_PTHREAD_H
			printf(", with thread support");
#endif
#ifdef WITH_DEBUG
			printf(", with debug support");
#endif
			printf("\n\n");
			exit(0);
			break;

		case '?':
			if (optopt == 'v') {
				/* using basic level */
				verbose = 2;
				break;
			}
			/* non recognized options: printing help */

		default:
			usage();
			exit(1);
			break;
		}
	}

	/* set the verbose level */
	SetDiagnosticLevel(verbose, stderr, stderr);

	/* now the next word should be the command */
	if (optind >= argc) {
		usage();
		exit(1);
	} 

	/* this is the command */
	command = strdup(argv[optind++]);
	if (command == NULL) {
		ABORT("out of memory\n");
	}

	/* now let's collect all the host/comamnds from command line/file */
	for (; optind < argc; optind++) {
		commandLine = REALLOC(commandLine, strlen(commandLine) + strlen(argv[optind]) + 2, 1);
		strcat(commandLine, argv[optind]);
		strcat(commandLine, "\n");
	}
	if (optionsFile != NULL) {
		/* read the file now */
		while (fgets(tmp, sizeof(tmp), optionsFile) != NULL) {
			/* remove empty lines */
			if (strlen(tmp) <= 1) {
				continue;
			}
			commandLine = REALLOC(commandLine, strlen(commandLine) + strlen(tmp) + 2, 1);
			strcat(commandLine, tmp);
			strcat(commandLine, "\n");
		}
		fclose(optionsFile);
	}
	if (commandLine[0] == '\0') {
		ABORT("There is no host/options specified\n");
		exit(1);
	}
	ptr = commandLine;

	if (strcmp(command, "register") == 0) {
	        if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a host\n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a nameserver\n");
			exit(1);
		}
		if (!HostChangeNameServer(&cookie, tmp, tout)) {
			printf("failed to change nameserver\n");
			exit(1);
		}
		DisconnectHost(&cookie);
	} else if (strcmp(command, "memory") == 0) {
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a host\n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a memory\n");
			exit(1);
		}
		if (!HostChangeMemory(&cookie, tmp, tout)) {
			printf("failed to change memory\n");
			exit(1);
		}
		DisconnectHost(&cookie);
	} else if (strcmp(command, "halt") == 0) {
		z = 0;

		while(GETWORD(tmp, ptr, (const char **)&ptr)) {
			z = 1;
			FillCookie(&cookie, tmp, &info, tout);
			if (!HostHalt(&cookie, password, tout)) {
				printf("failed to halt the %s\n", tmp);
			}
			DisconnectHost(&cookie);
		}
		if (z == 0) {
			printf("I need at least one host\n");
			exit(1);
		}
	} else if (strcmp(command, "log") == 0) {
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a host\n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}
		if (GETWORD(tmp, ptr, (const char **)&ptr)) {
			if (strcmp("error", tmp) == 0) {
				c = ALL_ERRORS;
			} else if (strcmp("log", tmp) == 0) {
				c = ALL_LOGS;
			}
		} else {
			c = ALL_DIAGNOSTICS;
		}
		if (!HostChangeDiagnostic(&cookie, c, tout)) {
			printf("failed to change diagnostics on the host\n");
			exit(1);
		}
		DisconnectHost(&cookie);
	} else if (strcmp(command, "add") == 0 || strcmp(command, "remove") == 0) {
		char *name, *host;

		/* ask a sensor to add a new member to a clique */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a host\n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}

		/* find the clique name */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a clique name\n");
			exit(1);
		}
		name = strdup(tmp);
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a sensor\n");
			exit(1);
		}
		host = strdup(tmp);
		if (name == NULL || host == NULL) {
			printf("out of memory\n");
			exit(2);
		}

		if (strcmp(command, "add") == 0) {
			/* ask the sensor to add a new member */
			if (!CliqueJoin(&cookie, name, host, tout)) {
				printf("failed to add sensor to clique\n");
				exit(2);
			}
		} else {
			/* ask the sensor to add a new member */
			if (!CliqueRemove(&cookie, name, host, tout)) {
				printf("failed to remove sensor from clique\n");
				exit(2);
			}
		}
		free(name);
		free(host);
		DisconnectHost(&cookie);
	} else if (strcmp(command, "start") == 0) {
		/* let's start a new activity */
		/*HostSpec s;
		ObjectSet objs; */
		char *name, 
		     /* *attr, */
		     *control;

		printf("not yet implemented\n");
		exit(2);

		/* let's get the host */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a host \n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}

		/* let's get the controlName */
		control = NwsAttributeValue_r(FindNwsAttribute(ptr, "controlName"));
		if (control == NULL) {
			printf("I need a controlName\n");
			exit(1);
		}

		/* let's get the activity name */
		name = NwsAttributeValue_r(FindNwsAttribute(ptr, "controlName"));
		if (name == NULL) {
			printf("I need an activity name\n");
			exit(1);
		}

		/* let's get the skillName */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a skill\n");
			exit(1);
		}

		/* let's build the list of options we understand: there
		 * are control and skill options */

		/* let's check if the activity is already registered */


	} else if (strcmp(command, "stop") == 0) {
		HostSpec s;
		ObjectSet objs;
		char filter[255 + 1];

		/* let's create a HostSpec for the sensor */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a host \n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}
		/* but first let's set the nameserver */
		s = *MakeHostSpec(info.nameServer, GetDefaultPort(NAME_SERVER_HOST));
		if (!UseNameServer(&s)) {
			printf("failed to set nameserver\n");
		}

		/* let's get the activity */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need an activity \n");
			exit(1);
		}

		/* let's see if we have such an activity registered */
		snprintf(filter, 256, "(name=%s)", tmp);
		objs = NewObjectSet();

		if (!GetObjectsTimeout(filter, &objs, tout) ||
				FirstObject(objs) == NULL) {
			printf("failed to find activity on the nameserver: trying to stop it anyway\n");
		}

		/* is host a nameserver? */
		if (info.hostType == NAME_SERVER_HOST) {
			Object obj;
			char *hostName;

			obj = FirstObject(objs);
			hostName = NwsAttributeValue_r(FindNwsAttribute(obj, "host"));
			if (hostName == NULL) {
				printf("failed to find sensor\n");
				exit (2);
			}
			s = *MakeHostSpec(hostName, GetDefaultPort(SENSOR_HOST));
			free(hostName);
		} else {
			s = *MakeHostSpec(cookie.name, cookie.port);
		}

		FreeObjectSet(&objs);

		/* let's stop it */
		if (!HaltActivity(&s, tmp)) {
			printf("failed to stop activty %s\n", tmp);
			exit(2);
		}
	} else if (!strcmp(command, "fetch") || !strcmp(command, "fetchClique")) {
		HostSpec proxy;
		char *hosts, *res;
		int howMany, len;

		/* first argument is the proxy for the fetch part */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			ABORT("I need a proxy\n");
		}
		proxy = *MakeHostSpec(tmp, GetDefaultPort(PROXY_HOST));

		res = hosts = NULL;
		if (!strcmp(command, "fetchClique")) {
			/* let's get the clique name */
			GETWORD(tmp, ptr, (const char **)&ptr);
		} else {
			/* then is the resource */
			if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
				ABORT("I need a resource\n");
			}
			res = strdup(tmp);
			if (res == NULL) {
				ABORT("out of memory\n");
			}
			if (IsResourceValid(res) == -1) {
				ABORT1("Unknown resource (%s)\n", res);
			}

			/* turns the hosts into the comma-separated list */
			hosts = NULL;
			len = 0;
			for (c = 0; GETWORD(tmp, ptr, (const char **)&ptr); c++) {
				hosts = (char *)realloc(hosts, len + strlen(tmp) + 2);
				if (hosts == NULL) {
					ABORT("out of memory\n");
				}
				if (len > 0) {
					hosts[len] = ',';
					len++;
				}
				memcpy(hosts + len, tmp, strlen(tmp) + 1);
				len += strlen(tmp);
				c++;
			}
			if (c == 0) {
				ABORT("Need to specify at least one target host!\n");
			}
		}

		howMany = 0;
		if (!strcmp(command, "fetch")) {
			if (!StartFetching(&proxy, res, NULL, hosts, tout)) {
				ERROR("failed to start fetching\n");
			}
		} else {
			/* we are fetching series for the whole clique */
			if (!StartCliqueFetching(&proxy, tmp, tout)) {
				ERROR("failed to start fetching for clique\n");
			}
		}
		FREE(hosts);
	} else if (!strcmp(command, "clean")) {
		unsigned long idle;
		char *s;

		/* we need a memory */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need a memory\n");
			exit(1);
		}
		if (!FillCookie(&cookie, tmp, &info, tout)) {
			exit(2);
		}

		/* now next is the idle */
		if (!GETWORD(tmp, ptr, (const char **)&ptr)) {
			printf("I need an idle time\n");
			exit(1);
		}
		idle = strtol(tmp, NULL, 10);

		/* everything else is a series: we need now to change
		 * '\n' into ',' */
		if (*ptr == '\n') {
			ptr++;
		}
		if (strlen(ptr) < 2) {
			printf("I need a series name\n");
			exit(1);
		}

		for (s = strchr(ptr, '\n'); s; s = strchr(s, '\n')) {
			*s = '\t';
		}

		if (MemoryClean(&cookie, ptr, idle)) {
			exit(1);
		}
	} else {
		/* unknown command */
		usage();
		exit(1);
	}
	free(commandLine);

	return(0);
}
