// -*- indent-tabs-mode: nil -*-

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cstdlib>
// NOTE: On Solaris errno is not working properly if cerrno is included first
#include <cerrno>

#include <dirent.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <glibmm.h>

#include <gfal_api.h>

#include <arc/Thread.h>
#include <arc/Logger.h>
#include <arc/URL.h>
#include <arc/StringConv.h>
#include <arc/data/DataBuffer.h>
#include <arc/data/DataCallback.h>
#include <arc/CheckSum.h>
#include <arc/FileUtils.h>
#include <arc/FileAccess.h>
#include <arc/Utils.h>

#include "DataPointGFAL.h"

namespace Arc {

  /// Class for locking environment while calling gfal functions.
  class GFALEnvLocker: public CertEnvLocker {
  public:
    static Logger logger;
    GFALEnvLocker(const UserConfig& usercfg, const std::string& lfc_host): CertEnvLocker(usercfg) {
      EnvLockUnwrap(false);
      // if root, we have to set X509_USER_CERT and X509_USER_KEY to
      // X509_USER_PROXY to force GFAL to use the proxy. If they are undefined
      // it uses the host cert and key.
      if (getuid() == 0 && !GetEnv("X509_USER_PROXY").empty()) {
        SetEnv("X509_USER_KEY", GetEnv("X509_USER_PROXY"), true);
        SetEnv("X509_USER_CERT", GetEnv("X509_USER_PROXY"), true);
      }
      logger.msg(DEBUG, "Using proxy %s", GetEnv("X509_USER_PROXY"));
      logger.msg(DEBUG, "Using key %s", GetEnv("X509_USER_KEY"));
      logger.msg(DEBUG, "Using cert %s", GetEnv("X509_USER_CERT"));

      if (!lfc_host.empty()) {
        // set LFC retry env variables (don't overwrite if set already)
        // connection timeout
        SetEnv("LFC_CONNTIMEOUT", "30", false);
        // number of retries
        SetEnv("LFC_CONRETRY", "1", false);
        // interval between retries
        SetEnv("LFC_CONRETRYINT", "10", false);

        // set host name env var
        SetEnv("LFC_HOST", lfc_host);
      }

      EnvLockWrap(false);
    }
  };

  Logger GFALEnvLocker::logger(Logger::getRootLogger(), "GFALEnvLocker");

  Logger DataPointGFAL::logger(Logger::getRootLogger(), "DataPoint.GFAL");

  DataPointGFAL::DataPointGFAL(const URL& u, const UserConfig& usercfg, PluginArgument* parg)
    : DataPointDirect(u, usercfg, parg), fd(-1), reading(false), writing(false) {
      LogLevel loglevel = logger.getThreshold();
      if (loglevel == DEBUG)
        gfal_set_verbose (GFAL_VERBOSE_VERBOSE | GFAL_VERBOSE_DEBUG | GFAL_VERBOSE_TRACE);
      if (loglevel == VERBOSE)
        gfal_set_verbose (GFAL_VERBOSE_VERBOSE);
      // lfc:// needs to be converted to lfn:/path or guid:abcd...
      if (url.Protocol() == "lfc") {
        lfc_host = url.Host();
        url.ChangeHost("");
        url.ChangePort(-1);
        if (url.MetaDataOption("guid").empty()) {
          url.ChangeProtocol("lfn");
        }
        else {
          url.ChangeProtocol("guid");
          // To fix: this call forces leading / on path
          url.ChangePath(url.MetaDataOption("guid"));
        }
      }
  }

  DataPointGFAL::~DataPointGFAL() {
    StopReading();
    StopWriting();
  }

  Plugin* DataPointGFAL::Instance(PluginArgument *arg) {
    DataPointPluginArgument *dmcarg = dynamic_cast<DataPointPluginArgument*>(arg);
    if (!dmcarg)
      return NULL;
    if (((const URL &)(*dmcarg)).Protocol() != "rfio" &&
        ((const URL &)(*dmcarg)).Protocol() != "dcap" &&
        ((const URL &)(*dmcarg)).Protocol() != "gsidcap" &&
        ((const URL &)(*dmcarg)).Protocol() != "root" &&
        ((const URL &)(*dmcarg)).Protocol() != "gsiftp" &&
        ((const URL &)(*dmcarg)).Protocol() != "srm" &&
        ((const URL &)(*dmcarg)).Protocol() != "lfc")
      return NULL;
    return new DataPointGFAL(*dmcarg, *dmcarg, dmcarg);
  }

  DataStatus DataPointGFAL::StartReading(DataBuffer& buf) {
    if (reading) return DataStatus::IsReadingError;
    if (writing) return DataStatus::IsWritingError;
    reading = true;
    
    // Open the file
    {
      GFALEnvLocker gfal_lock(usercfg, lfc_host);
      fd = gfal_open(url.plainstr().c_str(), O_RDONLY, 0);
    }
    if (fd < 0) {
      logger.msg(ERROR, "gfal_open failed: %s", StrError(errno));
      log_gfal_err();
      reading = false;
      return DataStatus::ReadStartError;
    }
    
    // Remember the DataBuffer we got: the separate reading thread will use it
    buffer = &buf;
    // StopReading will wait for this condition,
    // which will be signalled by the separate reading thread
    // Create the separate reading thread
    if (!CreateThreadFunction(&DataPointGFAL::read_file_start, this, &transfer_condition)) {
      logger.msg(ERROR, "Failed to create reading thread");
      if (fd != -1) {
        if (gfal_close(fd) < 0) {
          logger.msg(WARNING, "gfal_close failed: %s", StrError(errno));
        }
      }
      reading = false;
      return DataStatus::ReadStartError;
    }
    return DataStatus::Success;
  }
  
  void DataPointGFAL::read_file_start(void *object) {
    ((DataPointGFAL*)object)->read_file();
  }
  
  void DataPointGFAL::read_file() {
    int handle;
    unsigned int length;
    unsigned long long int offset = 0;
    unsigned int bytes_read;
    for (;;) {
      // Ask the DataBuffer for a buffer to read into
      if (!buffer->for_read(handle, length, true)) {
        buffer->error_read(true);
        break;
      }

      // Read into the buffer
      bytes_read = gfal_read(fd, (*(buffer))[handle], length);
            
      // If there was an error
      if (bytes_read < 0) {
        logger.msg(ERROR, "gfal_read failed: %s", StrError(errno));
        log_gfal_err();
        buffer->error_read(true);
        break;
      }
      
      // If there was no more to read
      if (bytes_read == 0) {
        buffer->is_read(handle, 0, offset);
        break;
      }
      
      // Tell the DataBuffer that we read something into it
      buffer->is_read(handle, bytes_read, offset);
      // Keep track of where we are in the file
      offset += bytes_read;
    }
    // We got out of the loop, which means we read the whole file
    // or there was an error, either case the reading is finished
    buffer->eof_read(true);
    // Close the file
    if (fd != -1) {
      if (gfal_close(fd) < 0) {
        logger.msg(WARNING, "gfal_close failed: %s", StrError(errno));
      }
      fd = -1;
    }
  }
  
  DataStatus DataPointGFAL::StopReading() {
    if (!reading) return DataStatus::ReadStopError;
    reading = false;
    if (!buffer) return DataStatus::ReadStopError;
    // If the reading is not finished yet trigger reading error
    if (!buffer->eof_read()) buffer->error_read(true);

    // Wait for the reading thread to finish
    logger.msg(DEBUG, "StopReading starts waiting for transfer_condition.");
    transfer_condition.wait();
    logger.msg(DEBUG, "StopReading finished waiting for transfer_condition.");

    // Close the file if not already done
    if (fd != -1) {
      if (gfal_close(fd) < 0) {
        logger.msg(WARNING, "gfal_close failed: %s", StrError(errno));
      }
      fd = -1;
    }
    // If there was an error (maybe we triggered it)
    if (buffer->error_read()) {
      buffer = NULL;
      return DataStatus::ReadError;
    }
    // If there was no error (the reading already finished)
    buffer = NULL;
    return DataStatus::Success;
  }
  
  DataStatus DataPointGFAL::StartWriting(DataBuffer& buf, DataCallback *space_cb) {
    if (reading) return DataStatus::IsReadingError;
    if (writing) return DataStatus::IsWritingError;
    writing = true;

    {
      GFALEnvLocker gfal_lock(usercfg, lfc_host);
      // Open the file
      fd = gfal_open(url.plainstr().c_str(), O_WRONLY | O_CREAT, 0600);
    }
    if (fd < 0) {
      // If no entry try to create parent directories
      if (errno == ENOENT) {
        URL parent_url = URL(url.plainstr());
        // For SRM the path can be given as SFN HTTP option
        if ((url.Protocol() == "srm" && !url.HTTPOption("SFN").empty())) {
          parent_url.AddHTTPOption("SFN", Glib::path_get_dirname(url.HTTPOption("SFN")), true);
        } else {
          parent_url.ChangePath(Glib::path_get_dirname(url.Path()));
        }

        {
          GFALEnvLocker gfal_lock(usercfg, lfc_host);
          // gfal_mkdir is always recursive
          if (gfal_mkdir(parent_url.plainstr().c_str(), 0700) == 0) {
            fd = gfal_open(url.plainstr().c_str(), O_WRONLY | O_CREAT, 0600);
          } else {
            logger.msg(ERROR, "Failed to create parent directories: ", StrError(errno));
          }
        }
      }
      if (fd < 0) {
        logger.msg(ERROR, "gfal_open failed: %s", StrError(errno));
        log_gfal_err();
        writing = false;
        return DataStatus::WriteStartError;
      }
    }
    
    // Remember the DataBuffer we got, the separate writing thread will use it
    buffer = &buf;
    // StopWriting will wait for this condition,
    // which will be signalled by the separate writing thread
    // Create the separate writing thread
    if (!CreateThreadFunction(&DataPointGFAL::write_file_start, this, &transfer_condition)) {
      logger.msg(ERROR, "Failed to create writing thread");
      if (fd != -1) {
        if (gfal_close(fd) < 0) {
          logger.msg(WARNING, "gfal_close failed: %s", StrError(errno));
        }
      }
      writing = false;
      return DataStatus::WriteStartError;
    }    
    return DataStatus::Success;
  }
  
  void DataPointGFAL::write_file_start(void *object) {
    ((DataPointGFAL*)object)->write_file();
  }
  
  void DataPointGFAL::write_file() {
    int handle;
    unsigned int length;
    unsigned long long int position;
    unsigned long long int offset = 0;
    unsigned int bytes_written;
    unsigned int chunk_offset;
    
    for (;;) {
      // Ask the DataBuffer for a buffer with data to write,
      // and the length and position where to write to
      if (!buffer->for_write(handle, length, position, true)) {
        // no more data from the buffer, did the other side finished?
        if (!buffer->eof_read()) {
          // the other side hasn't finished yet, must be an error
          buffer->error_write(true);
        }
        break;
      }      

      // if the buffer gives different position than we are currently in the
      // destination, then we have to seek there
      if (position != offset) {
        logger.msg(DEBUG, "DataPointGFAL::write_file got position %d and offset %d, has to seek", position, offset);
        gfal_lseek(fd, position, SEEK_SET);
        offset = position;
      }
      
      // we want to write the chunk we got from the buffer,
      // but we may not be able to write it in one shot
      chunk_offset = 0;
      while (chunk_offset < length) {
        bytes_written = gfal_write(fd, (*(buffer))[handle] + chunk_offset, length - chunk_offset);
        if (bytes_written < 0) break; // there was an error
        // calculate how far we got into to the chunk
        chunk_offset += bytes_written;
        // if the new chunk_offset is still less then the length of the chunk,
        // we have to continue writing
      }

      // we finished with writing (even if there was an error)
      buffer->is_written(handle);
      offset += length;

      // if there was an error during writing
      if (bytes_written < 0) {
        logger.msg(ERROR, "gfal_write failed: %s", StrError(errno));
        log_gfal_err();
        buffer->error_write(true);
        break;
      }
    }
    buffer->eof_write(true);
    // Close the file
    if (fd != -1) {
      if (gfal_close(fd) < 0) {
        logger.msg(WARNING, "gfal_close failed: %s", StrError(errno));
      }
      fd = -1;
    }
  }
    
  DataStatus DataPointGFAL::StopWriting() {
    if (!writing) return DataStatus::WriteStopError;
    writing = false;
    if (!buffer) return DataStatus::WriteStopError;
    
    // If the writing is not finished, trigger writing error
    if (!buffer->eof_write()) buffer->error_write(true);

    // Wait until the writing thread finishes
    logger.msg(DEBUG, "StopWriting starts waiting for transfer_condition.");
    transfer_condition.wait();
    logger.msg(DEBUG, "StopWriting finished waiting for transfer_condition.");

    // Close the file if not done already
    if (fd != -1) {
      if (gfal_close(fd) < 0) {
        logger.msg(WARNING, "gfal_close failed: %s", StrError(errno));
      }
      fd = -1;
    }
    // If there was an error (maybe we triggered it)
    if (buffer->error_write()) {
      buffer = NULL;
      return DataStatus::WriteError;
    }
    buffer = NULL;
    return DataStatus::Success;
  }  
  
  DataStatus DataPointGFAL::do_stat(const URL& stat_url, FileInfo& file) {
    struct stat st;
    int res;

    {
      GFALEnvLocker gfal_lock(usercfg, lfc_host);
      res = gfal_stat(stat_url.plainstr().c_str(), &st);
    }
    if (res < 0) {
      logger.msg(ERROR, "gfal_stat failed: %s", StrError(errno));
      log_gfal_err();
      return DataStatus::StatError;
    }

    if(S_ISREG(st.st_mode)) {
      file.SetType(FileInfo::file_type_file);
      file.SetMetaData("type", "file");
    } else if(S_ISDIR(st.st_mode)) {
      file.SetType(FileInfo::file_type_dir);
      file.SetMetaData("type", "dir");
    } else {
      file.SetType(FileInfo::file_type_unknown);
    }

    std::string path = stat_url.Path();
    // For SRM the path can be given as SFN HTTP Option
    if ((stat_url.Protocol() == "srm" && !stat_url.HTTPOption("SFN").empty())) path = stat_url.HTTPOption("SFN");

    std::string name = Glib::path_get_basename(path);
    file.SetMetaData("path", path);
    file.SetName(name);

    file.SetSize(st.st_size);
    file.SetMetaData("size", tostring(st.st_size));
    file.SetCreated(st.st_mtime);
    file.SetMetaData("mtime", (Time(st.st_mtime)).str());
    file.SetMetaData("atime", (Time(st.st_atime)).str());
    file.SetMetaData("ctime", (Time(st.st_ctime)).str());

    std::string perms;
    if (st.st_mode & S_IRUSR) perms += 'r'; else perms += '-';
    if (st.st_mode & S_IWUSR) perms += 'w'; else perms += '-';
    if (st.st_mode & S_IXUSR) perms += 'x'; else perms += '-';
    if (st.st_mode & S_IRGRP) perms += 'r'; else perms += '-';
    if (st.st_mode & S_IWGRP) perms += 'w'; else perms += '-';
    if (st.st_mode & S_IXGRP) perms += 'x'; else perms += '-';
    if (st.st_mode & S_IROTH) perms += 'r'; else perms += '-';
    if (st.st_mode & S_IWOTH) perms += 'w'; else perms += '-';
    if (st.st_mode & S_IXOTH) perms += 'x'; else perms += '-';
    file.SetMetaData("accessperm", perms);

    return DataStatus::Success;    
  }

  DataStatus DataPointGFAL::Check() {
    if (reading) return DataStatus::IsReadingError;
    if (writing) return DataStatus::IsWritingError;
    
    FileInfo file;
    DataStatus status_from_stat = do_stat(url, file);
    
    if (status_from_stat != DataStatus::Success) {
      return DataStatus::CheckError;
    }
    
    SetSize(file.GetSize());
    SetCreated(file.GetCreated());
    return DataStatus::Success;
  }
  
  DataStatus DataPointGFAL::Stat(FileInfo& file, DataPointInfoType verb) {
    return do_stat(url, file);
  }

  DataStatus DataPointGFAL::List(std::list<FileInfo>& files, DataPointInfoType verb) {
    // Open the directory
    struct dirent *d;
    DIR *dir;    
    {
      GFALEnvLocker gfal_lock(usercfg, lfc_host);
      dir = gfal_opendir(url.plainstr().c_str());
    }
    if (!dir) {
      logger.msg(ERROR, "gfal_opendir failed: %s", StrError(errno));
      log_gfal_err();
      return DataStatus::ListError;
    }
    
    // Loop over the content of the directory
    while ((d = gfal_readdir (dir))) {
      // Create a new FileInfo object and add it to the list of files
      std::list<FileInfo>::iterator f = files.insert(files.end(), FileInfo(d->d_name));
      // If information about times, type or access was also requested, do a stat
      if (verb & (INFO_TYPE_TIMES | INFO_TYPE_ACCESS | INFO_TYPE_TYPE)) {
        URL child_url = URL(url.plainstr() + '/' + d->d_name);
        logger.msg(DEBUG, "List will stat the URL %s", child_url.plainstr());
        do_stat(child_url, *f);
      }
    }
    
    // Then close the dir
    if (gfal_closedir (dir) < 0) {
      logger.msg(WARNING, "gfal_closedir failed: %s", StrError(errno));
      return DataStatus::ListError;
    }
    
    return DataStatus::Success;
  }
  
  DataStatus DataPointGFAL::Remove() {
    if (reading) return DataStatus::IsReadingError;
    if (writing) return DataStatus::IsReadingError;
    FileInfo file;
    DataStatus status_from_stat = do_stat(url, file);
    if (status_from_stat != DataStatus::Success)
      return DataStatus::DeleteError;

    int res;
    {
      GFALEnvLocker gfal_lock(usercfg, lfc_host);

      if (file.GetType() == FileInfo::file_type_dir) {
        res = gfal_rmdir(url.plainstr().c_str());
      } else {
        res = gfal_unlink(url.plainstr().c_str());
      }
    }
    if (res < 0) {
      if (file.GetType() == FileInfo::file_type_dir) logger.msg(ERROR, "gfal_rmdir failed: %s", StrError(errno));
      else logger.msg(ERROR, "gfal_unlink failed: %s", StrError(errno));
      log_gfal_err();
      return DataStatus::DeleteError;
    }
    return DataStatus::Success;
  }
  
  DataStatus DataPointGFAL::CreateDirectory(bool with_parents) {

    int res;
    {
      GFALEnvLocker gfal_lock(usercfg, lfc_host);
      // gfal_mkdir is always recursive
      res = gfal_mkdir(url.plainstr().c_str(), 0700);
    }
    if (res < 0) {
      logger.msg(ERROR, "gfal_mkdir failed: %s", StrError(errno));
      log_gfal_err();
      return DataStatus::CreateDirectoryError;
    }
    return DataStatus::Success;    
  }
  
  void DataPointGFAL::log_gfal_err() {
    char errbuf[2048];
    gfal_posix_strerror_r(errbuf, sizeof(errbuf));
    logger.msg(ERROR, errbuf);
    gfal_posix_clear_error();
  }

} // namespace Arc

Arc::PluginDescriptor PLUGINS_TABLE_NAME[] = {
  { "gfal2", "HED:DMC", "Grid File Access Library 2", 0, &Arc::DataPointGFAL::Instance },
  { NULL, NULL, NULL, 0, NULL }
};
